/*
 * © Copyright 2015 -  SourceClear Inc
 */

package com.srcclr.sdk.build;

import static org.apache.commons.lang3.StringUtils.isBlank;

import com.srcclr.sdk.CoordinateType;
import com.srcclr.sdk.Coords;
import com.srcclr.sdk.LibraryGraph;
import com.srcclr.sdk.LibraryJGraphT;
import org.gradle.api.Project;
import org.gradle.api.artifacts.Configuration;
import org.gradle.api.artifacts.ResolvedConfiguration;
import org.gradle.api.artifacts.ResolvedDependency;
import org.gradle.api.artifacts.UnknownConfigurationException;
import com.veracode.security.logging.SecureLogger;

import javax.annotation.Nonnull;
import javax.annotation.concurrent.Immutable;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;


/**
 * Utility class for building ComponentGraphs from Gradle projects.
 *
 * This class is immutable and maintains no internal state (other than the configurationName) so it may be used
 * repeatedly and concurrently.
 */
@Immutable
public class GradleComponentGraphBuilder {

  ///////////////////////////// Class Attributes \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  private static final SecureLogger LOGGER = SecureLogger.getLogger(GradleComponentGraphBuilder.class);

  ////////////////////////////// Class Methods \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  /*
   * Each Gradle Configuration (aka scope) can extend from other Configurations.  For
   * example, 'testCompile' extends from 'compile', such that to get an accurate depiction
   * of the 'testCompile' scope we need to ensure we include all 'compile' scopes too.  Gradle
   * is smart enough to take care of this with direct dependencies, but does not seem to automatically
   * do this for us with transitives.
   *
   * To combat this, let's build up our own internal Set of ancestor configurations that should be included
   * with any given scope.
   */
  private Set<String> buildAncestorConfigurations(Configuration configuration) {
    Set<String> parentConfigs = new HashSet<>();

    // `compile` and `runtime` Configurations were deprecated in gradle 7
    // They need to be added manually for configs that extends from them
    parentConfigs.add("runtime");
    parentConfigs.add("compile");

    for (Configuration parent : configuration.getExtendsFrom()) {
      parentConfigs.add(parent.getName());
      parentConfigs.addAll(buildAncestorConfigurations(parent));
    }

    return parentConfigs;
  }

  //////////////////////////////// Attributes \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  private final String configurationName;

  /////////////////////////////// Constructors \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  /**
   * Instantiate a new GradleComponentGraphBuilder with a specified configuration.  This is analogous to a Maven
   * profile.
   * @param configurationName The dependency configuration to use when building the tree.
   */
  public GradleComponentGraphBuilder(String configurationName) {
    this.configurationName = configurationName;
  }

  ////////////////////////////////// Methods \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  /**
   * Generate a Set of one or more LibraryGraph instances. This method will not traverse sub-projects, nor will it
   * look at any other projects except those passed with the projects parameter.
   * @param projects The projects to restrict graph generation to.
   * @param useJGraphT use the JGraphT implementation to build graphs.
   */
  public Set<LibraryGraph> generateGraphs(@Nonnull Collection<Project> projects, Boolean useJGraphT) {
    Set<LibraryGraph> graphs = new HashSet<>();

    for (Project p : projects) {
      try {
        if (useJGraphT) {
          graphs.add(buildGraphWithJGraphT(p));
        } else {
          graphs.add(buildGraph(p));
        }
      } catch (UnknownConfigurationException ex) {
        LOGGER.debug(String.format("No configuration named %s found for %s, skipping", configurationName, p.getName()), ex);
      }
    }

    return graphs;
  }

  /**
   * Generate a Set of one or more LibraryGraph instances.  This will iterate over the supplied Project and all sub
   * projects owned by it, returning a LibraryGraph for each.
   *
   * @param project The Gradle project instance to scan.
   * @param useJGraphT use the JGraphT implementation to build graphs.
   */
  public Set<LibraryGraph> generateGraphs(@Nonnull Project project, Boolean useJGraphT) {

    Set<LibraryGraph> graphs = new HashSet<>();

    for (Project p : project.getAllprojects()) {
      try {
        if (useJGraphT) {
          graphs.add(buildGraphWithJGraphT(p));
        } else {
          graphs.add(buildGraph(p));
        }
      } catch (UnknownConfigurationException ex) {
        LOGGER.debug(String.format("No configuration named %s found for %s, skipping", configurationName, p.getName()), ex);
      }
    }

    return graphs;
  }

  //------------------------ Implements:

  //------------------------ Overrides:

  //---------------------------- Abstract Methods -----------------------------

  //---------------------------- Utility Methods ------------------------------

  LibraryGraph buildGraph(Project project) {
    final String buildFileName = project.getBuildFile() != null ? project.getBuildFile().getName() : null;

    final LibraryGraph.Builder module = new LibraryGraph.Builder()
      .withCoords(new Coords.Builder()
        .withCoordinateType(CoordinateType.MAVEN)
        .withCoordinate1(project.getGroup().toString())
        .withCoordinate2(project.getName())
        .withVersion(project.getVersion().toString())
        .build())
      .withModuleName(project.getName())
      .withFilename(buildFileName);

    final Configuration config = project.getConfigurations().getByName(configurationName);

    if (config == null) {
      return module.build();
    }

    final Set<String> permittedScopes = buildAncestorConfigurations(config);
    permittedScopes.add(configurationName);

    final ResolvedConfiguration resolvedConfig = config.getResolvedConfiguration();

    for (ResolvedDependency d : resolvedConfig.getFirstLevelModuleDependencies()) {
      final LibraryGraph libraryGraph = buildGraph(d, buildFileName, Collections.unmodifiableCollection(permittedScopes), new HashSet<ResolvedDependency>());
      if (libraryGraph != null) {
        module.withDirect(libraryGraph);
      }
    }

    return module.build();
  }

  /**
   * Recursive!
   */
  private LibraryGraph buildGraph(ResolvedDependency d, String buildFileName, Collection<String> permittedScopes,
                                  Set<ResolvedDependency> seen) {

    if (seen.contains(d)) {
      return null;
    } else if (hasUnmatchableCoordinate(d)) {
      LOGGER.debug("Coordinate {}:{}:{} cannot be matched and was dropped",
        d.getModuleGroup(), d.getModuleName(), d.getModuleVersion());
      return null;
    }

    LibraryGraph.Builder builder = new LibraryGraph.Builder()
      .withCoords(new Coords.Builder().withCoordinateType(CoordinateType.MAVEN).withCoordinates(d.getModuleGroup(), d.getModuleName()).withVersion(d.getModuleVersion()).withScope(configurationName).build())
      .withFilename(buildFileName);

    for (ResolvedDependency dep : d.getChildren()) {
      if (permittedScopes.contains(dep.getConfiguration()) || "default".equals(dep.getConfiguration())) {
        seen.add(d);
        final LibraryGraph libraryGraph = buildGraph(dep, buildFileName, permittedScopes, seen);
        if (libraryGraph != null) {
          builder.withDirect(libraryGraph);
        }
        seen.remove(d);
      }
    }

    return builder.build();
  }

  LibraryGraph buildGraphWithJGraphT(Project project) {
    final String buildFileName = project.getBuildFile() != null ? project.getBuildFile().getName() : null;
    LibraryJGraphT graph = new LibraryJGraphT(new LibraryGraph.Builder()
      .withCoords(new Coords.Builder()
        .withCoordinateType(CoordinateType.MAVEN)
        .withCoordinate1(project.getGroup().toString())
        .withCoordinate2(project.getName())
        .withVersion(project.getVersion().toString())
        .build())
      .withModuleName(project.getName())
      .withFilename(buildFileName));

    final Configuration config = project.getConfigurations().getByName(configurationName);

    if (config == null) {
      return graph.toLibraryGraph();
    }

    final Set<String> permittedScopes = buildAncestorConfigurations(config);
    permittedScopes.add(configurationName);

    final ResolvedConfiguration resolvedConfig = config.getResolvedConfiguration();

    for (ResolvedDependency d : resolvedConfig.getFirstLevelModuleDependencies()) {
      buildGraphWithJGraphT(graph, graph.root(), d, buildFileName, Collections.unmodifiableCollection(permittedScopes));
    }

    return graph.toLibraryGraph();
  }

  /**
   * Recursive!
   */
  private void buildGraphWithJGraphT(LibraryJGraphT graph, LibraryGraph.Builder parent, ResolvedDependency d,
                                      String buildFileName, Collection<String> permittedScopes) {
    if (hasUnmatchableCoordinate(d)) {
      LOGGER.debug("Coordinate {}:{}:{} cannot be matched and was dropped",
        d.getModuleGroup(), d.getModuleName(), d.getModuleVersion());
      return;
    }

    LibraryGraph.Builder builder = new LibraryGraph.Builder()
      .withCoords(new Coords.Builder().withCoordinateType(CoordinateType.MAVEN)
        .withCoordinates(d.getModuleGroup(), d.getModuleName()).withVersion(d.getModuleVersion())
        .withScope(configurationName).build())
      .withFilename(buildFileName);

    boolean isAcyclic = graph.dependsOn(parent, builder);
    // If isAcyclic returns false, that means we do not want to add a relationship between parent and builder,
    // since that would cause the graph to be cyclic.
    if (!isAcyclic) {
      // If it is cyclic, we should not continue.
      return;
    }

    for (ResolvedDependency dep : d.getChildren()) {
      if (permittedScopes.contains(dep.getConfiguration()) || "default".equals(dep.getConfiguration())) {
        buildGraphWithJGraphT(graph, builder, dep, buildFileName, permittedScopes);
      }
    }

    return;
  }

  /**
   * Filters coordinates that we will never be able to match against the registry.
   * These typically appear when non-Maven repositories are used.
   *
   * @return true if the given dependency has a coordinate we cannot match
   */
  private static boolean hasUnmatchableCoordinate(ResolvedDependency d) {
    return isBlank(d.getModuleGroup()) || isBlank(d.getModuleName()) || isBlank(d.getModuleVersion());
  }

  //---------------------------- Property Methods -----------------------------

}
