/*
 * © Copyright 2015 -  SourceClear Inc
 */

package com.srcclr.sdk.build;

import com.srcclr.sdk.CoordinateType;
import com.srcclr.sdk.Coords;
import com.srcclr.sdk.LibraryGraph;
import com.srcclr.sdk.LibraryJGraphT;
import org.apache.maven.artifact.Artifact;
import org.apache.maven.artifact.resolver.filter.ArtifactFilter;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.shared.dependency.graph.DependencyNode;

import javax.annotation.Nullable;
import javax.annotation.concurrent.Immutable;
import java.util.*;

import static com.srcclr.sdk.Coords.UNDEFINED_VALUE;

/**
 * Utility class for building ComponentGraphs from Maven projects.
 *
 * This class is immutable and maintains no internal state (other than the configurationName) so it may be used
 * repeatedly and concurrently.
 */
@Immutable
public class MavenComponentGraphBuilder {

  ///////////////////////////// Class Attributes \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  ////////////////////////////// Class Methods \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  //////////////////////////////// Attributes \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  /////////////////////////////// Constructors \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  ////////////////////////////////// Methods \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

  /**
   * Builds a LibraryGraph object from a DependencyNode
   *
   * @param node              The root node for the project.
   * @param relativePathToPom The relative file path to the Pom file, from the
   *                          root of the project.
   *
   * @return A fully built (and immutable) dependency graph.
   * @throws MojoExecutionException Thrown when null node is found in the
   *                                dependency tree
   */
  public LibraryGraph buildGraph(final DependencyNode node, final String relativePathToPom)
      throws MojoExecutionException {
    return buildGraph(node, relativePathToPom, null);
  }

  /**
   * Builds a LibraryGraph object from a DependencyNode
   *
   * @param node              The root node for the project.
   * @param relativePathToPom The relative file path to the Pom file, from the
   *                          root of the project.
   * @param filter            The filter of the dependencies to be harvested. When
   *                          <code>null</code>, filtering will be disabled.
   *
   * @return A fully built (and immutable) dependency graph.
   * @throws MojoExecutionException Thrown when null node is found in the
   *                                dependency tree
   */
  public LibraryGraph buildGraph(final DependencyNode node,
                                 final String relativePathToPom,
                                 @Nullable final ArtifactFilter filter)
      throws MojoExecutionException {
    if (node == null) {
      throw new MojoExecutionException("Maven unexpectedly found a null dependency graph node");
    }

    return convertToLibraryGraphBuilder(node, relativePathToPom)
        .withDirects(buildGraphRecursively(node.getChildren(), relativePathToPom, filter, true))
        .build();
  }

  private Set<LibraryGraph> buildGraphRecursively(final Collection<DependencyNode> nodes,
                                                  final String relativePathToPom,
                                                  @Nullable final ArtifactFilter filter,
                                                  final boolean isDirect)
      throws MojoExecutionException {
    Set<LibraryGraph> result = new HashSet<>();

    if (nodes == null) {
      return result;
    }

    for (DependencyNode currentNode : nodes) {
      if (currentNode == null) {
        throw new MojoExecutionException("Maven unexpectedly found a null dependency graph node");
      }

      final Artifact nodeArtifact = currentNode.getArtifact();
      final List<DependencyNode> children = currentNode.getChildren();

      if (filter == null || filter.include(nodeArtifact)) {
        result.add(convertToLibraryGraphBuilder(currentNode, relativePathToPom)
            .withDirects(buildGraphRecursively(children, relativePathToPom, filter, false))
            .build());
        continue;
      }

      /*
       * At this point, we know that the current node has an unwanted scope. If it is
       * a direct dependency, we replace it with a node that cannot be matched, to
       * prevent the children becoming direct dependencies themselves, due to the
       * reporting differences between direct and transitive dependencies.
       */
      if (isDirect) {
        Set<LibraryGraph> childrenLibraryGraphs = buildGraphRecursively(children, relativePathToPom, filter, false);

        if (!childrenLibraryGraphs.isEmpty()) {
          result.add(createUndefinedGraphBuilder(currentNode, relativePathToPom)
              .withDirects(childrenLibraryGraphs)
              .build());
        }

        continue;
      }

      /*
       * At this point, we know that the current node has an unwanted scope. We skip
       * it, and we stitch its children directly to its parent.
       */
      result.addAll(buildGraphRecursively(children, relativePathToPom, filter, false));
    }

    return result;
  }

  /**
   * Builds a LibraryGraph object from a DependencyNode using JGraphT.
   *
   * @param node              The root node for the project.
   * @param relativePathToPom The relative file path to the Pom file, from the
   *                          root of the project.
   *
   * @return A fully built (and immutable) dependency graph.
   * @throws MojoExecutionException Thrown when null node is found in the
   *                                dependency tree
   */
  public LibraryGraph buildGraphWithJGraphT(final DependencyNode node,
                                            final String relativePathToPom)
      throws MojoExecutionException {
    return buildGraphWithJGraphT(node, relativePathToPom, null);
  }

  /**
   * Builds a LibraryGraph object from a DependencyNode using JGraphT.
   *
   * @param node              The root node for the project.
   * @param relativePathToPom The relative file path to the Pom file, from the
   *                          root of the project.
   * @param filter            The filter of the dependencies to be harvested. When
   *                          <code>null</code>, filtering will be disabled.
   *
   * @return A fully built (and immutable) dependency graph.
   * @throws MojoExecutionException Thrown when null node is found in the
   *                                dependency tree
   */
  public LibraryGraph buildGraphWithJGraphT(final DependencyNode node,
                                            final String relativePathToPom,
                                            @Nullable final ArtifactFilter filter)
      throws MojoExecutionException {
    if (node == null) {
      throw new MojoExecutionException("Maven unexpectedly found a null dependency graph node");
    }
    LibraryJGraphT graph = new LibraryJGraphT(convertToLibraryGraphBuilder(node, relativePathToPom));
    buildJGraphT(graph, graph.root(), node.getChildren(), relativePathToPom, filter, true);
    return graph.toLibraryGraph();
  }

  /**
   * Recursively builds the dependency graph with JGraphT.
   *
   * @param graph    The graph
   * @param parent   Parent of <code>nodes</code>
   * @param nodes    The sibling dependencies to iterate on
   * @param filename The file responsible for the dependency
   * @param filter   The filter of the dependencies to be harvested. When
   *                 <code>null</code>, filtering will be disabled.
   * @param isDirect Set to <code>true</code> if the current node is a direct
   *                 dependency. Set to <code>false</code> otherwise.
   *
   * @return The size of the created dependency subgraphs whose roots are in the
   *         set <code>nodes</code>.
   * @throws MojoExecutionException Thrown when null nodes is found in the
   *                                dependency tree
   */
  private long buildJGraphT(LibraryJGraphT graph,
                            LibraryGraph.Builder parent,
                            final Collection<DependencyNode> nodes,
                            final String filename,
                            @Nullable final ArtifactFilter filter,
                            boolean isDirect)
      throws MojoExecutionException {
    long graphSize = 0L;

    if (nodes == null) {
      return graphSize;
    }

    for (DependencyNode currentNode : nodes) {
      if (currentNode == null) {
        throw new MojoExecutionException("Maven unexpectedly found a null dependency graph node");
      }

      final Artifact nodeArtifact = currentNode.getArtifact();
      final List<DependencyNode> children = currentNode.getChildren();

      if (filter == null || filter.include(nodeArtifact)) {
        LibraryGraph.Builder builder = convertToLibraryGraphBuilder(currentNode, filename);

        graphSize += 1;

        /*
         * If the attempt to add the dependency gives false, it means that the the
         * relationship between parent and builder is not add because adding the
         * relationship will cause the graph to be cyclic. If it is cyclic, we should
         * not recurse further.
         */
        if (graph.dependsOn(parent, builder)) {
          graphSize += buildJGraphT(graph, builder, children, filename, filter, false);
        }

        continue;
      }

      /*
       * At this point, we know that the current node has an unwanted scope. If it is
       * a direct dependency, we replace it with a node that cannot be matched, to
       * prevent the children becoming direct dependencies themselves, due to the
       * reporting differences between direct and transitive dependencies.
       */
      if (isDirect) {
        LibraryGraph.Builder builder = createUndefinedGraphBuilder(currentNode, filename);

        long childrenGraphSize = buildJGraphT(graph, builder, children, filename, filter, false);

        if (childrenGraphSize > 0) {
          /*
           * We only add the current node, which is an undefined node, into the graph if
           * it has some children to be added.
           *
           * Usually, if the attempt to add the dependency gives false, it means that the
           * the relationship between parent and builder is not add because adding the
           * relationship will cause the graph to be cyclic. Here, we are sure that cycle
           * will not be formed, since we're still at the level of direct dependency, and
           * our new undefined node cannot be the same as the root. But just to be
           * paranoid, we use an assert here as a safeguard for unexpected behavior.
           */
          assert(graph.dependsOn(parent, builder));

          graphSize += (childrenGraphSize + 1);
        }

        continue;
      }

      /*
       * At this point, we know that the current node has an unwanted scope. We skip
       * it, and we stitch its children directly to its parent.
       */
      graphSize += buildJGraphT(graph, parent, children, filename, filter, false);
    }

    return graphSize;
  }

  /**
   * Creates a {@link LibraryGraph.Builder} object with undefined coordinates data
   * which cannot be matched, and whose scope is the same as that of
   * <code>node</code>.
   *
   * @param node     The {@link DependencyNode} object to retrieve the scope
   *                 information from.
   * @param filename The source filename of the dependency.
   *
   * @return A {@link LibraryGraph.Builder} object with undefined coordinates data
   *         which cannot be matched, and whose scope is the same as that of
   *         <code>node</code>.
   */
  private LibraryGraph.Builder createUndefinedGraphBuilder(final DependencyNode node, final String filename) {
    Artifact nodeArtifact = node.getArtifact();
    Coords coords = new Coords.Builder()
        .withCoordinateType(CoordinateType.MAVEN)
        .withCoordinates(UNDEFINED_VALUE, UNDEFINED_VALUE)
        .withVersion(UNDEFINED_VALUE)
        .withScope(nodeArtifact.getScope())
        .build();
    return new LibraryGraph.Builder()
        .withCoords(coords)
        .withFilename(filename);
  }

  /**
   * Creates a {@link LibraryGraph.Builder} object with coordinates data of
   * <code>node</code>.
   *
   * @param node     The {@link DependencyNode} object to retrieve the coordinates
   *                 from.
   * @param filename The source filename of the dependency.
   *
   * @return A {@link LibraryGraph.Builder} object with coordinates data of
   *         <code>node</code>.
   */
  private LibraryGraph.Builder convertToLibraryGraphBuilder(final DependencyNode node, final String filename) {
    Artifact nodeArtifact = node.getArtifact();
    Coords coords = new Coords.Builder()
        .withCoordinateType(CoordinateType.MAVEN)
        .withCoordinates(nodeArtifact.getGroupId(), nodeArtifact.getArtifactId())
        .withVersion(nodeArtifact.getVersion())
        .withScope(nodeArtifact.getScope())
        .build();
    return new LibraryGraph.Builder()
      .withCoords(coords)
      .withFilename(filename);
  }

  //------------------------ Implements:

  //------------------------ Overrides:

  //---------------------------- Abstract Methods -----------------------------

  //---------------------------- Utility Methods ------------------------------

  //---------------------------- Property Methods -----------------------------

}
