
package com.sta.mimages;

import java.util.Vector;

import java.awt.Font;
import java.awt.Color;
import java.awt.Image;
import java.awt.Graphics2D;
import java.awt.FontMetrics;

import java.awt.image.BufferedImage;

import com.sta.mlogger.MLogger;

/**
 * <p>Name: ImageBuffer</p>
 * <p>Description: Buffer fr Bilder.</p>
 * <p>Copyright: Copyright (c) 2012, 2014, 2016, 2017, 2021</p>
 * <p>Company: &gt;StA-Soft&lt;</p>
 * @author StA
 * @version 1.0
 */

public class ImageBuffer
{

  /**
   * Abstrakte Basisklasse fr Operationen.
   */

  public abstract class Operation
  {

    /**
     * Ausfhren der Operation.
     * @param ir Image-Renderer
     * @param x X-Koordinate
     * @param y Y-Koordinate
     */

    public abstract void draw(ImageRenderer ir, int x, int y);

  }

  /**
   * Operation: (Vordergrund-) Farbe setzen.
   */

  public class OpSetColor extends Operation
  {

    /**
     * Farbe.
     */

    private Color myColor;

    /**
     * Constructor mit Farbe.
     * @param color Farbe
     */

    public OpSetColor(Color color)
    {
      myColor = color;
    }

    @Override
    public void draw(ImageRenderer ir, int x, int y)
    {
      ir.setColor(myColor);
    }

  }

  /**
   * Operation: Hintergrundfarbe setzen.
   */

  public class OpSetBackground extends Operation
  {

    /**
     * Hintergrundfarbe.
     */

    private Color myColor;

    /**
     * Constructor mit Hintergrundfarbe.
     * @param color Hintergrundfarbe
     */

    public OpSetBackground(Color color)
    {
      myColor = color;
    }

    @Override
    public void draw(ImageRenderer ir, int x, int y)
    {
      ir.setBackground(myColor);
    }

  }

  /**
   * Operation: Linie zeichnen.
   */

  public class OpDrawLine extends Operation
  {

    /**
     * X1.
     */

    private int myX1;

    /**
     * Y1.
     */

    private int myY1;

    /**
     * X2.
     */

    private int myX2;

    /**
     * Y2.
     */

    private int myY2;

    /**
     * Constructor mit Linen-Koordinaten (Anfangs- und Endpunkt).
     * @param x1 X1
     * @param y1 Y1
     * @param x2 X2
     * @param y2 Y2
     */

    public OpDrawLine(int x1, int y1, int x2, int y2)
    {
      myX1 = x1;
      myY1 = y1;
      myX2 = x2;
      myY2 = y2;
    }

    @Override
    public void draw(ImageRenderer ir, int x, int y)
    {
      ir.drawLine(x + myX1, y + myY1, x + myX2, y + myY2);
    }

  }

  /**
   * Operation: Rechteck zeichnen (Breite/Hhe).
   */

  public class OpDrawRectWH extends Operation
  {

    /**
     * X.
     */

    private int myX;

    /**
     * Y.
     */

    private int myY;

    /**
     * W (Breite).
     */

    private int myW;

    /**
     * H (Hhe).
     */

    private int myH;

    /**
     * Constructor mit Basiskoordinaten und Breite/Hhe.
     * @param x X
     * @param y Y
     * @param w W (Breite)
     * @param h H (Hhe)
     */

    public OpDrawRectWH(int x, int y, int w, int h)
    {
      myX = x;
      myY = y;
      myW = w;
      myH = h;
    }

    @Override
    public void draw(ImageRenderer ir, int x, int y)
    {
      ir.drawRectWH(x + myX, y + myY, myW, myH);
    }

  }

  /**
   * Operation: Rechteck fllen (Breite/Hhe).
   */

  public class OpFillRect extends Operation
  {

    /**
     * X.
     */

    private int myX;

    /**
     * Y.
     */

    private int myY;

    /**
     * DX (Breite).
     */

    private int myDX;

    /**
     * DY (Hhe).
     */

    private int myDY;

    /**
     * Constructor mit Basiskoordinaten und Breite/Hhe.
     * @param x X
     * @param y Y
     * @param dx DX (Breite)
     * @param dy DY (Hhe)
     */

    public OpFillRect(int x, int y, int dx, int dy)
    {
      myX = x;
      myY = y;
      myDX = dx;
      myDY = dy;
    }

    @Override
    public void draw(ImageRenderer ir, int x, int y)
    {
      ir.fillRect(x + myX, y + myY, myDX, myDY);
    }

  }

  /**
   * Operation: Rechteck lschen (Breite/Hhe).
   */

  public class OpClearRect extends Operation
  {

    /**
     * X.
     */

    private int myX;

    /**
     * Y.
     */

    private int myY;

    /**
     * DX (Breite).
     */

    private int myDX;

    /**
     * DY (Hhe).
     */

    private int myDY;

    /**
     * Constructor mit Basiskoordinaten und Breite/Hhe.
     * @param x X
     * @param y Y
     * @param dx DX (Breite)
     * @param dy DY (Hhe)
     */

    public OpClearRect(int x, int y, int dx, int dy)
    {
      myX = x;
      myY = y;
      myDX = dx;
      myDY = dy;
    }

    @Override
    public void draw(ImageRenderer ir, int x, int y)
    {
      ir.clearRect(x + myX, y + myY, myDX, myDY);
    }

  }

  /**
   * Operation: Polygon zeichnen.
   */

  public class OpDrawPolygon extends Operation
  {

    /**
     * X-Koordinaten.
     */

    private int[] myXX;

    /**
     * Y-Koordinaten.
     */

    private int[] myYY;

    /**
     * Anzahl der Punkte.
     */

    private int myN;

    /**
     * Constructor mit Koordinaten und Anzahl der Punkte.
     * @param xx X-Koordinaten
     * @param yy Y-Koordinaten
     * @param n Anzahl der Punkte
     */

    public OpDrawPolygon(int[] xx, int[] yy, int n)
    {
      myXX = xx;
      myYY = yy;
      myN = n;
    }

    @Override
    public void draw(ImageRenderer ir, int x, int y)
    {
      int sizexx = myXX.length;
      int sizeyy = myYY.length;
      int[] xx = new int[sizexx];
      int[] yy = new int[sizeyy];
      for (int i = 0; i < sizexx; i++)
      {
        xx[i] = x + myXX[i];
      }
      for (int i = 0; i < sizeyy; i++)
      {
        yy[i] = y + myYY[i];
      }
      ir.drawPolygon(xx, yy, myN);
    }

  }

  /**
   * Operation: Polygon fllen.
   */

  public class OpFillPolygon extends Operation
  {

    /**
     * X-Koordinaten.
     */

    private int[] myXX;

    /**
     * Y-Koordinaten.
     */

    private int[] myYY;

    /**
     * Anzahl der Punkte.
     */

    private int myN;

    /**
     * Constructor mit Koordinaten und Anzahl der Punkte.
     * @param xx X-Koordinaten
     * @param yy Y-Koordinaten
     * @param n Anzahl der Punkte
     */

    public OpFillPolygon(int[] xx, int[] yy, int n)
    {
      myXX = xx;
      myYY = yy;
      myN = n;
    }

    @Override
    public void draw(ImageRenderer ir, int x, int y)
    {
      int sizexx = myXX.length;
      int sizeyy = myYY.length;
      int[] xx = new int[sizexx];
      int[] yy = new int[sizeyy];
      for (int i = 0; i < sizexx; i++)
      {
        xx[i] = x + myXX[i];
      }
      for (int i = 0; i < sizeyy; i++)
      {
        yy[i] = y + myYY[i];
      }
      ir.fillPolygon(xx, yy, myN);
    }

  }

  /**
   * Operation: Oval zeichnen.
   */

  public class OpDrawOval extends Operation
  {

    /**
     * X.
     */

    private int myX;

    /**
     * Y.
     */

    private int myY;

    /**
     * DX (Breite).
     */

    private int myDX;

    /**
     * DY (Hhe).
     */

    private int myDY;

    /**
     * Constructor mit Basiskoordinaten und Breite/Hhe.
     * @param x X
     * @param y Y
     * @param dx DX (Breite)
     * @param dy DY (Hhe)
     */

    public OpDrawOval(int x, int y, int dx, int dy)
    {
      myX = x;
      myY = y;
      myDX = dx;
      myDY = dy;
    }

    @Override
    public void draw(ImageRenderer ir, int x, int y)
    {
      ir.drawOval(x + myX, y + myY, myDX, myDY);
    }

  }

  /**
   * Operation: Oval fllen.
   */

  public class OpFillOval extends Operation
  {

    /**
     * X.
     */

    private int myX;

    /**
     * Y.
     */

    private int myY;

    /**
     * DX (Breite).
     */

    private int myDX;

    /**
     * DY (Hhe).
     */

    private int myDY;

    /**
     * Constructor mit Basiskoordinaten und Breite/Hhe.
     * @param x X
     * @param y Y
     * @param dx DX (Breite)
     * @param dy DY (Hhe)
     */

    public OpFillOval(int x, int y, int dx, int dy)
    {
      myX = x;
      myY = y;
      myDX = dx;
      myDY = dy;
    }

    @Override
    public void draw(ImageRenderer ir, int x, int y)
    {
      ir.fillOval(x + myX, y + myY, myDX, myDY);
    }

  }

  /**
   * Operation: Bild zeichnen.
   */

  public class OpDrawImage extends Operation
  {

    /**
     * X.
     */

    private int myX;

    /**
     * Y.
     */

    private int myY;

    /**
     * Bild.
     */

    private Image myImage;

    /**
     * Constructor mit Bild und Koordinaten.
     * @param img Bild
     * @param x X
     * @param y Y
     */

    public OpDrawImage(Image img, int x, int y)
    {
      myX = x;
      myY = y;
      myImage = img;
    }

    @Override
    public void draw(ImageRenderer ir, int x, int y)
    {
      ir.drawImage(myImage, x + myX, y + myY);
    }

  }

  /**
   * Operation: Bild zentriert zeichnen.
   */

  public class OpDrawImageCenter extends Operation
  {

    /**
     * X.
     */

    private int myX;

    /**
     * Y.
     */

    private int myY;

    /**
     * DX (Breite).
     */

    private int myDX;

    /**
     * DY (Hhe).
     */

    private int myDY;

    /**
     * Bild.
     */

    private Image myImage;

    /**
     * Constructor mit Bild, Basiskoordinaten (Mittelpunkt) und Breite/Hhe.
     * @param img Bild
     * @param x X
     * @param y Y
     * @param dx DX (Breite)
     * @param dy DY (Hhe)
     */

    public OpDrawImageCenter(Image img, int x, int y, int dx, int dy)
    {
      myX = x;
      myY = y;
      myDX = dx;
      myDY = dy;
      myImage = img;
    }

    @Override
    public void draw(ImageRenderer ir, int x, int y)
    {
      ir.drawImageCenter(myImage, x + myX, y + myY, myDX, myDY);
    }

  }

  /**
   * Operation: Bild gezoomt zeichnen (Breite/Hhe).
   */

  public class OpDrawImageZoom extends Operation
  {

    /**
     * X.
     */

    private int myX;

    /**
     * Y.
     */

    private int myY;

    /**
     * DX (Breite).
     */

    private int myDX;

    /**
     * DY (Hhe).
     */

    private int myDY;

    /**
     * Bild.
     */

    private Image myImage;

    /**
     * Constructor mit Bild, Basiskoordinaten (Mittelpunkt) und Breite/Hhe.
     * @param img Bild
     * @param x X
     * @param y Y
     * @param dx DX (Breite)
     * @param dy DY (Hhe)
     */

    public OpDrawImageZoom(Image img, int x, int y, int dx, int dy)
    {
      myX = x;
      myY = y;
      myDX = dx;
      myDY = dy;
      myImage = img;
    }

    @Override
    public void draw(ImageRenderer ir, int x, int y)
    {
      ir.drawImageZoom(myImage, x + myX, y + myY, myDX, myDY);
    }

  }

  /**
   * Operation: Image-Buffer-Inhalt zeichnen.
   */

  public class OpDrawImageBuffer extends Operation
  {

    /**
     * X.
     */

    private int myX;

    /**
     * Y.
     */

    private int myY;

    /**
     * Image-Buffer.
     */

    private ImageBuffer myImageBuffer;

    /**
     * Constructor mit Image-Buffer und Koordinaten.
     * @param ib Image-Buffer
     * @param x X
     * @param y Y
     */

    public OpDrawImageBuffer(ImageBuffer ib, int x, int y)
    {
      myX = x;
      myY = y;
      myImageBuffer = ib;
    }

    @Override
    public void draw(ImageRenderer ir, int x, int y)
    {
      myImageBuffer.renderImage(ir, x + myX, y + myY);
    }

  }

  /**
   * Operation: Bildinhalt kopieren.
   */

  public class OpCopyImage extends Operation
  {

    /**
     * X1.
     */

    private int myX1;

    /**
     * Y1.
     */

    private int myY1;

    /**
     * X2.
     */

    private int myX2;

    /**
     * Y2.
     */

    private int myY2;

    /**
     * X.
     */

    private int myX;

    /**
     * Y.
     */

    private int myY;

    /**
     * Constructor mit Anfangs- und Endkoordinaten des Bildausschnitts und Zielkoordinaten (relativ).
     * @param x1 X1 (Anfang des Bildausschnitts)
     * @param y1 Y1 (Anfang des Bildausschnitts)
     * @param x2 X2 (Ende des Bildausschnitts)
     * @param y2 Y2 (Ende des Bildausschnitts)
     * @param x X (Ziel, relativ)
     * @param y Y (Ziel, relativ)
     */

    public OpCopyImage(int x1, int y1, int x2, int y2, int x, int y)
    {
      myX1 = x1;
      myY1 = y1;
      myX2 = x2;
      myY2 = y2;
      myX = x;
      myY = y;
    }

    @Override
    public void draw(ImageRenderer ir, int x, int y)
    {
      // ARGB-Daten aus Bild ermitteln (x + myX1, y + myY1) - (x + myX2, y + myY2)
      BufferedImage img = ir.getImage();
      int w = myX2 - myX1;
      int h = myY2 - myY1;
      int[] ia = img.getRGB(x + myX1, y + myY1, w, h, null, 0, w);
      // ARGB-Daten in Bild eintragen (x + myX, y + myY)
      img.setRGB(x + myX, y + myY, w, h, ia, 0, w);
    }

  }

  /**
   * Operation: Text schreiben.
   */

  public class OpPrintAt extends Operation
  {

    /**
     * X.
     */

    private int myX;

    /**
     * Y.
     */

    private int myY;

    /**
     * Text.
     */

    private String myText;

    /**
     * Constructor mit Koordinaten und Text.
     * @param x X
     * @param y Y
     * @param text Text
     */

    public OpPrintAt(int x, int y, String text)
    {
      myX = x;
      myY = y;
      myText = text;
    }

    @Override
    public void draw(ImageRenderer ir, int x, int y)
    {
      ir.setCurX(x + myX);
      ir.setCurY(y + myY);
      try
      {
        ir.print(myText);
      }
      catch (Exception ex)
      {
        MLogger.err("", ex);
      }
    }

  }

  /**
   * Operation: Schatten zeichnen (Breite/Hhe).
   */

  public class OpDrawShadowWH extends Operation
  {

    /**
     * Farbe fr Schatten.
     */

    private int myShadowColor = 0x808080;

    /**
     * X.
     */

    private int myX;

    /**
     * Y.
     */

    private int myY;

    /**
     * W (Breite).
     */

    private int myW;

    /**
     * H (Hhe).
     */

    private int myH;

    /**
     * Constructor mit Basiskoordinaten und Breite/Hhe.
     * @param x X
     * @param y Y
     * @param w W (Breite)
     * @param h H (Hhe)
     */

    public OpDrawShadowWH(int x, int y, int w, int h)
    {
      myX = x;
      myY = y;
      myW = w;
      myH = h;
    }

    @Override
    public void draw(ImageRenderer ir, int x, int y)
    {
      BufferedImage img = ir.getImage();
      int[] rgbArray = img.getRGB(x + myX, y + myY, myW, myH, null, 0, myW);
      for (int i = 0; i < rgbArray.length; i++)
      {
        int rgb = rgbArray[i];
        if ((rgb & 0xffffff) == 0xffffff)
        {
          rgbArray[i] = rgb & 0xff000000 | myShadowColor;
        }
      }
      img.setRGB(x + myX, y + myY, myW, myH, rgbArray, 0, myW);
    }

  }

  //===========================================================================

  /**
   * Operationen in diesem Image-Buffer.
   */

  private Vector<Operation> myOperations = new Vector<>();

  /**
   * Breite.
   */

  private int mySizeX;

  /**
   * Hhe.
   */

  private int mySizeY;

  /**
   * Aktuelle Farbe. Wird beim Setzen der (Vordergrund-) Farbe zum spteren Ermitteln aufbewahrt.
   */

  private Color myColor = new Color(0, 0, 0, 255);

  //===========================================================================

  /**
   * Standard-Constructor.
   */

  public ImageBuffer()
  {
  }

  //===========================================================================

  /**
   * Neues Bild bzw. neue Bildgre festlegen.
   * @param sizex Breite
   * @param sizey Hhe
   */

  public void newImage(int sizex, int sizey)
  {
    mySizeX = sizex;
    mySizeY = sizey;
  }

  /**
   * Neues Bild mit Bildgre und Hintergrundfarbe.
   * @param sizex Breite
   * @param sizey Hhe
   * @param c Hintergrundfarbe
   */

  public void newImage(int sizex, int sizey, Color c)
  {
    mySizeX = sizex;
    mySizeY = sizey;
    setBackground(c);
    clearRect(0, 0, sizex, sizey);
  }

  /**
   * (Vordergrund-) Farbe setzen.
   * @param c (Vordergrund-) Farbe
   */

  public void setColor(Color c)
  {
    myOperations.add(new OpSetColor(c));
    myColor = c;
  }

  /**
   * (Vordergrund-) Farbe ermitteln. Funktioniert nur, falls die Farbe vorher gesetzt wurde.
   * @return (Vordergrund-) Farbe
   */

  public Color getColor()
  {
    return myColor;
  }

  /**
   * Hintergrundfarbe setzen.
   * @param c Hintergrundfarbe
   */

  public void setBackground(Color c)
  {
    myOperations.add(new OpSetBackground(c));
  }

  /**
   * Linie zeichnen.
   * @param x1 X1
   * @param y1 Y1
   * @param x2 X2
   * @param y2 Y2
   */

  public void drawLine(int x1, int y1, int x2, int y2)
  {
    myOperations.add(new OpDrawLine(x1, y1, x2, y2));
  }

  /**
   * Rechteck zeichnen.
   * @param x X
   * @param y Y
   * @param width Breite
   * @param height Hhe
   */

  public void drawRectWH(int x, int y, int width, int height)
  {
    myOperations.add(new OpDrawRectWH(x, y, width, height));
  }

  /**
   * Rechteck fllen.
   * @param x X
   * @param y Y
   * @param dx Breite
   * @param dy Hhe
   */

  public void fillRect(int x, int y, int dx, int dy)
  {
    myOperations.add(new OpFillRect(x, y, dx, dy));
  }

  /**
   * Rechteck lschen.
   * @param x X
   * @param y Y
   * @param dx Breite
   * @param dy Hhe
   */

  public void clearRect(int x, int y, int dx, int dy)
  {
    myOperations.add(new OpClearRect(x, y, dx, dy));
  }

  /**
   * Polygon zeichnen.
   * @param xx X-Koordinaten
   * @param yy Y-Koordinaten
   * @param n Anzahl der Punkte
   */

  public void drawPolygon(int[] xx, int[] yy, int n)
  {
    myOperations.add(new OpDrawPolygon(xx, yy, n));
  }

  /**
   * Polygon fllen.
   * @param xx X-Koordinaten
   * @param yy Y-Koordinaten
   * @param n Anzahl der Punkte
   */

  public void fillPolygon(int[] xx, int[] yy, int n)
  {
    myOperations.add(new OpFillPolygon(xx, yy, n));
  }

  /**
   * Oval zeichnen.
   * @param x Y
   * @param y Y
   * @param dx Breite
   * @param dy Hhe
   */

  public void drawOval(int x, int y, int dx, int dy)
  {
    myOperations.add(new OpDrawOval(x, y, dx, dy));
  }

  /**
   * Oval fllen.
   * @param x Y
   * @param y Y
   * @param dx Breite
   * @param dy Hhe
   */

  public void fillOval(int x, int y, int dx, int dy)
  {
    myOperations.add(new OpFillOval(x, y, dx, dy));
  }

  /**
   * Bild zeichnen.
   * @param img Bild
   * @param x X
   * @param y Y
   */

  public void drawImage(Image img, int x, int y)
  {
    myOperations.add(new OpDrawImage(img, x, y));
  }

  /**
   * Bild zentriert zeichnen.
   * @param img Bild
   * @param x X
   * @param y Y
   * @param dx Breite
   * @param dy Hhe
   */

  public void drawImageCenter(Image img, int x, int y, int dx, int dy)
  {
    myOperations.add(new OpDrawImageCenter(img, x, y, dx, dy));
  }

  /**
   * Bild gezoomt zeichnen.
   * @param img Bild
   * @param x X
   * @param y Y
   * @param dx Breite
   * @param dy Hhe
   */

  public void drawImageZoom(Image img, int x, int y, int dx, int dy)
  {
    myOperations.add(new OpDrawImageZoom(img, x, y, dx, dy));
  }

  /**
   * Image-Buffer zeichnen.
   * @param ib Image-Buffer
   * @param x X
   * @param y Y
   */

  public void drawImageBuffer(ImageBuffer ib, int x, int y)
  {
    myOperations.add(new OpDrawImageBuffer(ib, x, y));
  }

  /**
   * Bildbereich kopieren.
   * @param x1 X1
   * @param y1 Y1
   * @param x2 X2
   * @param y2 Y2
   * @param x X (Ziel)
   * @param y Y (Ziel)
   */

  public void copyImage(int x1, int y1, int x2, int y2, int x, int y)
  {
    myOperations.add(new OpCopyImage(x1, y1, x2, y2, x, y));
  }

  /**
   * Bildbereich kopieren.
   * @param x1 X1
   * @param y1 Y1
   * @param w Breite
   * @param h Hhe
   * @param x X (Ziel)
   * @param y Y (Ziel)
   */

  public void copyImageWH(int x1, int y1, int w, int h, int x, int y)
  {
    myOperations.add(new OpCopyImage(x1, y1, x1 + w, y1 + h, x, y));
  }

  /**
   * Text schreiben.
   * @param x X
   * @param y Y
   * @param text Text
   */

  public void printat(int x, int y, String text)
  {
    myOperations.add(new OpPrintAt(x, y, text));
  }

  /**
   * Schatten zeichnen.
   * @param x X
   * @param y Y
   * @param width Breite
   * @param height Hhe
   */

  public void drawShadowWH(int x, int y, int width, int height)
  {
    myOperations.add(new OpDrawShadowWH(x, y, width, height));
  }

  //---------------------------------------------------------------------------

  /**
   * Zeichensatz-/Schriftgre.
   */

  private int myFontSize = 12;

  /**
   * Font-Metrics-Objekt.
   */

  private FontMetrics myFM = null;

  /**
   * Zeichensatz-/Schriftgre ermitteln.
   * @return Zeichensatz-/Schriftgre
   */

  public int getFontSize()
  {
    return myFontSize;
  }

  // Bei Font-nderung msste man myFM = null setzen, damit es neu ermittelt wird

  /**
   * Textbreite ermitteln.
   * @param text Text
   * @return Textbreite
   */

  public int getWidth(String text)
  {
    if (myFM == null)
    {
      BufferedImage bi = new BufferedImage(10, 10, BufferedImage.TYPE_INT_ARGB);
      Graphics2D myG2D = (Graphics2D) bi.getGraphics();
      Font font = new Font("Arial", Font.BOLD, myFontSize);
      myG2D.setFont(font);
      myFM = myG2D.getFontMetrics();
    }
    return myFM.stringWidth(text);
  }

  /**
   * Zeilenhhe ermitteln.
   * @return Zeilenhhe
   */

  public int getHeight()
  {
    if (myFM == null)
    {
      BufferedImage bi = new BufferedImage(10, 10, BufferedImage.TYPE_INT_ARGB);
      Graphics2D myG2D = (Graphics2D) bi.getGraphics();
      Font font = new Font("Arial", Font.BOLD, myFontSize);
      myG2D.setFont(font);
      myFM = myG2D.getFontMetrics();
    }
    return myFM.getHeight();
  }

  //---------------------------------------------------------------------------

  /**
   * Inhalt dieses Image-Buffers in den Image-Renderer an bestimmte Position bertragen.
   * @param ir Image-Renderer
   * @param x X
   * @param y Y
   */

  protected void renderImage(ImageRenderer ir, int x, int y)
  {
    for (Operation op : myOperations)
    {
      op.draw(ir, x, y);
    }
  }

  //===========================================================================

  // * BufferedImage.TYPE_INT_RGB --> wichtig, damit die Darstellung im Panel wirklich "schnell" geht
  // * BufferedImage.TYPE_INT_ARGB --> Verwendung bei Zeichnen mit Transparenz, dann aber Darstellung (deutlich) langsam(er)

  /**
   * Bild erzeugen, Inhalt dieses Image-Buffers in den Image-Renderer bertragen und Bild zurckliefern.
   * @param imagetype Bildtyp
   * @return erzeugtes Bild
   */

  public BufferedImage getImage(int imagetype)
  {
    ImageRenderer ir = new ImageRenderer();
    ir.setAntialiasing(false);
    ir.setBorderBottom(0);
    ir.setBorderLeft(0);
    ir.setBorderRight(0);
    ir.setBorderTop(0);
    ir.setFontName("Arial");
    ir.setFontSize(12);
    ir.setFontStyleIndex(1);
    // ir.setImageTypeIndex(2);
    // ir.setImageTypeIndex(3);
    // ir.setImageType(BufferedImage.TYPE_INT_RGB);
    // ir.setImageType(BufferedImage.TYPE_INT_ARGB);
    ir.setImageType(imagetype);

    ir.resetImage();
    ir.setImageWidth(mySizeX);
    ir.setImageHeight(mySizeY);
    // ir.setBackground(new Color(255, 255, 255, 255));
    // ir.clearRect(0, 0, mySizeX, mySizeY);

    renderImage(ir, 0, 0);
    BufferedImage bi = ir.getImage();

    return bi;
  }

  /**
   * Bild erzeugen, Inhalt dieses Image-Buffers in den Image-Renderer bertragen und Bild zurckliefern.
   * @return erzeugtes Bild
   */

  public BufferedImage getImage()
  {
    // wichtig, damit die Darstellung im Panel wirklich "schnell" geht
    return getImage(BufferedImage.TYPE_INT_RGB);
  }

  /**
   * Main-Methode zu Test-Zwecken.
   * @param args Kommandozeilenargumente
   */

  public static void main(String[] args)
  {
    ImageBuffer ib1 = new ImageBuffer();
    ib1.newImage(200, 100, new Color(255, 255, 255, 255));
    ib1.drawLine(0, 0, 199, 0);
    ib1.drawLine(0, 0, 0, 99);
    ib1.drawLine(0, 99, 199, 99);
    ib1.drawLine(199, 0, 199, 99);
    ib1.drawLine(20, 10, 180, 80);

    ImageBuffer ib2 = new ImageBuffer();
    ib2.newImage(200, 100, new Color(255, 255, 255, 255));
    ib2.drawLine(0, 0, 199, 0);
    ib2.drawLine(0, 0, 0, 99);
    ib2.drawLine(0, 99, 199, 99);
    ib2.drawLine(199, 0, 199, 99);
    ib2.drawLine(180, 10, 20, 80);

    ImageBuffer ib3 = new ImageBuffer();
    ib3.newImage(200, 100, new Color(255, 0, 0, 255));
    ib3.drawLine(0, 0, 199, 0);
    ib3.drawLine(0, 0, 0, 99);
    ib3.drawLine(0, 99, 199, 99);
    ib3.drawLine(199, 0, 199, 99);
    ib3.drawLine(20, 10, 180, 80);
    ib3.drawLine(180, 10, 20, 80);

    ImageBuffer ib4 = new ImageBuffer();
    ib4.newImage(200, 100);
    ib4.setColor(new Color(0, 255, 0, 128));
    ib4.fillRect(0, 0, 200, 100);
    ib4.setColor(new Color(0, 0, 0, 255));
    ib4.drawLine(0, 0, 199, 0);
    ib4.drawLine(0, 0, 0, 99);
    ib4.drawLine(0, 99, 199, 99);
    ib4.drawLine(199, 0, 199, 99);
    ib4.drawLine(20, 10, 180, 80);
    ib4.drawLine(180, 10, 20, 80);

    ImageBuffer ib = new ImageBuffer();
    ib.newImage(2000, 1000, new Color(255, 255, 255, 255));
    ib.drawLine(0, 0, 1999, 0);
    ib.drawLine(0, 0, 0, 999);
    ib.drawLine(0, 999, 1999, 999);
    ib.drawLine(1999, 0, 1999, 999);

    ib.drawImageBuffer(ib1, 100, 100);
    ib.drawImageBuffer(ib2, 400, 100);
    ib.drawImageBuffer(ib3, 250, 300);
    ib.drawImageBuffer(ib4, 350, 350);

    BufferedImage bi = ib.getImage();
    try
    {
      ImageRenderer.saveImage("test.png", bi);
    }
    catch (Exception ex)
    {
      MLogger.err("", ex);
    }
  }

}
