
package com.sta.mimages;

import java.io.File;

import java.util.Vector;

import java.awt.Font;
import java.awt.Color;
import java.awt.Image;
import java.awt.Graphics2D;
import java.awt.FontMetrics;
import java.awt.RenderingHints;

import java.awt.image.BufferedImage;

import javax.imageio.ImageIO;

import com.sta.mlogger.MLogger;

/**
 * <p>Name: ImageRenderer</p>
 * <p>Description: Renderer zur Erzeugung von Bildern, allgemeiner Ansatz,
 * jedoch speziell fr TIF-Bilder gedacht.
 * </p>
 * <p>Copyright: Copyright (c) 2004-2008, 2010-2012, 2014, 2016-2020</p>
 * <p>Company: &gt;StA-Soft&lt;</p>
 * @author StA
 * @version 1.2
 */

public class ImageRenderer implements Cloneable
{

  /**
   * Standard-Bildbreite.
   */

  private static final int STDIMAGEWIDTH = 2480;

  /**
   * Standard-Bildhhe.
   */

  private static final int STDIMAGEHEIGHT = 3508;

  /**
   * Standardwert fr linken Rand.
   */

  private static final int STDBORDERLEFT = STDIMAGEWIDTH / 10;

  /**
   * Standardwert fr rechten Rand.
   */

  private static final int STDBORDERRIGHT = STDIMAGEWIDTH / 10;

  /**
   * Standardwert fr oberen Rand.
   */

  private static final int STDBORDERTOP = STDIMAGEHEIGHT / 15;

  /**
   * Standardwert fr unteren Rand.
   */

  private static final int STDBORDERBOTTOM = STDIMAGEHEIGHT / 15;

  /**
   * Verwendeter Bildtyp.
   */

  // private static final int STDIMAGETYPE = BufferedImage.TYPE_4BYTE_ABGR;
  // private static final int STDIMAGETYPE = BufferedImage.TYPE_INT_RGB; // 1
  private static final int STDIMAGETYPE = BufferedImage.TYPE_BYTE_GRAY; // 10 ok.
  // private static final int STDIMAGETYPE = BufferedImage.TYPE_BYTE_BINARY; // 12

  /**
   * Standardzeichensatz.
   */

  private static final String STDFONTNAME = "Courier";

  /**
   * Standard-Style.
   */

  private static final int STDFONTSTYLE = Font.PLAIN;
  // private static final int FONTSTYLE = Font.BOLD;

  /**
   * Standard-Schriftgre.
   */

  private static final int STDFONTSIZE = STDIMAGEWIDTH / 70;

  /**
   * Standardvorgabe fr Antialiasing.
   */

  private static final boolean STDANTIALIASING = true; // false;

  /**
   * Muster fr Standard-Dateiname.
   */

  protected static final String STDFILENAME = "image_%d.png";

  /**
   * Image-Typen.
   */

  private static final int[] IMG_TYPES = { BufferedImage.TYPE_BYTE_BINARY,
                                           BufferedImage.TYPE_BYTE_GRAY,
                                           BufferedImage.TYPE_INT_RGB,
                                           BufferedImage.TYPE_INT_ARGB };

  /**
   * Font-Styles.
   */

  private static final int[] FNT_STYLES = { Font.PLAIN, Font.BOLD, Font.ITALIC, Font.BOLD + Font.ITALIC };

  /**
   * Standard-Auflsung in DPI.
   */

  private static final int STDRESOLUTION = 300; // DPI

  //===========================================================================

  /**
   * Aktuelles Bild.
   */

  protected BufferedImage myImage = null;

  /**
   * Aktuelle Bildbreite.
   */

  private int myImageWidth = STDIMAGEWIDTH;

  /**
   * Aktuelle Bildhhe.
   */

  private int myImageHeight = STDIMAGEHEIGHT;

  /**
   * Aktueller linker Rand.
   */

  private int myBorderLeft = STDBORDERLEFT;

  /**
   * Aktueller rechter Rand.
   */

  private int myBorderRight = STDBORDERRIGHT;

  /**
   * Aktueller oberer Rand.
   */

  private int myBorderTop = STDBORDERTOP;

  /**
   * Aktueller unterer Rand.
   */

  private int myBorderBottom = STDBORDERBOTTOM;

  /**
   * Aktueller Bildtyp.
   */

  private int myImageType = STDIMAGETYPE;

  /**
   * Aktueller Zeichensatzname.
   */

  private String myFontName = STDFONTNAME;

  /**
   * Aktueller Style.
   */

  private int myFontStyle = STDFONTSTYLE;

  /**
   * Aktuelle Zeichensatzgre.
   */

  private int myFontSize = STDFONTSIZE;

  /**
   * Angabe, ob aktuell Antialiasing verwendet werden soll (true) oder nicht
   * (false).
   */

  private boolean myAntialiasing = STDANTIALIASING;

  /**
   * Aktuelles Graphics-Object.
   */

  private Graphics2D myG2D = null;

  /**
   * Aktuelle Zeichensatzdaten.
   */

  private FontMetrics myFM = null;

  /**
   * Aktuelle Seitennummer.
   */

  private int myPage = 0;

  /**
   * Aktuelle X-Position.
   */

  private int myCurX;

  /**
   * Aktuelle Y-Position.
   */

  private int myCurY;

  /**
   * Aktueller Dateiname.
   */

  private String myFileName = STDFILENAME;

  /**
   * Vector mit den erzeugten Images.
   */

  private Vector myImages = new Vector();

  /**
   * Auflsung in DPI.
   */

  private int myResolution = STDRESOLUTION;

  //===========================================================================

  /**
   * Standard-Constructor.
   */

  protected ImageRenderer()
  {
  }

  /**
   * Constructor mit einem Dateinamen.
   * @param fn Dateiname
   */

  public ImageRenderer(String fn)
  {
    setFileName(fn);
  }

  //===========================================================================

  /**
   * Kopie des Objekts erstellen. Dabei wird jedes Feld kopiert.
   * @return Kopie des Objekts
   */

  public ImageRenderer cloneIR()
  {
    try
    {
      return (ImageRenderer) super.clone();
    }
    catch (CloneNotSupportedException e)
    {
      throw new RuntimeException(e);
    }
  }

  //---------------------------------------------------------------------------

  /**
   * Bildbreite vorgeben.
   * @param width Bildbreite
   */

  public void setImageWidth(int width)
  {
    myImageWidth = width;
  }

  /**
   * Bildbreite ermitteln.
   * @return Bildbreite
   */

  public int getImageWidth()
  {
    return myImageWidth;
  }

  /**
   * Bildhhe vorgeben.
   * @param height Bildhhe
   */

  public void setImageHeight(int height)
  {
    myImageHeight = height;
  }

  /**
   * Bildhhe ermittln.
   * @return Bildhhe
   */

  public int getImageHeight()
  {
    return myImageHeight;
  }

  /**
   * Gre des linken Randes vorgeben.
   * @param left Gre des linken Randes
   */

  public void setBorderLeft(int left)
  {
    myBorderLeft = left;
  }

  /**
   * Gre des linken Randes ermiteln.
   * @return Gre des linken Randes
   */

  public int getBorderLeft()
  {
    return myBorderLeft;
  }

  /**
   * Gre des rechten Randes vorgeben.
   * @param right Gre des rechten Randes
   */

  public void setBorderRight(int right)
  {
    myBorderRight = right;
  }

  /**
   * Gre des rechten Randes ermiteln.
   * @return Gre des rechten Randes
   */

  public int getBorderRight()
  {
    return myBorderRight;
  }

  /**
   * Gre des oberen Randes vorgeben.
   * @param top Gre des oberen Randes
   */

  public void setBorderTop(int top)
  {
    myBorderTop = top;
  }

  /**
   * Gre des oberen Randes ermiteln.
   * @return Gre des oberen Randes
   */

  public int getBorderTop()
  {
    return myBorderTop;
  }

  /**
   * Gre des unteren Randes vorgeben.
   * @param bottom Gre des unteren Randes
   */

  public void setBorderBottom(int bottom)
  {
    myBorderBottom = bottom;
  }

  /**
   * Gre des unteren Randes ermiteln.
   * @return Gre des unteren Randes
   */

  public int getBorderBottom()
  {
    return myBorderBottom;
  }

  /**
   * Bildtyp vorgeben.
   * @param imagetype Bildtyp
   */

  public void setImageType(int imagetype)
  {
    myImageType = imagetype;
  }

  /**
   * Bildtyp ermitteln.
   * @return Bildtyp
   */

  public int getImageType()
  {
    return myImageType;
  }

  /**
   * Bildtyp per Index setzen.
   * @param i Index
   */

  public void setImageTypeIndex(int i)
  {
    int it = BufferedImage.TYPE_BYTE_GRAY;
    if ((i >= 0) && (i < IMG_TYPES.length))
    {
      it = IMG_TYPES[i];
    }
    setImageType(it);
  }

  /**
   * Index des Bildtyps ermitteln.
   * @return Index
   */

  public int getImageTypeIndex()
  {
    int it = getImageType();
    for (int i = 0; i < IMG_TYPES.length; i++)
    {
      if (it == IMG_TYPES[i])
      {
        return i;
      }
    }
    return -1;
  }

  /**
   * Zeichsatzname vorgeben.
   * @param fontname Zeichsatzname
   */

  public void setFontName(String fontname)
  {
    myFontName = fontname;
  }

  /**
   * Zeichsatzname ermitteln.
   * @return Zeichsatzname
   */

  public String getFontName()
  {
    return myFontName;
  }

  /**
   * Style vorgeben.
   * @param fontstyle Style
   */

  public void setFontStyle(int fontstyle)
  {
    myFontStyle = fontstyle;
  }

  /**
   * Style ermitteln.
   * @return Style
   */

  public int getFontStyle()
  {
    return myFontStyle;
  }

  /**
   * Style per Index vorgeben.
   * @param i Index
   */

  public void setFontStyleIndex(int i)
  {
    if ((i >= 0) && (i < FNT_STYLES.length))
    {
      setFontStyle(FNT_STYLES[i]);
    }
  }

  /**
   * Style-Index ermitteln.
   * @return Index
   */

  public int getFontStyleIndex()
  {
    int fs = getFontStyle();
    for (int i = 0; i < FNT_STYLES.length; i++)
    {
      if (fs == FNT_STYLES[i])
      {
        return i;
      }
    }
    return -1;
  }

  /**
   * Aktuelle Zeichsatzgre vorgeben.
   * @param fontsize aktuelle Zeichsatzgre
   */

  public void setFontSize(int fontsize)
  {
    myFontSize = fontsize;
  }

  /**
   * Aktuelle Zeichsatzgre ermitteln.
   * @return aktuelle Zeichsatzgre
   */

  public int getFontSize()
  {
    return myFontSize;
  }

  /**
   * Vorgeben, ob Antialiasing verwendet werden soll.
   * @param antialiasing true: ja, false: nein
   */

  public void setAntialiasing(boolean antialiasing)
  {
    myAntialiasing = antialiasing;
  }

  /**
   * Ermitteln, ob Antialiasing verwendet wird.
   * @return true: ja, false: nein
   */

  public boolean getAntialiasing()
  {
    return myAntialiasing;
  }

  /**
   * Aktuellen Dateinamen vorgeben.
   * @param fn aktueller Dateiname
   */

  public void setFileName(String fn)
  {
    if (fn == null)
    {
      throw new NullPointerException("Can't set filename to null. Need a filename to save images.");
    }
    myFileName = fn;
  }

  /**
   * Aktuellen Dateinamen ermitteln.
   * @return aktueller Dateiname
   */

  public String getFileName()
  {
    return myFileName;
  }

  /**
   * Aktuelle Seite vorgeben.
   * @param page aktuelle Seite
   */

  public void setPage(int page)
  {
    myPage = page;
  }

  /**
   * Aktuelle X-Position vorgeben.
   * @param x aktuelle X-Position
   */

  public void setCurX(int x)
  {
    myCurX = x;
  }

  /**
   * Aktuelle X-Position ermitteln.
   * @return aktuelle X-Position
   */

  public int getCurX()
  {
    return myCurX;
  }

  /**
   * Aktuelle Y-Position vorgeben.
   * @param y aktuelle Y-Position
   */

  public void setCurY(int y)
  {
    myCurY = y;
  }

  /**
   * Aktuelle Y-Position ermitteln.
   * @return aktuelle Y-Position
   */

  public int getCurY()
  {
    return myCurY;
  }

  /**
   * Aktuelle Seite ermittln.
   * @return Nummer der aktuellen Seite
   */

  public int getPage()
  {
    return myPage;
  }

/*
  public void setImage(BufferedImage img)
  {
    myImage = img;
  }
*/

  /**
   * Image ermitteln.
   * @return Image
   */

  public BufferedImage getImage()
  {
    return myImage;
  }

  /**
   * Alle Images ermitteln.
   * @return Vector mit allen Images
   */

  public Vector getImages()
  {
    return myImages;
  }

  /**
   * Auflsung ermitteln.
   * @return Auflsung in DPI
   */

  public int getResolution()
  {
    return myResolution;
  }

  /**
   * Auflsung setzen.
   * @param res Auflsung in DPI
   */

  public void setResolution(int res)
  {
    myResolution = res;
  }

  //===========================================================================

  /**
   * Zeichensatz einstellen.
   */

  public void refreshFont()
  {
    if (myG2D != null)
    {
      // Wichtig: Alle G2D-betreffenden Operationen mssen ausgefhrt werden
      // bevor der Font gesetzt und die FontMetrices ermittelt werden!!!
      if (myAntialiasing)
      {
        myG2D.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        myG2D.setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
      }
      myG2D.setFont(new Font(myFontName, myFontStyle, myFontSize));
      myFM = myG2D.getFontMetrics();
    }
  }

  /**
   * Image zurcksetzen.
   */

  public void resetImage()
  {
    myImage = null;
  }

  /**
   * Alle Images zurcksetzen.
   */

  public void resetImages()
  {
    myImages.clear();
    myPage = 0;
    resetImage();
  }

  /**
   * Prfen, ob ein Bild erzeugt wurde, falls nicht: eines anlegen sowie
   * Zeichensatz, Farben usw. korrekt einstellen.
   */

  private void checkImage()
  {
    if (myImage == null)
    {
      myImage = new BufferedImage(myImageWidth, myImageHeight, myImageType);
      myCurX = myBorderLeft;
      myCurY = myBorderTop;
      myG2D = (Graphics2D) myImage.getGraphics();
      myG2D.setBackground(Color.white);
      myG2D.setColor(Color.black);
      myG2D.clearRect(0, 0, myImageWidth, myImageHeight);
      refreshFont();
    }
  }

  /**
   * Image vorgeben.
   * @param img Image
   */

  public void setImage(BufferedImage img)
  {
    myImage = img;
    myImageWidth = img.getWidth();
    myImageHeight = img.getHeight();
    myCurX = myBorderLeft;
    myCurY = myBorderTop;
    myG2D = (Graphics2D) myImage.getGraphics();
    myG2D.setBackground(Color.white);
    myG2D.setColor(Color.black);
    // myG2D.clearRect(0, 0, myImageWidth, myImageHeight);
    refreshFont();
  }

//---

  /**
   * Bild mit bestimmter Auflsung speichern.
   * @param fn Dateiname
   * @param img das Bild
   * @param resolution Auflsung in DPI
   * @throws Exception im Fehlerfall
   */

  protected static void saveImage(String fn, BufferedImage img, int resolution) throws Exception
  {
    // Man knnte die Liste der verfgbaren Writer ermitteln und gegen die Extension prfen.
    // Falls Writer vorhanden -> ImageIO aufrufen
    // Problem: Spezial-Optionen bei TIFF-Bildern
    if (fn.endsWith(".png"))
    {
      ImageIO.write(img, "png", new File(fn));
    }
    else if (fn.endsWith(".jpg"))
    {
      // Sonderbehandlung fr JPEG: hier ist ARGB nicht erlaubt, es muss RGB sein, andernfalls haben die Bilder einen Rotstich
      if (img.getType() != BufferedImage.TYPE_INT_RGB)
      {
        BufferedImage img2save = new BufferedImage(img.getWidth(null), img.getHeight(null), BufferedImage.TYPE_INT_RGB);
        img2save.getGraphics().drawImage(img, 0, 0, null);
        img = img2save;
      }
      ImageIO.write(img, "jpg", new File(fn));
    }
    else if (fn.endsWith(".gif"))
    {
      ImageIO.write(img, "gif", new File(fn));
    }
    /*
    else if (fn.endsWith(".bmp"))
    {
      MLogger.inf("BMP");
      ImageIO.write(img, "bmp", new File(fn));
    }
    else if (fn.endsWith(".tif"))
    {
      MLogger.inf("TIF");
      ImageIO.write(img, "tif", new File(fn));
    }
    */
    else
    {
      /*
      try
      {
      */
        OldImageHandler.saveImage(fn, img, resolution);
      /*
      }
      catch (Exception ex)
      {
        ImageIO.write(img, "tiff", new File(fn));
      }
      */
    }
  }

  /**
   * Bild mit Standardauflsung speichern.
   * @param fn Dateiname
   * @param img das Bild
   * @throws Exception im Fehlerfall
   */

  public static void saveImage(String fn, BufferedImage img) throws Exception
  {
    saveImage(fn, img, STDRESOLUTION);
  }

  /**
   * Aktuelles Bild mit aktueller Auflsung speichern.
   * @param fn Dateiname
   * @throws Exception im Fehlerfall
   */

  protected void saveImage(String fn) throws Exception
  {
    saveImage(fn, myImage, myResolution);
  }

  /**
   * Aktuelles Bild speichern.
   * @throws Exception im Fehlerfall
   */

  public void saveImage() throws Exception
  {
    String fn = myFileName;
    int fni = fn.indexOf("%d");
    if (fni >= 0)
    {
      fn = fn.substring(0, fni) + ("" + (10000 + myPage)).substring(1) + (fni < fn.length() ? fn.substring(fni + 2) : "");
    }
    try
    {
      MLogger.deb("Saving picture(" + myPage + ")... ");
      saveImage(fn);
      MLogger.deb("Picture saved okay!");
    }
    catch (Exception e)
    {
      // an exception probably means out of disk space, etc
      MLogger.err("Picture(" + myPage + ") save failed: " + e.getMessage(), e);
      throw e;
    }
    myImages.add(fn);
    myPage++;
    myImage = null;
    myG2D = null;
    myFM = null;
  }

  /**
   * Ein ggf. begonnenes Bild speichern.
   * @throws Exception im Fehlerfall (insbesondere beim Speichern eines Bildes)
   */

  public void flush() throws Exception
  {
    if ((myImage != null) && ((myCurX != myBorderLeft) || (myCurY != myBorderTop)))
    {
      saveImage();
    }
  }

  /**
   * Zeilenumbruch ausgeben.
   * @throws Exception im Fehlerfall (insbesondere beim Speichern eines Bildes)
   */

  public void println() throws Exception
  {
    checkImage();
    myCurX = myBorderLeft;
    myCurY += myFM.getHeight();
    if (myCurY >= myImageHeight - myBorderBottom - myFM.getHeight())
    {
      saveImage();
    }
  }

  /**
   * Test ausgeben.
   * @param s Text
   * @throws Exception im Fehlerfall (insbesondere beim Speichern eines Bildes)
   */

  public void print(String s) throws Exception
  {
    checkImage();
    int space = myImageWidth - myBorderRight - myCurX;
    String ss = s;
    int w;
    while ((w = myFM.stringWidth(ss)) > space)
    {
      // schneide letztes Wort von ss incl. Leerzeichen ab
      int li = ss.lastIndexOf(' ');
      while ((li > 1) && (ss.charAt(li - 1) == ' '))
      {
        li--;
      }
      if (li > 0)
      {
        ss = ss.substring(0, li);
      }
      else
      {
        // bleibt nur ein leerer String, nimm nur so viele Zeichen, bis Zeilenende erreicht
        // ---
        // Problem: bei extrem langen Zeilen dauert das fortlaufende Abschneiden
        // eines Zeichens am Ende extrem lange. Man msste daher andersherum
        // vorgehen: vom String-Anfang so viele Zeichen nehmen, bis Zeilenende
        // berschritten, dann ein Zeichen abschneiden
        String ss1 = ss; // Original merken
        int cnt = 0;
        do
        {
          cnt++;
          ss = ss1.substring(0, cnt);
        }
        while (myFM.stringWidth(ss) <= space);
        ss = ss.substring(0, ss.length() - 1);
      }
    }
    // myG2D.drawString(ss, myCurX, myCurY + myFM.getHeight());
    myG2D.drawString(ss, myCurX, myCurY + myFM.getAscent());
    myCurX += w;
    // Wurde s komplett geschrieben?
    int lens = s.length();
    int lenss = ss.length();
    if (lenss != lens)
    {
      // nein: dann schreibe Rest ohne fhrende Spaces
      while (lenss < lens && s.charAt(lenss) == ' ')
      {
        lenss++;
      }
      if (lenss != lens)
      {
        ss = s.substring(lenss);
        println();
        print(ss);
      }
    }
  }

  /**
   * Text mit Zeilenumbruch ausgeben.
   * @param s Text
   * @throws Exception im Fehlerfall (insbesondere beim Speichern eines Bildes)
   */

  public void println(String s) throws Exception
  {
    print(s);
    println();
  }

  /**
   * Breite des Textes ermitteln.
   * @param s Text
   * @return Breite
   */

  public int getWidth(String s)
  {
    checkImage();
    return myFM.stringWidth(s);
  }

  /**
   * Hhe einer Zeile ermitteln.
   * @return Hhe
   */

  public int getHeight()
  {
    checkImage();
    return myFM.getHeight();
  }

  /**
   * Hhe eines Zeichens ermitteln.
   * @return Hhe
   */

  public int getAscent()
  {
    checkImage();
    return myFM.getAscent();
  }

  /**
   * Aktuelle Farbe ermitteln.
   * @return Aktuelle Farbe
   */

  public Color getColor()
  {
    checkImage();
    return myG2D.getColor();
  }

  /**
   * Aktuelle Farbe vorgeben.
   * @param c Aktuelle Farbe
   */

  public void setColor(Color c)
  {
    checkImage();
    myG2D.setColor(c);
  }

  /**
   * Aktuelle Hintergrundfarbe ermitteln.
   * @return Aktuelle Hintergrundfarbe
   */

  public Color getBackground()
  {
    checkImage();
    return myG2D.getBackground();
  }

  /**
   * Aktuelle Hintergrundfarbe vorgeben.
   * @param c Aktuelle Hintergrundfarbe
   */

  public void setBackground(Color c)
  {
    checkImage();
    myG2D.setBackground(c);
  }

  /**
   * Linie zeichnen.
   * @param x1 X1
   * @param y1 Y1
   * @param x2 X3
   * @param y2 Y3
   */

  public void drawLine(int x1, int y1, int x2, int y2)
  {
    checkImage();
    myG2D.drawLine(x1, y1, x2, y2);
  }

  /**
   * Rechteck zeichnen, incl. x/y, excl. x+width/y+height
   * (Aufruf mit 0/0, ImageSizeX/ImageSizeY ergibt also genau einen Rahmen um
   * das gesamte Bild).
   * @param x X-Start
   * @param y Y-Staer
   * @param width Breite
   * @param height Hhe
   */

  public void drawRectWH(int x, int y, int width, int height)
  {
    checkImage();
    myG2D.drawRect(x, y, width - 1, height - 1);
  }

  /**
   * Rechteck fllen.
   * @param x X
   * @param y Y
   * @param dx DX
   * @param dy DY
   */

  public void fillRect(int x, int y, int dx, int dy)
  {
    checkImage();
    myG2D.fillRect(x, y, dx, dy);
  }

  /**
   * Rechteck mit Hintergrundfarbe fllen.
   * @param x X
   * @param y Y
   * @param dx DX
   * @param dy DY
   */

  public void clearRect(int x, int y, int dx, int dy)
  {
    checkImage();
    myG2D.clearRect(x, y, dx, dy);
  }

  /**
   * Polygon zeichnen.
   * @param xx X-Koordinaten
   * @param yy Y-Koordinaten
   * @param n Anzahl der Punkte
   */

  public void drawPolygon(int[] xx, int[] yy, int n)
  {
    checkImage();
    myG2D.drawPolygon(xx, yy, n);
  }

  /**
   * Polygon fllen.
   * @param xx X-Koordinaten
   * @param yy Y-Koordinaten
   * @param n Anzahl der Punkte
   */

  public void fillPolygon(int[] xx, int[] yy, int n)
  {
    checkImage();
    myG2D.fillPolygon(xx, yy, n);
  }

  /**
   * Oval zeichnen.
   * @param x X
   * @param y Y
   * @param dx DX
   * @param dy DY
   */

  public void drawOval(int x, int y, int dx, int dy)
  {
    checkImage();
    myG2D.drawOval(x - dx, y - dy, 2 * dx - 1 + 1, 2 * dy - 1 + 1);
  }

  /**
   * Oval fllen.
   * @param x X
   * @param y Y
   * @param dx DX
   * @param dy DY
   */

  public void fillOval(int x, int y, int dx, int dy)
  {
    checkImage();
    myG2D.fillOval(x - dx, y - dy, 2 * dx + 1, 2 * dy + 1);
  }

  /**
   * Bild einfgen.
   * @param img das Bild
   * @param x X
   * @param y Y
   */

  public void drawImage(Image img, int x, int y)
  {
    checkImage();
    // Variante 1
    // myG2D.drawImage(img, x, y, null);
    // Variante 2
    if (img instanceof BufferedImage)
    {
      BufferedImage bi = (BufferedImage) img;
      int w = bi.getWidth();
      int h = bi.getHeight();
      int[] ia = bi.getRGB(0, 0, w, h, null, 0, w);
      myImage.setRGB(x, y, w, h, ia, 0, w);
    }
    else
    {
      myG2D.drawImage(img, x, y, null);
    }
  }

  /**
   * Bild zentriert zeichnen.
   * @param img das Bild
   * @param x X
   * @param y Y
   * @param dx DX
   * @param dy DY
   */

  public void drawImageCenter(Image img, int x, int y, int dx, int dy)
  {
    checkImage();
    int w = img.getWidth(null);
    int h = img.getHeight(null);
    x += (dx - w) / 2;
    y += (dy - h) / 2;
    myG2D.drawImage(img, x, y, null);
  }

  /**
   * Bild ggf. in der Gre angepasst zeichnen.
   * @param img das Bild
   * @param x X
   * @param y Y
   * @param dx DX
   * @param dy DY
   */

  public void drawImageZoom(Image img, int x, int y, int dx, int dy)
  {
    checkImage();
    myG2D.drawImage(img, x, y, x + dx, y + dy, 0, 0, img.getWidth(null), img.getHeight(null), null);
  }

  //===========================================================================

  /**
   * Test-Methode.
   * @param args Kommandozeilenparameter
   */

  public static void main(String[] args)
  {
    try
    {
      ImageRenderer ir = new ImageRenderer("test.tif");
      ir.setImageType(BufferedImage.TYPE_BYTE_BINARY);
      // ir.setTIFFCompressionGroup4();
      ir.setCurX(10);
      ir.setCurY(10);
      ir.println("Test");
      ir.flush();
      // Split
      // TIFSplitter.splitTIF_ImageIO("test.tif", "test.tif");
    }
    catch (Exception ex)
    {
      MLogger.err("", ex);
    }
  }

}
