
package com.sta.mimages;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.image.BufferedImage;

import javax.imageio.ImageIO;

import com.sta.mlogger.MLogger;

import com.sta.cts.UniTypeConv;

/**
 * <p>Name: ImageUtils</p>
 * <p>Description: Diese Klasse soll alle Methoden zur eigentlichen Behandlung von Bildern enthalten, u. a. laden/speichern,
 * zoomen, kombinieren usw. In allen anderen Klassen sowie in abhngigen Projekten kann und soll dann ausschlielich diese
 * Klasse bzw. eine Klasse, die wiederum diese Klasse verwendet, benutzt werden, um Bilder zu behandeln.
 * Die Idee geht auf die Klasse ImageHelper im WAR-Modul vom Generator3 zurck, in der bereits einige Methoden gesammelt wurden.
 * Statt die Quelltexte zu kopieren soll hier eine zentrale Stelle fr alle Verwendungen erstellt werden.
 * </p>
 * <p>Comment: Original-Kommentar: Hilfsklasse fr Bildverarbeitung.
 * Die umgesetzten Operationen entsprechen inhaltlich denen aus dem Image-Processor (MImgProc). Prinzipiell knnte man also
 * auch in MImages eine solche Hilfsklasse erstellen und in MImgProc und an den Verwendungsstellen hier im Projekt benutzen.
 * [ (c) 2020, 2021, 2023 ]
 * </p>
 * <p>Copyright: Copyright (c) 2023</p>
 * <p>Company: &gt;StA-Soft&lt;</p>
 * @author StA
 * @version 1.0
 */

public class ImageUtils
{

  /**
   * Bild aus Stream laden.
   * @param is Input-Stream
   * @return Bild
   * @throws IOException im Fehlerfall
   */

  public static BufferedImage loadImage(InputStream is) throws IOException
  {
    return ImageIO.read(is);
  }

  /**
   * Bild aus Byte-Array laden.
   * @param imagedata Byte-Array
   * @return Bild
   * @throws IOException im Fehlerfall
   */

  public static BufferedImage loadImage(byte[] imagedata) throws IOException
  {
    return loadImage(new ByteArrayInputStream(imagedata));
  }

  /**
   * Bild aus Datei laden.
   * @param f Datei
   * @return Bild
   * @throws IOException im Fehlerfall
   */

  public static BufferedImage loadImage(File f) throws IOException
  {
    return loadImage(new BufferedInputStream(new FileInputStream(f)));
  }

  /**
   * Bild aus Datei laden.
   * @param fn Dateiname
   * @return Bild
   * @throws IOException im Fehlerfall
   */

  public static BufferedImage loadImage(String fn) throws IOException
  {
    return loadImage(new File(fn));
  }

  //---------------------------------------------------------------------------

  /**
   * Bild in Stream speichern.
   * @param os Output-Stream
   * @param img Bild
   * @param format Format (png, jpg, gif, ...)
   * @throws IOException im Fehlerfall
   */

  public static void saveImage(OutputStream os, BufferedImage img, String format) throws IOException
  {
    if ("png".equals(format))
    {
      ImageIO.write(img, "png", os);
    }
    else if ("jpg".equals(format))
    {
      // Sonderbehandlung fr JPEG: hier ist ARGB nicht erlaubt, es muss RGB sein, andernfalls haben die Bilder einen Rotstich
      if (img.getType() != BufferedImage.TYPE_INT_RGB)
      {
        BufferedImage img2save = new BufferedImage(img.getWidth(null), img.getHeight(null), BufferedImage.TYPE_INT_RGB);
        img2save.getGraphics().drawImage(img, 0, 0, null);
        img = img2save;
      }
      ImageIO.write(img, "jpg", os);
    }
    else if ("gif".equals(format))
    {
      ImageIO.write(img, "gif", os);
    }
    /*
    else if ("bmp".equals(format))
    {
      MLogger.inf("BMP");
      ImageIO.write(img, "bmp", os);
    }
    else if ("tif".equals(format))
    {
      MLogger.inf("TIF");
      ImageIO.write(img, "tif", os);
    }
    */
    // else
    // {
      /*
      try
      {
      */
      // OldImageHandler.saveImage(fn, img, resolution); // Standard: resolution = 300 (DPI)
      /*
      }
      catch (Exception ex)
      {
        ImageIO.write(img, "tiff", new File(fn));
      }
      */
    // }
  }

  /**
   * Bild in Byte-Array speichern.
   * @param img Bild
   * @param format Format (png, jpg, gif, ...)
   * @return Byte-Array
   * @throws IOException im Fehlerfall
   */

  public static byte[] saveImage(BufferedImage img, String format) throws IOException
  {
    ByteArrayOutputStream baos = new ByteArrayOutputStream();
    saveImage(baos, img, format);
    return baos.toByteArray();
  }

  /**
   * Bild in Datei speichern.
   * @param f Datei
   * @param img Bild
   * @throws IOException im Fehlerfall
   */

  public static void saveImage(File f, BufferedImage img) throws IOException
  {
    File dir = f.getParentFile();
    if (dir != null)
    {
      dir.mkdirs();
    }
    String format = null;
    String name = f.getName();
    int i = name.lastIndexOf(".");
    if (i >= 0)
    {
      format = name.substring(i + 1).toLowerCase();
    }
    saveImage(new BufferedOutputStream(new FileOutputStream(f)), img, format);
  }

  /**
   * Bild in Datei speichern.
   * @param fn Dateiname
   * @param img Bild
   * @throws IOException im Fehlerfall
   */

  public static void saveImage(String fn, BufferedImage img) throws IOException
  {
    saveImage(new File(fn), img);
  }

  //---------------------------------------------------------------------------

  /**
   * Neues Bild erzeugen (ARGB, also 32 Bit incl. Alpha-Channel).
   * @param x Breite
   * @param y Hhe
   * @param cc (Hintergrund-) Farbe (optional, kann null sein)
   * @return das erzeugte Bild
   */

  public static BufferedImage newImage(int x, int y, Long cc)
  {
    BufferedImage img = new BufferedImage(x, y, BufferedImage.TYPE_INT_ARGB);
    if (cc != null)
    {
      Graphics g = img.getGraphics();
      g.setColor(new Color(cc.intValue(), true));
      g.fillRect(0, 0, x, y);
    }
    return img;
  }

  /**
   * Bild auf die angegebene Breite und Hhe zoomen.
   * @param img Bild
   * @param x neue Breite
   * @param y neue Hhe
   * @return Ergebnisbild
   */

  public static BufferedImage zoomImage(BufferedImage img, int x, int y)
  {
    Image scaledimg = img.getScaledInstance(x, y, Image.SCALE_SMOOTH);
    if (!(scaledimg instanceof BufferedImage))
    {
      img = new BufferedImage(scaledimg.getWidth(null), scaledimg.getHeight(null), BufferedImage.TYPE_INT_ARGB);
      img.getGraphics().drawImage(scaledimg, 0, 0, null);
    }
    else
    {
      img = (BufferedImage) scaledimg;
    }
    return img;
  }

  /**
   * Bild auf die angegebene Breite und Hhe zoomen. Entweder Breite oder Hhe knnen "*" sein, was bedeutet, dass das Bild
   * ohne Verzerrung (ohne Vernderung des Seitenverhltnisses) gezoomt werden soll.
   * @param img Bild
   * @param sx neue Breite
   * @param sy neue Hhe
   * @return Ergebnisbild
   */

  public static BufferedImage zoomImage(BufferedImage img, String sx, String sy)
  {
    Integer xx = null;
    Integer yy = null;
    if (!"*".equals(sx) && !"*".equals(sy))
    {
      xx = UniTypeConv.convString2Int(sx);
      yy = UniTypeConv.convString2Int(sy);
    }
    else if (!"*".equals(sx) && "*".equals(sy))
    {
      xx = UniTypeConv.convString2Int(sx);
      yy = img.getHeight() * xx / img.getWidth();
    }
    else if ("*".equals(sx) && !"*".equals(sy))
    {
      yy = UniTypeConv.convString2Int(sy);
      xx = img.getWidth() * yy / img.getHeight();
    }
    int x = xx != null ? xx : 1;
    int y = yy != null ? yy : 1;
    return zoomImage(img, x, y);
  }

  /**
   * Bilder kombinieren. Es wird ein neues Bild in der Gre des Hintergrundbilds erstellt, das Hintergrundbild wird an
   * Position (0, 0) und das Vordergrundbild an der angegebenen Position daraufkopiert.
   * @param background Hintergrundbild
   * @param foreground Vordergrundbild
   * @param x Position-X
   * @param y Position-Y
   * @return Ergebnisbild
   */

  public static BufferedImage combine(BufferedImage background, BufferedImage foreground, int x, int y)
  {
    // ev. genau falsch herum?
    BufferedImage img = new BufferedImage(background.getWidth(null), background.getHeight(null), BufferedImage.TYPE_INT_ARGB);
    img.getGraphics().drawImage(background, 0, 0, null);
    img.getGraphics().drawImage(foreground, x, y, null);
    return img;
  }

  /**
   * Bild in eine Box einpassen und dabei ohne Verzerrung auf die maximale Gre zoomen.
   * @param img Bild
   * @param x Breite der Box
   * @param y Hhe der Box
   * @param cc Farbe der ggf. verbleibenden Rnder
   * @return neues Bild
   */

  public static BufferedImage mboxImage(BufferedImage img, int x, int y, Long cc)
  {
    int ximg = img.getWidth();
    int yimg = img.getHeight();
    double dimg = 1.0 * ximg / yimg;
    BufferedImage box = newImage(x, y, cc);
    int xbox = box.getWidth();
    int ybox = box.getHeight();
    double dbox = 1.0 * xbox / ybox;
    BufferedImage res = null;
    if (dimg > dbox)
    {
      // Rand oben/unten
      BufferedImage n = zoomImage(img, "" + xbox, "*");
      res = combine(box, n, 0, (ybox - n.getHeight()) / 2);
    }
    else if (dimg < dbox)
    {
      // Rand links/rechts
      BufferedImage n = zoomImage(img, "*", "" + ybox);
      res = combine(box, n, (xbox - n.getWidth()) / 2, 0);
    }
    else
    {
      // kein Rand
      BufferedImage n = zoomImage(img, "" + xbox, "" + ybox);
      res = combine(box, n, 0, 0);
    }
    return res;
  }

  /**
   * Rahmen hinzufgen.
   * @param orgimg Originalbild
   * @param x1 Rahmenbreite links
   * @param y1 Rahmenhhe oben
   * @param x2 Rahmenbreite rechts
   * @param y2 Rahmenhhe unten
   * @param cc Rahmenfarbe (kann null sein, Standard: schwarz)
   * @return neues Bild mit Rahmen
   */

  public static BufferedImage addBorder(BufferedImage orgimg, int x1, int y1, int x2, int y2, Long cc)
  {
    int color = cc != null ? cc.intValue() : 0;
    int width = x1 + orgimg.getWidth() + x2;
    int height = y1 + orgimg.getHeight() + y2;
    BufferedImage newimg = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
    Graphics g = newimg.getGraphics();
    g.setColor(new Color(color, true));
    // g.fillRect(0, 0, width, height);
    if (x1 != 0)
    {
      g.fillRect(0, 0, x1, height);
    }
    if (y1 != 0)
    {
      g.fillRect(0, 0, width, y1);
    }
    if (x2 != 0)
    {
      g.fillRect(width - x2, 0, x2, height);
    }
    if (y2 != 0)
    {
      g.fillRect(0, height - y2, width, y2);
    }
    g.drawImage(orgimg, x1, y1, null);
    return newimg;
  }

  /**
   * Farbwerte eines Bildes skalieren.
   * @param img Bild
   * @param ff Faktor als RGBA (8 Bit pro Komponente, 255 = keine nderung, 128 = ca. halbieren)
   */

  public static void scale(BufferedImage img, int ff)
  {
    int blue = ff & 0xff;
    int green = (ff >> 8) & 0xff;
    int red = (ff >> 16) & 0xff;
    int alpha = (ff >> 24) & 0xff;
    MLogger.deb(() -> "argb = " + alpha + " / " + red + " / " + green + " / " + blue);
    int w = img.getWidth();
    int h = img.getHeight();
    for (int y = 0; y < h; y++)
    {
      for (int x = 0; x < w; x++)
      {
        int argb = img.getRGB(x, y);
        int b = argb & 0xff;
        int g = (argb >> 8) & 0xff;
        int r = (argb >> 16) & 0xff;
        int a = (argb >> 24) & 0xff;
        a = a * alpha / 255;
        r = r * red / 255;
        g = g * green / 255;
        b = b * blue / 255;
        argb = (a << 24) + (r << 16) + (g << 8) + b;
        img.setRGB(x, y, argb);
      }
    }
  }

  /**
   * Farbwerte eines Bildes skalieren.
   * Bedeutung der Werte: 255 = keine nderung, 128 = ca. halbieren, 512 = ca. verdoppeln, Ergebnis wird bei 255 gedeckelt.
   * @param orgimg Bild
   * @param alpha A
   * @param red R
   * @param green G
   * @param blue B
   * @return Bild mit skalierten Farbwerten
   */

  public static BufferedImage scale4(BufferedImage orgimg, int alpha, int red, int green, int blue)
  {
    BufferedImage img = new BufferedImage(orgimg.getWidth(null), orgimg.getHeight(null), BufferedImage.TYPE_INT_ARGB);
    img.getGraphics().drawImage(orgimg, 0, 0, null);
    MLogger.deb(() -> "argb = " + alpha + " / " + red + " / " + green + " / " + blue);
    int w = img.getWidth();
    int h = img.getHeight();
    for (int y = 0; y < h; y++)
    {
      for (int x = 0; x < w; x++)
      {
        int argb = img.getRGB(x, y);
        int b = argb & 0xff;
        int g = (argb >> 8) & 0xff;
        int r = (argb >> 16) & 0xff;
        int a = (argb >> 24) & 0xff;
        a = a * alpha / 255;
        r = r * red / 255;
        g = g * green / 255;
        b = b * blue / 255;
        argb = ((a <= 255 ? a : 255) << 24) + ((r <= 255 ? r : 255) << 16) + ((g <= 255 ? g : 255) << 8) + (b <= 255 ? b : 255);
        img.setRGB(x, y, argb);
      }
    }
    return img;
  }

  //===========================================================================

  /**
   * Dummy-Constructor.
   */

  protected ImageUtils()
  {
  }

}
