package com.sta.mimages;

import java.awt.Point;
import java.awt.geom.NoninvertibleTransformException;
import java.awt.image.BufferedImage;
import java.awt.image.renderable.ParameterBlock;

import javax.media.jai.Interpolation;
import javax.media.jai.JAI;
import javax.media.jai.PerspectiveTransform;
import javax.media.jai.PlanarImage;
import javax.media.jai.WarpPerspective;

import com.sta.mlogger.MLogger;

/**
 * <p>Name: ImageWarper</p>
 * <p>Description: Warper module for ImageGrabber using JAI.
 * </p>
 * <p>Comment:
 * This additional module requires JAI lib http://download.java.net/media/jai/builds/release/1_1_3/
 * You can download native lib of JAI, you can get more higher speed of calculating,
 * but you can run image-grabber without native lib with slower speed.
 * </p>
 * <p>Copyright: Copyright (c) 2019</p>
 * <p>Company: &gt;StA-Soft&lt;</p>
 * @author Tom Misawa (riversun.org@gmail.com), StA
 * @version 1.0
 */

public class ImageWarper
{

  /**
   * Construct rectangle image from cropped area that was specified by
   * supplied points by WARP operation.
   * @param bufImage Bufferd-Image
   * @param p0 the point of the corner no.0(upper,left)
   * @param p1 the point of the corner no.1(upper,right)
   * @param p2 the point of the corner no.2(bottom,right)
   * @param p3 The point of the corner no.3(bottomleft)
   * @param destRectWidth Ziel-Rechteck-Breite
   * @param destRectHeight Ziel-Rechteck-Hhe
   * @return Warp-Result
   */

  public WarpResult execWarpToRect(BufferedImage bufImage, Point p0, Point p1, Point p2, Point p3, int destRectWidth, int destRectHeight)
  {
    final WarpResult result = new WarpResult();
    final Point[] srcImageEdgePoints = new Point[]{p0, p1, p2, p3};
    final Point[] destSizePoints = new Point[4];
    destSizePoints[0] = new Point(0, 0);
    destSizePoints[1] = new Point(destRectWidth, 0);
    destSizePoints[2] = new Point(destRectWidth, destRectHeight);
    destSizePoints[3] = new Point(0, destRectHeight);
    final PerspectiveTransform perspectiveTransform = getWarpTransform(srcImageEdgePoints, destSizePoints);
    final Point offset = new Point(0, 0);
    final Point[] bufImagePoints = new Point[4];
    bufImagePoints[0] = new Point(0, 0);
    bufImagePoints[1] = new Point(bufImage.getWidth(), 0);
    bufImagePoints[2] = new Point(bufImage.getWidth(), bufImage.getHeight());
    bufImagePoints[3] = new Point(0, bufImage.getHeight());
    final Point bufImageTopLeft = getUpperLeftPointOf(bufImage, bufImagePoints, perspectiveTransform);
    offset.x = -bufImageTopLeft.x;
    offset.y = -bufImageTopLeft.y;
    final BufferedImage warpedBufferedImage = warpImage(bufImage, perspectiveTransform);
    result.offset = offset;
    result.bufImage = warpedBufferedImage;
    return result;
  }

  /**
   * Warp Result.
   */

  static class WarpResult
  {

    /**
     * Warped image.
     */

    private BufferedImage bufImage;

    /**
     * Warped image offset for fitting parent image (always parent image
     * means background image).
     */

    private Point offset;

    /**
     * Gezerrtes Bild ermitteln.
     * @return gezerrtes Bild
     */

    public BufferedImage getImage()
    {
      return bufImage;
    }

    /**
     * Offset ermitteln.
     * @return Offset
     */

    public Point getOffset()
    {
      return offset;
    }

  }

  /**
   * Execute the operation of WARP from rectangle image.
   * @param bufImage Buffered-Image
   * @param p0 The point of the corner no.0(upper,left)
   * @param p1 The point of the corner no.1(upper,right)
   * @param p2 The point of the corner no.2(bottom,right)
   * @param p3 The point of the corner no.3(bottomleft)
   * @return Warp-Result
   */

  public WarpResult execWarpFromRect(BufferedImage bufImage, Point p0, Point p1, Point p2, Point p3)
  {
    final WarpResult result = new WarpResult();
    final Point[] srcImageEdgePoints = getCornerPointsOfImage(bufImage);
    final Point[] moveToEdgePointsOnBackgroundImage = new Point[4];
    moveToEdgePointsOnBackgroundImage[0] = p0;
    moveToEdgePointsOnBackgroundImage[1] = p1;
    moveToEdgePointsOnBackgroundImage[2] = p2;
    moveToEdgePointsOnBackgroundImage[3] = p3;
    final PerspectiveTransform perspectiveTransform = getWarpTransform(srcImageEdgePoints, moveToEdgePointsOnBackgroundImage);
    final Point topLeft = getUpperLeftPointOf(bufImage, srcImageEdgePoints, perspectiveTransform);
    final BufferedImage warpedBufferedImage = warpImage(bufImage, perspectiveTransform);
    result.offset = topLeft;
    result.bufImage = warpedBufferedImage;
    return result;
  }

  /**
   * Comuputes the supplied warp transform and construct the image.
   * @param bufImage Buffered-Image
   * @param perspectiveTransform Transformation
   * @return Ziel-Bild
   */

  private BufferedImage warpImage(BufferedImage bufImage, PerspectiveTransform perspectiveTransform)
  {
    PerspectiveTransform inverseTransform = null;
    try
    {
      inverseTransform = perspectiveTransform.createInverse();
    }
    catch (NoninvertibleTransformException e1)
    {
      MLogger.err("", e1);
      return null;
    }
    catch (CloneNotSupportedException e1)
    {
      MLogger.err("", e1);
      return null;
    }
    final WarpPerspective warpPerspective = new WarpPerspective(inverseTransform);
    if (warpPerspective == null)
    {
      return null;
    }
    final ParameterBlock parameterBlock = new ParameterBlock();
    final PlanarImage planarImage = PlanarImage.wrapRenderedImage(bufImage);
    parameterBlock.addSource(planarImage);
    parameterBlock.add(warpPerspective);
    if (false)
    {
      // BINLINER:higher speed, low quality
      parameterBlock.add(Interpolation.getInstance(Interpolation.INTERP_BILINEAR));
    }
    if (true)
    {
      // BICUBIC:lower speed, high quality
      parameterBlock.add(Interpolation.getInstance(Interpolation.INTERP_BICUBIC)); //
    }
    // DO OPERATIONS[begin]
    final String OP_NAME = "warp";
    final PlanarImage warpedPlanarImage = JAI.create(OP_NAME, parameterBlock);
    // DO OPERATIONS[end]
    final BufferedImage wapedBufferedImage = warpedPlanarImage.getAsBufferedImage();
    return wapedBufferedImage;
  }

  /**
   * Returns corner points of the image.
   * @param bufImage Buffered-Image
   * @return Feld von Punkten
   */

  private Point[] getCornerPointsOfImage(BufferedImage bufImage)
  {
    final int srcWidth = bufImage.getWidth();
    final int srcHeight = bufImage.getHeight();
    final Point[] srcImageEdgePoints = new Point[4];
    srcImageEdgePoints[0] = new Point(0, 0);
    srcImageEdgePoints[1] = new Point(srcWidth, 0);
    srcImageEdgePoints[2] = new Point(srcWidth, srcHeight);
    srcImageEdgePoints[3] = new Point(0, srcHeight);
    return srcImageEdgePoints;
  }

  /**
   * Get upper-left point of warped object.
   * @param boundedIm Bild
   * @param points Punkte
   * @param persTF Transformation
   * @return Punkt
   */

  private Point getUpperLeftPointOf(BufferedImage boundedIm, Point[] points, PerspectiveTransform persTF)
  {
    final int numOfPoints = 4;
    final Point[] transformedPoints = new Point[numOfPoints];
    for (int i = 0; i < numOfPoints; i++)
    {
      transformedPoints[i] = new Point();
      persTF.transform(points[i], transformedPoints[i]);
    }
    int minX = Integer.MAX_VALUE;
    int minY = Integer.MAX_VALUE;
    for (int i = 0; i < points.length; i++)
    {
      if (transformedPoints[i].x < minX)
      {
        minX = transformedPoints[i].x;
      }
      if (points[i].y < minY)
      {
        minY = transformedPoints[i].y;
      }
    }
    return new Point(minX, minY);
  }

  /**
   * Create PerspectiveTransform from coords.
   * @param srcImageEdgePoints Eckpunkte im Quell-Bild
   * @param moveToEdgePoints Eckpunkte im Ziel-Bild
   * @return Transformation
   */

  private PerspectiveTransform getWarpTransform(Point[] srcImageEdgePoints, Point[] moveToEdgePoints)
  {
    final PerspectiveTransform perspectiveTransform = PerspectiveTransform.getQuadToQuad(
      // Source rectangle vertex coordinates placed clockwise from the
      // upper left corner of the rectangle ordered
      srcImageEdgePoints[0].x,
      srcImageEdgePoints[0].y,
      srcImageEdgePoints[1].x,
      srcImageEdgePoints[1].y,
      srcImageEdgePoints[2].x,
      srcImageEdgePoints[2].y,
      srcImageEdgePoints[3].x,
      srcImageEdgePoints[3].y,
      // Dest vertex coordinates placed clockwise from the upper left
      // corner of the rectangle ordered
      moveToEdgePoints[0].x,
      moveToEdgePoints[0].y,
      moveToEdgePoints[1].x,
      moveToEdgePoints[1].y,
      moveToEdgePoints[2].x,
      moveToEdgePoints[2].y,
      moveToEdgePoints[3].x,
      moveToEdgePoints[3].y
    );
    return perspectiveTransform;
  }

}
