
package com.sta.mimages;

import java.io.File;

import java.util.Arrays;
import java.util.Vector;

import java.net.URL;

import java.awt.Color;
import java.awt.Image;
import java.awt.MediaTracker;

import java.awt.Point;
import java.awt.image.BufferedImage;

import javax.swing.JPanel;
import javax.swing.ImageIcon;

import javax.imageio.ImageIO;

import com.sta.mlogger.MLogger;
import com.sta.mlogger.MLogEntry;

/**
 * <p>Name: MImages</p>
 * <p>Description: MImages.
 * Version 1.23: Umstellung des Converter2 auf com.sta.cts, anschlieend Umstellung auf Converter3.
 * Version 1.27: Umstellung auf Maven.
 * Version 1.34: Verteilung ber Maven-Central.
 * </p>
 * <p>Copyright: Copyright (c) 2005-2023</p>
 * <p>Company: &gt;StA-Soft&lt;</p>
 * @author StA
 * @version 1.44
 */

public class MImages
{

  /**
   * MImages-Version.
   */

  public static final String VERSION = MImagesVersionHelper.VERSION;

  /**
   * MImages-Copyright.
   */

  public static final String COPYRIGHT = MImagesVersionHelper.COPYRIGHT;

  /**
   * MImages-Company.
   */

  public static final String COMPANY = MImagesVersionHelper.COMPANY;

  //===========================================================================

  /**
   * Version ermitteln.
   * @return Version
   */

  public static String getVersion()
  {
    return VERSION;
  }

  /**
   * Copyright ermitteln.
   * @return Copyright
   */

  public static String getCopyright()
  {
    return COPYRIGHT;
  }

  /**
   * Company ermitteln.
   * @return Company
   */

  public static String getCompany()
  {
    return COMPANY;
  }

  //===========================================================================

  /**
   * Allgemeine Methode zum Holen eines Icons. Diese Methode wurde am
   * 25.01.2004 fr die Mglichenkeiten von Applets angepat. Zuerst wird
   * versucht, das Icon als Ressource ber den ClassPath zu laden. Gelingt dies
   * nicht, wird versucht, das Icon direkt aus dem File-System zu laden.
   * Letzteres wird im Falle von Applets nicht funktionieren.
   * @param pSrcFileName Dateiname mit Pfad und Extension (relativ ClassPath)
   * @return das ermittelte Icon
   */

  public static ImageIcon loadImageIcon(String pSrcFileName)
  {
    ImageIcon ii = null;
    Exception ex = null;
    try
    {
      if (new File(pSrcFileName).exists())
      {
        ii = new ImageIcon(pSrcFileName);
      }
    }
    catch (Exception e)
    {
      ex = e;
    }
    if (ii == null)
    {
      URL url = MImages.class.getResource("/" + pSrcFileName);
      if (url != null)
      {
        MLogger.deb("Resource found, load via URL (" + pSrcFileName + ")!");
        ii = new ImageIcon(url);
      }
    }
    if (ii == null)
    {
      if (ex != null)
      {
        MLogger.err("Error loading image via file system: " + pSrcFileName, ex);
      }
      else
      {
        MLogger.err("Error loading image via resources: " + pSrcFileName);
      }
    }
    return ii;
  }

  /**
   * Gre eines Bildes ermitteln.
   * Diese Methode wird beim Einfgen von Bildern bei der Report-Erstellung verwendet, um die Bilder auf eine passende Gre
   * zu bringen. Hier ist es wichtig, dass Probleme nicht durch Exceptions und nicht durch die Rckgabe von null gemeldet
   * werden, da beides nicht ohne greren Aufwand abgefangen werden kann.
   * @param pSrcFileName Dateiname mit Pfad und Extension (relativ ClassPath)
   * @return die ermittelte Gre als Point (also praktisch der erste Punkt rechts unten auerhalb vom Bild) bzw. (0, 0), falls
   * die Datei nicht gefunden wurde oder aus anderen Grnden nicht geladen werden konnte
   */

  public static Point getImageSize(String pSrcFileName)
  {
    ImageIcon ii = loadImageIcon(pSrcFileName);
    // Image iimg = ii.getImage();
    if (ii == null)
    {
      MLogger.inf("Problem loading image: " + pSrcFileName);
      return new Point(0, 0);
    }
    return new Point(ii.getIconWidth(), ii.getIconHeight());
  }

  /**
   * Auf Image warten.
   * @param img das Image, auf das gewartet werden soll
   */

  public static void wait4Image(Image img)
  {
    MediaTracker tracker = new MediaTracker(new JPanel());
    tracker.addImage(img, 0);
    try
    {
      tracker.waitForID(0);
    }
    catch (Exception ex)
    {
    }
  }

  /**
   * Nchsten (echten) Parameter aus dem Vector holen. "Echt" = keine Option.
   * Der Parameter wird aus dem Vector entfernt.
   * @param params Parameter-Vector
   * @return der Parameter oder null, falls keiner vorhanden oder eine Option
   * folgt
   */

  protected static String getNextParam(Vector params)
  {
    if (params.size() == 0)
    {
      return null;
    }
    String s = (String) params.get(0);
    if (s.length() == 0)
    {
      return null;
    }
    /*
    if (s.charAt(0) == '-')
    {
      return null;
    }
    */
    params.remove(0);
    return s;
  }

  /**
   * Log-Level prfen.
   * @param params Parameter-Vector
   */

  protected static void checkLogLevel(Vector params)
  {
    String s = getNextParam(params);
    if ("-err".equals(s) || "-error".equals(s))
    {
      MLogger.getMLogEntry(0).setMaxLevel(MLogEntry.ERROR);
    }
    else if ("-wrn".equals(s) || "-warning".equals(s))
    {
      MLogger.getMLogEntry(0).setMaxLevel(MLogEntry.WARNING);
    }
    else if ("-inf".equals(s) || "-info".equals(s))
    {
      MLogger.getMLogEntry(0).setMaxLevel(MLogEntry.INFO);
    }
    else if ("-msg".equals(s) || "-message".equals(s))
    {
      MLogger.getMLogEntry(0).setMaxLevel(MLogEntry.MESSAGE);
    }
    else if ("-deb".equals(s) || "-debug".equals(s))
    {
      MLogger.getMLogEntry(0).setMaxLevel(MLogEntry.DEBUG);
    }
    else if ("-all".equals(s))
    {
      MLogger.getMLogEntry(0).setMaxLevel(MLogEntry.ALL);
    }
    else if (s != null)
    {
      params.add(0, s);
    }
  }

  /**
   * Hilfe ausgeben.
   */

  public static void help()
  {
    MLogger.err("Syntax Error.");
    MLogger.msg("Usage: mimages ( PDM | TPM | TPM2 ) XMLFileName");
    MLogger.msg("       mimages UML srcpng ClassName dstpng");
    MLogger.msg("       mimages print filename fontname fontsize fontstyle color Test");
    MLogger.msg("       mimages mimgproc filename");
    MLogger.msg(" PDM  ... Physical Data Model");
    MLogger.msg(" TPM  ... Technical Process Model");
    MLogger.msg(" UML  ... Extract one class diagramm");
    System.exit(1);
  }

  /**
   * Bild erzeugen, Text in Bild schreiben.
   * @param fontname Zeichensatz
   * @param fontsize Schriftgre
   * @param fontstyleindex Index fr Font-Styles
   * @param color Farbe
   * @param y Y-Koordinate
   * @param text Text
   * @return Bild
   * @throws Exception im Fehlerfall
   */

  protected static BufferedImage print(String fontname, int fontsize, int fontstyleindex, Color color, int y, String text) throws Exception
  {
    ImageRenderer ir = new ImageRenderer();
    ir.setFontName(fontname);
    ir.setFontSize(fontsize);
    ir.setFontStyleIndex(fontstyleindex);
    ir.setImageTypeIndex(2); // RGB
    ir.setBorderBottom(0);
    ir.setBorderLeft(0);
    ir.setBorderRight(0);
    ir.setBorderTop(0);
    ir.setColor(color);
    ir.setAntialiasing(true);
    ir.setCurX(0);
    ir.setCurY(y);
    ir.print(text);
    int xx = ir.getCurX();
    ir.println();
    int yy = ir.getCurY();
    MLogger.deb("X/Y: " + xx + " / " + yy);
    MLogger.deb("w/h: " + ir.getWidth(text) + " / " + ir.getHeight());
    int ya = search4nn(ir);
    int ye = search4nn_(ir);
    MLogger.deb("ya/ye: " + ya + " / " + ye + " --> " + (ye - ya + 1));
    BufferedImage img = ir.getImage();
    img = img.getSubimage(0, ya, xx, ye - ya + 1);
    int w = img.getWidth();
    int h = img.getHeight();
    BufferedImage img2 = new BufferedImage(w, h, BufferedImage.TYPE_INT_ARGB);
    for (int yp = 0; yp < h; yp++)
    {
      for (int xp = 0; xp < w; xp++)
      {
        int rgb = img.getRGB(xp, yp);
        rgb = (rgb & 0xffffff) | ((255 - (rgb & 0xff)) << 24);
        img2.setRGB(xp, yp, rgb);
      }
    }
    return img2;
  }

  /**
   * Erste Zeile suchen, die nicht wei ist.
   * @param ir Image-Renderer
   * @return Zeile
   */

  private static int search4nn(ImageRenderer ir)
  {
    // Suche nach der ersten nicht-leeren Zeile
    int width = ir.getImageWidth();
    int height = ir.getImageHeight();
    int y = 0;
    boolean isnn = false;
    BufferedImage img = ir.getImage();
    while (!isnn && (y < height))
    {
      int x = 0;
      while ((x < width) && ((img.getRGB(x, y) & 0x00ffffff) == 0x00ffffff))
      {
        x++;
      }
      isnn = x < width;
      if (!isnn)
      {
        y++;
      }
    }
    return y;
  }

  /**
   * Letzte Zeile suchen, die nicht wei ist.
   * @param ir Image-Renderer
   * @return Zeile
   */

  private static int search4nn_(ImageRenderer ir)
  {
    // Suche nach der letzten nicht-leeren Zeile
    int width = ir.getImageWidth();
    int height = ir.getImageHeight();
    int y = height - 1;
    boolean isnn = false;
    BufferedImage img = ir.getImage();
    while (!isnn && (y >= 0))
    {
      int x = 0;
      while ((x < width) && ((img.getRGB(x, y) & 0x00ffffff) == 0x00ffffff))
      {
        x++;
      }
      isnn = x < width;
      if (!isnn)
      {
        y--;
      }
    }
    return y;
  }

  /**
   * Main-Methode.
   * @param args Kommandozeilenargumente
   */

  public static void main(String[] args)
  {
    MLogger.msg("MImages  Version " + VERSION + "  Copyright (c) " + COPYRIGHT + " " + COMPANY);
    Vector params = new Vector(Arrays.asList(args));
    checkLogLevel(params);
    String s = getNextParam(params);
    if ("PDM".equals(s))
    {
      new MImagesPDM().run(params);
    }
    else if ("TPM".equals(s))
    {
      new MImagesTPM().run(params);
    }
    else if ("UML".equals(s))
    {
      new MImagesUML().run(params);
    }
    else if ("print".equals(s))
    {
      try
      {
        String filename = getNextParam(params);
        String fontname = getNextParam(params);
        int fontsize = new Integer(getNextParam(params)).intValue();
        int fontstyleindex = new Integer(getNextParam(params)).intValue();
        String c = getNextParam(params);
        Color color;
        if (c.startsWith("$"))
        {
          color = new Color(Integer.parseInt(c.substring(1), 16));
        }
        else
        {
          color = new Color(Integer.parseInt(c));
        }
        String text = getNextParam(params);
        MLogger.deb("Text: " + text);

        BufferedImage img = print(fontname, fontsize, fontstyleindex, color, 0, text);
        // ir.myImage = img;
        // ir.resetImage();
        // ir.setImageWidth(xx);
        // ir.setImageHeight(yy - y / 2);
        // print(ir, fontname, fontsize, fontstyleindex, color, -y / 2, text);
        // ir.flush();
        // ir.saveImage(filename, img, ir.STDRE);
        ImageRenderer.saveImage(filename, img);
      }
      catch (Exception e)
      {
        MLogger.err("", e);
      }
    }
    else if ("mimgproc".equals(s))
    {
      try
      {
        String filename = getNextParam(params);
        if (filename == null)
        {
          throw new NullPointerException("Missing file name.");
        }
        MImgProc mimgproc = new MImgProc();
        mimgproc.run(filename);
      }
      catch (Exception e)
      {
        MLogger.err("", e);
      }
    }
    else if ("info".equals(s))
    {
      String[] rfn = ImageIO.getReaderFormatNames();
      MLogger.deb("Readers:");
      for (int i = 0; i < rfn.length; i++)
      {
        MLogger.deb("* " + rfn[i]);
      }
      String[] wfn = ImageIO.getWriterFormatNames();
      MLogger.deb("Writer:");
      for (int i = 0; i < wfn.length; i++)
      {
        MLogger.deb("* " + wfn[i]);
      }
    }
    else
    {
      help();
    }
  }

  //===========================================================================

  /**
   * Dummy-Constructor.
   */

  protected MImages()
  {
  }

}
