
package com.sta.mimages;

import java.util.Arrays;
import java.util.Vector;

import java.io.File;

import java.awt.Color;
import java.awt.Image;
import java.awt.Insets;

import javax.swing.ImageIcon;

import com.sta.mlogger.MLogger;

import com.sta.cts.XMLTag;
import com.sta.cts.XMLScanner;

/**
 * <p>Name: MImagesPDM</p>
 * <p>Description: Bilder fr PDM.</p>
 * <p>Copyright: Copyright (c) 2005, 2010-2012, 2014, 2016-2019, 2021</p>
 * <p>Company: &gt;StA-Soft&lt;</p>
 * @author StA
 * @version 1.0
 */

public class MImagesPDM extends MImages
{

  /**
   * Basisklasse fr DB-Darstellungen.
   */

  public static class DBBase
  {

    // fields

    /**
     * X.
     */

    protected int myX = -1;

    /**
     * Y.
     */

    protected int myY = -1;

    /**
     * Breite.
     */

    protected int mySizeX;

    /**
     * Hhe.
     */

    protected int mySizeY;

    // get/set

    /**
     * X ermitteln.
     * @return X
     */

    public int getX()
    {
      return myX;
    }

    /**
     * X vorgeben.
     * @param x X
     */

    public void setX(int x)
    {
      myX = x;
    }

    /**
     * Y ermitteln.
     * @return Y
     */

    public int getY()
    {
      return myY;
    }

    /**
     * Y vorgeben.
     * @param y Y
     */

    public void setY(int y)
    {
      myY = y;
    }

    /**
     * Breite ermitteln.
     * @return Breite
     */

    public int getSizeX()
    {
      return mySizeX;
    }

    /**
     * Hhe ermitteln.
     * @return Hhe
     */

    public int getSizeY()
    {
      return mySizeY;
    }

    // load

    /**
     * Boolean aus XML-Struktur holen.
     * @param t XML-Tag
     * @param name Attributname
     * @param std Standardwert
     * @return true (yes) oder false (no)
     */

    public boolean loadBoolean(XMLTag t, String name, boolean std)
    {
      String s = t.getAttr(name);
      if (s == null)
      {
        return std;
      }
      return s.equals("yes");
    }

    /**
     * String aus XML-Struktur holen.
     * @param t XML-Tag
     * @param name Attributname
     * @param std Standardwert
     * @return Text
     */

    public String loadString(XMLTag t, String name, String std)
    {
      return t.getAttr(name, std);
    }

    /**
     * Integer aus XML-Struktur holen.
     * @param t XML-Tag
     * @param name Attributname
     * @param std Standardwert
     * @return Integer
     */

    public int loadInt(XMLTag t, String name, int std)
    {
      return new Integer(t.getAttr(name, "" + std)).intValue();
    }

    // run

    // save

  }

  //---------------------------------------------------------------------------

  /**
   * Darstellung DB-Field.
   */

  public static class DBField extends DBBase
  {
    // fields

    /**
     * DB-Field-Name.
     */

    private String myDBFieldName;

    /**
     * DB-Field-Type.
     */

    private String myDBFieldType;

    /**
     * DB-Field-Null.
     */

    private String myDBFieldNull;

    /**
     * DB-Field-PK.
     */

    private boolean myDBFieldPK;

    /**
     * DB-Field-ID.
     */

    private boolean myDBFieldID;

    /**
     * DB-Field-OCA.
     */

    private boolean myDBFieldOCA;

    /**
     * DB-Field-FK.
     */

    private boolean myDBFieldFK;

    /**
     * DB-Field-FK-Table-Name.
     */

    private String myDBFieldFKTableName;

    /**
     * DB-Field-FK-Field-Name.
     */

    private String myDBFieldFKFieldName;

    /**
     * DB-Field-FK-On-Delete.
     */

    private String myDBFieldFKOnDelete;

    /**
     * DB-Field-Text.
     */

    private String myDBFieldText;

    // constructors

    /**
     * Standard-Constructor.
     */

    public DBField()
    {
    }

    /**
     * Constructor mit diversen Werten.
     * @param dbfieldname DB-Field-Name
     * @param dbfieldtype DB-Field-Type
     * @param dbfieldnull DB-Field-Null
     * @param dbfieldpk DB-Field-PK
     * @param dbfieldid DB-Field-ID
     * @param dbfieldoca DB-Field-OCA.
     * @param dbfieldfk DB-Field-FK
     * @param dbfieldfktablename DB-Field-FK-Table-Name
     * @param dbfieldfkfieldname DB-Field-FK-Field-Name
     * @param dbfieldfkondelete DB-Field-FK-On-Delete
     */

    public DBField(String dbfieldname, String dbfieldtype, String dbfieldnull, String dbfieldpk, String dbfieldid, String dbfieldoca, String dbfieldfk, String dbfieldfktablename, String dbfieldfkfieldname, String dbfieldfkondelete)
    {
      myDBFieldName = dbfieldname;
      myDBFieldType = dbfieldtype;
      myDBFieldNull = dbfieldnull;
      myDBFieldPK = "yes".equals(dbfieldpk);
      myDBFieldID = "yes".equals(dbfieldid);
      myDBFieldOCA = "yes".equals(dbfieldoca);
      myDBFieldFK = "yes".equals(dbfieldfk);
      myDBFieldFKTableName = dbfieldfktablename;
      myDBFieldFKFieldName = dbfieldfkfieldname;
      myDBFieldFKOnDelete = dbfieldfkondelete;
    }

    // get/set

    /**
     * DB-Field-Name ermitteln.
     * @return DB-Field-Name
     */

    public String getDBFieldName()
    {
      return myDBFieldName;
    }

    /**
     * DB-Field-Null ermitteln.
     * @return DB-Field-Null
     */

    public boolean getDBFieldNull()
    {
      return !"no".equals(myDBFieldNull);
    }

    /**
     * DB-Field-PK ermitteln.
     * @return DB-Field-PK
     */

    public boolean getDBFieldPK()
    {
      return myDBFieldPK;
    }

    /**
     * DB-Field-FK ermitteln.
     * @return DB-Field-FK
     */

    public boolean getDBFieldFK()
    {
      return myDBFieldFK;
    }

    /**
     * DB-Field-FK-Table-Name ermitteln.
     * @return DB-Field-FK-Table-Name
     */

    public String getDBFieldFKTableName()
    {
      return myDBFieldFKTableName;
    }
    /**
     * DB-Field-FK-Field-Name ermitteln.
     * @return DB-Field-FK-Field-Name
     */

    public String getDBFieldFKFieldName()
    {
      return myDBFieldFKFieldName;
    }

    /**
     * DB-Field-Text ermitteln.
     * @return DB-Field-Text
     */

    public String getDBFieldText()
    {
      if (myDBFieldText == null)
      {
        myDBFieldText = myDBFieldName + ": " + myDBFieldType +
                       ("no".equals(myDBFieldNull) ? " NOT" : "") + " NULL" +
                       (myDBFieldPK ? " (PK)" : "") +
                       (myDBFieldID ? " (ID)" : "") +
                       (myDBFieldFK ? " (FK:" + (myDBFieldFKOnDelete != null ? myDBFieldFKOnDelete.toUpperCase() : "RESTRICT") + ")" : "") +
                       (myDBFieldOCA ? " (OCA)" : "");

      }
      return myDBFieldText;
    }

    // load

    /**
     * Feld laden.
     * @param t XML-Tag
     */

    public void load(XMLTag t)
    {
      myDBFieldName = loadString(t, "dbfieldname", "???");
      myDBFieldType = loadString(t, "dbfieldtype", "???");
      myDBFieldNull = loadString(t, "dbfieldnull", "yes");
      myDBFieldPK = loadBoolean(t, "dbfieldpk", false);
      myDBFieldID = loadBoolean(t, "dbfieldid", false);
      myDBFieldOCA = loadBoolean(t, "dbfieldoca", false);
      myDBFieldFK = loadBoolean(t, "dbfieldfk", false);
      myDBFieldFKTableName = loadString(t, "dbfieldfktablename", null);
      myDBFieldFKFieldName = loadString(t, "dbfieldfkfieldname", null);
      myDBFieldFKOnDelete = loadString(t, "dbfieldfkondelete", null);
      myDBFieldText = loadString(t, "dbfieldtext", "???");
    }

    // run

    /**
     * Gre ermitteln.
     * @param ib Image-Buffer
     */

    public void calcSize(ImageBuffer ib)
    {
      mySizeX = ib.getWidth(getDBFieldText());
      mySizeY = ib.getHeight();
    }

    // save

  }

  //---------------------------------------------------------------------------

  /**
   * Basisklasse fr Bilder.
   */

  public static class DBImgBase extends DBBase
  {

    /**
     * Farbe fr Schatten.
     */

    protected static final Color SHADOW = new Color(136, 136, 136);

    // fields

    // ImageBuffer

    /**
     * Image-Buffer.
     */

    protected ImageBuffer myImageBuffer = new ImageBuffer();

    // helper

    /**
     * Prfen, ob es eine berschneidung gibt.
     * @param r "anderes" Bild
     * @return true: ja, false: nein
     */

    public boolean intersects(DBImgBase r)
    {
      int tw = mySizeX;
      int th = mySizeY;
      int rw = r.mySizeX;
      int rh = r.mySizeY;
      if (rw <= 0 || rh <= 0 || tw <= 0 || th <= 0)
      {
        return false;
      }
      int tx = myX;
      int ty = myY;
      int rx = r.myX;
      int ry = r.myY;
      rw += rx;
      rh += ry;
      tw += tx;
      th += ty;
      //      overflow || intersect
      return ((rw < rx || rw > tx) &&
              (rh < ry || rh > ty) &&
              (tw < tx || tw > rx) &&
              (th < ty || th > ry));
    }

    // liefert x, y, w (!), h (!)

    /**
     * berschneidung ermitteln.
     * @param r "anderes" Bild
     * @return berschneidung
     */

    public Insets intersection(DBImgBase r)
    {
      int tx1 = myX;
      int ty1 = myY;
      int rx1 = r.myX;
      int ry1 = r.myY;
      long tx2 = tx1;
      tx2 += mySizeX;
      long ty2 = ty1;
      ty2 += mySizeY;
      long rx2 = rx1;
      rx2 += r.mySizeX;
      long ry2 = ry1;
      ry2 += r.mySizeY;
      if (tx1 < rx1)
      {
        tx1 = rx1;
      }
      if (ty1 < ry1)
      {
        ty1 = ry1;
      }
      if (tx2 > rx2)
      {
        tx2 = rx2;
      }
      if (ty2 > ry2)
      {
        ty2 = ry2;
      }
      tx2 -= tx1;
      ty2 -= ty1;
      // tx2,ty2 will never overflow (they will never be
      // larger than the smallest of the two source w,h)
      // they might underflow, though...
      if (tx2 < Integer.MIN_VALUE)
      {
        tx2 = Integer.MIN_VALUE;
      }
      if (ty2 < Integer.MIN_VALUE)
      {
        ty2 = Integer.MIN_VALUE;
      }
      // return new Insets(tx1, ty1, (int) tx2, (int) ty2);
      return new Insets(ty1, tx1, (int) ty2, (int) tx2);
    }

    // get/set

    /**
     * Image-Buffer ermitteln.
     * @return Image-Buffer
     */

    public ImageBuffer getImageBuffer()
    {
      return myImageBuffer;
    }

    /*
    public void setImage(BufferedImage img)
    {
      myImage = img;
    }
    */

    // load

    // run

    // save

  }

  //---------------------------------------------------------------------------

  /**
   * Darstellung DB-Table.
   */

  public static class DBTable extends DBImgBase
  {

    // fields

    /**
     * Absolut-X.
     */

    protected int myAbsX = -1;

    /**
     * Absolut-Y.
     */

    protected int myAbsY = -1;

    /**
     * DB-Table-Name.
     */

    private String myDBTableName;

    /**
     * Felder.
     */

    private Vector myFields = new Vector();

    // constructors

    /**
     * Standard-Constructor.
     */

    public DBTable()
    {
    }

    /**
     * Constructor mit diversen Werten.
     * @param dbtablename DB-Table-Name
     * @param x X
     * @param y Y
     */

    public DBTable(String dbtablename, int x, int y)
    {
      myDBTableName = dbtablename;
      myX = x;
      myY = y;
    }

    // get/set

    /**
     * Absolut-X ermitteln.
     * @return Absolut-X
     */

    public int getAbsX()
    {
      return myAbsX;
    }

    /**
     * Absolut-Y ermitteln.
     * @return Absolut-Y
     */

    public int getAbsY()
    {
      return myAbsY;
    }

    /**
     * DB-Table-Name ermitteln.
     * @return DB-Table-Name
     */

    public String getDBTableName()
    {
      return myDBTableName;
    }

    /**
     * DB-Table-Name vorgeben.
     * @param dbtablename DB-Table-Name
     */

    public void setDBTableName(String dbtablename)
    {
      myDBTableName = dbtablename;
    }

    // add

    /**
     * DB-Field hinzufgen.
     * @param dbfield DB-Field
     */

    public void addDBField(DBField dbfield)
    {
      myFields.add(dbfield);
    }

    /**
     * DB-Field an bestimmtem Index hinzufgen.
     * @param i Index
     * @param dbfield DB-Field
     */

    public void addDBField(int i, DBField dbfield)
    {
      myFields.add(i, dbfield);
    }

    // load

    /**
     * DB-Table laden.
     * @param t XML-Tag
     * @param dbmodel DB-Model
     */

    public void load(XMLTag t, DBModel dbmodel)
    {
      myDBTableName = loadString(t, "dbtablename", "???");
      MLogger.deb("  DBTable: " + myDBTableName);
      myX = loadInt(t, "x", -1) * dbmodel.getRaster();
      myY = loadInt(t, "y", -1) * dbmodel.getRaster();
    }

    /**
     * DB-Fields laden.
     * @param xs XML-Scanner
     * @throws Exception im Fehlerfall
     */

    public void load(XMLScanner xs) throws Exception
    {
      XMLTag t;
      while ((t = xs.getLeafStart("field")) != null)
      {
        DBField dbfield = new DBField();
        dbfield.load(t);
        myFields.add(dbfield);
        t = xs.getLeafEnd("field");
        if (t == null)
        {
          throw new Exception("Missing </field>.");
        }
      }
    }

    // run

    /**
     * FK-Anzahl ermitteln.
     * @return FK-Anzahl
     */

    public int getFKCount()
    {
      int res = 0;
      int fcnt = myFields.size();
      for (int i = 0; i < fcnt; i++)
      {
        DBField dbfield = (DBField) myFields.get(i);
        if (dbfield.getDBFieldFK())
        {
          res++;
        }
      }
      return res;
    }

    /**
     * DB-Tabelle verarbeiten.
     * @param dbmodel DB-Model
     */

    public void run(DBModel dbmodel)
    {
      MLogger.deb("  DBTable: " + myDBTableName);
      // Berechnung
      int fcnt = myFields.size();
      int width = 0;
      int fontheight = myImageBuffer.getHeight();
      int fontwidth = myImageBuffer.getHeight(); // nicht fr Zeichenbreitenberechnung geeignet!
      int blackborder = dbmodel.getBlackBorder();
      int whiteborder = dbmodel.getWhiteBorder();
      int shadowx = dbmodel.getShadowX();
      int shadowy = dbmodel.getShadowY();
      int lineadd = dbmodel.getLineAdd();
      int lineheight = fontheight + lineadd;
      int lineadd1 = lineadd / 2;
      int lineadd2 = lineheight - fontheight - lineadd1;
      int linedelta = dbmodel.getLineDelta();
      int pk = fontheight + lineadd;
      int xt = blackborder + whiteborder + pk + fontheight / 4;
      int yt = fontheight + blackborder + whiteborder; // + lineadd1;
      int x = xt;
      int y = yt;
      int x0 = 0;
      int y0 = fontheight;
      boolean wasPK = false;
      for (int i = 0; i < fcnt; i++)
      {
        DBField dbfield = (DBField) myFields.get(i);
        boolean isPK = dbfield.getDBFieldPK();
        if ((i != 0) && (wasPK != isPK))
        {
          y += whiteborder + blackborder + whiteborder;
        }
        wasPK = isPK;
        dbfield.setX(x);
        dbfield.setY(y);
        dbfield.calcSize(myImageBuffer);
        int dx = dbfield.getSizeX();
        int mw = dbmodel.getMinWidth();
        dx = dx > mw ? dx : mw;
        width = dx > width ? dx : width;
        y += lineheight;
      }
      mySizeX = x + width + fontwidth + whiteborder + blackborder + shadowx;
      mySizeY = y + whiteborder + blackborder + shadowy;
      // Bilderzeugung
      myImageBuffer.newImage(mySizeX, mySizeY);
      try
      {
        myImageBuffer.setColor(Color.black); // hier wird schon checkImage aufgerufen!
        myImageBuffer.printat(0, 0 + linedelta, myDBTableName);
        int dx = mySizeX - shadowx;
        int dy = mySizeY - shadowy - y0;
        myImageBuffer.drawRectWH(x0, y0, dx, blackborder);
        myImageBuffer.drawRectWH(x0, y0, blackborder, dy);
        myImageBuffer.drawRectWH(dx - blackborder, y0, blackborder, dy);
        myImageBuffer.drawRectWH(x0, y0 + dy - blackborder, dx, blackborder);
        if ((shadowx > 0) || (shadowy > 0))
        {
          myImageBuffer.setColor(SHADOW);
          if (shadowx > 0)
          {
            myImageBuffer.fillRect(dx, y0 + shadowy, shadowx, dy);
          }
          if (shadowy > 0)
          {
            myImageBuffer.fillRect(x0 + shadowx, y0 + dy, dx, shadowy);
          }
          myImageBuffer.setColor(Color.black);
        }
        x = xt;
        y = yt;
        int xpk = blackborder + whiteborder;
        wasPK = false;
        for (int i = 0; i < fcnt; i++)
        {
          DBField dbfield = (DBField) myFields.get(i);
          boolean isPK = dbfield.getDBFieldPK();
          if (isPK)
          {
            Image pki = dbmodel.getPKImage(pk - 2, pk - 2);
            if (pki != null)
            {
              myImageBuffer.drawImageCenter(pki, blackborder + whiteborder + 1, y + 1, pk - 2, pk - 2);
            }
            else
            {
              MLogger.wrn("Missing PK-Image!");
            }
          }
          if ((i != 0) && (wasPK != isPK))
          {
            y += whiteborder;
            myImageBuffer.drawRectWH(blackborder + whiteborder, y, dx - 2 * blackborder - 2 * whiteborder, blackborder);
            y += blackborder + whiteborder;
          }
          wasPK = isPK;
          if ((x != dbfield.getX()) || (y != dbfield.getY()))
          {
            MLogger.wrn("Coordinate missmatch (" + i + "): " + x + "/" + y + " <--> " + dbfield.getX() + "/" + dbfield.getY());
          }
          myImageBuffer.printat(x, y + linedelta, dbfield.getDBFieldText());
          y += lineheight;
        }
      }
      catch (Exception e)
      {
        MLogger.err("Error while creating image for table " + myDBTableName + ": " + e.getMessage(), e);
      }
    }

    /**
     * Absolut-Werte berechnen.
     * @param x0 X0
     * @param y0 Y0
     * @param dbmodel DB-Model
     */

    public void calcAbs(int x0, int y0, DBModel dbmodel)
    {
      // x0 und y0 sind bereits relativ zur aktuellen Tabelle angegeben!!!
      myAbsX = x0;
      myAbsY = y0;
    }

    /**
     * Pfeil nach links zeichnen.
     * @param ib Image-Buffer
     * @param xe XE
     * @param ye YE
     * @param ptrSize Gre
     * @param ptrWidth Breite
     * @throws Exception im Fehlerfall
     */

    protected void drawPtr2Left(ImageBuffer ib, int xe, int ye, int ptrSize, int ptrWidth) throws Exception
    {
      for (int y = 1; y <= ptrWidth; y++)
      {
        ib.drawLine(xe + 2 * y, ye - y, xe + ptrSize, ye - y);
        ib.drawLine(xe + 2 * y, ye + y, xe + ptrSize, ye + y);
      }
    }

    /**
     * Pfeil nach rechts zeichnen.
     * @param ib Image-Buffer
     * @param xe XE
     * @param ye YE
     * @param ptrSize Gre
     * @param ptrWidth Breite
     * @throws Exception im Fehlerfall
     */

    protected void drawPtr2Right(ImageBuffer ib, int xe, int ye, int ptrSize, int ptrWidth) throws Exception
    {
      for (int y = 1; y <= ptrWidth; y++)
      {
        ib.drawLine(xe - 2 * y, ye - y, xe - ptrSize, ye - y);
        ib.drawLine(xe - 2 * y, ye + y, xe - ptrSize, ye + y);
      }
    }

    /**
     * Pfeil nach oben zeichnen.
     * @param ib Image-Buffer
     * @param xe XE
     * @param ye YE
     * @param ptrSize Gre
     * @param ptrWidth Breite
     * @throws Exception im Fehlerfall
     */

    protected void drawPtr2Top(ImageBuffer ib, int xe, int ye, int ptrSize, int ptrWidth) throws Exception
    {
      for (int x = 1; x <= ptrWidth; x++)
      {
        ib.drawLine(xe - x, ye + 2 * x, xe - x, ye + ptrSize);
        ib.drawLine(xe + x, ye + 2 * x, xe + x, ye + ptrSize);
      }
    }

    /**
     * Pfeil nach unten zeichnen.
     * @param ib Image-Buffer
     * @param xe XE
     * @param ye YE
     * @param ptrSize Gre
     * @param ptrWidth Breite
     * @throws Exception im Fehlerfall
     */

    protected void drawPtr2Bottom(ImageBuffer ib, int xe, int ye, int ptrSize, int ptrWidth) throws Exception
    {
      for (int x = 1; x <= ptrWidth; x++)
      {
        ib.drawLine(xe - x, ye - 2 * x, xe - x, ye - ptrSize);
        ib.drawLine(xe + x, ye - 2 * x, xe + x, ye - ptrSize);
      }
    }

    /**
     * Linie zeichnen.
     * @param ib Image-Buffer
     * @param xa XA
     * @param ya YA
     * @param xe XE
     * @param ye YE
     * @throws Exception im Fehlerfall
     */

    protected void drawLine2(ImageBuffer ib, int xa, int ya, int xe, int ye) throws Exception
    {
      // ib.drawLine(xa, ya, xe, ya);
      // ib.drawLine(xe, ya, xe, ye);
      int d = 4;
      int xx = xe > xa ? xe - d : xe + d;
      int yy = ye > ya ? ya + d : ya - d;
      ib.drawLine(xa, ya, xx, ya);
      ib.drawLine(xx, ya, xe, yy);
      ib.drawLine(xe, yy, xe, ye);
    }

    /**
     * Linie zeichnen.
     * @param ib Image-Buffer
     * @param xa XA
     * @param ya YA
     * @param xe XE
     * @param ye YE
     * @param xm XM
     * @throws Exception im Fehlerfall
     */

    protected void drawLine3(ImageBuffer ib, int xa, int ya, int xe, int ye, int xm) throws Exception
    {
      // ib.drawLine(xa, ya, xm, ya);
      // ib.drawLine(xm, ya, xm, ye);
      // ib.drawLine(xm, ye, xe, ye);
      int d = 4;
      int xx1 = xm > xa ? xm - d : xm + d;
      int xx2 = xe > xm ? xm + d : xm - d;
      int yy1 = ye > ya ? ya + d : ya - d;
      int yy2 = ye > ya ? ye - d : ye + d;
      ib.drawLine(xa, ya, xx1, ya);
      ib.drawLine(xx1, ya, xm, yy1);
      ib.drawLine(xm, yy1, xm, yy2);
      ib.drawLine(xm, yy2, xx2, ye);
      ib.drawLine(xx2, ye, xe, ye);
    }

    /**
     * FK's abarbeiten.
     * @param ib Image-Buffer
     * @param base Model-/Modul-Basis
     * @param dbmodel DB-Model
     * @throws Exception im Fehlerfall
     */

    public void runFKs(ImageBuffer ib, DBModBase base, DBModel dbmodel) throws Exception
    {
      int shadowx = dbmodel.getShadowX();
      int shadowy = dbmodel.getShadowY();
      // x0 und y0 sind bereits relativ zur aktuellen Tabelle angegeben!!!
      int fksright = 2; // 0;
      int fksleft = 2; // 0;
      int fcnt = myFields.size();
      for (int i = 0; i < fcnt; i++)
      {
        DBField dbfield = (DBField) myFields.get(i);
        if (dbfield.getDBFieldFK())
        {
          String dbfieldfktablename = dbfield.getDBFieldFKTableName();
          String dbfieldfkfieldname = dbfield.getDBFieldFKFieldName();
          DBTable dbfieldfktable = base.findDBTable(dbfieldfktablename);
          if (dbfieldfktable != null)
          {
            int dstTableX0 = dbfieldfktable.getAbsX();
            int dstTableY0 = dbfieldfktable.getAbsY();
            int dstTableX1 = dstTableX0 + dbfieldfktable.getSizeX() - shadowx;
            int dstTableY1 = dstTableY0 + dbfieldfktable.getSizeY() - shadowy;
            int dstFieldY = dbfieldfktable.getFieldY(dbfieldfkfieldname);
            if (dstFieldY >= 0)
            {
              dstFieldY += dstTableY0;
              // Quelltabelle: myAbsX, myAbsY (mySizeX, mySizeY)
              // Quellfeld:    myAbsX, myAbsY + dbfield.getY() + dbfield.getSizeY() / 2
              // Zieltabelle:  dstTableX0, dstTableY0 --> dstTableX1, dstTableY1
              // Zielfeld:     dstTableX0, dstFieldY | dstTableX1, dstFieldY
              int srcTableX0 = myAbsX;
              int srcTableY0 = myAbsY;
              int srcTableX1 = myAbsX + mySizeX - shadowx;
              int srcTableY1 = myAbsY + mySizeY - shadowy;
              int srcFieldY = srcTableY0 + dbfield.getY() + dbfield.getSizeY() / 2;
              int fkraster = dbmodel.getFKRaster();
              int deltax = /* shadowx + */ 2 * fkraster;
              int deltay = /* shadowy + */ 2 * fkraster;
              int fy = dbfield.getSizeY();
              int ptrSize = fy / 2;
              int ptrWidth = fy / 4;
              int dpk = fy / 3;
              int xa;
              int ya = srcFieldY;
              int xe;
              int ye = dstFieldY;
              if (dstTableX1 + shadowx + deltax <= srcTableX0)
              {
                // Quelle rechts
                xa = srcTableX0;
                xe = dstTableX1 - 1;
                ye = ye + (srcFieldY > dstFieldY ? dpk : srcFieldY < dstFieldY ? -dpk : 0);
                int xm = (xa + xe) / 2;
                // Verbindung
                drawLine3(ib, xa, ya, xe, ye, xm);
                // Pfeil
                drawPtr2Left(ib, xe, ye, ptrSize, ptrWidth);
              }
              else if (srcTableX1 + shadowx + deltax <= dstTableX0)
              {
                // Quelle links
                xa = srcTableX1 - 1;
                xe = dstTableX0;
                ye = ye + (srcFieldY > dstFieldY ? dpk : srcFieldY < dstFieldY ? -dpk : 0);
                int xm = (xa + xe) / 2;
                // Verbindung
                drawLine3(ib, xa, ya, xe, ye, xm);
                // Pfeil
                drawPtr2Right(ib, xe, ye, ptrSize, ptrWidth);
              }
              else if ((dstTableY1 + shadowy + deltay <= srcFieldY) && (dstTableX0 + deltax <= srcTableX0))
              {
                // Quelle unterhalb rechts
                xa = srcTableX0;
                xe = dstTableX0 + fkraster;
                ye = dstTableY1 - 1;
                // Verbindung
                drawLine2(ib, xa, ya, xe, ye);
                // Pfeil
                drawPtr2Top(ib, xe, ye, ptrSize, ptrWidth);
              }
              else if ((dstTableY1 + shadowy + deltay <= srcFieldY) && (srcTableX1 + shadowx + deltax <= dstTableX1))
              {
                // Quelle unterhalb links
                xa = srcTableX1 - 1;
                xe = dstTableX1 - fkraster;
                ye = dstTableY1 - 1;
                // Verbindung
                drawLine2(ib, xa, ya, xe, ye);
                // Pfeil
                drawPtr2Top(ib, xe, ye, ptrSize, ptrWidth);
              }
              else if (dstFieldY <= srcFieldY) // if (dstTableY1 + shadowy + deltay <= srcFieldY)
              {
                // Quelle unterhalb in etwa mittig
                xa = srcTableX1 - 1;
                xe = dstTableX1 - 1;
                ye = ye + (srcFieldY > dstFieldY ? dpk : srcFieldY < dstFieldY ? -dpk : 0);
                int xm = Math.max(srcTableX1 + shadowx + fksright * fkraster, dstTableX1 + shadowx + fkraster);
                fksright++;
                // Verbindung
                drawLine3(ib, xa, ya, xe, ye, xm);
                // Pfeil
                drawPtr2Left(ib, xe, ye, ptrSize, ptrWidth);
              }
              else if ((srcTableY1 + shadowy + deltay <= dstFieldY) && (srcTableX1 + shadowx + deltax <= dstTableX1))
              {
                MLogger.deb("     TopLeft: " + myDBTableName + "." + dbfield.getDBFieldName());
                // Quelle oberhalb links
                xa = srcTableX1 - 1;
                xe = dstTableX1 - fkraster;
                ye = dstTableY0 + fy;
                // Verbindung
                drawLine2(ib, xa, ya, xe, ye);
                // Pfeil
                drawPtr2Bottom(ib, xe, ye, ptrSize, ptrWidth);
              }
              else if (srcFieldY <= dstFieldY) // if (srcTableY1 + shadowy + deltay <= dstFieldY)
              {
                MLogger.deb("     TopMiddle: " + myDBTableName + "." + dbfield.getDBFieldName());
                // Quelle unterhalb in etwa mittig
                xa = srcTableX0;
                xe = dstTableX0;
                ye = ye + (srcFieldY > dstFieldY ? dpk : srcFieldY < dstFieldY ? -dpk : 0);
                int xm = Math.min(srcTableX0 - fksleft * fkraster, dstTableX0 - fkraster);
                fksleft++;
                // Verbindung
                drawLine3(ib, xa, ya, xe, ye, xm);
                // Pfeil
                drawPtr2Right(ib, xe, ye, ptrSize, ptrWidth);
              }
              else
              {
                MLogger.wrn("      Can't route FK: " + myDBTableName + "." + dbfield.getDBFieldName());
                xa = srcTableX0;
                xe = dstTableX0;
                ib.drawLine(xa, ya, xe, ye);
              }
              // Kreis
              // int OVL_DIM = 2;
              if (dbfield.getDBFieldNull())
              {
                // ib.drawOval(xa, ya, OVL_DIM, OVL_DIM);
                ib.drawLine(xa - 1, ya - 2, xa + 1, ya - 2);
                ib.drawLine(xa - 1, ya + 2, xa + 1, ya + 2);
                ib.drawLine(xa - 2, ya - 1, xa - 2, ya + 1);
                ib.drawLine(xa + 2, ya - 1, xa + 2, ya + 1);
              }
              else
              {
                // ib.fillOval(xa, ya, OVL_DIM, OVL_DIM);
                ib.drawLine(xa - 1, ya - 2, xa + 1, ya - 2);
                for (int y = -1; y <= 1; y++)
                {
                  ib.drawLine(xa - 2, ya + y, xa + 2, ya + y);
                }
                ib.drawLine(xa - 1, ya + 2, xa + 1, ya + 2);
              }
              /*
              if (base instanceof DBModul)
              {
                ib.setColor(Color.black);
              }
              */
            }
          }
        }
      }
    }

    /**
     * Field-Y ermitteln.
     * @param name Name
     * @return Field-Y
     */

    public int getFieldY(String name)
    {
      int fcnt = myFields.size();
      for (int i = 0; i < fcnt; i++)
      {
        DBField dbfield = (DBField) myFields.get(i);
        if (dbfield.getDBFieldName().equals(name))
        {
          return dbfield.getY() + dbfield.getSizeY() / 2;
        }
      }
      return -1;
    }

    // save

    /**
     * Bild speichern.
     * @param dst Ziel
     * @param dbmodel DB-Model
     */

    public void save(String dst, DBModel dbmodel)
    {
      dst = dst + File.separator + dbmodel.getImagePrefix() + myDBTableName.toLowerCase() + "." + dbmodel.getImageType().toLowerCase();
      MLogger.deb("  DBTable: " + myDBTableName + " --> " + dst);
      try
      {
        ImageRenderer.saveImage(dst, myImageBuffer.getImage());
      }
      catch (Exception e)
      {
        MLogger.err("Error while saving image for table " + myDBTableName + ": " + e.getMessage(), e);
      }
    }

  }

  //---------------------------------------------------------------------------

  /**
   * Basisklasse fr Modul-/Modell-Darstellungen.
   */

  public static abstract class DBModBase extends DBImgBase
  {

    // fields

    /**
     * Module.
     */

    protected Vector myModuls = new Vector();

    /**
     * Tabellen.
     */

    protected Vector myTables = new Vector();

    // get/set

    /**
     * Name ermitteln.
     * @return Name
     */

    protected abstract String getName();

    // add

    /**
     * DB-Modul hinzufgen.
     * @param dbmodul DB-Modul
     */

    public void addDBModul(DBModul dbmodul)
    {
      myModuls.add(dbmodul);
    }

    /**
     * DB-Table hinzufgen.
     * @param dbtable DB-Table
     */

    public void addDBTable(DBTable dbtable)
    {
      myTables.add(dbtable);
    }

    // load

    // run

    /**
     * Untergeordnete Elemente verarbeiten.
     * @param name Name
     * @param dbmodel DB-Model
     * @throws Exception im Fehlerfall
     */

    protected void runsub(String name, DBModel dbmodel) throws Exception
    {
      int mcnt = myModuls.size();
      for (int i = 0; i < mcnt; i++)
      {
        DBModul dbmodul = (DBModul) myModuls.get(i);
        dbmodul.run(dbmodel);
      }
      int tcnt = myTables.size();
      for (int i = 0; i < tcnt; i++)
      {
        DBTable dbtable = (DBTable) myTables.get(i);
        dbtable.run(dbmodel);
      }
      // Gesamtbild
      MLogger.deb(" Full picture...");
      mySizeX = 0;
      mySizeY = 0;
      int fontheight4name = myImageBuffer.getHeight();
      int blackborder = dbmodel.getBlackBorder();
      int borderleft = dbmodel.getBorderLeft();
      int borderright = dbmodel.getBorderRight();
      int bordertop = dbmodel.getBorderTop();
      int borderbottom = dbmodel.getBorderBottom();
      int shadowx = dbmodel.getShadowX();
      int shadowy = dbmodel.getShadowY();
      int fkraster = dbmodel.getFKRaster();
      // Durchlaufe alle Module+Tabellen, bei denen X und Y angegeben sind
      // * Modul: direkter Grenabgleich
      // * Tabellen: Breite um Anzahl FK's * FK-Rasterbreite erhhen, dann Grenabgleich
      boolean needadd = false;
      for (int i = 0; i < mcnt; i++)
      {
        DBModul dbmodul = (DBModul) myModuls.get(i);
        int xa = dbmodul.getX();
        int ya = dbmodul.getY();
        if ((xa >= 0) && (ya >= 0))
        {
          int xs = dbmodul.getSizeX();
          int ys = dbmodul.getSizeY();
          int xe = xa + xs;
          int ye = ya + ys;
          mySizeX = xe > mySizeX ? xe : mySizeX;
          mySizeY = ye > mySizeY ? ye : mySizeY;
          needadd = true;
        }
      }
      for (int i = 0; i < tcnt; i++)
      {
        DBTable dbtable = (DBTable) myTables.get(i);
        int xa = dbtable.getX();
        int ya = dbtable.getY();
        if ((xa >= 0) && (ya >= 0))
        {
          int xs = dbtable.getSizeX();
          int ys = dbtable.getSizeY();
          int fkcnt = dbtable.getFKCount();
          int xe = xa + xs + fkcnt * fkraster + fkraster;
          int ye = ya + ys;
          mySizeX = xe > mySizeX ? xe : mySizeX;
          mySizeY = ye > mySizeY ? ye : mySizeY;
          needadd = true;
        }
      }
      // Durchlaufe alle Module+Tabellen, bei denen X und/oder Y nicht angegeben ist
      // * Modul: An Bild anfgen (horizontal oder vertikal anordnen?), dann Grenabgleich
      // * Tabelle: Breite  um Anzahl FK's * FK-Rasterbreite erhhen, Bild anfgen, dann Grenabgleich
      for (int i = 0; i < mcnt; i++)
      {
        DBModul dbmodul = (DBModul) myModuls.get(i);
        int xa = dbmodul.getX();
        int ya = dbmodul.getY();
        if ((xa < 0) || (ya < 0))
        {
          if (needadd)
          {
            mySizeY += bordertop;
          }
          xa = 0;
          ya = mySizeY;
          dbmodul.setX(xa);
          dbmodul.setY(ya);
          int xs = dbmodul.getSizeX();
          int ys = dbmodul.getSizeY();
          int xe = xa + xs;
          int ye = ya + ys;
          mySizeX = xe > mySizeX ? xe : mySizeX;
          mySizeY = ye > mySizeY ? ye : mySizeY;
          needadd = true;
        }
      }
      int orgxs = mySizeX;
      int xx = 0;
      int yy = mySizeY;
      for (int i = 0; i < tcnt; i++)
      {
        DBTable dbtable = (DBTable) myTables.get(i);
        int xa = dbtable.getX();
        int ya = dbtable.getY();
        if ((xa < 0) || (ya < 0))
        {
          if (needadd)
          {
            mySizeY += bordertop;
            yy += bordertop;
            needadd = false;
          }
          xa = xx;
          ya = yy;
          dbtable.setX(xa);
          dbtable.setY(ya);
          int xs = dbtable.getSizeX();
          int ys = dbtable.getSizeY();
          int fkcnt = dbtable.getFKCount();
          int xe = xa + xs + fkcnt * fkraster + fkraster;
          int ye = ya + ys;
          xx = xe;
          if ((xa != 0) && (xe > orgxs))
          {
            mySizeY += bordertop;
            xx = 0;
            yy = mySizeY;
            xa = xx;
            ya = yy;
            dbtable.setX(xa);
            dbtable.setY(ya);
            xe = xa + xs + fkcnt * fkraster + fkraster;
            ye = ya + ys;
            xx = xe;
          }
          mySizeX = xe > mySizeX ? xe : mySizeX;
          mySizeY = ye > mySizeY ? ye : mySizeY;
        }
      }
      mySizeX += blackborder + borderleft + borderright + blackborder + shadowx;
      mySizeY += blackborder + bordertop + borderbottom + blackborder + shadowy + fontheight4name;
      int x0 = blackborder + borderleft;
      int y0 = blackborder + bordertop + fontheight4name;
      MLogger.deb("  size: " + mySizeX + "/" + mySizeY);
      // Gesamtbild erzeugen
      MLogger.deb("  create...");
      myImageBuffer.newImage(mySizeX, mySizeY);
      // neu(12.10.2009)...
      myImageBuffer.setColor(Color.black); // hier wird schon checkImage aufgerufen!
      myImageBuffer.printat(0, 0 + dbmodel.getLineDelta(), getName());
      // ...neu(12.10.2009)
      myImageBuffer.drawRectWH(0, fontheight4name + 0, mySizeX - shadowx, blackborder);
      myImageBuffer.drawRectWH(0, fontheight4name + 0, blackborder, mySizeY - shadowy - fontheight4name);
      myImageBuffer.drawRectWH(mySizeX - shadowx - blackborder, fontheight4name + 0, blackborder, mySizeY - shadowy - fontheight4name);
      myImageBuffer.drawRectWH(0, mySizeY - shadowy - blackborder, mySizeX - shadowx, blackborder);
      if ((shadowx > 0) || (shadowy > 0))
      {
        myImageBuffer.setColor(SHADOW);
        if (shadowx > 0)
        {
          myImageBuffer.fillRect(mySizeX - shadowx, fontheight4name + shadowy, shadowx, mySizeY - shadowy - fontheight4name);
        }
        if (shadowy > 0)
        {
          myImageBuffer.fillRect(shadowx, mySizeY - shadowy, mySizeX - shadowx, shadowy);
        }
        myImageBuffer.setColor(Color.black);
      }
      for (int i = 0; i < mcnt; i++)
      {
        DBModul dbmodul = (DBModul) myModuls.get(i);
        int x = dbmodul.getX();
        int y = dbmodul.getY();
        myImageBuffer.drawImageBuffer(dbmodul.getImageBuffer(), x0 + x, y0 + y);
        // dbmodul.setImage(null);
      }
      for (int i = 0; i < tcnt; i++)
      {
        DBTable dbtable = (DBTable) myTables.get(i);
        int x = dbtable.getX();
        int y = dbtable.getY();
        myImageBuffer.drawImageBuffer(dbtable.getImageBuffer(), x0 + x, y0 + y);
        // dbtable.setImage(null);
      }
      // FK's einzeichnen
      MLogger.deb("  fk's...");
      // * Durchlaufe alle Tabellen incl. denen in untergeordneten Modulen
      //   und ermittle dabei die X-Rnder (links und rechts) der Tabellenbilder
      //   (absolut im hier betrachteten Modell/Modul)
      // * Durchlaufe alle FK-Felder in diesen Tabellen und ermittle dabei die
      //   Y-Position in der Mitte des Feldtextes (absolut im hier betrachteten
      //   Modell/Modul) sowie den FK-Tabellen- und den FK-Feldnamen
      // * Suche die Tabelle ausgehend von dem hier betrachteten Modell/Modul
      //   und ermittle (falls gefunden) die absolute Position + Gre der
      //   Tabelle sowie die Y-Position in der Mitte des Feldtextes des
      //   Zielfelds (absolut im hier betrachteten Modell/Modul)
      // * Prfe die Position/Lage der Tabellen/Feldpositionen zueinander und
      //   lege fest, welche Art von Verbinder verwendet werden soll:
      //   entweder 3 Linien (2 waagerechte und eine senkrechte in der Mitte)
      //   oder 2 Linien (1 waagerechte und eine senkrechte)
      // * Zeichne die Linien, den Pfeil an der Zielposition, optional mit
      //   Abrundungen an den Richtungswechseln, ggf. mit verbessertem
      //   Routing-Algorithmus
      // calcAbs(blackborder + borderleft, blackborder + bordertop, dbmodel);
      calcAbs(x0, y0, fontheight4name, dbmodel);
      runFKs(myImageBuffer, this, dbmodel);
      MLogger.deb(" Full picture: Ok.");
      // Intersect
      // zuerst mal ein Feld mit allen Modulen und Tabellen aufbauen
      int size = mcnt + tcnt;
      DBImgBase[] ia = new DBImgBase[size];
      for (int i = 0; i < mcnt; i++)
      {
        DBModul dbmodul = (DBModul) myModuls.get(i);
        ia[i] = dbmodul;
      }
      for (int i = 0; i < tcnt; i++)
      {
        DBTable dbtable = (DBTable) myTables.get(i);
        ia[i + mcnt] = dbtable;
      }
      for (int i = 0; i < size; i++)
      {
        DBImgBase ib = ia[i];
        String typeb = ib instanceof DBModul ? "Modul" : "Table";
        String nameb = ib instanceof DBModul ? ((DBModul) ib).getName() : ((DBTable) ib).getDBTableName();
        for (int j = i + 1; j < size; j++)
        {
          DBImgBase ic = ia[j];
          // prfen, ob ib und ic sich berschneiden
          if (ib.intersects(ic))
          {
            String typec = ic instanceof DBModul ? "Modul" : "Table";
            String namec = ic instanceof DBModul ? ((DBModul) ic).getName() : ((DBTable) ic).getDBTableName();
            MLogger.wrn(typeb + " " + nameb + " intersects " + typec + " " + namec);
            Insets r = ib.intersection(ic); // liefert x, y, w (!), h (!)
            myImageBuffer.setColor(Color.RED);
            myImageBuffer.drawRectWH(x0 + r.left, y0 + r.top, r.right, r.bottom);
            myImageBuffer.drawRectWH(x0 + r.left - 1, y0 + r.top - 1, r.right + 2, r.bottom + 2);
          }
        }
      }
    }

    /**
     * Absolut-Werte ermitteln.
     * @param x0 X0
     * @param y0 Y0
     * @param fontheight4name Font-Hhe fr Name
     * @param dbmodel DB-Model
     */

    public void calcAbs(int x0, int y0, int fontheight4name, DBModel dbmodel)
    {
      int blackborder = dbmodel.getBlackBorder();
      int borderleft = dbmodel.getBorderLeft();
      int bordertop = dbmodel.getBorderTop();
      // x0 und y0 sind bereits relativ zum aktuellen Modell/Modul angegeben!!!
      int mcnt = myModuls.size();
      for (int i = 0; i < mcnt; i++)
      {
        DBModul dbmodul = (DBModul) myModuls.get(i);
        dbmodul.calcAbs(x0 + dbmodul.getX() + blackborder + borderleft, y0 + dbmodul.getY() + blackborder + bordertop + fontheight4name, fontheight4name, dbmodel);
      }
      int tcnt = myTables.size();
      for (int i = 0; i < tcnt; i++)
      {
        DBTable dbtable = (DBTable) myTables.get(i);
        dbtable.calcAbs(x0 + dbtable.getX(), y0 + dbtable.getY(), dbmodel);
      }
    }

    /**
     * FK's abarbeiten.
     * @param ib Image-Buffer
     * @param base Model-/Modul-Basis
     * @param dbmodel DB-Model
     * @throws Exception im Fehlerfall
     */

    protected void runFKs(ImageBuffer ib, DBModBase base, DBModel dbmodel) throws Exception
    {
      // x0 und y0 sind bereits relativ zum aktuellen Modell/Modul angegeben!!!
      int mcnt = myModuls.size();
      for (int i = 0; i < mcnt; i++)
      {
        DBModul dbmodul = (DBModul) myModuls.get(i);
        dbmodul.runFKs(ib, base, dbmodel);
      }
      int tcnt = myTables.size();
      for (int i = 0; i < tcnt; i++)
      {
        DBTable dbtable = (DBTable) myTables.get(i);
        dbtable.runFKs(ib, base, dbmodel);
      }
    }

    /**
     * DB-Table suchen.
     * @param name Name
     * @return DB-Table
     */

    protected DBTable findDBTable(String name)
    {
      int mcnt = myModuls.size();
      for (int i = 0; i < mcnt; i++)
      {
        DBModul dbmodul = (DBModul) myModuls.get(i);
        DBTable dbtable = dbmodul.findDBTable(name);
        if (dbtable != null)
        {
          return dbtable;
        }
      }
      int tcnt = myTables.size();
      for (int i = 0; i < tcnt; i++)
      {
        DBTable dbtable = (DBTable) myTables.get(i);
        if (dbtable.getDBTableName().equals(name))
        {
          return dbtable;
        }
      }
      return null;
    }

    // save

    /**
     * Bilder fr untergeordnete Elemente speichern.
     * @param dst Ziel
     * @param name Name
     * @param dbmodel DB-Model
     * @throws Exception im Fehlerfall
     */

    protected void savesub(String dst, String name, DBModel dbmodel) throws Exception
    {
      int mcnt = myModuls.size();
      for (int i = 0; i < mcnt; i++)
      {
        DBModul dbmodul = (DBModul) myModuls.get(i);
        dbmodul.save(dst + name, dbmodel);
      }
      int tcnt = myTables.size();
      for (int i = 0; i < tcnt; i++)
      {
        DBTable dbtable = (DBTable) myTables.get(i);
        dbtable.save(dst + name, dbmodel);
      }
      // Gesamtbild
      MLogger.deb("  Save full picture...");
      ImageRenderer.saveImage(dst + dbmodel.getImagePrefix() + name + "." + dbmodel.getImageType().toLowerCase(), myImageBuffer.getImage());
      /*
      // Falls das Bild nicht sinnvoll auf A4 druckbar ist, wird es in Einzelteile zerlegt
      int xmax = myImage.getWidth();
      int ymax = myImage.getHeight();
      // Die Breite entscheidet, wie zerlegt wird
      // Voraussetzung: die Tabellen stehen im 600er Raster
      // * max. 3 Tabellen (1800) auf A4 hoch
      // * max. 4 Tabellen (2400) auf A4 quer
      // int tables = xmax / 600;
      // 1 => A4 hoch
      // 2 => A4 hoch
      // 3 => A4 hoch
      // 4 => A4 quer
      // int columns = (xmax + 1800 - 1) / 1800;
      // int dx = (xmax + 1) / columns;
      // int lines = (ymax + 2
      if (((double) xmax) / ((double) ymax) < 1.4)
      {
        int y = 0;
        int line = 0;
        int dh = (int) (((double) xmax) / 1.4);
        while (y < ymax)
        {
          BufferedImage img = myImage.getSubimage(0, y, xmax, y + dh > ymax ? ymax - y : dh);
          ImageRenderer.saveImage(dst + dbmodel.getImagePrefix() + name + "_" + ("" + (line + 100)).substring(1) + "." + dbmodel.getImageType().toLowerCase(), img);
          y += dh;
          line++;
        }
      }
      */
      MLogger.deb("  Save full picture: Ok.");
    }

  }

  //---------------------------------------------------------------------------

  /**
   * Darstellung DB-Modul.
   */

  public static class DBModul extends DBModBase
  {

    // fields

    /**
     * DB-Modul-Name.
     */

    private String myDBModulName;

    // Constructor

    /**
     * Standard-Constructor.
     */

    public DBModul()
    {
    }

    /**
     * Constructor mit diversen Werten.
     * @param dbmodulname DB-Modul-Name
     * @param x X
     * @param y Y
     */

    public DBModul(String dbmodulname, int x, int y)
    {
      myDBModulName = dbmodulname;
      myX = x;
      myY = y;
    }

    // set/get

    @Override
    protected String getName()
    {
      return myDBModulName;
    }

    /**
     * Name vorgeben.
     * @param name Name
     */

    public void setName(String name)
    {
      myDBModulName = name;
    }

    // load

    /**
     * DB-Modul laden.
     * @param t XML-Tag
     * @param dbmodel DB-Model
     */

    public void load(XMLTag t, DBModel dbmodel)
    {
      myDBModulName = loadString(t, "dbmodulname", "???");
      MLogger.deb(" DBModul: " + myDBModulName);
      myX = loadInt(t, "x", -1) * dbmodel.getRaster();
      myY = loadInt(t, "y", -1) * dbmodel.getRaster();
    }

    /**
     * DB-Moduls und DB-Tables laden.
     * @param xs XML-Scanner
     * @param dbmodel DB-Model
     * @throws Exception im Fehlerfall
     */

    public void load(XMLScanner xs, DBModel dbmodel) throws Exception
    {
      XMLTag t;
      while ((t = xs.getLeafStart("modul")) != null)
      {
        DBModul dbmodul = new DBModul();
        dbmodul.load(t, dbmodel);
        myModuls.add(dbmodul);
        dbmodul.load(xs, dbmodel);
        t = xs.getLeafEnd("modul");
        if (t == null)
        {
          throw new Exception("Missing </modul>.");
        }
      }
      while ((t = xs.getLeafStart("table")) != null)
      {
        DBTable dbtable = new DBTable();
        dbtable.load(t, dbmodel);
        myTables.add(dbtable);
        dbtable.load(xs);
        t = xs.getLeafEnd("table");
        if (t == null)
        {
          throw new Exception("Missing </table>.");
        }
      }
    }

    // run

    /**
     * DB-Modul abarbeiten.
     * @param dbmodel DB-Model
     */

    public void run(DBModel dbmodel)
    {
      String name = myDBModulName.toLowerCase();
      MLogger.deb(" DBModul: " + myDBModulName);
      try
      {
        runsub(name, dbmodel);
      }
      catch (Exception e)
      {
        MLogger.err("Error while creating image for modul " + myDBModulName + ": " + e.getMessage(), e);
      }
    }

    // save

    /**
     * Bild(er) speichern.
     * @param dst Ziel
     * @param dbmodel DB-Model
     */

    public void save(String dst, DBModel dbmodel)
    {
      String name = myDBModulName.toLowerCase();
      String dir = dst + File.separator + name;
      new File(dir).mkdirs();
      MLogger.deb(" DBModul: " + myDBModulName + " --> " + dir);
      try
      {
        savesub(dst + File.separator, name, dbmodel);
      }
      catch (Exception e)
      {
        MLogger.err("Error while saving image for modul " + myDBModulName + ": " + e.getMessage(), e);
      }
    }

  }

  //---------------------------------------------------------------------------

  /**
   * Darstellung DB-Modell.
   */

  public static class DBModel extends DBModBase
  {

    // fields

    /**
     * DB-Model-Name.
     */

    private String myDBModelName;

    /**
     * Project-Name.
     */

    private String myPrjName;

    // private String myFontName = "Arial"; // Schriftart: Courier|Arial
    // private int myFontStyle = 1; // Schriftstil: 0=normal, 1=fett, 2=kursiv, 3=fett+kursiv
    // private int myFontSize = 12; // Schriftgrad = Gre der Zeichen in 1/72 inch (z. B. 12)
    // private int myColorDepth = 2; // Farbtiefe: 0=Binr, 1=Graustufen, 2=TrueColor
    // private boolean myAntializing = false; // false true

    /**
     * Image-Type.
     */

    private String myImageType = "PNG"; // TIF|JPG|GIF|PNG

    /**
     * Linker Rand fr Gesamtbilder.
     */

    private int myBorderLeft = 20;

    /**
     * Rechter Rand fr Gesamtbilder.
     */

    private int myBorderRight = 20;

    /**
     * Oberer Rand fr Gesamtbilder.
     */

    private int myBorderTop = 20;

    /**
     * Unterer Rand fr Gesamtbilder.
     */

    private int myBorderBottom = 20;

    /**
     * Zeilen-Versatz.
     */

    private int myLineAdd = 0;

    /**
     * Zeilen-Versatz. Aktuell: 0, alt: -4, Arial: -4.
     */

    private int myLineDelta = 0;

    // private int myTopBorder = 8; // =FntY

    /**
     * Black-Border.
     */

    private int myBlackBorder = 1;

    /**
     * White-Border.
     */

    private int myWhiteBorder = 1;

    /**
     * Shadow-X (=FntY).
     */

    private int myShadowX = 8;

    /**
     * Shadow-Y (=FntX).
     */

    private int myShadowY = 8;

    // private int myPKY = 10; // Gre des PK-Bildes

    /**
     * Min-Width.
     */

    private int myMinWidth = 400;

    /**
     * Raster.
     */

    private int myRaster = 1;

    /**
     * FK-Raster.
     */

    private int myFKRaster = 10;

    /**
     * PK-Image.
     */

    private Image myPKImage;

    /**
     * PK-Image-Zoom.
     */

    private Image myPKImageZoom;

    /**
     * Image-Prefix.
     */

    private String myImagePrefix = "";

    /* *
     * Standard-Constructor.
     */
/*
    public DBModel()
    {
    }
*/
    /**
     * Constructor mit Modellname.
     * @param dbmodelname DB-Model-Name
     */

    public DBModel(String dbmodelname)
    {
      myDBModelName = dbmodelname;
    }

    /**
     * Constructor mit Projektname.
     * @param dbmodelname DB-Model-Name
     * @param pPrjName DBProjektname
     */

    public DBModel(String dbmodelname, String pPrjName)
    {
      myDBModelName = dbmodelname;
      myPrjName = pPrjName;
    }

    // get/set

    @Override
    protected String getName()
    {
      return myPrjName != null ? myPrjName : myDBModelName;
    }

    /**
     * Image-Type ermitteln.
     * @return Image-Type
     */

    public String getImageType()
    {
      return myImageType;
    }

    /**
     * Border-Left ermitteln.
     * @return Border-Left
     */

    public int getBorderLeft()
    {
      return myBorderLeft;
    }

    /**
     * Border-Right ermitteln.
     * @return Border-Right
     */

    public int getBorderRight()
    {
      return myBorderRight;
    }

    /**
     * Border-Top ermitteln.
     * @return Border-Top
     */

    public int getBorderTop()
    {
      return myBorderTop;
    }

    /**
     * Border-Bottom ermitteln.
     * @return Border-Bottom
     */

    public int getBorderBottom()
    {
      return myBorderBottom;
    }

    /**
     * Zeilenversatz ermitteln.
     * @return Zeilenversatz
     */

    public int getLineAdd()
    {
      return myLineAdd;
    }

    /**
     * Zeilenversatz ermitteln.
     * @return Zeilenversatz
     */

    public int getLineDelta()
    {
      return myLineDelta;
    }

    /**
     * Black-Border ermitteln.
     * @return Black-Border
     */

    public int getBlackBorder()
    {
      return myBlackBorder;
    }

    /**
     * White-Border ermitteln.
     * @return White-Border
     */

    public int getWhiteBorder()
    {
      return myWhiteBorder;
    }

    /**
     * Shadow-X ermitteln.
     * @return Shadow-X
     */

    public int getShadowX()
    {
      return myShadowX;
    }

    /**
     * Shadow-Y ermitteln.
     * @return Shadow-Y
     */

    public int getShadowY()
    {
      return myShadowY;
    }

    /**
     * PK-Image ermitteln.
     * @return PK-Image
     */

    public Image getPKImage()
    {
      if (myPKImage == null)
      {
        ImageIcon icon = loadImageIcon("resources/icons/" + "pk13.png");
        if (icon == null)
        {
          icon = loadImageIcon("../gen/source/icons/pk13.png");
        }
        if (icon != null)
        {
          myPKImage = icon.getImage();
        }
        if (myPKImage == null)
        {
          throw new NullPointerException("Can't find pk13.png.");
        }
      }
      return myPKImage;
    }

    /**
     * PK-Image-Zoom ermitteln.
     * @param x Breite
     * @param y Hhe
     * @return PK-Image-Zoom
     */

    public Image getPKImage(int x, int y)
    {
      if (myPKImageZoom != null)
      {
        if ((myPKImageZoom.getWidth(null) == x) && (myPKImageZoom.getHeight(null) == y))
        {
          return myPKImageZoom;
        }
        myPKImageZoom = null;
      }
      Image pki = getPKImage();
      if (pki != null)
      {
        MLogger.deb("Zoom to (x/y): " + x + "/" + y);
        myPKImageZoom = pki.getScaledInstance(x, y, Image.SCALE_DEFAULT);
        wait4Image(myPKImageZoom);
      }
      return myPKImageZoom;
    }

    /**
     * Raster ermitteln.
     * @return Raster
     */

    public int getRaster()
    {
      return myRaster;
    }

    /**
     * FK-Raster ermitteln.
     * @return FK-Raster
     */

    public int getFKRaster()
    {
      return myFKRaster;
    }

    /**
     * Min-Width ermitteln.
     * @return Min-Width
     */

    public int getMinWidth()
    {
      return myMinWidth;
    }

    /**
     * Image-Prefix ermitteln.
     * @return Image-Prefix
     */

    public String getImagePrefix()
    {
      return myImagePrefix;
    }

    /**
     * Image-Prefix vorgeben.
     * @param pre Image-Prefix
     */

    public void setImagePrefix(String pre)
    {
      myImagePrefix = pre != null ? pre : "";
    }

    // load

    /**
     * DB-Model laden.
     * @param t XML-Tag
     */

    public void load(XMLTag t)
    {
      myDBModelName = loadString(t, "physdatamodelname", "???");
      MLogger.deb("DBModel: " + myDBModelName);
    }

    /**
     * DB-Moduls und DB-Tables laden.
     * @param xs XML-Scanner
     * @throws Exception im Fehlerfall
     */

    public void load(XMLScanner xs) throws Exception
    {
      XMLTag t;
      while ((t = xs.getLeafStart("modul")) != null)
      {
        DBModul dbmodul = new DBModul();
        dbmodul.load(t, this);
        myModuls.add(dbmodul);
        dbmodul.load(xs, this);
        t = xs.getLeafEnd("modul");
        if (t == null)
        {
          throw new Exception("Missing </modul>.");
        }
      }
      MLogger.deb(" DBModul: generated");
      while ((t = xs.getLeafStart("table")) != null)
      {
        DBTable dbtable = new DBTable();
        dbtable.load(t, this);
        myTables.add(dbtable);
        dbtable.load(xs);
        t = xs.getLeafEnd("table");
        if (t == null)
        {
          throw new Exception("Missing </table>.");
        }
      }
    }

    // run

    /*
    public void initIR(ImageRenderer ir)
    {
      ir.setAntialiasing(myAntializing);
      ir.setBorderBottom(0);
      ir.setBorderLeft(0);
      ir.setBorderRight(0);
      ir.setBorderTop(0);
      ir.setFontName(myFontName);
      ir.setFontSize(myFontSize);
      ir.setFontStyleIndex(myFontStyle);
      ir.setImageTypeIndex(myColorDepth);
    }
    */

    /**
     * DB-Model abarbeiten.
     */

    public void run()
    {
      String name = myDBModelName.toLowerCase();
      MLogger.deb("DBModel: " + myDBModelName);
      try
      {
        runsub(name, this);
      }
      catch (Exception e)
      {
        MLogger.err("Error while creating image for model " + myDBModelName + ": " + e.getMessage(), e);
      }
    }

    // save

    /**
     * Bild(er) speichern.
     * @param dst Ziel
     */

    public void save(String dst)
    {
      String name = myDBModelName.toLowerCase();
      String dir = dst + File.separator + name;
      new File(dir).mkdirs();
      MLogger.deb("DBModel: " + myDBModelName + " --> " + dir);
      try
      {
        savesub(dst + File.separator, name, this);
      }
      catch (Exception e)
      {
        MLogger.err("Error while saving image for model " + myDBModelName + ": " + e.getMessage(), e);
      }
    }

  }

  //===========================================================================

  /**
   * Liste der gelesenen und zu verarbeitenden Modelle (PDM's).
   */

  private static Vector statDBModels = new Vector();

  /**
   * Projektname.
   */

  private static String statPrjName;

  //===========================================================================

  /**
   * XML-Datei lesen und Struktur der bentigten Daten im Speicher aufbauen.
   * @param fn Name der Quell-XML-Datei
   * @param pre Prfix fr Bilddateinamen, z. B. "db_"
   * @throws Exception im Fehlerfall
   */

  public static void load(String fn, String pre) throws Exception
  {
    XMLScanner xs = new XMLScanner();
    xs.init(fn);
    try
    {
      xs.initH(null);
      XMLTag t;
      t = xs.getLeafStart("root");
      if (t == null)
      {
        throw new Exception("Missing <root>.");
      }
      statPrjName = t.getAttr("prjname");
      while ((t = xs.getLeafStart("model")) != null)
      {
        DBModel dbmodel = new DBModel(null, statPrjName);
        dbmodel.setImagePrefix(pre);
        dbmodel.load(t);
        statDBModels.add(dbmodel);
        dbmodel.load(xs);
        t = xs.getLeafEnd("model");
        if (t == null)
        {
          throw new Exception("Missing </model>.");
        }
      }

      t = xs.getLeafEnd("root");
      if (t == null)
      {
        throw new Exception("Missing </root>.");
      }
    }
    finally
    {
      xs.done();
    }
  }

  /**
   * Run-Methode zum Erzeugen der Bilder.
   * @throws Exception im Fehlerfall
   */

  public static void run() throws Exception
  {
    int mcnt = statDBModels.size();
    for (int i = 0; i < mcnt; i++)
    {
      DBModel dbmodel = (DBModel) statDBModels.get(i);
      dbmodel.run();
    }
  }

  /**
   * Save-Methode zum Speichern der Bilder.
   * @param dst Ziel-Basispfad fr erzeugte Bilder
   * @throws Exception im Fehlerfall
   */

  public static void save(String dst) throws Exception
  {
    int mcnt = statDBModels.size();
    for (int i = 0; i < mcnt; i++)
    {
      DBModel dbmodel = (DBModel) statDBModels.get(i);
      dbmodel.save(dst);
    }
  }

  //===========================================================================

  /**
   * Spezielle Run-Methode.
   * @param fn Name der Quell-XML-Datei
   * @param dst Ziel-Basispfad fr erzeugte Bilder
   * @param pre Prfix fr Bilddateinamen, z. B. "db_"
   * @throws Exception im Fehlerfall
   */

  public static void run(String fn, String dst, String pre) throws Exception
  {
    MLogger.deb("***** load *****");
    load(fn, pre);
    MLogger.deb("***** run *****");
    run();
    MLogger.deb("***** save *****");
    save(dst);
    MLogger.deb("***** ok *****");
  }

  /**
   * Zentrale Run-Methode, mit Fehlerbehandlung.
   * @param params Kommandozeilenargumente
   */

  public static void run(Vector params)
  {
    String fn = getNextParam(params);
    if (fn == null)
    {
      MLogger.err("Missing file name.");
      return;
    }
    String dst = getNextParam(params);
    if (dst == null)
    {
      dst = ".";
    }
    if (dst.endsWith("/") || dst.endsWith("\\"))
    {
      dst = dst.substring(0, dst.length() - 1);
    }
    String pre = getNextParam(params);
    try
    {
      run(fn, dst, pre);
    }
    catch (Exception e)
    {
      MLogger.err("While processing: " + e.getMessage(), e);
    }
  }

  /**
   * Main-Methode.
   * @param args Kommandozeilenargumente
   */

  public static void main(String[] args)
  {
    MLogger.msg("MImagesPDM  Version " + VERSION + "  Copyright (c) " + COPYRIGHT + " " + COMPANY);
    Vector params = new Vector(Arrays.asList(args));
    checkLogLevel(params);
    run(params);
    System.exit(0);
  }

}
