
package com.sta.mimages;

import java.awt.Point;
import java.awt.geom.AffineTransform;

import java.util.ArrayList;
import java.util.List;
import java.util.Stack;
import java.util.Arrays;
import java.util.Vector;
import java.util.StringTokenizer;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.io.BufferedInputStream;

import java.awt.Font;
import java.awt.Color;
import java.awt.Image;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.RenderingHints;

import java.awt.image.BufferedImage;

import javax.swing.ImageIcon;

import javax.imageio.ImageIO;

import com.sta.mlogger.MLogger;

import com.sta.cts.UniTypeConv;

/**
 * <p>Name: MImgProc</p>
 * <p>Description: Versuch einer Umsetzung des MImgProc von Delphi auf Java.</p>
 * <p>Copyright: Copyright (c) 2011-2023</p>
 * <p>Company: &gt;StA-Soft&lt;</p>
 * @author StA
 * @version 1.0
 */

public class MImgProc extends MImages
{

  /**
   * Der Image-Stack.
   */

  private Stack<BufferedImage> myImages = new Stack<>();

  //===========================================================================

  /**
   * Standard-Constructor.
   */

  public MImgProc()
  {
  }

  //===========================================================================

  /**
   * Neues Bild erzeugen und auf Stack ablegen.
   * @param params Parameter (Breite, Hhe, Farbe)
   */

  private void newimage(Vector params)
  {
    Integer xx = UniTypeConv.convString2Int(getNextParam(params));
    Integer yy = UniTypeConv.convString2Int(getNextParam(params));
    Long cc = UniTypeConv.convString2Long(getNextParam(params));
    myImages.push(ImageUtils.newImage(xx != null ? xx : 1, yy != null ? yy : 1, cc));
  }

  /**
   * Bild laden und auf Stack ablegen.
   * @param params Parameter (Dateiname, optional: "ii" - Verwendung von ImageIcon)
   * @throws Exception im Fehlerfall
   */

  private void loadimage(Vector params) throws Exception
  {
    String fn = getNextParam(params);
    // if (fn.endsWith(".png"))
    boolean ii = true;
    if (ii || "ii".equals(getNextParam(params)))
    {
      //
      // Versuch:
      //
      ImageIcon icon = loadImageIcon(fn);
      Image iimg = icon.getImage();
      BufferedImage img = new BufferedImage(iimg.getWidth(null), iimg.getHeight(null), BufferedImage.TYPE_INT_ARGB);
      img.getGraphics().drawImage(iimg, 0, 0, null);
      myImages.push(img);
    }
    else
    {
      //
      // So war's...
      //
      BufferedImage img = ImageIO.read(new BufferedInputStream(new FileInputStream(fn)));
      myImages.push(img);
    }
    //
    // weiter...
    //
  }

  /**
   * Bild vom Stack nehmen und speichern.
   * @param params Parameter (Dateiname)
   * @throws Exception im Fehlerfall
   */

  private void saveimage(Vector params) throws Exception
  {
    BufferedImage img = myImages.pop();
    String fn = getNextParam(params);
    File f = new File(fn);
    File dir = f.getParentFile();
    if (dir != null)
    {
      dir.mkdirs();
    }
    /*
    if (fn.endsWith(".png"))
    {
      ImageIO.write(img, "png", new File(fn));
    }
    else
    {
    */
      ImageRenderer.saveImage(fn, img);
    /*
    }
    */
  }

  /**
   * Bild im Stack duplizieren.
   * @param params Parameter (optional: i-tes Bild)
   */

  private void duplimage(Vector params)
  {
    Integer ii = UniTypeConv.convString2Int(getNextParam(params));
    int i = ii != null ? ii.intValue() : 0;
    BufferedImage img = myImages.get(myImages.size() - 1 - i);
    myImages.push(img);
  }

  /**
   * Bild zoomen.
   * @param params Parameter (x-neu, y-neu)
   */

  private void zoomimage(Vector params)
  {
    BufferedImage img = myImages.pop();
    String sx = getNextParam(params);
    String sy = getNextParam(params);
    myImages.push(ImageUtils.zoomImage(img, sx, sy));
  }

  /**
   * Bild vom Stack auf darunterliegendes Bild vom Stack "drucken". Beide Bilder
   * werden vom Stack entfernt, das Ergebnis wird auf den Stack gelegt.
   * @param params Parameter (x, y)
   */

  private void combine(Vector params)
  {
    // ev. genau falsch herum?
    BufferedImage foreground = myImages.pop();
    BufferedImage background = myImages.pop();
    Integer xx = UniTypeConv.convString2Int(getNextParam(params));
    Integer yy = UniTypeConv.convString2Int(getNextParam(params));
    myImages.push(ImageUtils.combine(background, foreground, xx != null ? xx : 1, yy != null ? yy : 1));
  }

  /**
   * Bild in Box zoomen und einpassen (Seitenverhltnis beibehalten).
   *
   * xcenter ycenter ???
   *
   * @param params Parameter (x-neu, y-neu, optional Hintergrundfarbe)
   */

  private void mboximage(Vector params)
  {
    Integer xx = UniTypeConv.convString2Int(getNextParam(params));
    Integer yy = UniTypeConv.convString2Int(getNextParam(params));
    Long cc = UniTypeConv.convString2Long(getNextParam(params));
    BufferedImage img = myImages.pop();
    myImages.push(ImageUtils.mboxImage(img, xx != null ? xx : 1, yy != null ? yy : 1, cc));
  }

  /**
   * Bild zerren.
   * Es mssen 4 Punkte angegeben werden. Es wird ein neues Bild erzeugt, was alle 4 Punkte enthlt.
   * Das Bild vom Stack wird in das Viereck der angegebenen Punkte auf das neue Bild gezeichnet und dabei gezerrt bzw. gestaucht.
   * @param params Parameter (4 Punkte, je x/y: links oben, rechts oben, rechts unten, links unten)
   */

  private void warpimage(Vector params)
  {
    BufferedImage img = myImages.pop();
    Integer x0 = UniTypeConv.convString2Int(getNextParam(params));
    Integer y0 = UniTypeConv.convString2Int(getNextParam(params));
    Integer x1 = UniTypeConv.convString2Int(getNextParam(params));
    Integer y1 = UniTypeConv.convString2Int(getNextParam(params));
    Integer x2 = UniTypeConv.convString2Int(getNextParam(params));
    Integer y2 = UniTypeConv.convString2Int(getNextParam(params));
    Integer x3 = UniTypeConv.convString2Int(getNextParam(params));
    Integer y3 = UniTypeConv.convString2Int(getNextParam(params));
    img = new ImageWarper().execWarpFromRect(img, new Point(x0, y0), new Point(x1, y1), new Point(x2, y2), new Point(x3, y3)).getImage();
    myImages.push(img);
  }

  /**
   * Bild ausschneiden.
   * @param params Vector (x1, y1, x2, y2) = (x-links, y-oben, x-rechts, y-unten)
   */

  private void cropimage(Vector params)
  {
    BufferedImage orgimg = myImages.pop();
    Integer xx1 = UniTypeConv.convString2Int(getNextParam(params));
    Integer yy1 = UniTypeConv.convString2Int(getNextParam(params));
    Integer xx2 = UniTypeConv.convString2Int(getNextParam(params));
    Integer yy2 = UniTypeConv.convString2Int(getNextParam(params));
    int x1 = xx1 != null ? xx1.intValue() : 0;
    int y1 = yy1 != null ? yy1.intValue() : 0;
    int x2 = xx2 != null ? xx2.intValue() : 0;
    int y2 = yy2 != null ? yy2.intValue() : 0;
    int width = x2 - x1;
    int height = y2 - y1;
    BufferedImage newimg = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
    Graphics g = newimg.getGraphics();
    g.drawImage(orgimg.getSubimage(x1, y1, width, height), 0, 0, null);
    myImages.push(newimg);
  }

  /**
   * Farben ohne Alpha-Channel vergleichen.
   * @param rgb zu prfende Farbe, Alpha-Channel wird hiervon entfernt
   * @param cmp1 untere Grenze
   * @param cmp2 obere Grenze (derzeit nicht verwendet)
   * @return true: gleich bzw. zw. cmp1 und cmp2
   */

  private boolean cmpColor(int rgb, int cmp1, int cmp2)
  {
    // genau genommen: wenn rgb zwischen cmp1 (incl.) und cmp2 (incl.) liegt...
    return (rgb & 0x00ffffff) == cmp1;
  }

  /**
   * Erstes X finden.
   * @param img Bild
   * @return erstes X
   */

  private int findFirstX(BufferedImage img)
  {
    // Suche nach der ersten nicht-leeren Zeile
    int width = img.getWidth();
    int height = img.getHeight();
    int x = 0;
    boolean isnn = false;
    while (!isnn && (x < width))
    {
      int y = 0;
      while ((y < height) && cmpColor(img.getRGB(x, y), 0x00ffffff, 0x00ffffff))
      {
        y++;
      }
      isnn = y < height;
      if (!isnn)
      {
        x++;
      }
    }
    return x;
  }

  /**
   * Letztes X finden.
   * @param img Bild
   * @return letztes X
   */

  private int findLastX(BufferedImage img)
  {
    // Suche nach der letzten nicht-leeren Zeile
    int width = img.getWidth();
    int height = img.getHeight();
    int x = width - 1;
    boolean isnn = false;
    while (!isnn && (x >= 0))
    {
      int y = 0;
      while ((y < height) && cmpColor(img.getRGB(x, y), 0x00ffffff, 0x00ffffff))
      {
        y++;
      }
      isnn = y < height;
      if (!isnn)
      {
        x--;
      }
    }
    return x;
  }

  /**
   * Erstes Y finden.
   * @param img Bild
   * @return erstes Y
   */

  private int findFirstY(BufferedImage img)
  {
    // Suche nach der ersten nicht-leeren Zeile
    int width = img.getWidth();
    int height = img.getHeight();
    int y = 0;
    boolean isnn = false;
    while (!isnn && (y < height))
    {
      int x = 0;
      while ((x < width) && cmpColor(img.getRGB(x, y), 0x00ffffff, 0x00ffffff))
      {
        x++;
      }
      isnn = x < width;
      if (!isnn)
      {
        y++;
      }
    }
    return y;
  }

  /**
   * Letztes Y finden.
   * @param img Bild
   * @return letztes Y
   */

  private int findLastY(BufferedImage img)
  {
    // Suche nach der letzten nicht-leeren Zeile
    int width = img.getWidth();
    int height = img.getHeight();
    int y = height - 1;
    boolean isnn = false;
    while (!isnn && (y >= 0))
    {
      int x = 0;
      while ((x < width) && cmpColor(img.getRGB(x, y), 0x00ffffff, 0x00ffffff))
      {
        x++;
      }
      isnn = x < width;
      if (!isnn)
      {
        y--;
      }
    }
    return y;
  }

  /**
   * Bild beschneiden.
   * @param params (derzeit keine)
   */

  private void cutimage(Vector params)
  {
    BufferedImage orgimg = myImages.pop();
    int x1 = findFirstX(orgimg);
    int y1 = findFirstY(orgimg);
    int x2 = findLastX(orgimg);
    int y2 = findLastY(orgimg);
    MLogger.deb("x1,y1 / x2,y2 = " + x1 + "," + y1 + " / " + x2 + "," + y2);
    int width = x2 - x1 + 1;
    int height = y2 - y1 + 1;
    BufferedImage newimg = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
    Graphics g = newimg.getGraphics();
    g.drawImage(orgimg.getSubimage(x1, y1, width, height), 0, 0, null);
    myImages.push(newimg);
  }

  /**
   * Farbvergleich komponentenweise mit Tolleranz pro Komponente.
   * @param argb1 Farbe 1
   * @param argb2 Farbe 2
   * @param tollerance Tolleranz fr alle Komponenten
   * @return true: gleich bzw. innerhalb der Tolleranz, false: verschieden bzw. auerhalb der Tolleranz
   */

  private boolean cmpColorWithTollerance(int argb1, int argb2, int tollerance)
  {
    int b1 = argb1 & 0xff;
    int g1 = (argb1 >> 8) & 0xff;
    int r1 = (argb1 >> 16) & 0xff;
    int a1 = (argb1 >> 24) & 0xff;
    int b2 = argb2 & 0xff;
    int g2 = (argb2 >> 8) & 0xff;
    int r2 = (argb2 >> 16) & 0xff;
    int a2 = (argb2 >> 24) & 0xff;
    int bt = tollerance & 0xff;
    int gt = (tollerance >> 8) & 0xff;
    int rt = (tollerance >> 16) & 0xff;
    int at = (tollerance >> 24) & 0xff;
    int db = Math.abs(b2 - b1);
    int dg = Math.abs(g2 - g1);
    int dr = Math.abs(r2 - r1);
    int da = Math.abs(a2 - a1);
    return (db <= bt) && (dg <= gt) && (dr <= rt) && (da <= at);
  }

  /**
   * Rekursive Fill-Methode. Ausgehend von einem Punkt, dessen Farbe bereits ersetzt wurde, werden die Punkte oberhalb,
   * unterhalb, links und rechts von diesem geprft, bei Farbgleichheit ersetzt, und es wird fr diese Punkte dann wiederum
   * diese Metode aufgerufen.
   * @param img das Bild
   * @param x aktuelles X
   * @param y aktuelles Y
   * @param width Breite
   * @param height Hhe
   * @param oldcolor zu ersetzende Farbe
   * @param tollerance Tolleranz fr alle Komponenten
   * @param newcolor neue Farbe
   */

  private void fill(BufferedImage img, int x, int y, int width, int height, int oldcolor, int tollerance, int newcolor)
  {
    List<Point> fifo = new ArrayList<>();
    fifo.add(new Point(x, y));
    while (fifo.size() > 0)
    {
      Point point = fifo.remove(0);
      x = point.x;
      y = point.y;
      // top
      if ((y >= 1) && cmpColorWithTollerance(img.getRGB(x, y - 1), oldcolor, tollerance))
      {
        img.setRGB(x, y - 1, newcolor);
        fifo.add(new Point(x, y - 1));
      }
      // bottom
      if ((y < height - 1) && cmpColorWithTollerance(img.getRGB(x, y + 1), oldcolor, tollerance))
      {
        img.setRGB(x, y + 1, newcolor);
        fifo.add(new Point(x, y + 1));
      }
      // left
      if ((x >= 1) && cmpColorWithTollerance(img.getRGB(x - 1, y), oldcolor, tollerance))
      {
        img.setRGB(x - 1, y, newcolor);
        fifo.add(new Point(x - 1, y));
      }
      // right
      if ((x < width - 1) && cmpColorWithTollerance(img.getRGB(x + 1, y), oldcolor, tollerance))
      {
        img.setRGB(x + 1, y, newcolor);
        fifo.add(new Point(x + 1, y));
      }
    }
  }

  /**
   * Bild fllen.
   * @param params x y color Tolleranz (entweder 1 Wert fr alle Farbkanle oder je ein Wert pro Farbkanal)
   */

  private void fillimage(Vector params)
  {
    BufferedImage img = myImages.peek();
    Integer xx = UniTypeConv.convString2Int(getNextParam(params));
    Integer yy = UniTypeConv.convString2Int(getNextParam(params));
    Long cc = UniTypeConv.convString2Long(getNextParam(params));
    Long tt = UniTypeConv.convString2Long(getNextParam(params));
    int x = xx != null ? xx.intValue() : 0;
    int y = yy != null ? yy.intValue() : 0;
    int newcolor = cc != null ? cc.intValue() : 0;
    int tollerance = tt != null ? tt.intValue() : 0;
    int width = img.getWidth();
    int height = img.getHeight();
    int oldolor = img.getRGB(x, y);
    img.setRGB(x, y, newcolor);
    fill(img, x, y, width, height, oldolor, tollerance, newcolor);
  }

  /**
   * Farbe ersetzen.
   * @param params oldcolor Tolleranz newcolor
   */

  private void replcolor(Vector params)
  {
    BufferedImage img = myImages.peek();
    Long oc = UniTypeConv.convString2Long(getNextParam(params));
    Long tt = UniTypeConv.convString2Long(getNextParam(params));
    Long nc = UniTypeConv.convString2Long(getNextParam(params));
    int oldcolor = oc != null ? oc.intValue() : 0;
    int tollerance = tt != null ? tt.intValue() : 0;
    int newcolor = nc != null ? nc.intValue() : 0;
    int width = img.getWidth();
    int height = img.getHeight();
    for (int y = 0; y < height; y++)
    {
      for (int x = 0; x < width; x++)
      {
        int curcolor = img.getRGB(x, y);
        if (cmpColorWithTollerance(curcolor, oldcolor, tollerance))
        {
          img.setRGB(x, y, newcolor);
        }
      }
    }
  }

  /**
   * Bild um 90 Grad nach rechts rotieren.
   * @param orgimg Original-Bild
   * @return rotiertes Bild
   */

  private BufferedImage rot90right(BufferedImage orgimg)
  {
    int orgwidth = orgimg.getWidth();
    int orgheight = orgimg.getHeight();
    BufferedImage newimg = new BufferedImage(orgheight, orgwidth, BufferedImage.TYPE_INT_ARGB);
    for (int y = 0; y < orgheight; y++)
    {
      for (int x = 0; x < orgwidth; x++)
      {
        int argb = orgimg.getRGB(x, y);
        newimg.setRGB(orgheight - y - 1, x, argb);
      }
    }
    return newimg;
  }

  /**
   * Bild rotieren.
   * @param params Gradzahl, erstmal nur 90 (nach rechts) und -90 (nach links)
   */

  private void rotimage(Vector params)
  {
    Integer deg = UniTypeConv.convString2Int(getNextParam(params));
    if (deg == null)
    {
      return;
    }
    if (deg == 0)
    {
      return;
    }
    if ((deg != -270) && (deg != -180) && (deg != -90) && (deg != 90) && (deg != 180) && (deg != 270))
    {
      // throw new RuntimeException("Invalid rotation degrees: " + deg);
      MLogger.wrn("Possible problem with rotation degrees: " + deg + ". Check image size!");
    }
    deg = (deg + 360) % 360; // 90, 180, 270
    BufferedImage orgimg = myImages.pop();
    BufferedImage newimg;
    if (deg == 90)
    {
      newimg = rot90right(orgimg);
    }
    else
    {
      int orgwidth = orgimg.getWidth();
      int orgheight = orgimg.getHeight();
      int newwidth = orgwidth;
      int newheight = orgheight;
      int pos = 0;
      if (deg == 90)
      {
        // right
        newwidth = orgheight;
        newheight = orgwidth;
        pos = (orgwidth - orgheight) / 2;
      }
      else if (deg == 270)
      {
        // left
        newwidth = orgheight;
        newheight = orgwidth;
        pos = (orgheight - orgwidth) / 2;
      }
    /*
    else if (deg == 180)
    {
      pos = 0;
    }
    */
      newimg = new BufferedImage(newwidth, newheight, BufferedImage.TYPE_INT_ARGB);
      Graphics2D g = newimg.createGraphics();
      AffineTransform at = new AffineTransform();
      at.rotate(Math.toRadians(deg), orgwidth / 2, orgheight / 2);
      g.transform(at);
      g.drawImage(orgimg, pos, pos, orgwidth, orgheight, null); // , this);
      orgimg = null;
    }
    myImages.push(newimg);
  }

  /**
   * Bild spiegeln.
   * @param params "h" = horizontal, "v" = vertikal
   */

  private void flipimage(Vector params)
  {
    String dir = getNextParam(params);
    boolean hor = true;
    if ("h".equals(dir))
    {
      hor = true;
    }
    else if ("v".equals(dir))
    {
      hor = false;
    }
    else
    {
      return;
    }
    BufferedImage orgimg = myImages.pop();
    int orgwidth = orgimg.getWidth();
    int orgheight = orgimg.getHeight();
    BufferedImage newimg = new BufferedImage(orgwidth, orgheight, BufferedImage.TYPE_INT_ARGB);
    for (int y = 0; y < orgheight; y++)
    {
      for (int x = 0; x < orgwidth; x++)
      {
        int argb = orgimg.getRGB(x, y);
        if (hor)
        {
          newimg.setRGB(orgwidth - x - 1, y, argb);
        }
        else
        {
          newimg.setRGB(x, orgheight - y - 1, argb);
        }
      }
    }
    myImages.push(newimg);
  }

  /**
   * Rahmen hinzufgen.
   * @param params Parameter (x-links, y-oben, x-rechts, y-unten, Farbe)
   */

  private void addborder(Vector params)
  {
    BufferedImage orgimg = myImages.pop();
    Integer xx1 = UniTypeConv.convString2Int(getNextParam(params));
    Integer yy1 = UniTypeConv.convString2Int(getNextParam(params));
    Integer xx2 = UniTypeConv.convString2Int(getNextParam(params));
    Integer yy2 = UniTypeConv.convString2Int(getNextParam(params));
    Long cc = UniTypeConv.convString2Long(getNextParam(params));
    myImages.push(ImageUtils.addBorder(orgimg, xx1 != null ? xx1 : 0, yy1 != null ? yy1 : 0, xx2 != null ? xx2 : 0, yy2 != null ? yy2 : 0, cc));
  }

  //---------------------------------------------------------------------------

  /**
   * Grau-Skalierung des Bildes auf dem Stack.
   * @param params Parameter (keine)
   */

  private void grayscale(Vector params)
  {
    BufferedImage img = myImages.peek();
    int w = img.getWidth();
    int h = img.getHeight();
    for (int y = 0; y < h; y++)
    {
      for (int x = 0; x < w; x++)
      {
        int argb = img.getRGB(x, y);
        int b = argb & 0xff;
        int g = (argb >> 8) & 0xff;
        int r = (argb >> 16) & 0xff;
        int a = (argb >> 24) & 0xff;
        int m = (int) (0.299 * r + 0.587 * g + 0.114 * b);
        m = m <= 255 ? m : 255;
        argb = (a << 24) + (m << 16) + (m << 8) + m;
        img.setRGB(x, y, argb);
      }
    }
  }

  /**
   * Farbwerte des Bildes auf dem Stack skalieren.
   * @param params Parameter (ARGB)
   */

  private void scale(Vector params)
  {
    BufferedImage img = myImages.peek();
    Long factor = UniTypeConv.convString2Long(getNextParam(params));
    int ff = factor != null ? factor.intValue() : 0xffffffff;
    ImageUtils.scale(img, ff);
  }

  /**
   * Farbwerte des Bildes auf dem Stack skalieren.
   * Wichtig: Das Bild wird vom Stack entfernt, es wird ein neues Bild erzeugt, dieses wird dann auf dem Stack abgelegt.
   * @param params Parameter (A R G B)
   */

  private void scale4(Vector params)
  {
    BufferedImage orgimg = myImages.pop();
    Integer alpha = UniTypeConv.convString2Int(getNextParam(params));
    Integer red = UniTypeConv.convString2Int(getNextParam(params));
    Integer green = UniTypeConv.convString2Int(getNextParam(params));
    Integer blue = UniTypeConv.convString2Int(getNextParam(params));
    myImages.push(ImageUtils.scale4(orgimg, alpha, red, green, blue));
  }

  /**
   * Farbwerte des Bildes auf dem Stack xor-en.
   * @param params Parameter (ARGB)
   */

  private void scalex(Vector params)
  {
    BufferedImage img = myImages.peek();
    Long factor = UniTypeConv.convString2Long(getNextParam(params));
    int ff = factor != null ? factor.intValue() : 0xffffffff;
    int blue = ff & 0xff;
    int green = (ff >> 8) & 0xff;
    int red = (ff >> 16) & 0xff;
    int alpha = (ff >> 24) & 0xff;
    MLogger.deb("argb = " + alpha + " / " + red + " / " + green + " / " + blue);
    int w = img.getWidth();
    int h = img.getHeight();
    for (int y = 0; y < h; y++)
    {
      for (int x = 0; x < w; x++)
      {
        int argb = img.getRGB(x, y);
        int b = argb & 0xff;
        int g = (argb >> 8) & 0xff;
        int r = (argb >> 16) & 0xff;
        int a = (argb >> 24) & 0xff;
        a = (a ^ alpha) & 0xff;
        r = (r ^ red) & 0xff;
        g = (g ^ green) & 0xff;
        b = (b ^ blue) & 0xff;
        argb = (a << 24) + (r << 16) + (g << 8) + b;
        img.setRGB(x, y, argb);
      }
    }
  }

  /**
   * Farbwerte des Bildes auf dem Stack xor-en.
   * @param params Parameter (A R G B)
   */

  private void scale4x(Vector params)
  {
    // BufferedImage img = myImages.peek();
    BufferedImage img1 = myImages.pop();
    BufferedImage img = new BufferedImage(img1.getWidth(null), img1.getHeight(null), BufferedImage.TYPE_INT_ARGB);
    img.getGraphics().drawImage(img1, 0, 0, null);
    int alpha = UniTypeConv.convString2Int(getNextParam(params)).intValue();
    int red = UniTypeConv.convString2Int(getNextParam(params)).intValue();
    int green = UniTypeConv.convString2Int(getNextParam(params)).intValue();
    int blue = UniTypeConv.convString2Int(getNextParam(params)).intValue();
    MLogger.deb("argb = " + alpha + " / " + red + " / " + green + " / " + blue);
    int w = img.getWidth();
    int h = img.getHeight();
    for (int y = 0; y < h; y++)
    {
      for (int x = 0; x < w; x++)
      {
        int argb = img.getRGB(x, y);
        // int f1 = argb;
        int b = argb & 0xff;
        int g = (argb >> 8) & 0xff;
        int r = (argb >> 16) & 0xff;
        int a = (argb >> 24) & 0xff;
        a = (a ^ alpha) & 0xff;
        r = (r ^ red) & 0xff;
        g = (g ^ green) & 0xff;
        b = (b ^ blue) & 0xff;
        argb = ((a <= 255 ? a : 255) << 24) + ((r <= 255 ? r : 255) << 16) + ((g <= 255 ? g : 255) << 8) + (b <= 255 ? b : 255);
        img.setRGB(x, y, argb);
      }
    }
    myImages.push(img);
  }

  /**
   * Transformation des Bildes auf dem Stack.
   * @param params Parameter (Matrix: A R G B / A R G B / A R G B / A R G B fr A / R / G / B)
   */

  private void transform(Vector params)
  {
    BufferedImage img = myImages.peek();
    int[][] matrix = new int[4][4];
    for (int z = 0; z < 4; z++)
    {
      for (int s = 0; s < 4; s++)
      {
        matrix[z][s] = UniTypeConv.convString2Int(getNextParam(params));
      }
    }
    int w = img.getWidth();
    int h = img.getHeight();
    for (int y = 0; y < h; y++)
    {
      for (int x = 0; x < w; x++)
      {
        int argb = img.getRGB(x, y);
        int[] src = new int[4];
        src[0] = (argb >> 24) & 0xff;
        src[1] = (argb >> 16) & 0xff;
        src[2] = (argb >> 8) & 0xff;
        src[3] = argb & 0xff;
        int[] dst = new int[4];
        // Matrix-Multiplikation dst := matrix * src
        for (int z = 0; z < 4; z++)
        {
          int v = 0;
          for (int s = 0; s < 4; s++)
          {
            v += matrix[z][s] * src[s];
          }
          v = v >> 8;
          dst[z] = v <= 255 ? v : 255;
        }
        argb = (dst[0] << 24) + (dst[1] << 16) + (dst[2] << 8) + dst[3];
        img.setRGB(x, y, argb);
      }
    }
  }

  //---------------------------------------------------------------------------

  /**
   * Text auf oberstes Bild vom Stack "drucken".
   * @param params Parameter (x, y, Font, Gre, Style, Farbe, Text)
   */

  private void print(Vector params)
  {
    Integer xx = UniTypeConv.convString2Int(getNextParam(params));
    Integer yy = UniTypeConv.convString2Int(getNextParam(params));
    int x = xx != null ? xx.intValue() : 1;
    int y = yy != null ? yy.intValue() : 1;
    String fontname = getNextParam(params);
    Integer fontsize = UniTypeConv.convString2Int(getNextParam(params));
    Integer fontstyleindex = UniTypeConv.convString2Int(getNextParam(params));
    Integer cc = UniTypeConv.convString2Int(getNextParam(params));
    String text = getNextParam(params);
    if (text.startsWith("\""))
    {
      if (text.length() > 1)
      {
        text = text.substring(1);
      }
      String s;
      while (!text.endsWith("\"") && ((s = getNextParam(params)) != null))
      {
        text += " " + s;
      }
      if (text.endsWith("\""))
      {
        if (text.length() > 1)
        {
          text = text.substring(0, text.length() - 1);
        }
        else
        {
          text = "";
        }
      }
    }
    BufferedImage background = myImages.pop();
    Graphics g = background.getGraphics();
    if (g instanceof Graphics2D)
    {
      ((Graphics2D) g).setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
      ((Graphics2D) g).setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
    }
    g.setFont(new Font(fontname, fontstyleindex, fontsize));
    g.setColor(new Color(cc));
    g.drawString(text, x, y);
    myImages.push(background);
  }

  /**
   * Linie auf oberstes Bild vom Stack zeichnen.
   * @param params Parameter (x1, y1, x2, y2, Farbe)
   */

  private void drawline(Vector params)
  {
    Integer xx1 = UniTypeConv.convString2Int(getNextParam(params));
    Integer yy1 = UniTypeConv.convString2Int(getNextParam(params));
    int x1 = xx1 != null ? xx1.intValue() : 0;
    int y1 = yy1 != null ? yy1.intValue() : 0;
    Integer xx2 = UniTypeConv.convString2Int(getNextParam(params));
    Integer yy2 = UniTypeConv.convString2Int(getNextParam(params));
    int x2 = xx2 != null ? xx2.intValue() : 0;
    int y2 = yy2 != null ? yy2.intValue() : 0;
    Long cc = UniTypeConv.convString2Long(getNextParam(params));
    int color = cc != null ? cc.intValue() : 0;
    BufferedImage background = myImages.peek();
    Graphics g = background.getGraphics();
    if (g instanceof Graphics2D)
    {
      ((Graphics2D) g).setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
      ((Graphics2D) g).setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
    }
    g.setColor(new Color(color));
    g.drawLine(x1, y1, x2, y2);
  }

  /**
   * Rechteck auf oberstes Bild vom Stack zeichnen.
   * @param params Parameter (x1, y1, x2, y2, Farbe)
   * [1: links oben, 2: rechts unten]
   */

  private void drawrect(Vector params)
  {
    Integer xx1 = UniTypeConv.convString2Int(getNextParam(params));
    Integer yy1 = UniTypeConv.convString2Int(getNextParam(params));
    int x1 = xx1 != null ? xx1.intValue() : 0;
    int y1 = yy1 != null ? yy1.intValue() : 0;
    Integer xx2 = UniTypeConv.convString2Int(getNextParam(params));
    Integer yy2 = UniTypeConv.convString2Int(getNextParam(params));
    int x2 = xx2 != null ? xx2.intValue() : 0;
    int y2 = yy2 != null ? yy2.intValue() : 0;
    Long cc = UniTypeConv.convString2Long(getNextParam(params));
    int color = cc != null ? cc.intValue() : 0;
    BufferedImage background = myImages.peek();
    Graphics g = background.getGraphics();
    if (g instanceof Graphics2D)
    {
      ((Graphics2D) g).setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
      ((Graphics2D) g).setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
    }
    g.setColor(new Color(color));
    g.drawRect(x1, y1, x2 - x1, y2 - y1);
  }

  /**
   * Box auf oberstes Bild vom Stack zeichnen.
   * @param params Parameter (x1, y1, x2, y2, Farbe)
   * [1: links oben, 2: rechts unten]
   */

  private void drawbox(Vector params)
  {
    Integer xx1 = UniTypeConv.convString2Int(getNextParam(params));
    Integer yy1 = UniTypeConv.convString2Int(getNextParam(params));
    int x1 = xx1 != null ? xx1.intValue() : 0;
    int y1 = yy1 != null ? yy1.intValue() : 0;
    Integer xx2 = UniTypeConv.convString2Int(getNextParam(params));
    Integer yy2 = UniTypeConv.convString2Int(getNextParam(params));
    int x2 = xx2 != null ? xx2.intValue() : 0;
    int y2 = yy2 != null ? yy2.intValue() : 0;
    Long cc = UniTypeConv.convString2Long(getNextParam(params));
    int color = cc != null ? cc.intValue() : 0;
    BufferedImage background = myImages.peek();
    Graphics g = background.getGraphics();
    if (g instanceof Graphics2D)
    {
      ((Graphics2D) g).setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
      ((Graphics2D) g).setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
    }
    g.setColor(new Color(color));
    g.fillRect(x1, y1, x2 - x1, y2 - y1);
  }

  /**
   * Ausgeflltes Rechteck auf oberstes Bild vom Stack zeichnen.
   * @param params Parameter (x1, y1, x2, y2, Farbe)
   * [1: links oben, 2: rechts unten]
   */

  private void fillrect(Vector params)
  {
    drawbox(params);
  }

  /**
   * Oval in gedachtes Rechteck auf oberstes Bild vom Stack zeichnen.
   * @param params Parameter (x1, y1, x2, y2, Farbe)
   * [1: links oben, 2: rechts unten]
   */

  private void drawoval(Vector params)
  {
    Integer xx1 = UniTypeConv.convString2Int(getNextParam(params));
    Integer yy1 = UniTypeConv.convString2Int(getNextParam(params));
    int x1 = xx1 != null ? xx1.intValue() : 0;
    int y1 = yy1 != null ? yy1.intValue() : 0;
    Integer xx2 = UniTypeConv.convString2Int(getNextParam(params));
    Integer yy2 = UniTypeConv.convString2Int(getNextParam(params));
    int x2 = xx2 != null ? xx2.intValue() : 0;
    int y2 = yy2 != null ? yy2.intValue() : 0;
    Long cc = UniTypeConv.convString2Long(getNextParam(params));
    int color = cc != null ? cc.intValue() : 0;
    BufferedImage background = myImages.peek();
    Graphics g = background.getGraphics();
    if (g instanceof Graphics2D)
    {
      ((Graphics2D) g).setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
      ((Graphics2D) g).setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
    }
    g.setColor(new Color(color));
    g.drawOval(x1, y1, x2 - x1, y2 - y1);
  }

  /**
   * Geflltes Oval in gedachtes Rechteck auf oberstes Bild vom Stack zeichnen.
   * @param params Parameter (x1, y1, x2, y2, Farbe)
   * [1: links oben, 2: rechts unten]
   */

  private void filloval(Vector params)
  {
    Integer xx1 = UniTypeConv.convString2Int(getNextParam(params));
    Integer yy1 = UniTypeConv.convString2Int(getNextParam(params));
    int x1 = xx1 != null ? xx1.intValue() : 0;
    int y1 = yy1 != null ? yy1.intValue() : 0;
    Integer xx2 = UniTypeConv.convString2Int(getNextParam(params));
    Integer yy2 = UniTypeConv.convString2Int(getNextParam(params));
    int x2 = xx2 != null ? xx2.intValue() : 0;
    int y2 = yy2 != null ? yy2.intValue() : 0;
    Long cc = UniTypeConv.convString2Long(getNextParam(params));
    int color = cc != null ? cc.intValue() : 0;
    BufferedImage background = myImages.peek();
    Graphics g = background.getGraphics();
    if (g instanceof Graphics2D)
    {
      ((Graphics2D) g).setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
      ((Graphics2D) g).setRenderingHint(RenderingHints.KEY_TEXT_ANTIALIASING, RenderingHints.VALUE_TEXT_ANTIALIAS_ON);
    }
    g.setColor(new Color(color));
    g.fillOval(x1, y1, x2 - x1, y2 - y1);
  }

  //---------------------------------------------------------------------------

  /**
   * Farbe prfen.
   * @param argb int
   * @param a1 int
   * @param a2 int
   * @param r1 int
   * @param r2 int
   * @param g1 int
   * @param g2 int
   * @param b1 int
   * @param b2 int
   * @return true: ok, false: nicht ok
   */

  private boolean checkColor(int argb, int a1, int a2, int r1, int r2, int g1, int g2, int b1, int b2)
  {
    int a = (argb >> 24) & 0xff;
    int r = (argb >> 16) & 0xff;
    int g = (argb >> 8) & 0xff;
    int b = argb & 0xff;
    return (a1 <= a) && (a <= a2) && (r1 <= r) && (r <= r2) && (g1 <= g) && (g <= g2) && (b1 <= b) && (b <= b2);
  }

  /**
   * Text auf Bild vom Stack unkenntlich machen.
   *
   * mimages mimgproc killtext.dat
   *
   * @param params Parameter (x1, y1, x2, y2, Farbe ARGB, Farbtolleranz ARGB, Umkreisgre)
   * @throws Exception im Fehlerfall
   */

  private void killtext(Vector params) throws Exception
  {
    MLogger.deb("killtext...");
    BufferedImage img = myImages.peek();
    int w = img.getWidth();
    int h = img.getHeight();
    int x1 = UniTypeConv.convString2Int(getNextParam(params)).intValue();
    int y1 = UniTypeConv.convString2Int(getNextParam(params)).intValue();
    int x2 = UniTypeConv.convString2Int(getNextParam(params)).intValue();
    int y2 = UniTypeConv.convString2Int(getNextParam(params)).intValue();
    long color = UniTypeConv.convString2Long(getNextParam(params)).longValue();
    long toll = UniTypeConv.convString2Long(getNextParam(params)).longValue();
    int circ = UniTypeConv.convString2Int(getNextParam(params)).intValue();
    if (x1 < 0)
    {
      x1 = w + x1;
      if (x2 == 0)
      {
        x2 = w;
      }
    }
    if (x2 < 0)
    {
      x2 = w + x2;
    }
    if (y1 < 0)
    {
      y1 = h + y1;
      if (y2 == 0)
      {
        y2 = h;
      }
    }
    if (y2 < 0)
    {
      y2 = h + y2;
    }
    if (x1 >= x2)
    {
      throw new Exception("X-Problem: " + x1 + " / " + x2);
    }
    if (y1 >= y2)
    {
      throw new Exception("Y-Problem: " + y1 + " / " + y2);
    }
    // Durchlaufen aller Punkte im angegeben Rechteck
    // Farbe ermitteln, falls diese Farbe in Tolleranzgrenze fllt:
    // * Ermittle Farbe der Punkte im Umkreis entsprechend circ, die nicht in
    //   ARGB-Tolleranz fallen, und berechne die Farbe des betrachteten Punktes
    //   (Farbe merken, Farbe des Punktes anfangs nicht ndern!)
    // Farbe aller Punkte, fr die eine neue Farbe berechnet wurde, ndern.
    int a = ((int) color >> 24) & 0xff;
    int r = ((int) color >> 16) & 0xff;
    int g = ((int) color >> 8) & 0xff;
    int b = (int) color & 0xff;
    int at = ((int) toll >> 24) & 0xff;
    int rt = ((int) toll >> 16) & 0xff;
    int gt = ((int) toll >> 8) & 0xff;
    int bt = (int) toll & 0xff;

    int b1 = Math.max(b - bt, 0);
    int b2 = Math.min(b + bt, 255);
    int g1 = Math.max(g - gt, 0);
    int g2 = Math.min(g + gt, 255);
    int r1 = Math.max(r - rt, 0);
    int r2 = Math.min(r + rt, 255);
    int a1 = Math.max(a - at, 0);
    int a2 = Math.min(a + at, 255);

    int dx = x2 - x1;
    int dy = y2 - y1;

    MLogger.deb("x1/dx/x2: " + x1 + "/" + dx + "/" + x2);
    MLogger.deb("y1/dy/y2: " + y1 + "/" + dy + "/" + y2);
    MLogger.deb("toll: " + Long.toHexString(toll));
    MLogger.deb("argb: " + a + "/" + r + "/" + g + "/" + b);
    MLogger.deb("toll: " + at + "/" + rt + "/" + gt + "/" + bt);
    MLogger.deb("toll1: " + a1 + "/" + r1 + "/" + g1 + "/" + b1);
    MLogger.deb("toll2: " + a2 + "/" + r2 + "/" + g2 + "/" + b2);

    boolean[] ba = new boolean[dy * dx];
    Arrays.fill(ba, false);
    for (int y = 0; y < dy; y++)
    {
      for (int x = 0; x < dx; x++)
      {
        int argb = img.getRGB(x1 + x, y1 + y);
        int ii = y * dx + x;
        if (checkColor(argb, a1, a2, r1, r2, g1, g2, b1, b2))
        {
          // Farbe gefunden
          ba[ii] = true; // 0xff000000;
          if (x > 0)
          {
            ba[ii - 1] = true;
          }
          if (x < dx - 1)
          {
            ba[ii + 1] = true;
          }
          if (y > 0)
          {
            ba[ii - dx] = true;
          }
          if (y < dy - 1)
          {
            ba[ii + dx] = true;
          }
        }
        /*
        else
        {
          // ba[ii] = false;
        }
        */
      }
    }

    int[] ca = new int[dy * dx];
    for (int y = 0; y < dy; y++)
    {
      for (int x = 0; x < dx; x++)
      {
        int argb = img.getRGB(x1 + x, y1 + y);
        if (ba[y * dx + x])
        {
          ca[y * dx + x] = 0xff000000;
          // Punkte zusammensuchen, aus denen die neue Farbe berechnet wird
          int asum = 0;
          int rsum = 0;
          int gsum = 0;
          int bsum = 0;
          int cnt = 0;
          for (int yy = -circ; yy <= circ; yy++)
          {
            int dxx = circ - Math.abs(yy);
            for (int xx = -dxx; xx <= dxx; xx++)
            {
              if ((yy != 0) || (xx != 0))
              {
                if ((x + xx < 0) || (x + xx >= dx) || (y + yy < 0) || (y + yy >= dy) || !ba[(y + yy) * dx + (x + xx)] /* || !checkColor(argbx, a1, a2, r1, r2, g1, g2, b1, b2) */)
                {
                  int argbx = img.getRGB(x1 + x + xx, y1 + y + yy);
                  int ax = (argbx >> 24) & 0xff;
                  int rx = (argbx >> 16) & 0xff;
                  int gx = (argbx >> 8) & 0xff;
                  int bx = argbx & 0xff;
                  int f = 2 * circ - Math.abs(xx) - Math.abs(yy) + 1;
                  asum += ax * f;
                  rsum += rx * f;
                  gsum += gx * f;
                  bsum += bx * f;
                  cnt += f;
                }
              }
            }
          }
          if (cnt != 0)
          {
            ca[y * dx + x] = ((asum / cnt) << 24) + ((rsum / cnt) << 16) + ((gsum / cnt) << 8) + (bsum / cnt);
          }
          else
          {
            ca[y * dx + x] = argb;
          }
        }
        else
        {
          ca[y * dx + x] = argb;
        }
      }
    }
    /*
    for (int x = 0; x < dx; x++)
    {
      ca[0 * dx + x] = 0xffffffff;
      ca[(dy - 1) * dx + x] = 0xffffffff;
    }
    */
    img.setRGB(x1, y1, dx, dy, ca, 0, dx);
    MLogger.deb("killtext: Ok.");
  }

  //---------------------------------------------------------------------------

  /**
   * Zentrale run-Methode zur Abarbeitung einer Datei
   *
   * Folgende Kommandos sind erlaubt:
   *
   * * newimage width height [ color ]     -  neues Bild erzeugen, optional mit Hintergrundfarbe
   * * loadimage filename [ ii ]           -  Bild laden, "ii" = verw. ImageIcon zum Laden
   * * saveimage filename                  -  Bild speichern
   * * duplimage [ i ]                     -  i-tes Element vom Stack duplizieren
   * * zoomimage xx yy                     -  neue Gre
   * * combine x y                         -  oberstes Bild auf darunter liegendes
   * * mboximage width height [ color ]    -  oberstes Bild in neues Bild einpassen
   * * warpimage x0 y0 x1 y1 x2 y2 x3 y3   -  Bild zerren, 4 Punkte ab links oben im Uhrzeigersinn
   * * cropimage x1 y1 x2 y2               -  1: links oben, 2: rechts unten
   * * cutimage                            -  weien Rahmen abschneiden
   * * rotimage deg                        -  drehen um deg Grad
   * * flipimage h | v                     -  spiegeln horizontal / vertikal
   * * addborder x1 y1 x2 y2               -  1: links/oben, 2: rechts/unten (jeweils die Rahmenbreiten)
   * * grayscale
   * * scale ARGB                          -  Farbwerte skalieren (Parameter: 1 Farbwert)
   * * scale4 A R G B                      -  Farbwerte skalieren (Parameter: 4 Einzelfarbwerte)
   * * scalex ARGB                         -  Farbwerte xor-en (Parameter: 1 Farbwert)
   * * scale4x A R G B                     -  Farbwerte xor-en (Parameter: 4 Einzelfarbwerte)
   * * transform A R G B  A R G B  A R G B   A R G B  -  Matrix: A R G B / A R G B / A R G B / A R G B fr A / R / G / B
   * * print x y font size style color text
   * * drawline x1 y1 x2 y2 color          -  1: links oben, 2: rechts unten
   * * drawrect x1 y1 x2 y2 color          -  1: links oben, 2: rechts unten
   * * drawbox x1 y1 x2 y2 color           -  1: links oben, 2: rechts unten
   * * fillrect x1 y1 x2 y2 color          -  wie drawbox
   * * drawoval x1 y1 x2 y2 color          -  1: links oben, 2: rechts unten (gedachtes umschlieendes Rechteck)
   * * filloval x1 y1 x2 y2 color          -  1: links oben, 2: rechts unten (gedachtes umschlieendes Rechteck)
   * * fillimage x y color Tolleranz       -  Farbe und Tolleranz
   * * replcolor oldcolor Tolleranz newcolor - Farbe oldcolor mit Tolleranz durch newcolor ersetzen
   *
   * Komplex-Kommando(s):
   *
   * * killtext x1 y1 x2 y2 color colortoll circle   -  x1, y1, x2, y2, Farbe ARGB, Farbtolleranz ARGB, Umkreisgre
   *
   * Alte - aktuell nicht untersttzte - Kommandos:
   *
   * addshadow   mboximage   scalecomp   swapimages
   * loadfont    zoomfont    savefont    img2font    font2img    printat
   * echo        xdebug      run         runc        runu        sett     waitt
   * sleep       stop        goto        gpff        set
   * copywebext  delwebext   antiflicker gentempfilename
   *
   * @param filename Dateiname
   * @throws Exception im Fehlerfall
   */

  public void run(String filename) throws Exception
  {
    System.setProperty("com.sun.media.jai.disableMediaLib", "true");
    myImages.clear();
    BufferedReader br = new BufferedReader(new InputStreamReader(new FileInputStream(filename)));
    try
    {
      int linenr = 0;
      String line;
      while ((line = br.readLine()) != null)
      {
        try
        {
          linenr++;
          line = line.trim();
          if (!line.startsWith(";") && !line.startsWith("#") && (line.length() > 0))
          {
            StringTokenizer st = new StringTokenizer(line);
            Vector params = new Vector();
            while (st.hasMoreTokens())
            {
              params.add(st.nextToken());
            }
            String cmd = getNextParam(params);
            if ("xdebug".equals(cmd))
            {
              MLogger.deb("xdebug");
            }
            else if ("newimage".equals(cmd))
            {
              newimage(params);
            }
            else if ("loadimage".equals(cmd))
            {
              loadimage(params);
            }
            else if ("saveimage".equals(cmd))
            {
              saveimage(params);
            }
            else if ("duplimage".equals(cmd))
            {
              duplimage(params);
            }
            else if ("zoomimage".equals(cmd))
            {
              zoomimage(params);
            }
            else if ("combine".equals(cmd))
            {
              combine(params);
            }
            else if ("mboximage".equals(cmd))
            {
              mboximage(params);
            }
            else if ("warpimage".equals(cmd))
            {
              warpimage(params);
            }
            else if ("cropimage".equals(cmd))
            {
              cropimage(params);
            }
            else if ("cutimage".equals(cmd))
            {
              cutimage(params);
            }
            else if ("fillimage".equals(cmd))
            {
              fillimage(params);
            }
            else if ("replcolor".equals(cmd))
            {
              replcolor(params);
            }
            else if ("rotimage".equals(cmd))
            {
              rotimage(params);
            }
            else if ("flipimage".equals(cmd))
            {
              flipimage(params);
            }
            else if ("addborder".equals(cmd))
            {
              addborder(params);
            }
            else if ("addshadow".equals(cmd))
            {
              MLogger.deb("addshadow");
              // addshadow(params);
            }
            //-------------------------------------------------------------------
            else if ("grayscale".equals(cmd))
            {
              grayscale(params);
            }
            else if ("scale".equals(cmd))
            {
              scale(params);
            }
            else if ("scale4".equals(cmd))
            {
              scale4(params);
            }
            else if ("scalex".equals(cmd))
            {
              scalex(params);
            }
            else if ("scale4x".equals(cmd))
            {
              scale4x(params);
            }
            else if ("transform".equals(cmd))
            {
              transform(params);
            }
            //-------------------------------------------------------------------
            else if ("print".equals(cmd))
            {
              print(params);
            }
            else if ("drawline".equals(cmd))
            {
              drawline(params);
            }
            else if ("drawrect".equals(cmd))
            {
              drawrect(params);
            }
            else if ("drawbox".equals(cmd))
            {
              drawbox(params);
            }
            else if ("fillrect".equals(cmd))
            {
              fillrect(params);
            }
            else if ("drawoval".equals(cmd))
            {
              drawoval(params);
            }
            else if ("filloval".equals(cmd))
            {
              filloval(params);
            }
            //-------------------------------------------------------------------
            else if ("killtext".equals(cmd))
            {
              killtext(params);
            }
            else if ("goto".equals(cmd))
            {
              String dst = getNextParam(params);
              if (dst != null)
              {
                // MLogger.inf("goto " + dst);
                while ((line = br.readLine()) != null)
                {
                  linenr++;
                  line = line.trim();
                  if (line.equals(":" + dst))
                  {
                    // MLogger.inf("found " + dst);
                    break;
                  }
                }
              }
            }
            else
            {
              MLogger.wrn("Invalid command in line " + linenr + ": " + cmd);
            }
          }
        }
        catch (Exception ex)
        {
          throw new Exception("File: " + filename + " --> error in line " + linenr + "\n" + "Line: " + line, ex);
        }
      }
      MLogger.inf(" " + myImages.size());
    }
    finally
    {
      br.close();
    }
  }
/*
  public static void main(String... args)
  {
    MImgProc ip = new MImgProc();
    try
    {
      ip.loadimage(new Vector<>(Arrays.asList("D:\\Temp\\Governikus\\GeeC.png\\GeeC-001.png")));
      ip.cutimage(new Vector<>());
      ip.rotimage(new Vector<>(Arrays.asList("90")));
      // ip.saveimage(new Vector<>(Arrays.asList("D:\\Temp\\Governikus\\GeeC.png\\GeeC-001-tmp.png")));
      // ip.loadimage(new Vector<>(Arrays.asList("D:\\Temp\\Governikus\\GeeC.png\\GeeC-001-tmp.png")));
      ip.saveimage(new Vector<>(Arrays.asList("D:\\Temp\\Governikus\\GeeC.png\\GeeC-001-cut.png")));
    }
    catch (Exception ex)
    {
      MLogger.err("", ex);
    }
  }
*/
}
