/*
 * Decompiled with CFR 0.152.
 */
package com.staros.manager;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.protobuf.ByteString;
import com.staros.exception.ExceptionCode;
import com.staros.exception.InternalErrorStarException;
import com.staros.exception.InvalidArgumentStarException;
import com.staros.exception.NoAliveWorkersException;
import com.staros.exception.NotExistStarException;
import com.staros.exception.NotLeaderStarException;
import com.staros.exception.StarException;
import com.staros.filestore.FilePath;
import com.staros.filestore.FileStore;
import com.staros.filestore.FileStoreMgr;
import com.staros.heartbeat.HeartbeatManager;
import com.staros.journal.DelegateJournalSystem;
import com.staros.journal.DummyJournalSystem;
import com.staros.journal.Journal;
import com.staros.journal.JournalReplayer;
import com.staros.journal.JournalSystem;
import com.staros.journal.StarMgrJournal;
import com.staros.metrics.MetricsSystem;
import com.staros.proto.CreateMetaGroupInfo;
import com.staros.proto.CreateShardGroupInfo;
import com.staros.proto.CreateShardInfo;
import com.staros.proto.CreateShardJournalInfo;
import com.staros.proto.DeleteShardGroupInfo;
import com.staros.proto.FilePathInfo;
import com.staros.proto.FileStoreInfo;
import com.staros.proto.FileStoreType;
import com.staros.proto.LeaderInfo;
import com.staros.proto.MetaGroupInfo;
import com.staros.proto.MetaGroupJournalInfo;
import com.staros.proto.ReplicaInfo;
import com.staros.proto.ReplicaState;
import com.staros.proto.ReplicaUpdateInfo;
import com.staros.proto.ReplicationType;
import com.staros.proto.SectionType;
import com.staros.proto.ServiceInfo;
import com.staros.proto.ShardGroupInfo;
import com.staros.proto.ShardInfo;
import com.staros.proto.ShardInfoList;
import com.staros.proto.ShardReportInfo;
import com.staros.proto.StarManagerImageMetaFooter;
import com.staros.proto.StarManagerImageMetaHeader;
import com.staros.proto.UpdateMetaGroupInfo;
import com.staros.proto.UpdateShardGroupInfo;
import com.staros.proto.UpdateShardInfo;
import com.staros.proto.UpdateWorkerGroupInfo;
import com.staros.proto.WarmupLevel;
import com.staros.proto.WorkerGroupDetailInfo;
import com.staros.proto.WorkerGroupSpec;
import com.staros.proto.WorkerInfo;
import com.staros.schedule.Scheduler;
import com.staros.schedule.ShardSchedulerV2;
import com.staros.section.Section;
import com.staros.section.SectionReader;
import com.staros.section.SectionWriter;
import com.staros.service.Service;
import com.staros.service.ServiceManager;
import com.staros.service.ServiceTemplate;
import com.staros.shard.Shard;
import com.staros.shard.ShardChecker;
import com.staros.shard.ShardGroup;
import com.staros.shard.ShardManager;
import com.staros.util.Config;
import com.staros.util.IdGenerator;
import com.staros.util.LockCloseable;
import com.staros.util.LogRateLimiter;
import com.staros.util.LogUtils;
import com.staros.util.ReplicaStateComparator;
import com.staros.util.Utils;
import com.staros.worker.Worker;
import com.staros.worker.WorkerGroup;
import com.staros.worker.WorkerManager;
import io.prometheus.metrics.core.datapoints.DistributionDataPoint;
import io.prometheus.metrics.core.datapoints.Timer;
import io.prometheus.metrics.core.metrics.Histogram;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.InetSocketAddress;
import java.nio.charset.StandardCharsets;
import java.security.DigestInputStream;
import java.security.DigestOutputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class StarManager {
    public static final byte[] IMAGE_META_MAGIC_BYTES = "STARMGR1".getBytes(StandardCharsets.UTF_8);
    private static final Logger LOG = LogManager.getLogger(StarManager.class);
    private static final double LOG_LIMIT = 1.0;
    private static final LogRateLimiter logLimiter = new LogRateLimiter(LOG, 1.0);
    private static final Histogram HISTOGRAM_WORKER_HEARTBEAT = MetricsSystem.registerHistogram((String)"starmgr_process_worker_heartbeat", (String)"Histogram stats for starmgr processing worker heartbeat latency", (List)Lists.newArrayList((Object[])new String[]{"service_id"}));
    private AtomicBoolean isLeader;
    private ServiceManager serviceManager;
    private WorkerManager workerManager;
    private ShardSchedulerV2 shardScheduler;
    private ShardChecker shardChecker;
    private HeartbeatManager heartbeatManager;
    private JournalReplayer journalReplayer;
    private JournalSystem journalSystem;
    private IdGenerator idGenerator;
    private InetSocketAddress listenAddress;
    private LeaderInfo leaderInfo = LeaderInfo.newBuilder().build();

    public StarManager() {
        this(new DummyJournalSystem());
    }

    public StarManager(JournalSystem journalSystem) {
        this.journalSystem = new DelegateJournalSystem(journalSystem);
        this.idGenerator = new IdGenerator(this.journalSystem);
        this.isLeader = new AtomicBoolean(false);
        this.serviceManager = new ServiceManager(this.journalSystem, this.idGenerator);
        this.workerManager = new WorkerManager(this.journalSystem, this.idGenerator);
        this.shardScheduler = new ShardSchedulerV2(this.serviceManager, this.workerManager);
        this.shardChecker = new ShardChecker(this.serviceManager, this.workerManager, this.shardScheduler);
        this.heartbeatManager = new HeartbeatManager(this.workerManager);
        this.journalReplayer = new JournalReplayer(this);
        this.serviceManager.setShardScheduler(this.shardScheduler);
    }

    private void checkLeader() throws StarException {
        if (!this.isLeader.get()) {
            String leaderMsg = this.leaderInfo.getHost().isEmpty() ? "unknown leader" : this.leaderInfo.toString();
            throw new NotLeaderStarException(this.leaderInfo, "request rejected, current star manager is not leader. Current leader: " + leaderMsg);
        }
    }

    public void setListenAddressInfo(String address, int port) {
        this.listenAddress = InetSocketAddress.createUnresolved(address, port);
    }

    public void becomeLeader() {
        this.journalSystem.onBecomeLeader();
        this.isLeader.set(true);
        this.startBackgroundThreads();
        this.leaderInfo = this.myLeaderInfo();
        Journal journal = StarMgrJournal.logLeaderInfo(this.leaderInfo);
        this.journalSystem.write(journal);
        LOG.info("star manager background threads start, now is leader.");
    }

    public void becomeFollower() {
        this.journalSystem.onBecomeFollower();
        this.isLeader.set(false);
        this.stopBackgroundThreads();
        LOG.info("star manager background threads stop, now is follower.");
    }

    public void startBackgroundThreads() {
        this.workerManager.start();
        this.shardScheduler.start();
        this.shardChecker.start();
        this.heartbeatManager.start();
    }

    public void stopBackgroundThreads() {
        this.heartbeatManager.stop();
        this.shardChecker.stop();
        this.shardScheduler.stop();
        this.workerManager.stop();
    }

    public void registerService(String serviceTemplateName, List<String> serviceComponents) throws StarException {
        this.checkLeader();
        this.serviceManager.registerService(serviceTemplateName, serviceComponents);
    }

    public void deregisterService(String serviceTemplateName) throws StarException {
        this.checkLeader();
        this.serviceManager.deregisterService(serviceTemplateName);
    }

    public String bootstrapService(String serviceTemplateName, String serviceName) throws StarException {
        this.checkLeader();
        String serviceId = this.serviceManager.bootstrapService(serviceTemplateName, serviceName);
        this.workerManager.bootstrapService(serviceId);
        return serviceId;
    }

    public void shutdownService(String serviceId) throws StarException {
        this.checkLeader();
        this.serviceManager.shutdownService(serviceId);
    }

    public ServiceInfo getServiceInfoById(String serviceId) throws StarException {
        return this.serviceManager.getServiceInfoById(serviceId);
    }

    public ServiceInfo getServiceInfoByName(String serviceName) throws StarException {
        return this.serviceManager.getServiceInfoByName(serviceName);
    }

    public long createWorkerGroup(String serviceId, String owner, WorkerGroupSpec spec, Map<String, String> labels, Map<String, String> properties, int replicaNumber, ReplicationType replicationType, WarmupLevel warmupLevel) throws StarException {
        this.checkLeader();
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new NotExistStarException("service {} not exist.", new Object[]{serviceId});
            }
            long l = this.workerManager.createWorkerGroup(serviceId, owner, spec, labels, properties, replicaNumber, replicationType, warmupLevel);
            return l;
        }
    }

    public List<WorkerGroupDetailInfo> listWorkerGroups(String serviceId, List<Long> groupIds, Map<String, String> filterLabelsMap, boolean includeWorkersInfo) {
        if (!groupIds.isEmpty() && !filterLabelsMap.isEmpty()) {
            throw new InvalidArgumentStarException("Can only specify group id list or filtering labels but not both!");
        }
        if (!groupIds.isEmpty()) {
            return this.workerManager.listWorkerGroupsById(serviceId, groupIds, includeWorkersInfo);
        }
        return this.workerManager.listWorkerGroups(serviceId, filterLabelsMap, includeWorkersInfo);
    }

    public void deleteWorkerGroup(String serviceId, long groupId) throws StarException {
        this.checkLeader();
        this.workerManager.deleteWorkerGroup(serviceId, groupId);
    }

    public void updateWorkerGroup(String serviceId, long groupId, WorkerGroupSpec spec, Map<String, String> labels, Map<String, String> props, int replicaNumber, ReplicationType replicationType, WarmupLevel warmupLevel) throws StarException {
        this.checkLeader();
        this.workerManager.updateWorkerGroup(serviceId, groupId, spec, labels, props, replicaNumber, replicationType, warmupLevel);
    }

    public long addWorker(String serviceId, long groupId, String ipPort) throws StarException {
        this.checkLeader();
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new NotExistStarException("service {} not exist.", new Object[]{serviceId});
            }
            long l = this.workerManager.addWorker(serviceId, groupId, ipPort);
            return l;
        }
    }

    public void removeWorker(String serviceId, long groupId, long workerId) throws StarException {
        this.checkLeader();
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, String.format("service %s not exist.", serviceId));
            }
            this.workerManager.removeWorker(serviceId, groupId, workerId);
        }
    }

    public WorkerInfo getWorkerInfo(String serviceId, long workerId) throws StarException {
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, String.format("service %s not exist.", serviceId));
            }
            WorkerInfo workerInfo = this.workerManager.getWorkerInfo(workerId);
            if (!workerInfo.getServiceId().equals(serviceId)) {
                throw new StarException(ExceptionCode.INVALID_ARGUMENT, String.format("worker %d not belong to service %s.", workerId, serviceId));
            }
            WorkerInfo workerInfo2 = workerInfo;
            return workerInfo2;
        }
    }

    public WorkerInfo getWorkerInfo(String serviceId, String ipPort) throws StarException {
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, String.format("service %s not exist.", serviceId));
            }
            WorkerInfo workerInfo = this.workerManager.getWorkerInfo(ipPort);
            if (!workerInfo.getServiceId().equals(serviceId)) {
                throw new StarException(ExceptionCode.INVALID_ARGUMENT, String.format("worker %s not belong to service %s.", ipPort, serviceId));
            }
            WorkerInfo workerInfo2 = workerInfo;
            return workerInfo2;
        }
    }

    public List<ShardInfo> createShard(String serviceId, List<CreateShardInfo> createShardInfos) throws StarException {
        this.checkLeader();
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, String.format("service %s not exist.", serviceId));
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            Preconditions.checkNotNull((Object)shardManager);
            FileStoreMgr fsMgr = this.getFileStoreMgr(serviceId);
            List<ShardInfo> list = shardManager.createShard(createShardInfos, fsMgr);
            return list;
        }
    }

    public void deleteShard(String serviceId, List<Long> shardIds) throws StarException {
        this.checkLeader();
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, String.format("service %s not exist.", serviceId));
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            Preconditions.checkNotNull((Object)shardManager);
            shardManager.deleteShard(shardIds);
        }
    }

    public void updateShard(String serviceId, List<UpdateShardInfo> updateShardInfos) throws StarException {
        this.checkLeader();
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, String.format("service %s not exist.", serviceId));
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            Preconditions.checkNotNull((Object)shardManager);
            shardManager.updateShard(updateShardInfos);
        }
    }

    private List<ShardInfo> gatherWorkerInfoForShard(String serviceId, List<ShardInfo> shardInfos, long workerGroupId) throws StarException {
        List<Long> targetWorkerIds = Collections.emptyList();
        if (workerGroupId == 0L && !Config.ENABLE_ZERO_WORKER_GROUP_COMPATIBILITY) {
            try {
                targetWorkerIds = this.workerManager.getDefaultWorkerGroup(serviceId).getAllWorkerIds(true);
            }
            catch (StarException exception) {
                LOG.info("Fail to get default worker group for service:{}, error:{}.", (Object)serviceId, (Object)exception.getMessage());
            }
        } else {
            targetWorkerIds = this.workerManager.getWorkerGroup(serviceId, workerGroupId).getAllWorkerIds(true);
        }
        HashMap<Long, WorkerInfo> workerInfoMap = new HashMap<Long, WorkerInfo>();
        ArrayList<ShardInfo> shardInfosFinal = new ArrayList<ShardInfo>(shardInfos.size());
        for (ShardInfo shardInfo : shardInfos) {
            List<ReplicaInfo> validReplicaInfos = this.filterShardReplicaInfoByState(shardInfo.getReplicaInfoList(), targetWorkerIds);
            ShardInfo.Builder shardInfoBuilder = shardInfo.toBuilder().clearReplicaInfo();
            for (ReplicaInfo replicaInfo : validReplicaInfos) {
                try {
                    long workerId = replicaInfo.getWorkerInfo().getWorkerId();
                    WorkerInfo workerInfo = (WorkerInfo)workerInfoMap.get(workerId);
                    if (workerInfo == null) {
                        workerInfo = this.workerManager.getWorkerInfo(workerId);
                        workerInfoMap.put(workerId, workerInfo);
                    }
                    ReplicaInfo replicaInfoFinal = replicaInfo.toBuilder().setWorkerInfo(workerInfo).build();
                    shardInfoBuilder.addReplicaInfo(replicaInfoFinal);
                }
                catch (NotExistStarException workerId) {
                }
                catch (StarException e) {
                    if (e.getExceptionCode() == ExceptionCode.NOT_EXIST) continue;
                    throw e;
                }
            }
            shardInfosFinal.add(shardInfoBuilder.build());
        }
        return shardInfosFinal;
    }

    private List<ReplicaInfo> filterShardReplicaInfoByState(List<ReplicaInfo> replicaInfoList, List<Long> aliveWorkers) {
        List remainReplicas = replicaInfoList.stream().filter(x -> aliveWorkers.contains(x.getWorkerInfo().getWorkerId())).sorted((c1, c2) -> ReplicaStateComparator.compare(c1.getReplicaState(), c2.getReplicaState())).collect(Collectors.toList());
        ArrayList<ReplicaInfo> finalReplicas = new ArrayList<ReplicaInfo>();
        for (ReplicaInfo replicaInfo : remainReplicas) {
            if (replicaInfo.getReplicaState() == ReplicaState.REPLICA_SCALE_OUT) {
                if (!finalReplicas.isEmpty()) break;
                finalReplicas.add(replicaInfo);
                break;
            }
            finalReplicas.add(replicaInfo);
        }
        return finalReplicas;
    }

    public List<ShardInfo> getShardInfo(String serviceId, List<Long> shardIds, long workerGroupId) throws StarException {
        ShardManager shardManager;
        try (LockCloseable ignored = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, String.format("service %s not exist.", serviceId));
            }
            shardManager = this.getShardManagerNoLock(serviceId);
            Preconditions.checkNotNull((Object)shardManager);
        }
        List<ShardInfo> shardInfoList = this.gatherWorkerInfoForShard(serviceId, shardManager.getShardInfo(shardIds), workerGroupId);
        if (this.checkReplicaAndMayTriggerSchedule(serviceId, shardInfoList, workerGroupId)) {
            shardInfoList = this.gatherWorkerInfoForShard(serviceId, shardManager.getShardInfo(shardIds), workerGroupId);
        }
        return shardInfoList;
    }

    public List<ShardInfoList> listShardInfo(String serviceId, List<Long> groupIds, long workerGroupId, boolean withoutReplicaInfo) throws StarException {
        ShardManager shardManager;
        LockCloseable ignored = new LockCloseable(this.serviceManager.readLock());
        Object object = null;
        try {
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, String.format("service %s not exist.", serviceId));
            }
            shardManager = this.getShardManagerNoLock(serviceId);
            Preconditions.checkNotNull((Object)shardManager);
        }
        catch (Throwable throwable) {
            object = throwable;
            throw throwable;
        }
        finally {
            if (ignored != null) {
                if (object != null) {
                    try {
                        ignored.close();
                    }
                    catch (Throwable throwable) {
                        ((Throwable)object).addSuppressed(throwable);
                    }
                } else {
                    ignored.close();
                }
            }
        }
        ArrayList<ShardInfoList> results = new ArrayList<ShardInfoList>(groupIds.size());
        object = groupIds.iterator();
        while (object.hasNext()) {
            long groupId = (Long)object.next();
            List<Long> singleGroupList = Collections.singletonList(groupId);
            List<ShardInfo> shardInfoList = shardManager.listShardInfo(singleGroupList, withoutReplicaInfo).get(0);
            if (withoutReplicaInfo) {
                results.add(ShardInfoList.newBuilder().addAllShardInfos(shardInfoList).build());
                continue;
            }
            List<ShardInfo> updatedShardInfoList = this.gatherWorkerInfoForShard(serviceId, shardInfoList, workerGroupId);
            if (this.checkReplicaAndMayTriggerSchedule(serviceId, updatedShardInfoList, workerGroupId)) {
                shardInfoList = shardManager.listShardInfo(singleGroupList, withoutReplicaInfo).get(0);
                updatedShardInfoList = this.gatherWorkerInfoForShard(serviceId, shardInfoList, workerGroupId);
            }
            results.add(ShardInfoList.newBuilder().addAllShardInfos(updatedShardInfoList).build());
        }
        return results;
    }

    @VisibleForTesting
    public boolean checkReplicaAndMayTriggerSchedule(String serviceId, List<ShardInfo> shardInfoList, long workerGroupId) throws StarException {
        List<Long> shardToSchedule = shardInfoList.stream().filter(x -> x.getReplicaInfoCount() == 0).map(ShardInfo::getShardId).collect(Collectors.toList());
        if (!shardToSchedule.isEmpty()) {
            if (!this.isLeader.get()) {
                logLimiter.info("Need on-demand schedule {} to workerGroup:{}, but this is NOT the leader", new Object[]{shardToSchedule, workerGroupId});
                this.checkLeader();
                return false;
            }
            try {
                this.shardScheduler.scheduleAddToGroup(serviceId, shardToSchedule, workerGroupId);
            }
            catch (NoAliveWorkersException exception) {
                logLimiter.info("On-demand schedule {} failed. Error: {}", new Object[]{shardToSchedule, exception.getMessage()});
                return false;
            }
            catch (StarException exception) {
                logLimiter.info("On-demand schedule {} to workerGroup:{} failed. Error:{}", new Object[]{shardToSchedule, workerGroupId, exception.getMessage()});
            }
            return true;
        }
        return false;
    }

    public List<ShardGroupInfo> createShardGroup(String serviceId, List<CreateShardGroupInfo> createShardGroupInfos) throws StarException {
        this.checkLeader();
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, String.format("service %s not exist.", serviceId));
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            Preconditions.checkNotNull((Object)shardManager);
            List<ShardGroupInfo> list = shardManager.createShardGroup(createShardGroupInfos);
            return list;
        }
    }

    public void deleteShardGroup(String serviceId, List<Long> groupIds, boolean deleteShards) throws StarException {
        this.checkLeader();
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, String.format("service %s not exist.", serviceId));
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            Preconditions.checkNotNull((Object)shardManager);
            shardManager.deleteShardGroup(groupIds, deleteShards);
        }
    }

    public void updateShardGroup(String serviceId, List<UpdateShardGroupInfo> updateShardGroupInfos) throws StarException {
        this.checkLeader();
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, String.format("service %s not exist.", serviceId));
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            Preconditions.checkNotNull((Object)shardManager);
            shardManager.updateShardGroup(updateShardGroupInfos);
        }
    }

    public Pair<List<ShardGroupInfo>, Long> listShardGroupInfo(String serviceId, boolean includeAnonymousGroup, long startGroupId) throws StarException {
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, String.format("service %s not exist.", serviceId));
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            Preconditions.checkNotNull((Object)shardManager);
            Pair<List<ShardGroupInfo>, Long> pair = shardManager.listShardGroupInfo(includeAnonymousGroup, startGroupId);
            return pair;
        }
    }

    public List<ShardGroupInfo> getShardGroupInfo(String serviceId, List<Long> shardGroupIds) throws StarException {
        ShardManager shardManager;
        try (LockCloseable ignored = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, String.format("service %s not exist.", serviceId));
            }
            shardManager = this.getShardManagerNoLock(serviceId);
            Preconditions.checkNotNull((Object)shardManager);
        }
        return shardManager.getShardGroupInfo(shardGroupIds);
    }

    public MetaGroupInfo createMetaGroup(String serviceId, CreateMetaGroupInfo createMetaGroupInfo) throws StarException {
        this.checkLeader();
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, String.format("service %s not exist.", serviceId));
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            Preconditions.checkNotNull((Object)shardManager);
            MetaGroupInfo metaGroupInfo = shardManager.createMetaGroup(createMetaGroupInfo);
            return metaGroupInfo;
        }
    }

    public void deleteMetaGroup(String serviceId, long metaGroupId) throws StarException {
        this.checkLeader();
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, String.format("service %s not exist.", serviceId));
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            Preconditions.checkNotNull((Object)shardManager);
            shardManager.deleteMetaGroup(metaGroupId);
        }
    }

    public void updateMetaGroup(String serviceId, UpdateMetaGroupInfo updateMetaGroupInfo) throws StarException {
        this.checkLeader();
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, String.format("service %s not exist.", serviceId));
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            Preconditions.checkNotNull((Object)shardManager);
            shardManager.updateMetaGroup(updateMetaGroupInfo);
        }
    }

    public MetaGroupInfo getMetaGroupInfo(String serviceId, long metaGroupId) throws StarException {
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, String.format("service %s not exist.", serviceId));
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            Preconditions.checkNotNull((Object)shardManager);
            MetaGroupInfo metaGroupInfo = shardManager.getMetaGroupInfo(metaGroupId);
            return metaGroupInfo;
        }
    }

    public List<MetaGroupInfo> listMetaGroupInfo(String serviceId) throws StarException {
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, String.format("service %s not exist.", serviceId));
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            Preconditions.checkNotNull((Object)shardManager);
            List<MetaGroupInfo> list = shardManager.listMetaGroupInfo();
            return list;
        }
    }

    public boolean queryMetaGroupStable(String serviceId, long metaGroupId, long workerGroupId) throws StarException {
        ShardManager shardManager;
        List<Long> workerIds;
        try (LockCloseable ignored = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, String.format("service %s not exist.", serviceId));
            }
            workerIds = workerGroupId == 0L ? this.workerManager.getDefaultWorkerGroup(serviceId).getAllWorkerIds(false) : this.workerManager.getWorkerGroup(serviceId, workerGroupId).getAllWorkerIds(false);
            shardManager = this.getShardManagerNoLock(serviceId);
            Preconditions.checkNotNull((Object)shardManager);
        }
        return shardManager.isMetaGroupStable(metaGroupId, workerIds);
    }

    public void processWorkerHeartbeat(String serviceId, long workerId, long startTime, Map<String, String> workerProperties, List<Long> shardIds, List<ShardReportInfo> shardReportInfos) {
        this.checkLeader();
        long currentTime = System.currentTimeMillis();
        this.workerManager.submitAsyncTask(() -> this.processWorkerHeartbeatAsync(serviceId, workerId, startTime, workerProperties, shardIds, shardReportInfos, currentTime));
    }

    @VisibleForTesting
    void processWorkerHeartbeatAsync(String serviceId, long workerId, long startTime, Map<String, String> workerProperties, List<Long> shardIds, List<ShardReportInfo> shardReportInfos, long timestamp) {
        if (!this.isLeader.get()) {
            return;
        }
        DistributionDataPoint dp = (DistributionDataPoint)HISTOGRAM_WORKER_HEARTBEAT.labelValues(new String[]{serviceId});
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());
             Timer timer = dp.startTimer();){
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, String.format("service %s not exist.", serviceId));
            }
            LOG.debug("process heartbeat from worker {} for service {}.", (Object)workerId, (Object)serviceId);
            boolean isRestart = this.workerManager.processWorkerHeartbeat(serviceId, workerId, startTime, shardIds.size(), workerProperties, timestamp);
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            Preconditions.checkNotNull((Object)shardManager);
            if (isRestart) {
                shardManager.scheduleShardsBelongToWorker(workerId);
            }
            shardManager.validateWorkerReportedReplicas(shardIds, workerId);
            this.checkAndUpdateShard(this.serviceManager, serviceId, workerId, shardReportInfos);
        }
        catch (Throwable throwable) {
            LOG.info("Async processing worker heartbeat failed with exception:{}, message: {}", (Object)throwable.getClass().getName(), (Object)throwable.getMessage());
        }
    }

    protected void checkAndUpdateShard(ServiceManager serviceManager, String serviceId, long workerId, List<ShardReportInfo> shardReportInfos) {
        FileStoreMgr fsMgr = serviceManager.getFileStoreMgr(serviceId);
        Preconditions.checkNotNull((Object)fsMgr);
        ShardManager shardManager = serviceManager.getShardManagerNoLock(serviceId);
        Preconditions.checkNotNull((Object)shardManager);
        Worker worker = this.workerManager.getWorker(workerId);
        if (worker == null) {
            throw new NotExistStarException("Worker:{} does not exist", new Object[]{workerId});
        }
        WorkerGroup wg = this.workerManager.getWorkerGroup(serviceId, worker.getGroupId());
        List workerIds = wg.replicationEnabled() ? wg.getAllWorkerIds(false) : Collections.emptyList();
        ArrayList<Long> todoShardIds = new ArrayList<Long>();
        shardReportInfos.forEach(shardReportInfo -> {
            long shardId = shardReportInfo.getShardId();
            Shard shard = shardManager.getShard(shardId);
            if (shard == null) {
                return;
            }
            FilePath filePath = shard.getFilePath();
            if (filePath == null) {
                return;
            }
            FileStore fileStore = fsMgr.getFileStore(filePath.fs.key());
            if (fileStore == null) {
                return;
            }
            if (shardReportInfo.getHashCode() == 0) {
                if (shardReportInfo.getFileStoreVersion() < fileStore.getVersion()) {
                    todoShardIds.add(shardId);
                }
            } else if (shard.hashCodeWithReplicas(workerIds) != shardReportInfo.getHashCode()) {
                todoShardIds.add(shardId);
            }
        });
        if (!todoShardIds.isEmpty()) {
            this.shardScheduler.scheduleAsyncAddToWorker(serviceId, todoShardIds, workerId);
        }
    }

    public FilePathInfo allocateFilePath(String serviceId, FileStoreType fsType, String suffix, String fsKey) {
        this.checkLeader();
        FileStore fs = null;
        FileStoreMgr fsMgr = this.serviceManager.getFileStoreMgr(serviceId);
        fs = fsMgr.allocFileStore(fsKey);
        if (fs == null) {
            throw new InvalidArgumentStarException(String.format("file store with key %s not exists", fsKey));
        }
        FilePath fp = new FilePath(fs, String.format("%s/%s", serviceId, suffix));
        return fp.toProtobuf();
    }

    public String addFileStore(String serviceId, FileStoreInfo fsInfo) throws StarException {
        this.checkLeader();
        String fsKey = fsInfo.getFsKey();
        if (fsKey.isEmpty()) {
            fsKey = UUID.randomUUID().toString();
            fsInfo = fsInfo.toBuilder().setFsKey(fsKey).build();
        }
        FileStoreMgr fsMgr = this.serviceManager.getFileStoreMgr(serviceId);
        FileStore fs = FileStore.fromProtobuf(fsInfo);
        fsMgr.addFileStore(fs);
        return fsKey;
    }

    public void removeFileStore(String serviceId, String fsKey, String fsName) throws StarException {
        this.checkLeader();
        if (fsKey.isEmpty() && fsName.isEmpty()) {
            throw new InvalidArgumentStarException("neither fskey nor fsname is provided");
        }
        FileStoreMgr fsMgr = this.serviceManager.getFileStoreMgr(serviceId);
        if (!fsKey.isEmpty()) {
            fsMgr.removeFileStore(fsKey);
        } else {
            fsMgr.removeFileStoreByName(fsName);
        }
    }

    public List<FileStoreInfo> listFileStore(String serviceId, FileStoreType fsType) throws StarException {
        FileStoreMgr fsMgr = this.serviceManager.getFileStoreMgr(serviceId);
        List<FileStore> fileStores = fsMgr.listFileStore(fsType);
        return fileStores.stream().map(FileStore::toProtobuf).collect(Collectors.toList());
    }

    public void updateFileStore(String serviceId, FileStoreInfo fsInfo) throws StarException {
        this.checkLeader();
        this.serviceManager.updateFileStore(serviceId, fsInfo);
    }

    public FileStoreInfo getFileStore(String serviceId, String fsName, String fsKey) throws StarException {
        FileStore fileStore;
        if (fsKey.isEmpty() && fsName.isEmpty()) {
            throw new InvalidArgumentStarException("neither fskey nor fsname is provided");
        }
        FileStoreMgr fsMgr = this.serviceManager.getFileStoreMgr(serviceId);
        FileStore fileStore2 = fileStore = !fsKey.isEmpty() ? fsMgr.getFileStore(fsKey) : fsMgr.getFileStoreByName(fsName);
        if (fileStore == null) {
            throw new StarException(ExceptionCode.NOT_EXIST, String.format("file store with name %s not exists.", fsName));
        }
        return fileStore.toProtobuf();
    }

    public ShardManager getShardManager(String serviceId) {
        return this.serviceManager.getShardManager(serviceId);
    }

    private ShardManager getShardManagerNoLock(String serviceId) {
        return this.serviceManager.getShardManagerNoLock(serviceId);
    }

    public FileStoreMgr getFileStoreMgr(String serviceId) {
        return this.serviceManager.getFileStoreMgr(serviceId);
    }

    public ServiceManager getServiceManager() {
        return this.serviceManager;
    }

    public WorkerManager getWorkerManager() {
        return this.workerManager;
    }

    public IdGenerator getIdGenerator() {
        return this.idGenerator;
    }

    public HeartbeatManager getHeartbeatManager() {
        return this.heartbeatManager;
    }

    public void replay(Journal journal) {
        this.journalReplayer.replay(journal);
    }

    public void replayRegisterService(ServiceTemplate serviceTemplate) {
        this.serviceManager.replayRegisterService(serviceTemplate);
    }

    public void replayDeregisterService(String serviceTemplateName) {
        this.serviceManager.replayDeregisterService(serviceTemplateName);
    }

    public void replayBootstrapService(Service service) {
        this.serviceManager.replayBootstrapService(service);
        this.workerManager.bootstrapService(service.getServiceId());
    }

    public void replayShutdownService(Service service) {
        this.serviceManager.replayShutdownService(service);
    }

    public void replayCreateShard(String serviceId, CreateShardJournalInfo info) {
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                LogUtils.fatal(LOG, "service {} not exist when replay create shard, should not happen!", serviceId);
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            assert (shardManager != null);
            shardManager.replayCreateShard(info);
        }
    }

    public void replayDeleteShard(String serviceId, List<Long> shardIds) {
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                LogUtils.fatal(LOG, "service {} not exist when replay delete shard, should not happen!", serviceId);
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            assert (shardManager != null);
            shardManager.replayDeleteShard(shardIds);
        }
    }

    public void replayUpdateShard(String serviceId, List<Shard> shards) {
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                LogUtils.fatal(LOG, "service {} not exist when replay update shard, should not happen!", serviceId);
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            assert (shardManager != null);
            shardManager.replayUpdateShard(shards);
        }
    }

    public void replayCreateShardGroup(String serviceId, List<ShardGroup> shardGroups) {
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                LogUtils.fatal(LOG, "service {} not exist when replay create shard group, should not happen!", serviceId);
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            assert (shardManager != null);
            shardManager.replayCreateShardGroup(shardGroups);
        }
    }

    public void replayDeleteShardGroup(String serviceId, DeleteShardGroupInfo info) {
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                LogUtils.fatal(LOG, "service {} not exist when replay delete shard group, should not happen!", serviceId);
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            assert (shardManager != null);
            shardManager.replayDeleteShardGroup(info);
        }
    }

    public void replayUpdateShardGroup(String serviceId, List<ShardGroup> shardGroups) {
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                LogUtils.fatal(LOG, "service {} not exist when replay update shard group, should not happen!", serviceId);
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            assert (shardManager != null);
            shardManager.replayUpdateShardGroup(shardGroups);
        }
    }

    public void replayCreateMetaGroup(String serviceId, MetaGroupJournalInfo info) {
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                LogUtils.fatal(LOG, "service {} not exist when replay create meta group, should not happen!", serviceId);
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            assert (shardManager != null);
            shardManager.replayCreateMetaGroup(info);
        }
    }

    public void replayDeleteMetaGroup(String serviceId, MetaGroupJournalInfo info) {
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                LogUtils.fatal(LOG, "service {} not exist when replay delete meta group, should not happen!", serviceId);
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            assert (shardManager != null);
            shardManager.replayDeleteMetaGroup(info);
        }
    }

    public void replayUpdateMetaGroup(String serviceId, MetaGroupJournalInfo info) {
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                LogUtils.fatal(LOG, "service {} not exist when replay update meta group, should not happen!", serviceId);
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            assert (shardManager != null);
            shardManager.replayUpdateMetaGroup(info);
        }
    }

    public void replayCreateWorkerGroup(String serviceId, WorkerGroup group) {
        try (LockCloseable ignored = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                LogUtils.fatal(LOG, "service {} not exist when replay CreateWorkerGroup, should not happen!", serviceId);
                return;
            }
            this.workerManager.replayCreateWorkerGroup(serviceId, group);
        }
    }

    public void replayDeleteWorkerGroup(String serviceId, long groupId) throws StarException {
        this.workerManager.replayDeleteWorkerGroup(serviceId, groupId);
    }

    public void replayUpdateWorkerGroup(String serviceId, UpdateWorkerGroupInfo info) {
        this.workerManager.replayUpdateWorkerGroup(serviceId, info);
    }

    public void replayAddWorker(String serviceId, Worker worker) {
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                LogUtils.fatal(LOG, "service {} not exist when replay add worker, should not happen!", serviceId);
            }
            this.workerManager.replayAddWorker(serviceId, worker);
        }
    }

    public void replayUpdateWorker(String serviceId, List<Worker> workers) {
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                LogUtils.fatal(LOG, "service {} not exist when replay update worker, should not happen!", serviceId);
            }
            this.workerManager.replayUpdateWorker(serviceId, workers);
        }
    }

    public void replayRemoveWorker(String serviceId, long groupId, long workerId) {
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                LogUtils.fatal(LOG, "service {} not exist when replay remove worker, should not happen!", serviceId);
            }
            this.workerManager.replayRemoveWorker(serviceId, groupId, workerId);
        }
    }

    public void replaySetId(long id) {
        this.idGenerator.setNextId(id);
    }

    public void replayLeaderChange(LeaderInfo info) {
        this.leaderInfo = info.toBuilder().build();
    }

    public void replayAddFileStore(String serviceId, FileStoreInfo fsInfo) {
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                LogUtils.fatal(LOG, "service {} not exist when replay add file store, should not happen!", serviceId);
            }
            FileStoreMgr fsMgr = this.getFileStoreMgr(serviceId);
            assert (fsMgr != null);
            fsMgr.replayAddFileStore(fsInfo);
        }
    }

    public void replayRemoveFileStore(String serviceId, String fsKey) {
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                LogUtils.fatal(LOG, "service {} not exist when replay remove file store, should not happen!", serviceId);
            }
            FileStoreMgr fsMgr = this.getFileStoreMgr(serviceId);
            assert (fsMgr != null);
            fsMgr.replayRemoveFileStore(fsKey);
        }
    }

    public void replayUpdateFileStore(String serviceId, FileStoreInfo fsInfo) {
        this.serviceManager.replayUpdateFileStore(serviceId, fsInfo);
    }

    private LeaderInfo myLeaderInfo() {
        if (this.listenAddress == null) {
            return LeaderInfo.newBuilder().build();
        }
        return LeaderInfo.newBuilder().setHost(this.listenAddress.getHostString()).setPort(this.listenAddress.getPort()).build();
    }

    public void dumpMeta(OutputStream out) throws IOException {
        LOG.info("start dump star manager meta data ...");
        out.write(IMAGE_META_MAGIC_BYTES);
        StarManagerImageMetaHeader rawHeader = StarManagerImageMetaHeader.newBuilder().setGenerateTime(System.currentTimeMillis()).setReplayJournalId(this.journalSystem.getReplayId()).setNextGlobalId(this.idGenerator.getNextPersistentId()).setLeaderInfo(this.leaderInfo).setDigestAlgorithm("MD5").clearChecksum().build();
        byte[] checksum = null;
        try {
            checksum = MessageDigest.getInstance("MD5").digest(rawHeader.toByteArray());
        }
        catch (NoSuchAlgorithmException exception) {
            throw new InternalErrorStarException("Can't calculate checksum. Error:{}", new Object[]{exception.getMessage()});
        }
        StarManagerImageMetaHeader header = StarManagerImageMetaHeader.newBuilder().mergeFrom(rawHeader).setChecksum(ByteString.copyFrom((byte[])checksum)).build();
        header.writeDelimitedTo(out);
        DigestOutputStream mdStream = Utils.getDigestOutputStream(out, "MD5");
        try (SectionWriter writer = new SectionWriter((OutputStream)mdStream);){
            try (OutputStream stream = writer.appendSection(SectionType.SECTION_STARMGR_SERVICEMGR);){
                this.serviceManager.dumpMeta(stream);
            }
            stream = writer.appendSection(SectionType.SECTION_STARMGR_WORKERMGR);
            var9_12 = null;
            try {
                this.workerManager.dumpMeta(stream);
            }
            catch (Throwable throwable) {
                var9_12 = throwable;
                throw throwable;
            }
            finally {
                if (stream != null) {
                    if (var9_12 != null) {
                        try {
                            stream.close();
                        }
                        catch (Throwable throwable) {
                            var9_12.addSuppressed(throwable);
                        }
                    } else {
                        stream.close();
                    }
                }
            }
        }
        mdStream.flush();
        StarManagerImageMetaFooter.Builder footerBuilder = StarManagerImageMetaFooter.newBuilder();
        if (mdStream.getMessageDigest() != null) {
            footerBuilder.setChecksum(ByteString.copyFrom((byte[])mdStream.getMessageDigest().digest()));
        }
        footerBuilder.build().writeDelimitedTo(out);
        LOG.info("end dump star manager meta data.");
    }

    public void loadMeta(InputStream in) throws IOException {
        LOG.info("start load star manager meta data ...");
        int firstByte = in.read();
        if (firstByte == -1) {
            LOG.warn("load star manager meta data found empty stream, do nothing.");
            return;
        }
        int len = IMAGE_META_MAGIC_BYTES.length;
        byte[] magic = new byte[len];
        magic[0] = (byte)firstByte;
        int n = in.read(magic, 1, len - 1);
        Preconditions.checkState((n == len - 1 ? 1 : 0) != 0);
        if (!Arrays.equals(IMAGE_META_MAGIC_BYTES, magic)) {
            throw new IOException("verify star manager meta file raw header failed, meta is not valid.");
        }
        StarManagerImageMetaHeader header = StarManagerImageMetaHeader.parseDelimitedFrom((InputStream)in);
        if (header == null) {
            throw new EOFException();
        }
        ByteString expectedChecksum = header.getChecksum();
        if (expectedChecksum.isEmpty()) {
            throw new IOException("Data integrity check failed. Expect to have checksum in ImageMetaHeader");
        }
        StarManagerImageMetaHeader rawHeader = StarManagerImageMetaHeader.newBuilder().mergeFrom(header).clearChecksum().build();
        try {
            byte[] checksum = MessageDigest.getInstance("MD5").digest(rawHeader.toByteArray());
            if (!expectedChecksum.equals((Object)ByteString.copyFrom((byte[])checksum))) {
                throw new IOException("checksum mismatch");
            }
        }
        catch (NoSuchAlgorithmException exception) {
            LOG.warn("Failed to calculate checksum of the header, error: {}. Ignored for now!", (Object)exception.getMessage());
        }
        long generateTime = header.getGenerateTime();
        long replayJournalId = header.getReplayJournalId();
        this.journalSystem.setReplayId(replayJournalId);
        this.idGenerator.setNextId(header.getNextGlobalId());
        this.leaderInfo = header.getLeaderInfo();
        DigestInputStream digestInput = Utils.getDigestInputStream(in, header.getDigestAlgorithm());
        try (SectionReader reader = new SectionReader((InputStream)digestInput);){
            reader.forEach(this::loadSectionMeta);
        }
        StarManagerImageMetaFooter footer = StarManagerImageMetaFooter.parseDelimitedFrom((InputStream)in);
        if (footer == null) {
            throw new EOFException();
        }
        Utils.validateChecksum(digestInput.getMessageDigest(), footer.getChecksum());
        LOG.info("end load star manager meta data at {}, replay journal id {}.", (Object)new SimpleDateFormat("MM-dd HH:mm:ss").format(new Date(generateTime)), (Object)replayJournalId);
    }

    private void loadSectionMeta(Section section) throws IOException {
        switch (section.getHeader().getSectionType()) {
            case SECTION_STARMGR_WORKERMGR: {
                this.workerManager.loadMeta(section.getStream());
                break;
            }
            case SECTION_STARMGR_SERVICEMGR: {
                this.serviceManager.loadMeta(section.getStream());
                break;
            }
            default: {
                LOG.warn("Unknown section type:{} when loadMeta in StarManager, ignore it!", (Object)section.getHeader().getSectionType());
            }
        }
    }

    public String dump() throws IOException {
        String name = "star_manager_meta";
        File file = new File(name);
        DataOutputStream dos = new DataOutputStream(new FileOutputStream(file));
        this.serviceManager.dump(dos);
        this.workerManager.dump(dos);
        return Config.STARMGR_IP + ":" + file.getAbsolutePath();
    }

    public String getServiceIdByIdOrName(String service) throws StarException {
        String serviceId;
        try {
            ServiceInfo serviceInfo = this.getServiceInfoByName(service);
            serviceId = serviceInfo.getServiceId();
        }
        catch (StarException e) {
            ServiceInfo serviceInfo = this.getServiceInfoById(service);
            serviceId = serviceInfo.getServiceId();
        }
        return serviceId;
    }

    public void removeShardGroupReplicas(String serviceId, Long shardGroupId) throws StarException {
        this.checkLeader();
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, "service {} not exist", new Object[]{serviceId});
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            Preconditions.checkNotNull((Object)shardManager);
            shardManager.removeShardGroupReplicas(shardGroupId);
        }
    }

    @VisibleForTesting
    Scheduler getScheduler() {
        return this.shardScheduler;
    }

    public void updateShardReplicaInfo(String serviceId, long workerId, List<ReplicaUpdateInfo> replicaUpdateInfos) throws StarException {
        this.checkLeader();
        try (LockCloseable lock = new LockCloseable(this.serviceManager.readLock());){
            if (!this.serviceManager.existService(serviceId)) {
                throw new StarException(ExceptionCode.NOT_EXIST, "service {} not exist", new Object[]{serviceId});
            }
            ShardManager shardManager = this.getShardManagerNoLock(serviceId);
            Preconditions.checkNotNull((Object)shardManager);
            shardManager.updateShardReplicaInfo(workerId, replicaUpdateInfos);
        }
    }
}

