/*
 * Decompiled with CFR 0.152.
 */
package com.staros.filestore;

import com.google.protobuf.ByteString;
import com.google.protobuf.MessageOrBuilder;
import com.google.protobuf.util.JsonFormat;
import com.staros.credential.ADLS2Credential;
import com.staros.credential.AwsCredential;
import com.staros.credential.AwsCredentialMgr;
import com.staros.credential.AzBlobCredential;
import com.staros.exception.AlreadyExistsStarException;
import com.staros.exception.ExceptionCode;
import com.staros.exception.InvalidArgumentStarException;
import com.staros.exception.NotExistStarException;
import com.staros.exception.StarException;
import com.staros.filestore.ADLS2FileStore;
import com.staros.filestore.AzBlobFileStore;
import com.staros.filestore.FileStore;
import com.staros.filestore.HDFSFileStore;
import com.staros.filestore.S3FileStore;
import com.staros.journal.DummyJournalSystem;
import com.staros.journal.Journal;
import com.staros.journal.JournalSystem;
import com.staros.journal.StarMgrJournal;
import com.staros.proto.FileStoreInfo;
import com.staros.proto.FileStoreMgrImageMetaFooter;
import com.staros.proto.FileStoreMgrImageMetaHeader;
import com.staros.proto.FileStoreType;
import com.staros.proto.SectionType;
import com.staros.section.Section;
import com.staros.section.SectionReader;
import com.staros.section.SectionWriter;
import com.staros.util.Config;
import com.staros.util.Constant;
import com.staros.util.LockCloseable;
import com.staros.util.Utils;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.DigestInputStream;
import java.security.DigestOutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class FileStoreMgr {
    private static final Logger LOG = LogManager.getLogger(FileStoreMgr.class);
    private String serviceId;
    private Map<String, FileStore> fileStores;
    private ReentrantReadWriteLock lock;
    private JournalSystem journalSystem;

    public static FileStoreMgr createFileStoreMgrForTest(String serviceId) {
        return new FileStoreMgr(serviceId, new DummyJournalSystem());
    }

    public FileStoreMgr(String serviceId, JournalSystem journalSystem) {
        this.serviceId = serviceId;
        this.fileStores = new HashMap<String, FileStore>();
        this.lock = new ReentrantReadWriteLock();
        this.journalSystem = journalSystem;
        this.addFileStoresFromConfig();
    }

    public FileStore getFileStore(String fsKey) {
        try (LockCloseable lockCloseable = new LockCloseable((Lock)this.lock.readLock());){
            FileStore fileStore = this.fileStores.get(fsKey);
            return fileStore;
        }
    }

    public void addFileStore(FileStore fs) throws StarException {
        this.addFileStore(fs, false);
    }

    private void addFileStore(FileStore fs, boolean isReplay) throws StarException {
        if (!fs.isValid()) {
            throw new InvalidArgumentStarException("Invalid file store (fsKey: {}), please check", new Object[]{fs.key()});
        }
        try (LockCloseable lockCloseable = new LockCloseable((Lock)this.lock.writeLock());){
            if (this.fileStores.containsKey(fs.key())) {
                throw new AlreadyExistsStarException("File store with key {} already exist", new Object[]{fs.key()});
            }
            if (!isReplay && !fs.isBuiltin()) {
                Journal journal = StarMgrJournal.logAddFileStore(this.serviceId, fs.toProtobuf());
                this.journalSystem.write(journal);
            }
            Utils.executeNoExceptionOrDie(() -> this.fileStores.put(fs.key(), fs));
            LOG.info("Add file store {}", (Object)fs);
        }
    }

    public void removeFileStore(String fsKey) throws StarException {
        this.removeFileStore(fsKey, false);
    }

    private void removeFileStore(String fsKey, boolean isReplay) throws StarException {
        try (LockCloseable lockCloseable = new LockCloseable((Lock)this.lock.writeLock());){
            if (!this.fileStores.containsKey(fsKey)) {
                throw new NotExistStarException("File store with key {} not exist", new Object[]{fsKey});
            }
            FileStore fs = this.fileStores.get(fsKey);
            if (fs.isBuiltin()) {
                throw new InvalidArgumentStarException("Builtin file store can not be removed");
            }
            if (!isReplay) {
                Journal journal = StarMgrJournal.logRemoveFileStore(this.serviceId, fsKey);
                this.journalSystem.write(journal);
            }
            Utils.executeNoExceptionOrDie(() -> this.fileStores.remove(fsKey));
            LOG.info("Remove file store {}", (Object)fs);
        }
    }

    public void removeFileStoreByName(String fsName) throws StarException {
        this.removeFileStoreByName(fsName, false);
    }

    private void removeFileStoreByName(String fsName, boolean isReplay) throws StarException {
        try (LockCloseable lockCloseable = new LockCloseable((Lock)this.lock.writeLock());){
            FileStore fs = this.getFileStoreByName(fsName);
            if (fs == null) {
                throw new NotExistStarException("File store with name {} not exist", new Object[]{fsName});
            }
            if (fs.isBuiltin()) {
                throw new InvalidArgumentStarException("Builtin file store can not be removed");
            }
            if (!isReplay) {
                Journal journal = StarMgrJournal.logRemoveFileStore(this.serviceId, fs.key());
                this.journalSystem.write(journal);
            }
            Utils.executeNoExceptionOrDie(() -> this.fileStores.remove(fs.key()));
            LOG.info("Remove file store {}", (Object)fs);
        }
    }

    public void updateFileStore(FileStore fs) throws StarException {
        this.updateFileStore(fs, false);
    }

    private void updateFileStore(FileStore fs, boolean isReplay) throws StarException {
        try (LockCloseable lockCloseable = new LockCloseable((Lock)this.lock.writeLock());){
            if (!this.fileStores.containsKey(fs.key())) {
                throw new NotExistStarException("File store with key {} not exist", new Object[]{fs.key()});
            }
            if (fs.isBuiltin()) {
                throw new InvalidArgumentStarException("Builtin file store can not be updated");
            }
            if (!isReplay && !fs.isBuiltin()) {
                Journal journal = StarMgrJournal.logUpdateFileStore(this.serviceId, fs.toProtobuf());
                this.journalSystem.write(journal);
            }
            Utils.executeNoExceptionOrDie(() -> {
                FileStore oldFs = this.fileStores.get(fs.key());
                oldFs.mergeFrom(fs);
                oldFs.increaseVersion();
            });
            LOG.info("Update file store {}", (Object)fs);
        }
    }

    public void replaceFileStore(FileStore fs) throws StarException {
        this.replaceFileStore(fs, false);
    }

    private void replaceFileStore(FileStore fs, boolean isReplay) throws StarException {
        try (LockCloseable lockCloseable = new LockCloseable((Lock)this.lock.writeLock());){
            FileStore oldFs = this.fileStores.get(fs.key());
            if (oldFs == null) {
                throw new NotExistStarException("File store with key {} not exist", new Object[]{fs.key()});
            }
            if (oldFs.isBuiltin()) {
                throw new InvalidArgumentStarException("Builtin file store can not be replaced");
            }
            if (!isReplay) {
                Journal journal = StarMgrJournal.logReplaceFileStore(this.serviceId, fs.toProtobuf());
                this.journalSystem.write(journal);
            }
            Utils.executeNoExceptionOrDie(() -> {
                fs.setVersion(oldFs.getVersion());
                fs.increaseVersion();
                this.fileStores.put(fs.key(), fs);
            });
            LOG.info("Replace file store {}", (Object)fs);
        }
    }

    public List<FileStore> listFileStore(FileStoreType fsType) throws StarException {
        try (LockCloseable lockCloseable = new LockCloseable((Lock)this.lock.readLock());){
            if (fsType == Constant.FS_NOT_SET) {
                ArrayList<FileStore> arrayList = new ArrayList<FileStore>(this.fileStores.values());
                return arrayList;
            }
            List<FileStore> list = this.fileStores.values().stream().filter(fileStore -> fileStore.type() == fsType).collect(Collectors.toList());
            return list;
        }
    }

    public FileStore allocFileStore(String fsKey) throws StarException {
        try (LockCloseable lockCloseable = new LockCloseable((Lock)this.lock.readLock());){
            if (this.fileStores.isEmpty()) {
                throw new NotExistStarException("No any file store exist");
            }
            if (fsKey.isEmpty()) {
                Iterator enabledFileStores = this.fileStores.values().stream().filter(fileStore -> fileStore.getEnabled()).iterator();
                if (!enabledFileStores.hasNext()) {
                    throw new NotExistStarException("No any enabled file store exist");
                }
                FileStore fileStore2 = (FileStore)enabledFileStores.next();
                return fileStore2;
            }
            if (!this.fileStores.containsKey(fsKey)) {
                throw new NotExistStarException("file store {} not exists", new Object[]{fsKey});
            }
            FileStore fs = this.fileStores.get(fsKey);
            if (!fs.getEnabled()) {
                throw new InvalidArgumentStarException("file store {} is disabled", new Object[]{fsKey});
            }
            FileStore fileStore3 = fs;
            return fileStore3;
        }
    }

    public void addFileStoresFromConfig() {
        ArrayList<FileStore> fileStores = new ArrayList<FileStore>();
        fileStores.add(this.loadS3FileStoreFromConfig());
        fileStores.add(this.loadHDFSFileStoreFromConfig());
        fileStores.add(this.loadAzBlobFileStoreFromConfig());
        fileStores.add(this.loadADLS2FileStoreFromConfig());
        for (FileStore fileStore : fileStores) {
            if (fileStore == null) continue;
            try {
                this.addFileStore(fileStore);
                LOG.debug("Add hdfs file store with key {} from config", (Object)fileStore.key());
            }
            catch (StarException e) {
                if (e.getExceptionCode() == ExceptionCode.ALREADY_EXIST) {
                    this.updateFileStore(fileStore);
                    continue;
                }
                LOG.info("no default file store configured");
            }
        }
    }

    public FileStore loadAzBlobFileStoreFromConfig() {
        if (Config.AZURE_BLOB_ENDPOINT.isEmpty()) {
            LOG.info("Empty AZURE_BLOB_ENDPOINT configured, skip load");
            return null;
        }
        if (Config.AZURE_BLOB_PATH.isEmpty()) {
            LOG.info("Empty AZURE_BLOB_PATH configured, skip load");
            return null;
        }
        AzBlobCredential credential = new AzBlobCredential(Config.AZURE_BLOB_SHARED_KEY, Config.AZURE_BLOB_SAS_TOKEN, Config.AZURE_BLOB_TENANT_ID, Config.AZURE_BLOB_CLIENT_ID, Config.AZURE_BLOB_CLIENT_SECRET, Config.AZURE_BLOB_CLIENT_CERTIFICATE_PATH, Config.AZURE_BLOB_AUTHORITY_HOST);
        AzBlobFileStore azblob = new AzBlobFileStore(Constant.AZURE_BLOB_FSKEY_FOR_CONFIG, Constant.AZURE_BLOB_FSNAME_FOR_CONFIG, Config.AZURE_BLOB_ENDPOINT, Config.AZURE_BLOB_PATH, credential);
        azblob.setBuiltin(true);
        return azblob;
    }

    public FileStore loadADLS2FileStoreFromConfig() {
        if (Config.AZURE_ADLS2_ENDPOINT.isEmpty()) {
            LOG.info("Empty AZURE_ADLS2_ENDPOINT configured, skip load");
            return null;
        }
        if (Config.AZURE_ADLS2_PATH.isEmpty()) {
            LOG.info("Empty AZURE_ADLS2_PATH configured, skip load");
            return null;
        }
        ADLS2Credential credential = new ADLS2Credential(Config.AZURE_ADLS2_SHARED_KEY, Config.AZURE_ADLS2_SAS_TOKEN, Config.AZURE_ADLS2_TENANT_ID, Config.AZURE_ADLS2_CLIENT_ID, Config.AZURE_ADLS2_CLIENT_SECRET, Config.AZURE_ADLS2_CLIENT_CERTIFICATE_PATH, Config.AZURE_ADLS2_AUTHORITY_HOST);
        ADLS2FileStore adls2 = new ADLS2FileStore(Constant.AZURE_ADLS2_FSKEY_FOR_CONFIG, Constant.AZURE_ADLS2_FSNAME_FOR_CONFIG, Config.AZURE_ADLS2_ENDPOINT, Config.AZURE_ADLS2_PATH, credential);
        adls2.setBuiltin(true);
        return adls2;
    }

    public FileStore loadHDFSFileStoreFromConfig() {
        HDFSFileStore hdfs = new HDFSFileStore(Constant.HDFS_FSKEY_FOR_CONFIG, Constant.HDFS_FSNAME_FOR_CONFIG, Config.HDFS_URL);
        hdfs.setBuiltin(true);
        return hdfs;
    }

    public FileStore loadS3FileStoreFromConfig() {
        AwsCredential credential = AwsCredentialMgr.getCredentialFromConfig();
        if (credential == null) {
            LOG.warn("get credential from config error");
            return null;
        }
        if (Config.S3_BUCKET.isEmpty()) {
            LOG.info("Empty S3_BUCKET configured, skip load");
            return null;
        }
        if (Config.S3_BUCKET.contains("/")) {
            LOG.warn("Invalid S3_BUCKET configuration:{}", (Object)Config.S3_BUCKET);
            throw new InvalidArgumentStarException("Invalid S3_BUCKET configuration:{}", new Object[]{Config.S3_BUCKET});
        }
        S3FileStore s3fs = new S3FileStore(Constant.S3_FSKEY_FOR_CONFIG, Constant.S3_FSNAME_FOR_CONFIG, Config.S3_BUCKET, Config.S3_REGION, Config.S3_ENDPOINT, credential, Config.S3_PATH_PREFIX);
        s3fs.setBuiltin(true);
        return s3fs;
    }

    public FileStore getFileStoreByName(String fsName) {
        try (LockCloseable lockCloseable = new LockCloseable((Lock)this.lock.readLock());){
            for (FileStore fileStore : this.fileStores.values()) {
                if (!fileStore.name().equals(fsName)) continue;
                FileStore fileStore2 = fileStore;
                return fileStore2;
            }
        }
        return null;
    }

    public void replayAddFileStore(FileStoreInfo fsInfo) {
        this.addFileStore(FileStore.fromProtobuf(fsInfo), true);
    }

    public void replayRemoveFileStore(String fsKey) {
        this.removeFileStore(fsKey, true);
    }

    public void replayUpdateFileStore(FileStoreInfo fsInfo) {
        this.updateFileStore(FileStore.fromProtobuf(fsInfo), true);
    }

    public void replayReplaceFileStore(FileStoreInfo fsInfo) {
        this.replaceFileStore(FileStore.fromProtobuf(fsInfo), true);
    }

    public void dumpMeta(OutputStream out) throws IOException {
        try (LockCloseable ignored = new LockCloseable((Lock)this.lock.readLock());){
            LOG.debug("start dump filestore manager meta data ...");
            FileStoreMgrImageMetaHeader header = FileStoreMgrImageMetaHeader.newBuilder().setDigestAlgorithm("MD5").setNumFileStore((int)this.fileStores.values().stream().filter(fileStore -> !fileStore.isBuiltin()).count()).build();
            header.writeDelimitedTo(out);
            DigestOutputStream mdStream = Utils.getDigestOutputStream(out, "MD5");
            try (SectionWriter writer = new SectionWriter((OutputStream)mdStream);
                 OutputStream stream = writer.appendSection(SectionType.SECTION_FILESTOREMGR_FILESTORE);){
                this.dumpFileStores(stream);
            }
            mdStream.flush();
            FileStoreMgrImageMetaFooter.Builder footerBuilder = FileStoreMgrImageMetaFooter.newBuilder();
            if (mdStream.getMessageDigest() != null) {
                footerBuilder.setChecksum(ByteString.copyFrom((byte[])mdStream.getMessageDigest().digest()));
            }
            footerBuilder.build().writeDelimitedTo(out);
            LOG.debug("end dump filestore manager meta data.");
        }
    }

    private void dumpFileStores(OutputStream stream) throws IOException {
        for (FileStore fileStore : this.fileStores.values()) {
            if (fileStore.isBuiltin()) continue;
            fileStore.toProtobuf().writeDelimitedTo(stream);
        }
    }

    public void loadMeta(InputStream in) throws IOException {
        try (LockCloseable ignored = new LockCloseable((Lock)this.lock.writeLock());){
            LOG.debug("start load file store manager meta data ...");
            FileStoreMgrImageMetaHeader header = FileStoreMgrImageMetaHeader.parseDelimitedFrom((InputStream)in);
            if (header == null) {
                throw new EOFException();
            }
            DigestInputStream digestInput = Utils.getDigestInputStream(in, header.getDigestAlgorithm());
            try (SectionReader reader = new SectionReader((InputStream)digestInput);){
                reader.forEach(x -> this.loadFileStoreMgrSection((Section)x, header));
            }
            FileStoreMgrImageMetaFooter footer = FileStoreMgrImageMetaFooter.parseDelimitedFrom((InputStream)in);
            if (footer == null) {
                throw new EOFException();
            }
            Utils.validateChecksum(digestInput.getMessageDigest(), footer.getChecksum());
            LOG.debug("end load file store manager meta data.");
        }
    }

    private void loadFileStoreMgrSection(Section section, FileStoreMgrImageMetaHeader header) throws IOException {
        switch (section.getHeader().getSectionType()) {
            case SECTION_FILESTOREMGR_FILESTORE: {
                this.loadFileStores(section.getStream(), header.getNumFileStore());
                break;
            }
            default: {
                LOG.warn("Unknown section type:{} when loadMeta in FileStoreMgr, ignore it!", (Object)section.getHeader().getSectionType());
            }
        }
    }

    private void loadFileStores(InputStream stream, int numFileStore) throws IOException {
        for (int i = 0; i < numFileStore; ++i) {
            FileStoreInfo fsInfo = FileStoreInfo.parseDelimitedFrom((InputStream)stream);
            if (fsInfo == null) {
                throw new EOFException();
            }
            FileStore fs = FileStore.fromProtobuf(fsInfo);
            if (fs.type() == FileStoreType.INVALID) continue;
            this.addFileStore(fs, true);
        }
    }

    public void dump(DataOutputStream out) throws IOException {
        try (LockCloseable ignored = new LockCloseable((Lock)this.lock.readLock());){
            for (FileStore fileStore : this.fileStores.values()) {
                String s = JsonFormat.printer().print((MessageOrBuilder)fileStore.toDebugProtobuf()) + "\n";
                out.writeBytes(s);
            }
        }
    }

    public void clear() {
        try (LockCloseable ignored = new LockCloseable((Lock)this.lock.readLock());){
            this.fileStores.clear();
        }
    }
}

