/*
 * Decompiled with CFR 0.152.
 */
package com.sleepycat.je.rep.impl.networkRestore;

import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.EnvironmentFailureException;
import com.sleepycat.je.dbi.EnvironmentFailureReason;
import com.sleepycat.je.log.FileManager;
import com.sleepycat.je.rep.impl.RepImpl;
import com.sleepycat.je.rep.impl.networkRestore.FeederManager;
import com.sleepycat.je.rep.impl.networkRestore.Protocol;
import com.sleepycat.je.rep.net.DataChannel;
import com.sleepycat.je.rep.utilint.BinaryProtocol;
import com.sleepycat.je.rep.utilint.NamedChannel;
import com.sleepycat.je.rep.utilint.RepUtils;
import com.sleepycat.je.rep.vlsn.VLSNRange;
import com.sleepycat.je.util.DbBackup;
import com.sleepycat.je.utilint.LogVerifier;
import com.sleepycat.je.utilint.LoggerUtils;
import com.sleepycat.je.utilint.StoppableThread;
import com.sleepycat.je.utilint.VLSN;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.net.SocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedByInterruptException;
import java.nio.channels.FileChannel;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.Objects;
import java.util.logging.Logger;

public class LogFileFeeder
extends StoppableThread {
    private static final int SOCKET_TIMEOUT_MS = 300000;
    static final int TRANSFER_BYTES = 8192;
    private final FeederManager feederManager;
    private final NamedChannel namedChannel;
    private int clientId;
    private DbBackup dbBackup = null;
    final MessageDigest messageDigest;
    private final Logger logger;

    public LogFileFeeder(FeederManager feederManager, DataChannel channel) throws DatabaseException {
        super(feederManager.getEnvImpl(), "Log File Feeder");
        this.feederManager = feederManager;
        this.logger = feederManager.logger;
        this.namedChannel = new NamedChannel(channel, feederManager.nameIdPair);
        try {
            this.messageDigest = MessageDigest.getInstance("SHA1");
        }
        catch (NoSuchAlgorithmException e) {
            LoggerUtils.severe(this.logger, feederManager.getEnvImpl(), "The SHA1 algorithm was not made available by the security provider");
            throw EnvironmentFailureException.unexpectedException(e);
        }
    }

    public void shutdown() {
        if (this.shutdownDone(this.logger)) {
            return;
        }
        this.shutdownThread(this.logger);
        this.feederManager.feeders.remove(this.clientId);
        LoggerUtils.info(this.logger, this.feederManager.getEnvImpl(), "Log file feeder for client:" + this.clientId + " is shutdown.");
    }

    @Override
    protected int initiateSoftShutdown() {
        RepUtils.shutdownChannel(this.namedChannel);
        return 300000;
    }

    @Override
    public void run() {
        Protocol protocol = new Protocol(this.feederManager.nameIdPair, 2, this.feederManager.getEnvImpl());
        try {
            this.configureChannel();
            protocol = this.checkProtocol(protocol);
            this.checkFeeder(protocol);
            this.sendFileList(protocol);
            this.sendRequestedFiles(protocol);
            this.dbBackup.endBackup();
            this.dbBackup = null;
        }
        catch (ClosedByInterruptException e) {
            LoggerUtils.fine(this.logger, this.feederManager.getEnvImpl(), "Ignoring ClosedByInterruptException normal shutdown");
        }
        catch (IOException e) {
            LoggerUtils.warning(this.logger, this.feederManager.getEnvImpl(), " IO Exception: " + e.getMessage());
        }
        catch (BinaryProtocol.ProtocolException e) {
            LoggerUtils.severe(this.logger, this.feederManager.getEnvImpl(), " Protocol Exception: " + e.getMessage());
        }
        catch (Exception e) {
            throw new EnvironmentFailureException(this.feederManager.getEnvImpl(), EnvironmentFailureReason.UNCAUGHT_EXCEPTION, (Throwable)e);
        }
        finally {
            try {
                this.namedChannel.getChannel().close();
            }
            catch (IOException e) {
                LoggerUtils.warning(this.logger, this.feederManager.getEnvImpl(), "Log File feeder io exception on channel close: " + e.getMessage());
            }
            this.shutdown();
            if (this.dbBackup != null) {
                if (this.feederManager.shutdown.get()) {
                    this.dbBackup.endBackup();
                } else {
                    FeederManager feederManager = this.feederManager;
                    Objects.requireNonNull(feederManager);
                    FeederManager.Lease lease = feederManager.new FeederManager.Lease(this.clientId, this.feederManager.leaseDuration, this.dbBackup);
                    LoggerUtils.info(this.logger, this.feederManager.getEnvImpl(), "Lease created for node: " + this.clientId);
                }
            }
            LoggerUtils.info(this.logger, this.feederManager.getEnvImpl(), "Log file feeder for client: " + this.clientId + " exited");
        }
    }

    private void checkFeeder(Protocol protocol) throws IOException, DatabaseException {
        protocol.read(this.namedChannel.getChannel(), Protocol.FeederInfoReq.class);
        int feeders = this.feederManager.getActiveFeederCount() - 1;
        VLSN rangeFirst = VLSN.NULL_VLSN;
        VLSN rangeLast = VLSN.NULL_VLSN;
        if (this.feederManager.getEnvImpl() instanceof RepImpl) {
            RepImpl repImpl = (RepImpl)this.feederManager.getEnvImpl();
            feeders += repImpl.getRepNode().feederManager().activeReplicaCount();
            VLSNRange range = repImpl.getVLSNIndex().getRange();
            rangeFirst = range.getFirst();
            rangeLast = range.getLast();
        }
        Protocol protocol2 = protocol;
        Objects.requireNonNull(protocol2);
        protocol.write((BinaryProtocol.Message)protocol2.new Protocol.FeederInfoResp(feeders, rangeFirst, rangeLast), this.namedChannel);
    }

    private void sendRequestedFiles(Protocol protocol) throws IOException, BinaryProtocol.ProtocolException, DatabaseException {
        String prevFileName = null;
        try {
            while (true) {
                byte[] cachedDigest;
                Protocol.FileReq fileReq = protocol.read(this.namedChannel.getChannel(), Protocol.FileReq.class);
                String fileName = fileReq.getFileName();
                FileManager fMgr = this.feederManager.getEnvImpl().getFileManager();
                File file = new File(fMgr.getFullFileName(fileName));
                if (!file.exists()) {
                    throw EnvironmentFailureException.unexpectedState("Log file not found: " + fileName);
                }
                long length = file.length();
                long lastModified = file.lastModified();
                byte[] digest = null;
                Protocol.FileInfoResp resp = null;
                Protocol.FileInfoResp cachedResp = this.feederManager.statResponses.get(fileName);
                byte[] byArray = cachedDigest = cachedResp != null && cachedResp.getFileLength() == length && cachedResp.getLastModifiedTime() == lastModified ? cachedResp.getDigestSHA1() : null;
                if (fileReq instanceof Protocol.FileInfoReq) {
                    digest = cachedDigest != null ? cachedDigest : (((Protocol.FileInfoReq)fileReq).getNeedSHA1() ? LogFileFeeder.getSHA1Digest(file, length).digest() : new byte[]{});
                    Protocol protocol2 = protocol;
                    Objects.requireNonNull(protocol2);
                    resp = protocol2.new Protocol.FileInfoResp(fileName, length, lastModified, digest);
                } else {
                    if (prevFileName != null && !fileName.equals(prevFileName)) {
                        this.dbBackup.removeFileProtection(prevFileName);
                    }
                    prevFileName = fileName;
                    Protocol protocol3 = protocol;
                    Objects.requireNonNull(protocol3);
                    protocol.write((BinaryProtocol.Message)protocol3.new Protocol.FileStart(fileName, length, lastModified), this.namedChannel);
                    digest = this.sendFileContents(file, length);
                    if (cachedDigest != null && !Arrays.equals(cachedDigest, digest)) {
                        throw EnvironmentFailureException.unexpectedState("Inconsistent cached and computed digests");
                    }
                    Protocol protocol4 = protocol;
                    Objects.requireNonNull(protocol4);
                    resp = protocol4.new Protocol.FileEnd(fileName, length, lastModified, digest);
                }
                if (digest.length > 0) {
                    this.feederManager.statResponses.put(fileName, resp);
                }
                protocol.write((BinaryProtocol.Message)resp, this.namedChannel);
            }
        }
        catch (BinaryProtocol.ProtocolException pe) {
            if (pe.getUnexpectedMessage() instanceof Protocol.Done) {
                return;
            }
            throw pe;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static MessageDigest getSHA1Digest(File file, long length) throws IOException, DatabaseException {
        MessageDigest messageDigest = null;
        try {
            messageDigest = MessageDigest.getInstance("SHA1");
        }
        catch (NoSuchAlgorithmException e) {
            throw EnvironmentFailureException.unexpectedException(e);
        }
        try (FileInputStream fileStream = new FileInputStream(file);){
            int readBytes;
            ByteBuffer buffer = ByteBuffer.allocate(8192);
            for (long bytes = length; bytes > 0L; bytes -= (long)readBytes) {
                int readSize = (int)Math.min(8192L, bytes);
                readBytes = fileStream.read(buffer.array(), 0, readSize);
                if (readBytes == -1) {
                    throw new IOException("Premature EOF. Was expecting: " + readSize);
                }
                messageDigest.update(buffer.array(), 0, readBytes);
            }
        }
        return messageDigest;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private byte[] sendFileContents(File file, long length) throws IOException {
        LogVerifier verifier = new LogVerifier(this.feederManager.getEnvImpl(), file.getName(), -1L);
        try (FileInputStream fileStream = new FileInputStream(file);){
            FileChannel fileChannel = fileStream.getChannel();
            this.messageDigest.reset();
            ByteBuffer buffer = ByteBuffer.allocateDirect(8192);
            byte[] array = buffer.hasArray() ? buffer.array() : new byte[8192];
            int transmitBytes = 0;
            while (true) {
                int off;
                buffer.clear();
                if (fileChannel.read(buffer) < 0) break;
                buffer.flip();
                int lim = buffer.limit();
                if (buffer.hasArray()) {
                    off = buffer.arrayOffset();
                } else {
                    off = 0;
                    buffer.get(array, 0, lim);
                    buffer.rewind();
                }
                verifier.verify(array, off, lim);
                this.messageDigest.update(array, off, lim);
                transmitBytes += this.namedChannel.getChannel().write(buffer);
            }
            verifier.verifyAtEof();
            if ((long)transmitBytes != length) {
                String msg = "File length:" + length + " does not match the number of bytes that were transmitted:" + transmitBytes;
                throw new IllegalStateException(msg);
            }
            String msg = String.format("Sent file: %s Length:%,d bytes to client:%d", file, length, this.clientId);
            LoggerUtils.info(this.logger, this.feederManager.getEnvImpl(), msg);
        }
        return this.messageDigest.digest();
    }

    private void sendFileList(Protocol protocol) throws IOException, BinaryProtocol.ProtocolException, DatabaseException {
        protocol.read(this.namedChannel.getChannel(), Protocol.FileListReq.class);
        if (this.dbBackup == null) {
            this.dbBackup = new DbBackup(this.feederManager.getEnvImpl());
            this.dbBackup.setNetworkRestore();
            this.dbBackup.startBackup();
        } else {
            ++this.feederManager.leaseRenewalCount;
        }
        String[] files = this.dbBackup.getLogFilesInBackupSet();
        for (int i = 0; i < files.length; ++i) {
            if (!files[i].contains(File.separator)) continue;
            files[i] = files[i].substring(files[i].indexOf(File.separator) + 1, files[i].length());
        }
        Protocol protocol2 = protocol;
        Objects.requireNonNull(protocol2);
        protocol.write((BinaryProtocol.Message)protocol2.new Protocol.FileListResp(files), this.namedChannel);
    }

    private Protocol checkProtocol(Protocol protocol) throws IOException, BinaryProtocol.ProtocolException {
        LogFileFeeder prev;
        BinaryProtocol.ClientVersion clientVersion = protocol.read(this.namedChannel.getChannel(), BinaryProtocol.ClientVersion.class);
        this.clientId = clientVersion.getNodeId();
        FeederManager.Lease lease = this.feederManager.leases.get(this.clientId);
        if (lease != null) {
            this.dbBackup = lease.terminate();
        }
        if ((prev = this.feederManager.feeders.put(this.clientId, this)) != null) {
            SocketAddress prevFeederAddress = prev.namedChannel.getChannel().getRemoteAddress();
            LoggerUtils.warning(this.logger, this.feederManager.getEnvImpl(), "Log file feeder with client id:" + this.clientId + " already present; originated from " + prevFeederAddress + " new connection originated from:" + this.namedChannel.getChannel().getRemoteAddress());
        }
        if (clientVersion.getVersion() != protocol.getVersion()) {
            String message = "Client requested protocol version: " + clientVersion.getVersion() + " but the server version is " + protocol.getVersion();
            LoggerUtils.warning(this.logger, this.feederManager.getEnvImpl(), message);
        }
        protocol.write((BinaryProtocol.Message)protocol.new BinaryProtocol.ServerVersion(), this.namedChannel);
        return protocol;
    }

    private DataChannel configureChannel() throws IOException {
        LoggerUtils.fine(this.logger, this.feederManager.getEnvImpl(), "Log File Feeder accepted connection from " + this.namedChannel);
        this.namedChannel.getChannel().socket().setSoTimeout(300000);
        this.namedChannel.getChannel().socket().setTcpNoDelay(false);
        return this.namedChannel.getChannel();
    }

    @Override
    protected Logger getLogger() {
        return this.logger;
    }
}

