/*
 * Decompiled with CFR 0.152.
 */
package com.stratio.cassandra.lucene;

import com.google.common.base.Objects;
import com.stratio.cassandra.lucene.IndexException;
import com.stratio.cassandra.lucene.schema.Schema;
import com.stratio.cassandra.lucene.schema.SchemaBuilder;
import java.io.File;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.ColumnDefinition;
import org.apache.cassandra.db.Directories;
import org.apache.lucene.analysis.Analyzer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IndexConfig {
    private static final Logger logger = LoggerFactory.getLogger(IndexConfig.class);
    public static final String SCHEMA_OPTION = "schema";
    public static final String REFRESH_SECONDS_OPTION = "refresh_seconds";
    public static final double DEFAULT_REFRESH_SECONDS = 60.0;
    public static final String DIRECTORY_PATH_OPTION = "directory_path";
    public static final String INDEXES_DIR_NAME = "lucene";
    public static final String RAM_BUFFER_MB_OPTION = "ram_buffer_mb";
    public static final int DEFAULT_RAM_BUFFER_MB = 64;
    public static final String MAX_MERGE_MB_OPTION = "max_merge_mb";
    public static final int DEFAULT_MAX_MERGE_MB = 5;
    public static final String MAX_CACHED_MB_OPTION = "max_cached_mb";
    public static final int DEFAULT_MAX_CACHED_MB = 30;
    public static final String INDEXING_THREADS_OPTION = "indexing_threads";
    public static final int DEFAULT_INDEXING_THREADS = 0;
    public static final String INDEXING_QUEUES_SIZE_OPTION = "indexing_queues_size";
    public static final int DEFAULT_INDEXING_QUEUES_SIZE = 50;
    public static final String EXCLUDED_DATA_CENTERS_OPTION = "excluded_data_centers";
    public static final List<String> DEFAULT_EXCLUDED_DATA_CENTERS = Collections.emptyList();
    private final ColumnDefinition columnDefinition;
    private final CFMetaData metadata;
    private final Map<String, String> options;
    private Schema schema;
    private double refreshSeconds = 60.0;
    private Path path;
    private int ramBufferMB = 64;
    private int maxMergeMB = 5;
    private int maxCachedMB = 30;
    private int indexingThreads = 0;
    private int indexingQueuesSize = 50;
    private List<String> excludedDataCenters = DEFAULT_EXCLUDED_DATA_CENTERS;

    public IndexConfig(CFMetaData metadata, ColumnDefinition columnDefinition) {
        this.metadata = metadata;
        this.columnDefinition = columnDefinition;
        this.options = columnDefinition.getIndexOptions();
        this.parseRefresh();
        this.parseRamBufferMB();
        this.parseMaxMergeMB();
        this.parseMaxCachedMB();
        this.parsePath();
        this.parseIndexingThreads();
        this.parseIndexingQueuesSize();
        this.parseExcludedDataCenters();
        this.parseSchema();
        this.validate();
    }

    public CFMetaData getMetadata() {
        return this.metadata;
    }

    public ColumnDefinition getColumnDefinition() {
        return this.columnDefinition;
    }

    public String getKeyspaceName() {
        return this.columnDefinition.ksName;
    }

    public String getTableName() {
        return this.columnDefinition.cfName;
    }

    public String getIndexName() {
        return this.columnDefinition.getIndexName();
    }

    public String getName() {
        return String.format("%s.%s.%s", this.getKeyspaceName(), this.getTableName(), this.getIndexName());
    }

    public boolean isWide() {
        return this.metadata.clusteringColumns().size() > 0;
    }

    public Schema getSchema() {
        return this.schema;
    }

    public Analyzer getAnalyzer() {
        return this.schema.getAnalyzer();
    }

    public Path getPath() {
        return this.path;
    }

    public List<String> getExcludedDataCenters() {
        return this.excludedDataCenters;
    }

    public double getRefreshSeconds() {
        return this.refreshSeconds;
    }

    public int getRamBufferMB() {
        return this.ramBufferMB;
    }

    public int getMaxMergeMB() {
        return this.maxMergeMB;
    }

    public int getMaxCachedMB() {
        return this.maxCachedMB;
    }

    public int getIndexingThreads() {
        return this.indexingThreads;
    }

    public int getIndexingQueuesSize() {
        return this.indexingQueuesSize;
    }

    private void parseRefresh() {
        String refreshOption = this.options.get(REFRESH_SECONDS_OPTION);
        if (refreshOption != null) {
            try {
                this.refreshSeconds = Double.parseDouble(refreshOption);
            }
            catch (NumberFormatException e) {
                throw new IndexException("'%s' must be a strictly positive double", REFRESH_SECONDS_OPTION);
            }
            if (this.refreshSeconds <= 0.0) {
                throw new IndexException("'%s' must be strictly positive", REFRESH_SECONDS_OPTION);
            }
        } else {
            this.refreshSeconds = 60.0;
        }
    }

    private void parseRamBufferMB() {
        String ramBufferSizeOption = this.options.get(RAM_BUFFER_MB_OPTION);
        if (ramBufferSizeOption != null) {
            try {
                this.ramBufferMB = Integer.parseInt(ramBufferSizeOption);
            }
            catch (NumberFormatException e) {
                throw new IndexException("'%s' must be a strictly positive integer", RAM_BUFFER_MB_OPTION);
            }
            if (this.ramBufferMB <= 0) {
                throw new IndexException("'%s' must be strictly positive", RAM_BUFFER_MB_OPTION);
            }
        }
    }

    private void parseMaxMergeMB() {
        String maxMergeSizeMBOption = this.options.get(MAX_MERGE_MB_OPTION);
        if (maxMergeSizeMBOption != null) {
            try {
                this.maxMergeMB = Integer.parseInt(maxMergeSizeMBOption);
            }
            catch (NumberFormatException e) {
                throw new IndexException("'%s' must be a strictly positive integer", MAX_MERGE_MB_OPTION);
            }
            if (this.maxMergeMB <= 0) {
                throw new IndexException("'%s' must be strictly positive", MAX_MERGE_MB_OPTION);
            }
        }
    }

    private void parseMaxCachedMB() {
        String maxCachedMBOption = this.options.get(MAX_CACHED_MB_OPTION);
        if (maxCachedMBOption != null) {
            try {
                this.maxCachedMB = Integer.parseInt(maxCachedMBOption);
            }
            catch (NumberFormatException e) {
                throw new IndexException("'%s' must be a strictly positive integer", MAX_CACHED_MB_OPTION);
            }
            if (this.maxCachedMB <= 0) {
                throw new IndexException("'%s' must be strictly positive", MAX_CACHED_MB_OPTION);
            }
        }
    }

    private void parseSchema() {
        String schemaOption = this.options.get(SCHEMA_OPTION);
        if (schemaOption != null && !schemaOption.trim().isEmpty()) {
            try {
                this.schema = SchemaBuilder.fromJson(schemaOption).build();
            }
            catch (Exception e) {
                throw new IndexException(e, "'%s' is invalid : %s", SCHEMA_OPTION, e.getMessage());
            }
        } else {
            throw new IndexException("'%s' required", SCHEMA_OPTION);
        }
    }

    private void parsePath() {
        String pathOption = this.options.get(DIRECTORY_PATH_OPTION);
        if (pathOption == null) {
            Directories directories = new Directories(this.metadata);
            String basePath = directories.getDirectoryForNewSSTables().getAbsolutePath();
            this.path = Paths.get(basePath + File.separator + INDEXES_DIR_NAME, new String[0]);
        } else {
            this.path = Paths.get(pathOption, new String[0]);
        }
    }

    private void parseIndexingThreads() {
        String indexPoolNumQueuesOption = this.options.get(INDEXING_THREADS_OPTION);
        if (indexPoolNumQueuesOption != null) {
            try {
                this.indexingThreads = Integer.parseInt(indexPoolNumQueuesOption);
            }
            catch (NumberFormatException e) {
                throw new IndexException("'%s' must be a positive integer", INDEXING_THREADS_OPTION);
            }
        }
    }

    private void parseIndexingQueuesSize() {
        String indexPoolQueuesSizeOption = this.options.get(INDEXING_QUEUES_SIZE_OPTION);
        if (indexPoolQueuesSizeOption != null) {
            try {
                this.indexingQueuesSize = Integer.parseInt(indexPoolQueuesSizeOption);
            }
            catch (NumberFormatException e) {
                throw new IndexException("'%s' must be a strictly positive integer", INDEXING_QUEUES_SIZE_OPTION);
            }
            if (this.indexingQueuesSize <= 0) {
                throw new IndexException("'%s' must be strictly positive", INDEXING_QUEUES_SIZE_OPTION);
            }
        }
    }

    private void parseExcludedDataCenters() {
        String excludedDataCentersOption = this.options.get(EXCLUDED_DATA_CENTERS_OPTION);
        if (excludedDataCentersOption != null) {
            String[] array = excludedDataCentersOption.trim().split(",");
            this.excludedDataCenters = Arrays.asList(array);
        }
    }

    private void validate() {
        try {
            this.schema.validate(this.metadata);
        }
        catch (IndexException e) {
            if (Files.exists(this.path, new LinkOption[0])) {
                logger.error("Index '{}' schema is invalid probably due to schema changes in the indexed table: {}", (Object)this.columnDefinition.getIndexName(), (Object)e.getMessage());
            }
            throw new IndexException(e, "'%s' is invalid : %s", SCHEMA_OPTION, e.getMessage());
        }
    }

    public String toString() {
        return Objects.toStringHelper((Object)this).add(SCHEMA_OPTION, (Object)this.schema).add("refreshSeconds", this.refreshSeconds).add("path", (Object)this.path).add("ramBufferMB", this.ramBufferMB).add("maxMergeMB", this.maxMergeMB).add("maxCachedMB", this.maxCachedMB).add("excludedDataCenters", this.excludedDataCenters).toString();
    }
}

