/*
 * Decompiled with CFR 0.152.
 */
package com.stratio.cassandra.lucene.service;

import com.stratio.cassandra.lucene.IndexConfig;
import com.stratio.cassandra.lucene.service.LuceneIndexMBean;
import com.stratio.cassandra.lucene.service.NoIDFSimilarity;
import java.io.File;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.nio.file.Path;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import javax.management.MBeanException;
import javax.management.ObjectName;
import javax.management.OperationsException;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.IndexWriterConfig;
import org.apache.lucene.index.SortingMergePolicy;
import org.apache.lucene.index.Term;
import org.apache.lucene.index.TieredMergePolicy;
import org.apache.lucene.index.TrackingIndexWriter;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.ControlledRealTimeReopenThread;
import org.apache.lucene.search.EarlyTerminatingSortingCollector;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.SearcherFactory;
import org.apache.lucene.search.SearcherManager;
import org.apache.lucene.search.Sort;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.search.TopFieldCollector;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.NRTCachingDirectory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LuceneIndex
implements LuceneIndexMBean {
    private static final Logger logger = LoggerFactory.getLogger(LuceneIndex.class);
    private final Path path;
    private final String name;
    private final Sort mergeSort;
    private final Directory directory;
    private final IndexWriter indexWriter;
    private final SearcherManager searcherManager;
    private final ControlledRealTimeReopenThread<IndexSearcher> searcherReopener;
    private ObjectName objectName;

    public LuceneIndex(IndexConfig config, Sort mergeSort) throws IOException {
        this.path = config.getPath();
        this.name = config.getName();
        this.mergeSort = mergeSort;
        FSDirectory fsDirectory = FSDirectory.open(this.path);
        this.directory = new NRTCachingDirectory(fsDirectory, config.getMaxMergeMB(), config.getMaxCachedMB());
        TieredMergePolicy tieredMergePolicy = new TieredMergePolicy();
        SortingMergePolicy sortingMergePolicy = new SortingMergePolicy(tieredMergePolicy, mergeSort);
        IndexWriterConfig indexWriterConfig = new IndexWriterConfig(config.getAnalyzer());
        indexWriterConfig.setRAMBufferSizeMB(config.getRamBufferMB());
        indexWriterConfig.setOpenMode(IndexWriterConfig.OpenMode.CREATE_OR_APPEND);
        indexWriterConfig.setUseCompoundFile(true);
        indexWriterConfig.setMergePolicy(sortingMergePolicy);
        this.indexWriter = new IndexWriter(this.directory, indexWriterConfig);
        SearcherFactory searcherFactory = new SearcherFactory(){

            @Override
            public IndexSearcher newSearcher(IndexReader reader, IndexReader previousReader) {
                logger.debug("{} creating new index searcher", (Object)LuceneIndex.this.name);
                IndexSearcher searcher = new IndexSearcher(reader);
                searcher.setSimilarity(new NoIDFSimilarity());
                return searcher;
            }
        };
        TrackingIndexWriter trackingIndexWriter = new TrackingIndexWriter(this.indexWriter);
        this.searcherManager = new SearcherManager(this.indexWriter, true, searcherFactory);
        this.searcherReopener = new ControlledRealTimeReopenThread<IndexSearcher>(trackingIndexWriter, this.searcherManager, config.getRefreshSeconds(), config.getRefreshSeconds());
        this.searcherReopener.start();
        try {
            this.objectName = new ObjectName(String.format("com.stratio.cassandra.lucene:type=LuceneIndexes,keyspace=%s,table=%s,index=%s", config.getKeyspaceName(), config.getTableName(), this.name));
            ManagementFactory.getPlatformMBeanServer().registerMBean(this, this.objectName);
        }
        catch (MBeanException | OperationsException e) {
            logger.error("Error while registering MBean", (Throwable)e);
        }
    }

    public void upsert(Term term, Document document) throws IOException {
        logger.debug("{} update document {} with term {}", new Object[]{this.name, document, term});
        this.indexWriter.updateDocument(term, document);
    }

    public void upsert(Map<Term, Document> documents) throws IOException {
        for (Map.Entry<Term, Document> entry : documents.entrySet()) {
            this.upsert(entry.getKey(), entry.getValue());
        }
    }

    public void delete(Term term) throws IOException {
        logger.debug("{} delete by term {}", (Object)this.name, (Object)term);
        this.indexWriter.deleteDocuments(term);
    }

    public void delete(Query query) throws IOException {
        logger.debug("{} deleting by query {}", (Object)this.name, (Object)query);
        this.indexWriter.deleteDocuments(query);
    }

    public void truncate() throws IOException {
        this.indexWriter.deleteAll();
        this.indexWriter.commit();
        logger.info("{} truncated", (Object)this.name);
    }

    @Override
    public void commit() throws IOException {
        this.indexWriter.commit();
        logger.debug("{} committed", (Object)this.name);
    }

    public void close() throws IOException {
        this.searcherReopener.interrupt();
        this.searcherManager.close();
        this.indexWriter.close();
        this.directory.close();
        try {
            ManagementFactory.getPlatformMBeanServer().unregisterMBean(this.objectName);
        }
        catch (MBeanException | OperationsException e) {
            logger.error("Error while removing MBean", (Throwable)e);
        }
        logger.info("{} closed", (Object)this.name);
    }

    public void delete() throws IOException {
        this.close();
        FileUtils.deleteRecursive((File)this.path.toFile());
        logger.info("{} removed", (Object)this.name);
    }

    public SearcherManager getSearcherManager() {
        return this.searcherManager;
    }

    public Map<Document, ScoreDoc> search(IndexSearcher searcher, Query query, Sort sort, ScoreDoc after, Integer count, Set<String> fields) throws IOException {
        TopDocs topDocs;
        if (after == null && EarlyTerminatingSortingCollector.canEarlyTerminate(sort, this.mergeSort)) {
            TopFieldCollector collector = TopFieldCollector.create(sort, count, null, true, false, false);
            searcher.search(query, new EarlyTerminatingSortingCollector(collector, sort, count, this.mergeSort));
            topDocs = collector.topDocs();
        } else {
            topDocs = searcher.searchAfter(after, query, count, sort.rewrite(searcher));
        }
        LinkedHashMap<Document, ScoreDoc> searchResults = new LinkedHashMap<Document, ScoreDoc>();
        for (ScoreDoc scoreDoc : topDocs.scoreDocs) {
            Document document = searcher.doc(scoreDoc.doc, fields);
            searchResults.put(document, scoreDoc);
        }
        return searchResults;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public long getNumDocs() throws IOException {
        logger.debug("{} get num docs", (Object)this.name);
        IndexSearcher searcher = (IndexSearcher)this.searcherManager.acquire();
        try {
            long l = searcher.getIndexReader().numDocs();
            return l;
        }
        finally {
            this.searcherManager.release(searcher);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public long getNumDeletedDocs() throws IOException {
        logger.debug("{} get num deleted docs", (Object)this.name);
        IndexSearcher searcher = (IndexSearcher)this.searcherManager.acquire();
        try {
            long l = searcher.getIndexReader().numDeletedDocs();
            return l;
        }
        finally {
            this.searcherManager.release(searcher);
        }
    }

    @Override
    public void forceMerge(int maxNumSegments, boolean doWait) throws IOException {
        logger.info("{} merging index segments to {}", (Object)this.name, (Object)maxNumSegments);
        this.indexWriter.forceMerge(maxNumSegments, doWait);
        this.indexWriter.commit();
        logger.info("{} segments merge completed", (Object)this.name);
    }

    @Override
    public void forceMergeDeletes(boolean doWait) throws IOException {
        logger.info("{} merging index segments with deletions", (Object)this.name);
        this.indexWriter.forceMergeDeletes(doWait);
        this.indexWriter.commit();
        logger.info("{} merging index segments with deletions completed", (Object)this.name);
    }

    @Override
    public void refresh() throws IOException {
        logger.debug("{} refreshing index searcher", (Object)this.name);
        this.searcherManager.maybeRefreshBlocking();
    }

    static {
        BooleanQuery.setMaxClauseCount(Integer.MAX_VALUE);
    }
}

