package com.stripe.android.paymentsheet.model

import com.stripe.android.model.PaymentIntent
import com.stripe.android.model.PaymentIntent.ConfirmationMethod.Automatic
import com.stripe.android.model.SetupIntent
import com.stripe.android.model.StripeIntent
import com.stripe.android.model.StripeIntent.Status.Canceled
import com.stripe.android.model.StripeIntent.Status.RequiresCapture
import com.stripe.android.model.StripeIntent.Status.Succeeded
import com.stripe.android.paymentsheet.state.PaymentSheetLoadingException
import com.stripe.android.paymentsheet.state.asPaymentSheetLoadingException

internal fun StripeIntent.validate(): PaymentSheetLoadingException? {
    return runCatching { StripeIntentValidator.requireValid(this) }.exceptionOrNull()?.asPaymentSheetLoadingException
}

/**
 * Validator for [PaymentIntent] or [SetupIntent] instances used in PaymentSheet.
 */
internal object StripeIntentValidator {

    fun requireValid(
        stripeIntent: StripeIntent
    ): StripeIntent {
        val exception = when {
            stripeIntent is PaymentIntent && stripeIntent.confirmationMethod != Automatic -> {
                PaymentSheetLoadingException.InvalidConfirmationMethod(stripeIntent.confirmationMethod)
            }
            stripeIntent is PaymentIntent && stripeIntent.isInTerminalState -> {
                PaymentSheetLoadingException.PaymentIntentInTerminalState(stripeIntent.status)
            }
            stripeIntent is PaymentIntent && (stripeIntent.amount == null || stripeIntent.currency == null) -> {
                PaymentSheetLoadingException.MissingAmountOrCurrency
            }
            stripeIntent is SetupIntent && stripeIntent.isInTerminalState -> {
                PaymentSheetLoadingException.SetupIntentInTerminalState(stripeIntent.status)
            }
            else -> {
                // valid
                null
            }
        }

        if (exception != null) {
            throw exception
        }

        return stripeIntent
    }
}

private val PaymentIntent.isInTerminalState: Boolean
    get() = status in setOf(Canceled, Succeeded, RequiresCapture)

private val SetupIntent.isInTerminalState: Boolean
    get() = status in setOf(Canceled, Succeeded)
