package com.stripe.net;

import com.stripe.Stripe;

public class RequestOptions {
	public static RequestOptions getDefault() {
		return new RequestOptions(Stripe.apiKey, Stripe.apiVersion, null);
	}

	private final String apiKey;
	private final String stripeVersion;
	private final String idempotentRequest;

	private RequestOptions(String apiKey, String stripeVersion, String idempotentRequest) {
		this.apiKey = apiKey;
		this.stripeVersion = stripeVersion;
		this.idempotentRequest = idempotentRequest;
	}

	public String getApiKey() {
		return apiKey;
	}

	public String getStripeVersion() {
		return stripeVersion;
	}

	public String getIdempotentRequest() {
		return idempotentRequest;
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) return true;
		if (o == null || getClass() != o.getClass()) return false;

		RequestOptions that = (RequestOptions) o;

		if (apiKey != null ? !apiKey.equals(that.apiKey) : that.apiKey != null) {
			return false;
		}
		if (idempotentRequest != null ? !idempotentRequest.equals(that.idempotentRequest) : that.idempotentRequest != null) {
			return false;
		}
		if (stripeVersion != null ? !stripeVersion.equals(that.stripeVersion) : that.stripeVersion != null) {
			return false;
		}

		return true;
	}

	@Override
	public int hashCode() {
		int result = apiKey != null ? apiKey.hashCode() : 0;
		result = 31 * result + (stripeVersion != null ? stripeVersion.hashCode() : 0);
		result = 31 * result + (idempotentRequest != null ? idempotentRequest.hashCode() : 0);
		return result;
	}

	public static RequestOptionsBuilder builder() {
		return new RequestOptionsBuilder();
	}

	public RequestOptionsBuilder toBuilder() {
		return new RequestOptionsBuilder().setApiKey(this.apiKey).setStripeVersion(this.stripeVersion);
	}

	public static final class RequestOptionsBuilder {
		private String apiKey;
		private String stripeVersion;
		private String idempotentRequest;

		public RequestOptionsBuilder() {
			this.apiKey = Stripe.apiKey;
			this.stripeVersion = Stripe.apiVersion;
		}

		public String getApiKey() {
			return apiKey;
		}

		public RequestOptionsBuilder setApiKey(String apiKey) {
			this.apiKey = normalizeApiKey(apiKey);
			return this;
		}

		public RequestOptionsBuilder clearApiKey() {
			this.apiKey = null;
			return this;
		}

		public RequestOptionsBuilder setStripeVersion(String stripeVersion) {
			this.stripeVersion = normalizeStripeVersion(stripeVersion);
			return this;
		}

		public RequestOptionsBuilder clearStripeVersion() {
			this.stripeVersion = null;
			return this;
		}

		public RequestOptionsBuilder setIdempotentRequest(String idempotentRequest) {
			this.idempotentRequest = idempotentRequest;
			return this;
		}

		public RequestOptionsBuilder clearIdempotentRequest() {
			this.idempotentRequest = null;
			return this;
		}

		public String getIdempotentRequest() {
			return this.idempotentRequest;
		}

		public RequestOptions build() {
			return new RequestOptions(normalizeApiKey(this.apiKey), normalizeStripeVersion(this.stripeVersion), normalizeIdempotentRequest(this.idempotentRequest));
		}
	}

	private static String normalizeApiKey(String apiKey) {
		// null apiKeys are considered "valid"
		if (apiKey == null) {
			return null;
		}
		String normalized = apiKey.trim();
		if (normalized.isEmpty()) {
			throw new InvalidRequestOptionsException("Empty API key specified!");
		}
		return normalized;
	}

	private static String normalizeStripeVersion(String stripeVersion) {
		// null stripeVersions are considered "valid" and use Stripe.apiVersion
		if (stripeVersion == null) {
			return null;
		}
		String normalized = stripeVersion.trim();
		if (normalized.isEmpty()) {
			throw new InvalidRequestOptionsException("Empty Stripe version specified!");
		}
		return normalized;
	}

	private static String normalizeIdempotentRequest(String idempotentRequest) {
		if (idempotentRequest == null) {
			return null;
		}
		String normalized = idempotentRequest.trim();
		if (normalized.isEmpty()) {
			throw new InvalidRequestOptionsException("Empty Idempotent Request String Specified!");
		}
		if (normalized.length() > 255) {
			throw new InvalidRequestOptionsException(String.format("Idempotent Request length was %d, which is larger than the 255 character maximum!", normalized.length()));
		}
		return normalized;
	}

	public static class InvalidRequestOptionsException extends RuntimeException {
		public InvalidRequestOptionsException(String message) {
			super(message);
		}
	}
}
