package com.structurizr.io.plantuml;

import com.structurizr.model.*;
import com.structurizr.util.StringUtils;
import com.structurizr.view.*;

import java.io.IOException;
import java.io.Writer;
import java.util.*;
import java.util.stream.Collectors;

import static java.lang.String.format;

/**
 * A writer that outputs diagram definitions that can be used to create diagrams
 * using PlantUML (http://plantuml.com/plantuml/).
 *
 * System landscape, system context, container, component, dynamic and deployment diagrams are supported.
 */
public class StructurizrPlantUMLWriter extends PlantUMLWriter {

    /**
     * Creates a new PlantUMLWriter, with some default skin params.
     */
    public StructurizrPlantUMLWriter() {
        addSkinParam("shadowing", "false");
        addSkinParam("arrowFontSize", "10");
        addSkinParam("defaultTextAlignment", "center");
        addSkinParam("wrapWidth", "200");
        addSkinParam("maxMessageSize", "100");
    }

    @Override
    protected void write(ContainerView view, Writer writer) {
        try {
            writeHeader(view, writer);

            view.getElements().stream()
                    .filter(ev -> !(ev.getElement() instanceof Container))
                    .map(ElementView::getElement)
                    .sorted(Comparator.comparing(Element::getName))
                    .forEach(e -> write(view, e, writer, false));

            List<SoftwareSystem> softwareSystems = new ArrayList<>(view.getElements().stream().map(ElementView::getElement).filter(e -> e instanceof Container).map(c -> ((Container)c).getSoftwareSystem()).collect(Collectors.toSet()));
            softwareSystems.sort(Comparator.comparing(Element::getName));

            for (SoftwareSystem softwareSystem : softwareSystems) {
                writer.write(format("package \"%s\\n%s\" {", softwareSystem.getName(), typeOf(view, softwareSystem, true)));
                writer.write(System.lineSeparator());

                view.getElements().stream()
                        .filter(ev -> ev.getElement() instanceof Container && ev.getElement().getParent().equals(softwareSystem))
                        .map(ElementView::getElement)
                        .sorted(Comparator.comparing(Element::getName))
                        .forEach(e -> write(view, e, writer, true));

                writer.write("}");
                writer.write(System.lineSeparator());
            }

            writeRelationships(view, writer);

            writeFooter(writer);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    @Override
    protected void write(ComponentView view, Writer writer) {
        try {
            writeHeader(view, writer);

            view.getElements().stream()
                    .filter(ev -> !(ev.getElement() instanceof Component))
                    .map(ElementView::getElement)
                    .sorted(Comparator.comparing(Element::getName))
                    .forEach(e -> write(view, e, writer, false));

            List<Container> containers = new ArrayList<>(view.getElements().stream().map(ElementView::getElement).filter(e -> e instanceof Component).map(c -> ((Component)c).getContainer()).collect(Collectors.toSet()));
            containers.sort(Comparator.comparing(Element::getName));

            for (Container container : containers) {
                writer.write(format("package \"%s\\n%s\" {", container.getName(), typeOf(view, container, true)));
                writer.write(System.lineSeparator());

                view.getElements().stream()
                        .filter(ev -> ev.getElement() instanceof Component && ev.getElement().getParent().equals(container))
                        .map(ElementView::getElement)
                        .sorted(Comparator.comparing(Element::getName))
                        .forEach(e -> write(view, e, writer, true));

                writer.write("}");
                writer.write(System.lineSeparator());
            }

            writeRelationships(view, writer);

            writeFooter(writer);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    @Override
    protected void write(DynamicView view, Writer writer) {
        try {
            writeHeader(view, writer);

            Set<Element> elements = new LinkedHashSet<>();
            for (RelationshipView relationshipView : view.getRelationships()) {
                elements.add(relationshipView.getRelationship().getSource());
                elements.add(relationshipView.getRelationship().getDestination());
            }

            if (isUseSequenceDiagrams()) {
                elements.forEach(element -> {
                    try {
                        writer.write(format("%s \"%s\\n<size:10>%s</size>\" as %s <<%s>> %s%s",
                                plantumlSequenceType(view, element),
                                element.getName(),
                                typeOf(view, element, true),
                                idOf(element),
                                idOf(element),
                                backgroundOf(view, element),
                                System.lineSeparator()));
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                });

                view.getRelationships().forEach(relationship -> {
                    try {
                        String arrowStart = "-";
                        String arrowEnd = ">";

                        if (relationship.isResponse() != null && relationship.isResponse() == true) {
                            arrowStart = "<-";
                            arrowEnd = "-";
                        }

                        writer.write(
                                format("%s %s[%s]%s %s : %s. %s",
                                        idOf(relationship.getRelationship().getSource()),
                                        arrowStart,
                                        view.getViewSet().getConfiguration().getStyles().findRelationshipStyle(relationship.getRelationship()).getColor(),
                                        arrowEnd,
                                        idOf(relationship.getRelationship().getDestination()),
                                        relationship.getOrder(),
                                        hasValue(relationship.getDescription()) ? relationship.getDescription() : hasValue(relationship.getRelationship().getDescription()) ? relationship.getRelationship().getDescription() : ""
                                )
                        );
                        writer.write(System.lineSeparator());
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                });

            } else {
                elements.forEach(e -> write(view, e, writer, false));

                view.getRelationships().forEach(relationship -> {
                    writeRelationship(view, relationship, writer);
                });
            }

            writeFooter(writer);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    @Override
    protected void write(View view, DeploymentNode deploymentNode, Writer writer, int indent) {
        try {
            if (view.isElementInView(deploymentNode)) {
                writer.write(
                        format("%snode \"%s\\n%s\" <<%s>> as %s {",
                                calculateIndent(indent),
                                deploymentNode.getName() + (deploymentNode.getInstances() > 1 ? " (x" + deploymentNode.getInstances() + ")" : ""),
                                typeOf(view, deploymentNode, true),
                                idOf(deploymentNode),
                                idOf(deploymentNode)
                        )
                );

                writer.write(System.lineSeparator());

                List<DeploymentNode> children = new ArrayList<>(deploymentNode.getChildren());
                children.sort(Comparator.comparing(DeploymentNode::getName));
                for (DeploymentNode child : children) {
                    if (view.isElementInView(child)) {
                        write(view, child, writer, indent + 1);
                    }
                }

                List<InfrastructureNode> infrastructureNodes = new ArrayList<>(deploymentNode.getInfrastructureNodes());
                infrastructureNodes.sort(Comparator.comparing(InfrastructureNode::getName));
                for (InfrastructureNode infrastructureNode : infrastructureNodes) {
                    if (view.isElementInView(infrastructureNode)) {
                        write(view, infrastructureNode, writer, indent + 1);
                    }
                }

                List<ContainerInstance> containerInstances = new ArrayList<>(deploymentNode.getContainerInstances());
                containerInstances.sort(Comparator.comparing(ContainerInstance::getName));
                for (ContainerInstance containerInstance : containerInstances) {
                    if (view.isElementInView(containerInstance)) {
                        write(view, containerInstance, writer, indent + 1);
                    }
                }

                List<SoftwareSystemInstance> softwareSystemInstances = new ArrayList<>(deploymentNode.getSoftwareSystemInstances());
                softwareSystemInstances.sort(Comparator.comparing(SoftwareSystemInstance::getName));
                for (SoftwareSystemInstance softwareSystemInstance : softwareSystemInstances) {
                    if (view.isElementInView(softwareSystemInstance)) {
                        write(view, softwareSystemInstance, writer, indent + 1);
                    }
                }

                writer.write(
                        format("%s}", calculateIndent(indent))
                );
                writer.write(System.lineSeparator());
            }
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    protected void write(View view, Element element, Writer writer, int indent) {
        try {
            String shape = plantUMLShapeOf(view, element);
            if ("actor".equals(shape)) {
                shape = "rectangle";
            }
            String name = element.getName();
            String description = element.getDescription();
            String type = typeOf(view, element, true);

            if (element instanceof StaticStructureElementInstance) {
                StaticStructureElementInstance elementInstance = (StaticStructureElementInstance)element;
                name = elementInstance.getElement().getName();
                description = elementInstance.getElement().getDescription();
                type = typeOf(view, elementInstance.getElement(), true);
                shape = plantUMLShapeOf(view, elementInstance.getElement());
            }

            final String prefix = calculateIndent(indent);
            final String separator = System.lineSeparator();
            final String id = idOf(element);

            writer.write(format("%s%s \"==%s\\n<size:10>%s</size>%s\" <<%s>> as %s%s",
                    prefix,
                    shape,
                    name,
                    type,
                    (StringUtils.isNullOrEmpty(description) ? "" : "\\n\\n" + description),
                    id,
                    id,
                    separator)
            );

        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    @Override
    protected void writeRelationship(View view, RelationshipView relationshipView, Writer writer) {
        String description = "";

        if (!StringUtils.isNullOrEmpty(relationshipView.getOrder())) {
            description = relationshipView.getOrder() + ". ";
        }

        description += (hasValue(relationshipView.getDescription()) ? relationshipView.getDescription() : hasValue(relationshipView.getRelationship().getDescription()) ? relationshipView.getRelationship().getDescription() : "");

        try {
            Relationship relationship = relationshipView.getRelationship();
            RelationshipStyle style = relationshipStyleOf(view, relationship);
            if (style.getDashed() == null) {
                style.setDashed(true);
            }

            String arrowStart;
            String arrowEnd;
            String relationshipStyle = style.getColor();

            if (style.getThickness() != null) {
                relationshipStyle += ",thickness=" + style.getThickness();
            }

            if (relationshipView.isResponse() != null && relationshipView.isResponse()) {
                arrowStart = style.getDashed() ? "<." : "<-";
                arrowEnd = style.getDashed() ? "." : "-";
            } else {
                arrowStart = style.getDashed() ? "." : "-";
                arrowEnd = style.getDashed() ? ".>" : "->";
            }

            // 1 .[#rrggbb,thickness=n].> 2 : "...\n<size:8>...</size>
            writer.write(format("%s %s[%s]%s %s : \"%s%s\"%s",
                    idOf(relationship.getSource()),
                    arrowStart,
                    relationshipStyle,
                    arrowEnd,
                    idOf(relationship.getDestination()),
                    description,
                    (StringUtils.isNullOrEmpty(relationship.getTechnology()) ? "" : "\\n<size:8>[" + relationship.getTechnology() + "]</size>"),
                    System.lineSeparator()
                ));
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    @Override
    protected void writeHeader(View view, Writer writer) throws IOException {
        super.writeHeader(view, writer);

        writer.write("hide stereotype");
        writer.write(System.lineSeparator());

        if (view instanceof DynamicView && isUseSequenceDiagrams()) {
            // do nothing
        } else {
            if (view.getAutomaticLayout() != null) {
                switch (view.getAutomaticLayout().getRankDirection()) {
                    case LeftRight:
                        writer.write("left to right direction");
                        break;
                    default:
                        writer.write("top to bottom direction");
                        break;
                }
                writer.write(System.lineSeparator());
            }
        }

        for (ElementView elementView : view.getElements()) {
            Element element = elementView.getElement();

            String id = idOf(element);

            if (element instanceof StaticStructureElementInstance) {
                element = (( StaticStructureElementInstance)element).getElement();
            }

            String type = plantUMLShapeOf(view, element);
            if ("actor".equals(type)) {
                type = "rectangle"; // the actor shape is not supported in this implementation
            }
            String background = backgroundOf(view, element);
            String stroke = strokeOf(view, element);
            String color = colorOf(view, element);
            Shape shape = shapeOf(view, element);

            if (view instanceof DynamicView && isUseSequenceDiagrams()) {
                type = "sequenceParticipant";
            }

            writer.write(format("skinparam %s<<%s>> {%s", type, id, System.lineSeparator()));
            if (element instanceof DeploymentNode) {
                writer.write(format("  BackgroundColor #ffffff%s", System.lineSeparator()));
            } else {
                writer.write(format("  BackgroundColor %s%s", background, System.lineSeparator()));
            }
            writer.write(format("  FontColor %s%s", color, System.lineSeparator()));
            writer.write(format("  BorderColor %s%s", stroke, System.lineSeparator()));

            if (shape == Shape.RoundedBox) {
                writer.write(format("  roundCorner 20%s", System.lineSeparator()));
            }

            writer.write(format("}%s", System.lineSeparator()));
        }
    }

}