/*
 * Copyright 2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.stvconsultants.easygloss.footnotes;

import java.lang.annotation.Annotation;
import java.util.HashMap;

/**
 * A loose Annotation for matching against other annotations.
 *
 * @author Stephen Connolly
 */
public class Footnote {
    
    private final Annotation annotation;
    private final Class<? extends Annotation> annotationType;
    private final HashMap<String, FootnoteElement> criterion;
    
    /**
     * Creates a new instance of Footnote.
     *
     * @param annotationType the type of Annotation that this footnote will
     * match
     */
    public Footnote(Class<? extends Annotation> annotationType) {
        this.annotation = null;
        this.annotationType = annotationType;
        this.criterion = new HashMap<String, FootnoteElement>();
    }
    
    /**
     * Creates a new instance of Footnote.
     *
     * @param annotation the Annotation that this footnote will match
     */
    public Footnote(Annotation annotation) {
        this.annotation = annotation;
        this.annotationType = annotation.annotationType();
        this.criterion = new HashMap<String, FootnoteElement>();
    }
    
    /**
     * Required by Annotation interface.
     *
     * @return The annotation class
     */
    public Class<? extends Annotation> annotationType() {
        return annotationType;
    }
    
    /**
     * Returns true if the specified object represents an annotation that is
     * logically equivalent to this one. In other words, returns true if the
     * specified object is an instance of the same annotation type as this
     * instance, all of whose members are equal to the corresponding member of
     * this annotation, as defined below:
     * <ul>
     * <li>Two corresponding primitive typed members whose values are x and y
     * are considered equal if x == y, unless their type is float or double.</li>
     * <li>Two corresponding float members whose values are x and y are
     * considered equal if Float.valueOf(x).equals(Float.valueOf(y)). (Unlike
     * the == operator, NaN is considered equal to itself, and 0.0f unequal to
     * -0.0f.)</li>
     * <li>Two corresponding double members whose values are x and y are
     * considered equal if Double.valueOf(x).equals(Double.valueOf(y)).
     * (Unlike the == operator, NaN is considered equal to itself, and 0.0
     * unequal to -0.0.)</li>
     * <li>Two corresponding String, Class, enum, or annotation typed members
     * whose values are x and y are considered equal if x.equals(y). (Note
     * that2 this definition is recursive for annotation typed members.)</li>
     * <li>Two corresponding array typed members x and y are considered equal
     * if Arrays.equals(x, y), for the appropriate overloading of
     * Arrays.equals(long[], long[]).</li>
     * </ul>
     *
     * @param value The object to check for equality
     * @return If the object is logically equivalent to this one.
     */
    public boolean equals(Object value) {
        // we will compare based on key fields only
        final boolean retVal;
        if (this == value) {
            retVal = true;
        } else if (value == null) {
            retVal = false;
        } else if (value.getClass() == this.getClass()) {
            Footnote that = (Footnote)value;
            retVal = ((this.annotation == that.annotation ||
                    (this.annotation != null &&
                    this.annotation.equals(that.annotation)))) &&
                    (this.annotationType == that.annotationType ||
                    (this.annotationType != null &&
                    this.annotationType.equals(that.annotationType))) &&
                    (this.criterion == that.criterion ||
                    (this.criterion != null &&
                    this.criterion.equals(that.criterion)));
        } else {
            retVal = false;
        }
        return retVal;
    }
    
    /**
     * Returns true if the specified object represents an annotation that is
     * logically equivalent to this one. In other words, returns true if the
     * specified object is an instance of the same annotation type as this
     * instance, all of whose members are equal to the corresponding member of
     * this annotation, as defined below:
     * <ul>
     * <li>Two corresponding primitive typed members whose values are x and y
     * are considered equal if x == y, unless their type is float or double.</li>
     * <li>Two corresponding float members whose values are x and y are
     * considered equal if Float.valueOf(x).equals(Float.valueOf(y)). (Unlike
     * the == operator, NaN is considered equal to itself, and 0.0f unequal to
     * -0.0f.)</li>
     * <li>Two corresponding double members whose values are x and y are
     * considered equal if Double.valueOf(x).equals(Double.valueOf(y)).
     * (Unlike the == operator, NaN is considered equal to itself, and 0.0
     * unequal to -0.0.)</li>
     * <li>Two corresponding String, Class, enum, or annotation typed members
     * whose values are x and y are considered equal if x.equals(y). (Note
     * that2 this definition is recursive for annotation typed members.)</li>
     * <li>Two corresponding array typed members x and y are considered equal
     * if Arrays.equals(x, y), for the appropriate overloading of
     * Arrays.equals(long[], long[]).</li>
     * </ul>
     *
     * @param that The annotation to check for matching
     * @return If the annotation matches this one.
     * @since 1.1
     */
    public boolean matches(Annotation that) {
        boolean retVal;
        if (that == null) {
            retVal = false;
        } else if (!that.annotationType().equals(annotationType)) {
            retVal = false;
        } else if (this.annotation != null) {
            // if we were given an exact annotation to match, ensure we match it
            retVal = this.annotation.equals(that);
        } else {
            retVal = true;
            for (FootnoteElement criteria: this.criterion.values()) {
                if (!criteria.compare(that)) {
                    retVal = false;
                    break;
                }
            }
        }
        return retVal;
    }
    
    /**
     * Returns the hash code of this footnote.
     *
     * @return The hash code.
     */
    public int hashCode() {
        int hash = 7;
        hash = 31 * hash + (annotation == null ? 0 : annotation.hashCode());
        hash = 31 * hash + annotationType.hashCode();
        hash = 31 * hash + criterion.hashCode();
        return hash;
    }
    
    /**
     * Generate a string representation of the Footnote.
     * @return The string representation of this instance.
     */
    public String toString() {
        StringBuilder buf = new StringBuilder(this.getClass().getName());
        buf.append("(");
        if (annotation != null){
            buf.append(annotation);
            buf.append(",");
        }
        buf.append("@");
        buf.append(annotationType.getName());
        buf.append("(");
        boolean first = true;
        for (FootnoteElement ele: criterion.values()) {
            if (first) {
                first = false;
            } else {
                buf.append(",");
            }
            buf.append(ele.toString());
        }
        buf.append("))");
        return buf.toString();
    }
    
    /**
     * Add a criterion to the footnote.
     *
     * @param name The element name.
     * @param value The value that the element value must be equal to.
     * @return This instance for method chaining.
     */
    public Footnote with(String name, Object value) {
        FootnoteElement ele;
        boolean add;
        if (criterion.containsKey(name)) {
            ele = criterion.get(name);
            add = false;
        } else {
            ele = new FootnoteElement(name);
            add = true;
        }
        ele.setEquals(value);
        if (add) {
            criterion.put(ele.getName(), ele);
        }
        return this;
    }
    
    /**
     * Add a criterion to the footnote.
     *
     * @param name The element name.
     * @param value A value that the element value must not equal.
     * @return This instance for method chaining.
     */
    public Footnote without(String name, Object value) {
        FootnoteElement ele;
        boolean add;
        if (criterion.containsKey(name)) {
            ele = criterion.get(name);
            add = false;
        } else {
            ele = new FootnoteElement(name);
            add = true;
        }
        ele.addNotEqual(value);
        if (add) {
            criterion.put(ele.getName(), ele);
        }
        return this;
    }
    
    /**
     * Add a criterion to the footnote.
     *
     * @param name The element name.
     * @param value A value that the element array value must contain.
     * @return This instance for method chaining.
     */
    public Footnote including(String name, Object value) {
        FootnoteElement ele;
        boolean add;
        if (criterion.containsKey(name)) {
            ele = criterion.get(name);
            add = false;
        } else {
            ele = new FootnoteElement(name);
            add = true;
        }
        ele.addContains(value);
        if (add) {
            criterion.put(ele.getName(), ele);
        }
        return this;
    }
    
    /**
     * Add a criterion to the footnote.
     *
     * @param name The element name.
     * @param value A value that the element array value must not contain.
     * @return This instance for method chaining.
     */
    public Footnote excluding(String name, Object value) {
        FootnoteElement ele;
        boolean add;
        if (criterion.containsKey(name)) {
            ele = criterion.get(name);
            add = false;
        } else {
            ele = new FootnoteElement(name);
            add = true;
        }
        ele.addExcludes(value);
        if (add) {
            criterion.put(ele.getName(), ele);
        }
        return this;
    }
    
    /**
     * Add a criterion to the footnote.
     *
     * @param name The element name.
     * @param regex The regex that the element value must match.
     * @return This instance for method chaining.
     */
    public Footnote matching(String name, String regex) {
        FootnoteElement ele;
        boolean add;
        if (criterion.containsKey(name)) {
            ele = criterion.get(name);
            add = false;
        } else {
            ele = new FootnoteElement(name);
            add = true;
        }
        ele.setRegex(regex);
        if (add) {
            criterion.put(ele.getName(), ele);
        }
        return this;
    }
    
}
