/*
 * Copyright 2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.stvconsultants.easygloss;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.List;

import com.stvconsultants.easygloss.footnotes.Footnote;

/**
 * <code>AbstractGloss</code> is the base class for all glosses.
 *
 * @author Stephen Connolly
 */
public abstract class AbstractGloss {

    /**
     * Creates a new instance of AbstractGloss
     */
    protected AbstractGloss() {
        super();
    }

    private static void internalApply(Footnote annotation, Class instanceClass,
                                      Class injectionClass, List<Exception> exceptions,
                                      List<Field> fields, List<Method> setters) {
        if (annotation == null) {
            throw new NullPointerException("Annotation cannot be null");
        }
        instanceClass.getClass(); // throw NPE if null

        // get any matching annotated fields
        fields.clear();
        fields.addAll(filter(findFields(instanceClass, annotation),
                injectionClass));

        // now look for matching annotated methods
        List<Method> methods = findMethods(instanceClass, annotation);

        // finding the annotated setters is easy
        setters.clear();
        setters.addAll(filterSetters(methods, injectionClass));

        // if the getter was annotated, we have a bit more work
        List<Method> getters = filterGetters(methods, injectionClass);
        for (Method getter : getters) {
            Method setter = findSetter(instanceClass, getter);
            if (setter == null) {
                exceptions.add(new NoSuchMethodException(
                        new StringBuilder().append("Cannot find corresponding setter for ").append(getter.getName())
                                .append(" in class ").append(instanceClass.getName()).toString()));
            } else {
                setters.add(setter);
            }
        }
    }

    private static void rethrowExceptions(List<Exception> exceptions) {
        if (!exceptions.isEmpty()) {
            StringBuilder message = new StringBuilder();
            String prefix = "[";
            for (Exception e : exceptions) {
                message.append(prefix);
                prefix = ", ";
                message.append(e);
            }
            message.append("]");
            throw new GlossApplicationError(message.toString());
        }
    }

    /**
     * Apply the specified gloss to the static class.  Tries to inject any annotated static fields, properties or setter
     * methods with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation    The annotation to match
     * @param instanceClass The instance to search
     * @param injection     The value to inject
     *
     * @since 1.2
     */
    public static void applyStatic(Footnote annotation, Class instanceClass,
                                   Object injection) {
        if (injection == null) {
            throw new NullPointerException("Cannot inject a null value");
        }
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }

        List<Exception> exceptions = new ArrayList<Exception>();
        List<Field> fields = new ArrayList<Field>();
        List<Method> setters = new ArrayList<Method>();

        internalApply(annotation, instanceClass, injection.getClass(), exceptions,
                fields, setters);

        for (Field field : fields) {
            if (Modifier.isStatic(field.getModifiers())) {
                try {
                    if (!field.isAccessible()) {
                        field.setAccessible(true);
                    }
                    field.set(null, injection);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        for (Method setter : setters) {
            if (Modifier.isStatic(setter.getModifiers())) {
                try {
                    if (!setter.isAccessible()) {
                        setter.setAccessible(true);
                    }
                    setter.invoke(null, injection);
                } catch (InvocationTargetException e) {
                    exceptions.add(e);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        rethrowExceptions(exceptions);
    }

    /**
     * Apply the specified gloss to the static class.  Tries to inject any annotated static fields, properties or setter
     * methods with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation    The annotation to match
     * @param instanceClass The instance to search
     * @param injection     The value to inject
     *
     * @since 1.2
     */
    public static void applyStatic(Annotation annotation, Class instanceClass,
                                   Object injection) {
        if (annotation == null) {
            throw new NullPointerException();
        }
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        applyStatic(new Footnote(annotation), instanceClass, injection);
    }

    /**
     * Apply the specified gloss to the static class.  Tries to inject any annotated static fields, properties or setter
     * methods with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotationType The annotation class to match
     * @param instanceClass  The instance to search
     * @param injection      The value to inject
     *
     * @since 1.2
     */
    public static void applyStatic(Class<? extends Annotation> annotationType,
                                   Class instanceClass, Object injection) {
        if (annotationType == null) {
            throw new NullPointerException();
        }
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        applyStatic(new Footnote(annotationType), instanceClass, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation The annotation to match
     * @param instance   The instance to search
     * @param injection  The value to inject
     */
    public static void apply(Footnote annotation, Object instance,
                             Object injection) {
        if (injection == null) {
            throw new NullPointerException("Cannot inject a null value");
        }
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }

        List<Exception> exceptions = new ArrayList<Exception>();
        List<Field> fields = new ArrayList<Field>();
        List<Method> setters = new ArrayList<Method>();

        internalApply(annotation, instance.getClass(), injection.getClass(), exceptions,
                fields, setters);

        for (Field field : fields) {
            if (!Modifier.isStatic(field.getModifiers())) {
                try {
                    if (!field.isAccessible()) {
                        field.setAccessible(true);
                    }
                    field.set(instance, injection);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        for (Method setter : setters) {
            if (!Modifier.isStatic(setter.getModifiers())) {
                try {
                    if (!setter.isAccessible()) {
                        setter.setAccessible(true);
                    }
                    setter.invoke(instance, injection);
                } catch (InvocationTargetException e) {
                    exceptions.add(e);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        rethrowExceptions(exceptions);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation The annotation to match
     * @param instance   The instance to search
     * @param injection  The value to inject
     *
     * @since 1.1
     */
    public static void apply(Annotation annotation, Object instance,
                             Object injection) {
        if (annotation == null) {
            throw new NullPointerException();
        }
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        apply(new Footnote(annotation), instance, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotationType The annotation class to match
     * @param instance       The instance to search
     * @param injection      The value to inject
     *
     * @since 1.1
     */
    public static void apply(Class<? extends Annotation> annotationType,
                             Object instance, Object injection) {
        if (annotationType == null) {
            throw new NullPointerException();
        }
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        apply(new Footnote(annotationType), instance, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation The annotation to match
     * @param instance   The instance to search
     * @param injection  The value to inject
     */
    public static void apply(Footnote annotation, Object instance,
                             int injection) {
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        List<Exception> exceptions = new ArrayList<Exception>();
        List<Field> fields = new ArrayList<Field>();
        List<Method> setters = new ArrayList<Method>();

        internalApply(annotation, instance.getClass(), int.class, exceptions,
                fields, setters);

        for (Field field : fields) {
            if (!Modifier.isStatic(field.getModifiers())) {
                try {
                    if (!field.isAccessible()) {
                        field.setAccessible(true);
                    }
                    field.setInt(instance, injection);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        for (Method setter : setters) {
            if (!Modifier.isStatic(setter.getModifiers())) {
                try {
                    if (!setter.isAccessible()) {
                        setter.setAccessible(true);
                    }
                    setter.invoke(instance, injection);
                } catch (InvocationTargetException e) {
                    exceptions.add(e);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        rethrowExceptions(exceptions);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation The annotation to match
     * @param instance   The instance to search
     * @param injection  The value to inject
     *
     * @since 1.1
     */
    public static void apply(Annotation annotation, Object instance,
                             int injection) {
        if (annotation == null) {
            throw new NullPointerException();
        }
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        apply(new Footnote(annotation), instance, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotationType The annotation class to match
     * @param instance       The instance to search
     * @param injection      The value to inject
     *
     * @since 1.1
     */
    public static void apply(Class<? extends Annotation> annotationType,
                             Object instance, int injection) {
        if (annotationType == null) {
            throw new NullPointerException();
        }
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        apply(new Footnote(annotationType), instance, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation The annotation to match
     * @param instance   The instance to search
     * @param injection  The value to inject
     */
    public static void apply(Footnote annotation, Object instance,
                             short injection) {
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        List<Exception> exceptions = new ArrayList<Exception>();
        List<Field> fields = new ArrayList<Field>();
        List<Method> setters = new ArrayList<Method>();

        internalApply(annotation, instance.getClass(), short.class, exceptions,
                fields, setters);

        for (Field field : fields) {
            if (!Modifier.isStatic(field.getModifiers())) {
                try {
                    if (!field.isAccessible()) {
                        field.setAccessible(true);
                    }
                    field.setShort(instance, injection);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        for (Method setter : setters) {
            if (!Modifier.isStatic(setter.getModifiers())) {
                try {
                    if (!setter.isAccessible()) {
                        setter.setAccessible(true);
                    }
                    setter.invoke(instance, injection);
                } catch (InvocationTargetException e) {
                    exceptions.add(e);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        rethrowExceptions(exceptions);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation The annotation to match
     * @param instance   The instance to search
     * @param injection  The value to inject
     *
     * @since 1.1
     */
    public static void apply(Annotation annotation, Object instance,
                             short injection) {
        if (annotation == null) {
            throw new NullPointerException();
        }
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        apply(new Footnote(annotation), instance, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotationType The annotation class to match
     * @param instance       The instance to search
     * @param injection      The value to inject
     *
     * @since 1.1
     */
    public static void apply(Class<? extends Annotation> annotationType,
                             Object instance, short injection) {
        if (annotationType == null) {
            throw new NullPointerException();
        }
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        apply(new Footnote(annotationType), instance, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation The annotation to match
     * @param instance   The instance to search
     * @param injection  The value to inject
     */
    public static void apply(Footnote annotation, Object instance,
                             long injection) {
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        List<Exception> exceptions = new ArrayList<Exception>();
        List<Field> fields = new ArrayList<Field>();
        List<Method> setters = new ArrayList<Method>();

        internalApply(annotation, instance.getClass(), long.class, exceptions,
                fields, setters);

        for (Field field : fields) {
            if (!Modifier.isStatic(field.getModifiers())) {
                try {
                    if (!field.isAccessible()) {
                        field.setAccessible(true);
                    }
                    field.setLong(instance, injection);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        for (Method setter : setters) {
            if (!Modifier.isStatic(setter.getModifiers())) {
                try {
                    if (!setter.isAccessible()) {
                        setter.setAccessible(true);
                    }
                    setter.invoke(instance, injection);
                } catch (InvocationTargetException e) {
                    exceptions.add(e);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        rethrowExceptions(exceptions);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation The annotation to match
     * @param instance   The instance to search
     * @param injection  The value to inject
     *
     * @since 1.1
     */
    public static void apply(Annotation annotation, Object instance,
                             long injection) {
        if (annotation == null) {
            throw new NullPointerException();
        }
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        apply(new Footnote(annotation), instance, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotationType The annotation class to match
     * @param instance       The instance to search
     * @param injection      The value to inject
     *
     * @since 1.1
     */
    public static void apply(Class<? extends Annotation> annotationType,
                             Object instance, long injection) {
        if (annotationType == null) {
            throw new NullPointerException();
        }
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        apply(new Footnote(annotationType), instance, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation The annotation to match
     * @param instance   The instance to search
     * @param injection  The value to inject
     */
    public static void apply(Footnote annotation, Object instance,
                             float injection) {
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        List<Exception> exceptions = new ArrayList<Exception>();
        List<Field> fields = new ArrayList<Field>();
        List<Method> setters = new ArrayList<Method>();

        internalApply(annotation, instance.getClass(), float.class, exceptions,
                fields, setters);

        for (Field field : fields) {
            if (!Modifier.isStatic(field.getModifiers())) {
                try {
                    if (!field.isAccessible()) {
                        field.setAccessible(true);
                    }
                    field.setFloat(instance, injection);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        for (Method setter : setters) {
            if (!Modifier.isStatic(setter.getModifiers())) {
                try {
                    if (!setter.isAccessible()) {
                        setter.setAccessible(true);
                    }
                    setter.invoke(instance, injection);
                } catch (InvocationTargetException e) {
                    exceptions.add(e);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        rethrowExceptions(exceptions);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation The annotation to match
     * @param instance   The instance to search
     * @param injection  The value to inject
     *
     * @since 1.1
     */
    public static void apply(Annotation annotation, Object instance,
                             float injection) {
        if (annotation == null) {
            throw new NullPointerException();
        }
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        apply(new Footnote(annotation), instance, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotationType The annotation class to match
     * @param instance       The instance to search
     * @param injection      The value to inject
     *
     * @since 1.1
     */
    public static void apply(Class<? extends Annotation> annotationType,
                             Object instance, float injection) {
        if (annotationType == null) {
            throw new NullPointerException();
        }
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        apply(new Footnote(annotationType), instance, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation The annotation to match
     * @param instance   The instance to search
     * @param injection  The value to inject
     */
    public static void apply(Footnote annotation, Object instance,
                             double injection) {
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        List<Exception> exceptions = new ArrayList<Exception>();
        List<Field> fields = new ArrayList<Field>();
        List<Method> setters = new ArrayList<Method>();

        internalApply(annotation, instance.getClass(), double.class, exceptions,
                fields, setters);

        for (Field field : fields) {
            if (!Modifier.isStatic(field.getModifiers())) {
                try {
                    if (!field.isAccessible()) {
                        field.setAccessible(true);
                    }
                    field.setDouble(instance, injection);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        for (Method setter : setters) {
            if (!Modifier.isStatic(setter.getModifiers())) {
                try {
                    if (!setter.isAccessible()) {
                        setter.setAccessible(true);
                    }
                    setter.invoke(instance, injection);
                } catch (InvocationTargetException e) {
                    exceptions.add(e);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        rethrowExceptions(exceptions);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation The annotation to match
     * @param instance   The instance to search
     * @param injection  The value to inject
     *
     * @since 1.1
     */
    public static void apply(Annotation annotation, Object instance,
                             double injection) {
        if (annotation == null) {
            throw new NullPointerException();
        }
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        apply(new Footnote(annotation), instance, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotationType The annotation to match
     * @param instance       The instance to search
     * @param injection      The value to inject
     *
     * @since 1.1
     */
    public static void apply(Class<? extends Annotation> annotationType,
                             Object instance, double injection) {
        if (annotationType == null) {
            throw new NullPointerException();
        }
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        apply(new Footnote(annotationType), instance, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation The annotation to match
     * @param instance   The instance to search
     * @param injection  The value to inject
     */
    public static void apply(Footnote annotation, Object instance,
                             char injection) {
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        List<Exception> exceptions = new ArrayList<Exception>();
        List<Field> fields = new ArrayList<Field>();
        List<Method> setters = new ArrayList<Method>();

        internalApply(annotation, instance.getClass(), char.class, exceptions,
                fields, setters);

        for (Field field : fields) {
            if (!Modifier.isStatic(field.getModifiers())) {
                try {
                    if (!field.isAccessible()) {
                        field.setAccessible(true);
                    }
                    field.setChar(instance, injection);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        for (Method setter : setters) {
            if (!Modifier.isStatic(setter.getModifiers())) {
                try {
                    if (!setter.isAccessible()) {
                        setter.setAccessible(true);
                    }
                    setter.invoke(instance, injection);
                } catch (InvocationTargetException e) {
                    exceptions.add(e);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        rethrowExceptions(exceptions);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation The annotation to match
     * @param instance   The instance to search
     * @param injection  The value to inject
     *
     * @since 1.1
     */
    public static void apply(Annotation annotation, Object instance,
                             char injection) {
        if (annotation == null) {
            throw new NullPointerException();
        }
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        apply(new Footnote(annotation), instance, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotationType The annotation class to match
     * @param instance       The instance to search
     * @param injection      The value to inject
     *
     * @since 1.1
     */
    public static void apply(Class<? extends Annotation> annotationType,
                             Object instance, char injection) {
        if (annotationType == null) {
            throw new NullPointerException();
        }
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        apply(new Footnote(annotationType), instance, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation The annotation to match
     * @param instance   The instance to search
     * @param injection  The value to inject
     */
    public static void apply(Footnote annotation, Object instance,
                             byte injection) {
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        List<Exception> exceptions = new ArrayList<Exception>();
        List<Field> fields = new ArrayList<Field>();
        List<Method> setters = new ArrayList<Method>();

        internalApply(annotation, instance.getClass(), byte.class, exceptions,
                fields, setters);

        for (Field field : fields) {
            if (!Modifier.isStatic(field.getModifiers())) {
                try {
                    if (!field.isAccessible()) {
                        field.setAccessible(true);
                    }
                    field.setByte(instance, injection);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        for (Method setter : setters) {
            if (!Modifier.isStatic(setter.getModifiers())) {
                try {
                    if (!setter.isAccessible()) {
                        setter.setAccessible(true);
                    }
                    setter.invoke(instance, injection);
                } catch (InvocationTargetException e) {
                    exceptions.add(e);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        rethrowExceptions(exceptions);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation The annotation to match
     * @param instance   The instance to search
     * @param injection  The value to inject
     *
     * @since 1.1
     */
    public static void apply(Annotation annotation, Object instance,
                             byte injection) {
        if (annotation == null) {
            throw new NullPointerException();
        }
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        apply(new Footnote(annotation), instance, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotationType The annotation class to match
     * @param instance       The instance to search
     * @param injection      The value to inject
     *
     * @since 1.1
     */
    public static void apply(Class<? extends Annotation> annotationType,
                             Object instance, byte injection) {
        if (annotationType == null) {
            throw new NullPointerException();
        }
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        apply(new Footnote(annotationType), instance, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation The annotation to match
     * @param instance   The instance to search
     * @param injection  The value to inject
     */
    public static void apply(Footnote annotation, Object instance,
                             boolean injection) {
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        List<Exception> exceptions = new ArrayList<Exception>();
        List<Field> fields = new ArrayList<Field>();
        List<Method> setters = new ArrayList<Method>();

        internalApply(annotation, instance.getClass(), boolean.class, exceptions,
                fields, setters);

        for (Field field : fields) {
            if (!Modifier.isStatic(field.getModifiers())) {
                try {
                    if (!field.isAccessible()) {
                        field.setAccessible(true);
                    }
                    field.setBoolean(instance, injection);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        for (Method setter : setters) {
            if (!Modifier.isStatic(setter.getModifiers())) {
                try {
                    if (!setter.isAccessible()) {
                        setter.setAccessible(true);
                    }
                    setter.invoke(instance, injection);
                } catch (InvocationTargetException e) {
                    exceptions.add(e);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        rethrowExceptions(exceptions);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation The annotation to match
     * @param instance   The instance to search
     * @param injection  The value to inject
     *
     * @since 1.1
     */
    public static void apply(Annotation annotation, Object instance,
                             boolean injection) {
        if (annotation == null) {
            throw new NullPointerException();
        }
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        apply(new Footnote(annotation), instance, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotationType The annotation class to match
     * @param instance       The instance to search
     * @param injection      The value to inject
     *
     * @since 1.1
     */
    public static void apply(Class<? extends Annotation> annotationType,
                             Object instance, boolean injection) {
        if (annotationType == null) {
            throw new NullPointerException();
        }
        if (instance == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        apply(new Footnote(annotationType), instance, injection);
    }

    /**
     * Filter a list of fields, returning a list of only those that are assignment compatible with type.
     *
     * @param list The list of fields to filter.
     * @param type The type that the fields must be assignment compatible with.
     *
     * @return The filtered list.
     */
    protected static List<Field> filter(List<Field> list, Class type) {
        assert list != null;
        assert type != null;
        List<Field> filteredList = new ArrayList<Field>();
        for (Field field : list) {
            if (field.getType().isAssignableFrom(type)) {
                filteredList.add(field);
            }
        }
        return filteredList;
    }

    /**
     * Filter a list of methods, returning a list of only those that match the method signature.
     *
     * @param list       The list of methods to filter.
     * @param returnType The return type of the method signature.
     * @param paramTypes The parameters of the method signature.
     *
     * @return The filtered list.
     */
    protected static List<Method> filter(List<Method> list, Class returnType, Class[] paramTypes) {
        assert list != null;
        assert returnType != null;
        assert paramTypes != null;
        List<Method> filtered = new ArrayList<Method>();
        for (Method method : list) {
            if (method.getReturnType().isAssignableFrom(returnType) &&
                    matchParameters(method, paramTypes)) {
                filtered.add(method);
            }
        }
        return filtered;
    }

    /**
     * Filter a list of methods, returning a list of only those that are getters for type <i>type</i>.
     *
     * @param list The list of methods to filter.
     * @param type The type of getter to filter on.
     *
     * @return The filtered list.
     */
    protected static List<Method> filterGetters(List<Method> list, Class type) {
        assert list != null;
        assert type != null;
        List<Method> filtered = new ArrayList<Method>();
        for (Method method : list) {
            if (method.getReturnType().equals(type) &&
                    method.getParameterTypes().length == 0 && (method.getName().startsWith("get") ||
                    (type.equals(boolean.class) && method.getName().startsWith("is")))) {
                filtered.add(method);
            }
        }
        return filtered;
    }

    /**
     * Filter a list of methods, returning a list of only those that are setters for type <i>type</i>.
     *
     * @param list The list of methods to filter.
     * @param type The type of getter to filter on.
     *
     * @return The filtered list.
     */
    protected static List<Method> filterSetters(List<Method> list, Class type) {
        assert list != null;
        assert type != null;
        Class[] paramTypes = {type};
        List<Method> filtered = new ArrayList<Method>();
        for (Method method : list) {
            if (method.getReturnType().equals(void.class) && matchParameters(method, paramTypes) &&
                    method.getName().startsWith("set")) {
                filtered.add(method);
            }
        }
        return filtered;
    }

    /**
     * Get all the annotated fields for the class <i>clazz</i> including those from the superclasses.
     *
     * @param clazz      The class to search.
     * @param annotation The annotation that the fields must have.
     *
     * @return The list of fields with the required annotation.
     */
    protected static List<Field> findFields(Class clazz, Footnote annotation) {
        assert clazz != null;
        assert annotation != null;
        List<Field> result = new ArrayList<Field>();
        for (Class superclazz = clazz; superclazz != null; superclazz = superclazz.getSuperclass()) {
            Field[] fields = superclazz.getDeclaredFields();
            for (Field field : fields) {
                for (Annotation anno : field.getAnnotations()) {
                    if (annotation.matches(anno) &&
                            !result.contains(field)) {
                        result.add(field);
                    }
                }
            }
        }
        return result;
    }

    /**
     * Get all the annotated methods for the class <i>clazz</i> including those from the superclasses.
     *
     * @param clazz      The class to search.
     * @param annotation The annotation that the methods must have.
     *
     * @return The list of fields with the required annotation.
     */
    protected static List<Method> findMethods(Class clazz, Footnote annotation) {
        assert clazz != null;
        assert annotation != null;
        List<Method> result = new ArrayList<Method>();
        for (Class superclazz = clazz; superclazz != null; superclazz = superclazz.getSuperclass()) {
            Method[] methods = superclazz.getDeclaredMethods();
            for (Method method : methods) {
                for (Annotation anno : method.getAnnotations()) {
                    if (annotation.matches(anno) &&
                            !result.contains(method)) {
                        result.add(method);
                    }
                }
            }
        }
        return result;
    }

    /**
     * Find the setter for the corresponding getter in the specified class. Will look in superclasses if necessary.
     *
     * @param clazz  The class to search in.
     * @param getter The getter to match.
     *
     * @return The setter or <code>null</code> if it cannot be found.
     */
    protected static Method findSetter(Class clazz, Method getter) {
        // we are a private method that does not take null parameters
        assert clazz != null;
        assert getter != null;
        assert !getter.getReturnType().equals(void.class);
        assert getter.getName().startsWith("get") ||
                (getter.getReturnType().equals(boolean.class) && getter.getName().startsWith("is"));
        assert getter.getParameterTypes().length == 0 ||
                (getter.getParameterTypes().length == 1 && getter.getParameterTypes()[0].equals(int.class));
        Class type = getter.getReturnType();
        String setter;
        if (type.equals(boolean.class) && getter.getName().startsWith("is")) {
            setter = getter.getName().replaceFirst("is", "set");
        } else {
            setter = getter.getName().replaceFirst("get", "set");
        }
        Class[] parameterTypes = new Class[getter.getParameterTypes().length + 1];
        if (parameterTypes.length == 2) {
            parameterTypes[0] = int.class;
            parameterTypes[1] = type;
        } else {
            parameterTypes[0] = type;
        }
        Method result = null;
        for (Class superclazz = clazz; superclazz != null; superclazz = superclazz.getSuperclass()) {
            try {
                result = superclazz.getDeclaredMethod(setter, parameterTypes);
                if (result != null) {
                    break;
                }
            } catch (NoSuchMethodException e) {
                // we were expecting you Mr Bond
                result = null;
            }
            // if exiting the loop normally, ensure result is null
            result = null;
        }
        return result;
    }

    /**
     * Call all the matching annotated methods in the instance using the parameters specified. <b>Note:</b> will not
     * accept null parameters.
     *
     * @param annotation The annotation that the methods must have.
     * @param instance   The instance to invoke the methods on.
     * @param params     The parameters to invoke the methods with (also the signature for the methods is formed from
     *                   these parameters)
     */
    public static void invoke(Footnote annotation, Object instance,
                              Object... params) {
        if (annotation == null) {
            throw new NullPointerException("annotation cannot be null");
        }
        if (instance == null) {
            throw new NullPointerException("Cannot invoke on null");
        }
        Class[] paramTypes = new Class[params.length];
        for (int index = 0; index < params.length; index++) {
            if (params[index] == null) {
                throw new NullPointerException("Cannot invoke with a null parameter");
            }
            paramTypes[index] = params[index].getClass();
        }
        List<Exception> exceptions = new ArrayList<Exception>();
        List<Method> methods = filter(findMethods(instance.getClass(), annotation),
                void.class, paramTypes);
        for (Method method : methods) {
            try {
                if (!method.isAccessible()) {
                    method.setAccessible(true);
                }
                method.invoke(instance, params);
            } catch (InvocationTargetException e) {
                exceptions.add(e);
            } catch (IllegalAccessException e) {
                exceptions.add(e);
            }
        }
        if (!exceptions.isEmpty()) {
            StringBuilder message = new StringBuilder();
            for (Exception e : exceptions) {
                if (message.length() > 0) {
                    message.append(", ");
                } else {
                    message.append("[");
                }
                message.append(e);
            }
            message.append("]");
            throw new GlossApplicationError(message.toString());
        }
    }

    /**
     * Call all the matching annotated methods in the instance using the parameters specified. <b>Note:</b> will not
     * accept null parameters.
     *
     * @param annotation The annotation that the methods must have.
     * @param instance   The instance to invoke the methods on.
     * @param params     The parameters to invoke the methods with (also the signature for the methods is formed from
     *                   these parameters)
     *
     * @since 1.1
     */
    public static void invoke(Annotation annotation, Object instance,
                              Object... params) {
        if (annotation == null) {
            throw new NullPointerException();
        }
        invoke(new Footnote(annotation), instance, params);
    }

    /**
     * Call all the matching annotated methods in the instance using the parameters specified. <b>Note:</b> will not
     * accept null parameters.
     *
     * @param annotationType The annotation class that the methods must have.
     * @param instance       The instance to invoke the methods on.
     * @param params         The parameters to invoke the methods with (also the signature for the methods is formed
     *                       from these parameters)
     *
     * @since 1.1
     */
    public static void invoke(Class<? extends Annotation> annotationType,
                              Object instance, Object... params) {
        if (annotationType == null) {
            throw new NullPointerException();
        }
        invoke(new Footnote(annotationType), instance, params);
    }

    /**
     * Check if the method's parameters match paramTypes.  The parameters of the method need only be assignment
     * compatible for a match to be made.
     *
     * @param method     The method to check.
     * @param paramTypes The parameter types to match against.
     *
     * @return <code>true</code> if we have a match.
     */
    protected static boolean matchParameters(Method method, Class<?>[] paramTypes) {
        assert method != null;
        assert paramTypes != null;
        Class<?>[] mp = method.getParameterTypes();
        boolean retVal;
        if (mp.length != paramTypes.length) {
            retVal = false;
        } else {
            retVal = true;
            for (int index = 0; index < paramTypes.length; index++) {
                if (!mp[index].isAssignableFrom(paramTypes[index])) {
                    retVal = false;
                    break;
                }
            }
        }
        return retVal;
    }

    /**
     * Apply the gloss to the instance
     *
     * @param instance The object to which the gloss should be applied
     */
    abstract public void apply(Object instance);

    /**
     * Apply the gloss to the class
     *
     * @param instanceClass The object to which the gloss should be applied
     */
    abstract public void applyStatic(Class instanceClass);

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation    The annotation to match
     * @param instanceClass The instanceClass to search
     * @param injection     The value to inject
     */
    public static void applyStatic(Footnote annotation, Class instanceClass,
                                   int injection) {
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        List<Exception> exceptions = new ArrayList<Exception>();
        List<Field> fields = new ArrayList<Field>();
        List<Method> setters = new ArrayList<Method>();

        internalApply(annotation, instanceClass, int.class, exceptions,
                fields, setters);

        for (Field field : fields) {
            if (Modifier.isStatic(field.getModifiers())) {
                try {
                    if (!field.isAccessible()) {
                        field.setAccessible(true);
                    }
                    field.setInt(null, injection);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        for (Method setter : setters) {
            if (Modifier.isStatic(setter.getModifiers())) {
                try {
                    if (!setter.isAccessible()) {
                        setter.setAccessible(true);
                    }
                    setter.invoke(null, injection);
                } catch (InvocationTargetException e) {
                    exceptions.add(e);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        rethrowExceptions(exceptions);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation    The annotation to match
     * @param instanceClass The instance to search
     * @param injection     The value to inject
     *
     * @since 1.1
     */
    public static void applyStatic(Annotation annotation, Class instanceClass,
                                   int injection) {
        if (annotation == null) {
            throw new NullPointerException();
        }
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        applyStatic(new Footnote(annotation), instanceClass, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotationType The annotation class to match
     * @param instanceClass  The instanceClass to search
     * @param injection      The value to inject
     *
     * @since 1.1
     */
    public static void applyStatic(Class<? extends Annotation> annotationType,
                                   Class instanceClass, int injection) {
        if (annotationType == null) {
            throw new NullPointerException();
        }
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        applyStatic(new Footnote(annotationType), instanceClass, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation    The annotation to match
     * @param instanceClass The instanceClass to search
     * @param injection     The value to inject
     */
    public static void applyStatic(Footnote annotation, Class instanceClass,
                                   short injection) {
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        List<Exception> exceptions = new ArrayList<Exception>();
        List<Field> fields = new ArrayList<Field>();
        List<Method> setters = new ArrayList<Method>();

        internalApply(annotation, instanceClass, short.class, exceptions,
                fields, setters);

        for (Field field : fields) {
            if (Modifier.isStatic(field.getModifiers())) {
                try {
                    if (!field.isAccessible()) {
                        field.setAccessible(true);
                    }
                    field.setShort(null, injection);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        for (Method setter : setters) {
            if (Modifier.isStatic(setter.getModifiers())) {
                try {
                    if (!setter.isAccessible()) {
                        setter.setAccessible(true);
                    }
                    setter.invoke(null, injection);
                } catch (InvocationTargetException e) {
                    exceptions.add(e);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        rethrowExceptions(exceptions);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation    The annotation to match
     * @param instanceClass The instanceClass to search
     * @param injection     The value to inject
     *
     * @since 1.1
     */
    public static void applyStatic(Annotation annotation, Class instanceClass,
                                   short injection) {
        if (annotation == null) {
            throw new NullPointerException();
        }
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        applyStatic(new Footnote(annotation), instanceClass, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotationType The annotation class to match
     * @param instanceClass  The instanceClass to search
     * @param injection      The value to inject
     *
     * @since 1.1
     */
    public static void applyStatic(Class<? extends Annotation> annotationType,
                                   Class instanceClass, short injection) {
        if (annotationType == null) {
            throw new NullPointerException();
        }
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        applyStatic(new Footnote(annotationType), instanceClass, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation    The annotation to match
     * @param instanceClass The instanceClass to search
     * @param injection     The value to inject
     */
    public static void applyStatic(Footnote annotation, Class instanceClass,
                                   long injection) {
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        List<Exception> exceptions = new ArrayList<Exception>();
        List<Field> fields = new ArrayList<Field>();
        List<Method> setters = new ArrayList<Method>();

        internalApply(annotation, instanceClass, long.class, exceptions,
                fields, setters);

        for (Field field : fields) {
            if (Modifier.isStatic(field.getModifiers())) {
                try {
                    if (!field.isAccessible()) {
                        field.setAccessible(true);
                    }
                    field.setLong(null, injection);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        for (Method setter : setters) {
            if (Modifier.isStatic(setter.getModifiers())) {
                try {
                    if (!setter.isAccessible()) {
                        setter.setAccessible(true);
                    }
                    setter.invoke(null, injection);
                } catch (InvocationTargetException e) {
                    exceptions.add(e);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        rethrowExceptions(exceptions);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation    The annotation to match
     * @param instanceClass The instanceClass to search
     * @param injection     The value to inject
     *
     * @since 1.1
     */
    public static void applyStatic(Annotation annotation, Class instanceClass,
                                   long injection) {
        if (annotation == null) {
            throw new NullPointerException();
        }
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        applyStatic(new Footnote(annotation), instanceClass, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotationType The annotation class to match
     * @param instanceClass  The instanceClass to search
     * @param injection      The value to inject
     *
     * @since 1.1
     */
    public static void applyStatic(Class<? extends Annotation> annotationType,
                                   Class instanceClass, long injection) {
        if (annotationType == null) {
            throw new NullPointerException();
        }
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        applyStatic(new Footnote(annotationType), instanceClass, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation    The annotation to match
     * @param instanceClass The instanceClass to search
     * @param injection     The value to inject
     */
    public static void applyStatic(Footnote annotation, Class instanceClass,
                                   float injection) {
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        List<Exception> exceptions = new ArrayList<Exception>();
        List<Field> fields = new ArrayList<Field>();
        List<Method> setters = new ArrayList<Method>();

        internalApply(annotation, instanceClass, float.class, exceptions,
                fields, setters);

        for (Field field : fields) {
            if (Modifier.isStatic(field.getModifiers())) {
                try {
                    if (!field.isAccessible()) {
                        field.setAccessible(true);
                    }
                    field.setFloat(null, injection);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        for (Method setter : setters) {
            if (Modifier.isStatic(setter.getModifiers())) {
                try {
                    if (!setter.isAccessible()) {
                        setter.setAccessible(true);
                    }
                    setter.invoke(null, injection);
                } catch (InvocationTargetException e) {
                    exceptions.add(e);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        rethrowExceptions(exceptions);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation    The annotation to match
     * @param instanceClass The instanceClass to search
     * @param injection     The value to inject
     *
     * @since 1.1
     */
    public static void applyStatic(Annotation annotation, Class instanceClass,
                                   float injection) {
        if (annotation == null) {
            throw new NullPointerException();
        }
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        applyStatic(new Footnote(annotation), instanceClass, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotationType The annotation class to match
     * @param instanceClass  The instanceClass to search
     * @param injection      The value to inject
     *
     * @since 1.1
     */
    public static void applyStatic(Class<? extends Annotation> annotationType,
                                   Class instanceClass, float injection) {
        if (annotationType == null) {
            throw new NullPointerException();
        }
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        applyStatic(new Footnote(annotationType), instanceClass, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation    The annotation to match
     * @param instanceClass The instanceClass to search
     * @param injection     The value to inject
     */
    public static void applyStatic(Footnote annotation, Class instanceClass,
                                   double injection) {
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        List<Exception> exceptions = new ArrayList<Exception>();
        List<Field> fields = new ArrayList<Field>();
        List<Method> setters = new ArrayList<Method>();

        internalApply(annotation, instanceClass, double.class, exceptions,
                fields, setters);

        for (Field field : fields) {
            if (Modifier.isStatic(field.getModifiers())) {
                try {
                    if (!field.isAccessible()) {
                        field.setAccessible(true);
                    }
                    field.setDouble(null, injection);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        for (Method setter : setters) {
            if (Modifier.isStatic(setter.getModifiers())) {
                try {
                    if (!setter.isAccessible()) {
                        setter.setAccessible(true);
                    }
                    setter.invoke(null, injection);
                } catch (InvocationTargetException e) {
                    exceptions.add(e);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        rethrowExceptions(exceptions);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation    The annotation to match
     * @param instanceClass The instanceClass to search
     * @param injection     The value to inject
     *
     * @since 1.1
     */
    public static void applyStatic(Annotation annotation, Class instanceClass,
                                   double injection) {
        if (annotation == null) {
            throw new NullPointerException();
        }
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        applyStatic(new Footnote(annotation), instanceClass, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotationType The annotation to match
     * @param instanceClass  The instanceClass to search
     * @param injection      The value to inject
     *
     * @since 1.1
     */
    public static void applyStatic(Class<? extends Annotation> annotationType,
                                   Class instanceClass, double injection) {
        if (annotationType == null) {
            throw new NullPointerException();
        }
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        applyStatic(new Footnote(annotationType), instanceClass, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation    The annotation to match
     * @param instanceClass The instanceClass to search
     * @param injection     The value to inject
     */
    public static void applyStatic(Footnote annotation, Class instanceClass,
                                   char injection) {
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        List<Exception> exceptions = new ArrayList<Exception>();
        List<Field> fields = new ArrayList<Field>();
        List<Method> setters = new ArrayList<Method>();

        internalApply(annotation, instanceClass, char.class, exceptions,
                fields, setters);

        for (Field field : fields) {
            if (Modifier.isStatic(field.getModifiers())) {
                try {
                    if (!field.isAccessible()) {
                        field.setAccessible(true);
                    }
                    field.setChar(null, injection);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        for (Method setter : setters) {
            if (Modifier.isStatic(setter.getModifiers())) {
                try {
                    if (!setter.isAccessible()) {
                        setter.setAccessible(true);
                    }
                    setter.invoke(null, injection);
                } catch (InvocationTargetException e) {
                    exceptions.add(e);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        rethrowExceptions(exceptions);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation    The annotation to match
     * @param instanceClass The instanceClass to search
     * @param injection     The value to inject
     *
     * @since 1.1
     */
    public static void applyStatic(Annotation annotation, Class instanceClass,
                                   char injection) {
        if (annotation == null) {
            throw new NullPointerException();
        }
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        applyStatic(new Footnote(annotation), instanceClass, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotationType The annotation class to match
     * @param instanceClass  The instanceClass to search
     * @param injection      The value to inject
     *
     * @since 1.1
     */
    public static void applyStatic(Class<? extends Annotation> annotationType,
                                   Class instanceClass, char injection) {
        if (annotationType == null) {
            throw new NullPointerException();
        }
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        applyStatic(new Footnote(annotationType), instanceClass, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation    The annotation to match
     * @param instanceClass The instanceClass to search
     * @param injection     The value to inject
     */
    public static void applyStatic(Footnote annotation, Class instanceClass,
                                   byte injection) {
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        List<Exception> exceptions = new ArrayList<Exception>();
        List<Field> fields = new ArrayList<Field>();
        List<Method> setters = new ArrayList<Method>();

        internalApply(annotation, instanceClass, byte.class, exceptions,
                fields, setters);

        for (Field field : fields) {
            if (Modifier.isStatic(field.getModifiers())) {
                try {
                    if (!field.isAccessible()) {
                        field.setAccessible(true);
                    }
                    field.setByte(null, injection);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        for (Method setter : setters) {
            if (Modifier.isStatic(setter.getModifiers())) {
                try {
                    if (!setter.isAccessible()) {
                        setter.setAccessible(true);
                    }
                    setter.invoke(null, injection);
                } catch (InvocationTargetException e) {
                    exceptions.add(e);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        rethrowExceptions(exceptions);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation    The annotation to match
     * @param instanceClass The instanceClass to search
     * @param injection     The value to inject
     *
     * @since 1.1
     */
    public static void applyStatic(Annotation annotation, Class instanceClass,
                                   byte injection) {
        if (annotation == null) {
            throw new NullPointerException();
        }
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        applyStatic(new Footnote(annotation), instanceClass, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotationType The annotation class to match
     * @param instanceClass  The instanceClass to search
     * @param injection      The value to inject
     *
     * @since 1.1
     */
    public static void applyStatic(Class<? extends Annotation> annotationType,
                                   Class instanceClass, byte injection) {
        if (annotationType == null) {
            throw new NullPointerException();
        }
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        applyStatic(new Footnote(annotationType), instanceClass, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation    The annotation to match
     * @param instanceClass The instanceClass to search
     * @param injection     The value to inject
     */
    public static void applyStatic(Footnote annotation, Class instanceClass,
                                   boolean injection) {
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        List<Exception> exceptions = new ArrayList<Exception>();
        List<Field> fields = new ArrayList<Field>();
        List<Method> setters = new ArrayList<Method>();

        internalApply(annotation, instanceClass, boolean.class, exceptions,
                fields, setters);

        for (Field field : fields) {
            if (Modifier.isStatic(field.getModifiers())) {
                try {
                    if (!field.isAccessible()) {
                        field.setAccessible(true);
                    }
                    field.setBoolean(null, injection);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        for (Method setter : setters) {
            if (Modifier.isStatic(setter.getModifiers())) {
                try {
                    if (!setter.isAccessible()) {
                        setter.setAccessible(true);
                    }
                    setter.invoke(null, injection);
                } catch (InvocationTargetException e) {
                    exceptions.add(e);
                } catch (IllegalAccessException e) {
                    exceptions.add(e);
                }
            }
        }

        rethrowExceptions(exceptions);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotation    The annotation to match
     * @param instanceClass The instanceClass to search
     * @param injection     The value to inject
     *
     * @since 1.1
     */
    public static void applyStatic(Annotation annotation, Class instanceClass,
                                   boolean injection) {
        if (annotation == null) {
            throw new NullPointerException();
        }
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        applyStatic(new Footnote(annotation), instanceClass, injection);
    }

    /**
     * Apply the specified gloss to the instance.  Tries to inject any annotated fields, properties or setter methods
     * with the injection. <br/> Matching rules: <ul> <li><b>Fields:</b> Must be assignment compatible with
     * <i>injection</i> any access type (public, protected, default or private) will be identified, including those from
     * superclasses, however, if a required call to Field.setAccessible fails due to a SecurityManager or other
     * restriction then the injection will fail.</li> <li><b>Properties:</b> any access type (public, protected, default
     * or private). The annotation must be on the getter. The getter must return a value that is assignment compatible
     * with <i>injection</i>. The name must begin with <code>get</code> unless the return type is <code>boolean</code>
     * in which case <code>is</code> is acceptable.
     * <p/>
     * Cannot inject elements into an array property, must inject the entire array (i.e. will not match a
     * <code>get<i>XXX</i>(int index)</code> method. The corresponding setter must exist. If the setter is not
     * accessible and the required call to Method.setAccessible fails due to a SecurityManager or other restriction then
     * the injection will fail.</li> <li><b>Setter methods:</b>The name must begin with <code>set</code>. Cannot inject
     * elements into an array property, must inject the entire array (i.e. will not match a <code>set<i>XXX</i>(int
     * index, T value)</code> method. If the setter is not accessible and the required call to Method.setAccessible
     * fails due to a SecurityManager or other restriction then the injection will fail.</li> </ul>
     *
     * @param annotationType The annotation class to match
     * @param instanceClass  The instanceClass to search
     * @param injection      The value to inject
     *
     * @since 1.1
     */
    public static void applyStatic(Class<? extends Annotation> annotationType,
                                   Class instanceClass, boolean injection) {
        if (annotationType == null) {
            throw new NullPointerException();
        }
        if (instanceClass == null) {
            throw new NullPointerException("Cannot inject into null");
        }
        applyStatic(new Footnote(annotationType), instanceClass, injection);
    }

}
