/*
 * Copyright 2006 the original author or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.stvconsultants.easygloss;

import java.util.ArrayList;
import java.util.List;

/**
 * A <code>MultiGloss</code> is a decorator for annotated objects that applies multiple coats of gloss.
 *
 * @author Stephen Connolly
 */
public class MultiGloss extends AbstractGloss {

    private List<AbstractGloss> coats;

    /**
     * Creates a new instance of MultiGloss
     */
    public MultiGloss() {
        this.coats = new ArrayList<AbstractGloss>();
    }

    /**
     * Adds a coat to the gloss
     *
     * @param coat The gloss to add to the multigloss
     *
     * @return this instance for method chaining.
     */
    public MultiGloss add(AbstractGloss coat) {
        this.coats.add(coat);
        return this;
    }

    /**
     * Apply the gloss to the instance
     *
     * @param instance The object to which the gloss should be applied
     */
    public void apply(Object instance) {
        for (AbstractGloss coat : coats) {
            coat.apply(instance);
        }
    }

    /**
     * Apply the gloss to the class
     *
     * @param instanceClass The class to which the gloss should be applied
     */
    public void applyStatic(Class instanceClass) {
        for (AbstractGloss coat : coats) {
            coat.applyStatic(instanceClass);
        }
    }

}
