/*
 * Copyright 2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.stvconsultants.easygloss.footnotes;

import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

/**
 * An element of the matching rules for a footnote.
 *
 * @author Stephen Connolly
 */
class FootnoteElement {

    /**
     * The name of the element to match on
     */
    private final String name;
    /**
     * The value that the element must equal, or null if this rule is not to be enforced. This rule cannot be enforced
     * at the same time as any other rule.
     */
    Object mustEqual = null;
    /**
     * The a list of the values that the element must not equal, or null if this rule is not to be enforced.
     */
    List<Object> mustNotEqual = null;
    /**
     * The a list of the values that the element must contain assuming that the element is an array type, or null if
     * this rule is not to be enforced.
     */
    List<Object> mustContain = null;
    /**
     * The a list of the values that the element must not contain assuming that the element is an array type, or null if
     * this rule is not to be enforced.
     */
    List<Object> mustExclude = null;
    /**
     * The regex that the element value must match assuming that the element is a String, or null if this rule is not to
     * be enforced.
     */
    String regex = null;

    /**
     * Default constructor.
     *
     * @param name The name of the element to match
     */
    public FootnoteElement(String name) {
        if (name == null) {
            throw new NullPointerException("name cannot be null");
        }
        this.name = name;
    }

    /**
     * Set the value that the element value must be equal to.
     *
     * @param target The value that the element value must be equal to.
     */
    public void setEquals(Object target) {
        if (target != null &&
                (mustContain != null ||
                        mustNotEqual != null ||
                        mustExclude != null)) {
            throw new IllegalFootnoteMatchingRuleError();
        }
        this.mustEqual = target;
    }

    /**
     * Add a value that the element value must be not equal to.
     *
     * @param target A value that the element value must be not equal to.
     */
    public void addNotEqual(Object target) {
        if (target != null && mustEqual != null) {
            throw new IllegalFootnoteMatchingRuleError();
        }
        if (this.mustNotEqual == null) {
            this.mustNotEqual = new ArrayList<Object>();
        }
        this.mustNotEqual.add(target);
    }

    /**
     * Add a value that the element's array value must contain.
     *
     * @param target A value that the element's array value must contain.
     */
    public void addContains(Object target) {
        if (target != null && mustEqual != null) {
            throw new IllegalFootnoteMatchingRuleError();
        }
        if (this.mustContain == null) {
            this.mustContain = new ArrayList<Object>();
        }
        this.mustContain.add(target);
    }

    /**
     * Add a value that the element's array value must not contain.
     *
     * @param target A value that the element's array value must not contain.
     */
    public void addExcludes(Object target) {
        if (target != null && mustEqual != null) {
            throw new IllegalFootnoteMatchingRuleError();
        }
        if (this.mustExclude == null) {
            this.mustExclude = new ArrayList<Object>();
        }
        this.mustExclude.add(target);
    }

    /**
     * Set the regex that the element value must match.
     *
     * @param target The regex that the element value must match.
     */
    public void setRegex(String regex) {
        if (regex != null && mustEqual != null) {
            throw new IllegalFootnoteMatchingRuleError();
        }
        this.regex = regex;
    }

    /**
     * Compare the element on the actual annotation with the rules that have been defined.
     *
     * @param annotation The annotation to compare the rules to.
     *
     * @returns <code>true</code> if the annotation matches the rules.
     */
    @SuppressWarnings("unchecked")
    public boolean compare(Annotation annotation) {
        Class<? extends Annotation> clazz = annotation.annotationType();
        boolean retVal = true;
        try {
            Method method = clazz.getMethod(this.getName());
            Object value = method.invoke(annotation);
            if (mustEqual != null) {
                retVal = mustEqual.equals(value);
            }
            if (retVal && mustNotEqual != null) {
                retVal = !mustNotEqual.contains(value);
            }
            if (retVal && regex != null /*&& value instanceof String*/) {
                String valueStr = (String) value;
                retVal = valueStr.matches(regex);
            }
            if (retVal && (value instanceof Collection ||
                    value instanceof Object[])) {
                final Collection valueCollection = value instanceof Collection ?
                        (Collection) value : Arrays.asList((Object[]) value);
                if (retVal && mustExclude != null) {
                    retVal = nullIntersection(mustExclude, valueCollection);
                }
                if (retVal && mustContain != null) {
                    retVal = valueCollection.containsAll(mustContain);
                }
            }
        } catch (NoSuchMethodException e) {
            retVal = false;
        } catch (InvocationTargetException e) {
            retVal = false;
        } catch (IllegalAccessException e) {
            retVal = false;
        }
        return retVal;
    }

    /**
     * Determine if the intersection between two collections is null.
     *
     * @param a The first collection.
     * @param b The second collection.
     */
    private static boolean nullIntersection(Collection<?> a, Collection<?> b) {
        boolean retVal = true;
        for (Object element : a) {
            if (b.contains(element)) {
                retVal = false;
                break;
            }
        }
        return retVal;
    }

    /**
     * Get the element name that this element matches.
     *
     * @returns the name that this element matches
     */
    public String getName() {
        return name;
    }

    /**
     * Get the string representation of this element.
     *
     * @returns the string representation of this element.
     */
    public String toString() {
        StringBuilder buf = new StringBuilder();
        buf.append(getName());
        if (mustEqual != null) {
            buf.append("=");
            if (mustEqual instanceof String) {
                buf.append("\"");
            }
            buf.append(mustEqual);
            if (mustEqual instanceof String) {
                buf.append("\"");
            }
        } else {
            if (regex != null) {
                buf.append(" like \"");
                buf.append(regex);
                buf.append("\"");
            } else {
                buf.append("=*");
            }
            if (mustNotEqual != null) {
                buf.append(" not in [");
                boolean f = true;
                for (Object item : mustNotEqual) {
                    if (!f) {
                        buf.append(",");
                    }
                    if (item instanceof String) {
                        buf.append("\"");
                    }
                    buf.append(item);
                    if (item instanceof String) {
                        buf.append("\"");
                    }
                    f = false;
                }
                buf.append("]");
            }
            if (mustContain != null) {
                buf.append(" with elements [");
                boolean f = true;
                for (Object item : mustContain) {
                    if (!f) {
                        buf.append(",");
                    }
                    if (item instanceof String) {
                        buf.append("\"");
                    }
                    buf.append(item);
                    if (item instanceof String) {
                        buf.append("\"");
                    }
                    f = false;
                }
                buf.append("]");
            }
            if (mustExclude != null) {
                buf.append(" without elements [");
                boolean f = true;
                for (Object item : mustExclude) {
                    if (!f) {
                        buf.append(",");
                    }
                    if (item instanceof String) {
                        buf.append("\"");
                    }
                    buf.append(item);
                    if (item instanceof String) {
                        buf.append("\"");
                    }
                    f = false;
                }
                buf.append("]");
            }
        }
        return buf.toString();
    }

    /**
     * Compare this instance with another instance.
     *
     * @param obj The instance to compare with.
     *
     * @returns <code>true</code> if the two instances are logically equivalent.
     */
    public boolean equals(Object obj) {
        final boolean retVal;
        if (this == obj) {
            retVal = true;
        } else if ((obj == null) || (obj.getClass() != this.getClass())) {
            retVal = false;
        } else {
            // object must be a FootnoteElement at this point
            FootnoteElement that = (FootnoteElement) obj;

            retVal = (this.name == that.name ||
                    this.name.equals(that.name)) &&
                    (this.mustEqual == that.mustEqual ||
                            (this.mustEqual != null &&
                                    this.mustEqual.equals(that.mustEqual))) &&
                    (this.mustNotEqual == that.mustNotEqual ||
                            (this.mustNotEqual != null &&
                                    this.mustNotEqual.equals(that.mustNotEqual))) &&
                    (this.mustContain == that.mustContain ||
                            (this.mustContain != null &&
                                    this.mustContain.equals(that.mustContain))) &&
                    (this.mustExclude == that.mustExclude ||
                            (this.mustExclude != null &&
                                    this.mustExclude.equals(that.mustExclude))) &&
                    (this.regex == that.regex ||
                            (this.regex != null &&
                                    this.regex.equals(that.regex)));
        }
        return retVal;
    }

    /**
     * Calculate the hashCode for this instance.
     *
     * @returns the hash code.
     */
    public int hashCode() {
        int hash = 7;
        hash = 31 * hash + name.hashCode();
        hash = 31 * hash + ((mustEqual == null) ? 0 : mustEqual.hashCode());
        hash = 31 * hash + ((mustNotEqual == null) ? 0 : mustNotEqual.hashCode());
        hash = 31 * hash + ((mustContain == null) ? 0 : mustContain.hashCode());
        hash = 31 * hash + ((mustContain == null) ? 0 : mustContain.hashCode());
        hash = 31 * hash + ((regex == null) ? 0 : regex.hashCode());
        return hash;
    }

}