/*
 * Copyright 2006 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.stvconsultants.easygloss.javaee;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Map;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.annotation.Resource;
import javax.ejb.EJB;
import javax.ejb.PostActivate;
import javax.ejb.PrePassivate;
import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.PersistenceContext;
import javax.persistence.PersistenceUnit;
import javax.persistence.PostLoad;
import javax.persistence.PostPersist;
import javax.persistence.PostRemove;
import javax.persistence.PostUpdate;
import javax.persistence.PrePersist;
import javax.persistence.PreRemove;
import javax.persistence.PreUpdate;

import com.stvconsultants.easygloss.AbstractGloss;
import com.stvconsultants.easygloss.GlossApplicationError;
import com.stvconsultants.easygloss.footnotes.Footnote;

/**
 * A gloss for applying to JavaEE annotated classes.
 *
 * @author Stephen Connolly
 */
public class JavaEEGloss extends AbstractGloss {

    /**
     * Used to keep differnt classes which will match the same footnote separate in the HashMap.
     */
    static private class InjectableKey {

        private final Class type;
        private final Footnote annotation;

        public InjectableKey(Class type, Footnote annotation) {
            type.getClass(); // throw NPE if null
            annotation.getClass(); // throw NPE if null
            this.type = type;
            this.annotation = annotation;
        }

        public boolean equals(Object obj) {
            final boolean retVal;
            // we will compare based on key fields only
            if (this == obj) {
                retVal = true;
            } else if ((obj == null) || (obj.getClass() != this.getClass())) {
                retVal = false;
            } else {
                // object must be a InjectableKey at this point
                InjectableKey that = (InjectableKey) obj;
                retVal = (this.getType() == that.getType() ||
                        this.getType().equals(that.getType())) &&
                        (this.getAnnotation() == that.getAnnotation() ||
                                this.getAnnotation().equals(that.getAnnotation()));
            }
            return retVal;
        }

        public int hashCode() {
            int hash = 7;
            hash = 31 * hash + getType().hashCode();
            hash = 31 * hash + getAnnotation().hashCode();
            return hash;
        }

        public String toString() {
            StringBuilder buf = new StringBuilder(this.getClass().getSimpleName());
            buf.append("(");
            buf.append(getAnnotation());
            buf.append(", ");
            buf.append(getType());
            buf.append(")");
            return buf.toString();
        }

        public Class getType() {
            return type;
        }

        public Footnote getAnnotation() {
            return annotation;
        }
    }

    private final static String DEFAULT_UNIT_NAME = "";
    private final static String DEFAULT_EJB_NAME = "";
    private final static String DEFAULT_EJB_BEAN_NAME = "";
    private final static Class DEFAULT_EJB_BEAN_INTERFACE = Object.class;
    private final static String DEFAULT_RESOURCE_NAME = "";
    private final static Class DEFAULT_RESOURCE_TYPE = Object.class;
    HashMap<InjectableKey, Object> injectables;

    /**
     * Creates a new instance of JavaEEGloss
     */
    public JavaEEGloss() {
        super();
        injectables = new HashMap<InjectableKey, Object>();
    }

    /**
     * Create a new instance of <code>T</code> and apply the gloss to it. Throws <code>GlossApplicationError</code> if
     * anything goes wrong.
     *
     * @param type The class type to create.
     *
     * @return A freshly glossed instance of class T.
     */
    public <T> T make(Class<T> type) {
        try {
            Constructor<T> ctr = type.getConstructor();
            T instance = ctr.newInstance();
            apply(instance);
            return instance;
        } catch (NoSuchMethodException ex) {
            throw new GlossApplicationError(ex);
        } catch (InvocationTargetException ex) {
            throw new GlossApplicationError(ex);
        } catch (IllegalAccessException ex) {
            throw new GlossApplicationError(ex);
        } catch (InstantiationException ex) {
            throw new GlossApplicationError(ex);
        }
    }

    /**
     * Add a default entity manager to the gloss. That is, one that will only match <code>@PersistenceContext()
     * EntityManager</code> or appropriate setter method.
     *
     * @param em The <code>EntityManager</code> to inject
     */
    public void addEM(EntityManager em) {
        addEM(DEFAULT_UNIT_NAME, em);
    }

    /**
     * Add a general entity manager to the gloss. That is, one that will match any <code>@PersistenceContext</code>
     * annotated <code>EntityManager</code> or appropriate setter method.
     *
     * @param em The <code>EntityManager</code> to inject
     */
    public void addGenericEM(EntityManager em) {
        addEM(null, em);
    }

    /**
     * Add an entity manager to the gloss. That is, one that will only match <code>@PersistenceContext(unitName="<i>unitName</i>")
     * EntityManager</code> or appropriate setter method.
     *
     * @param unitName The <code>unitName</code> to inject for (if null => match any)
     * @param em       The <code>EntityManager</code> to inject
     */
    public void addEM(String unitName, EntityManager em) {
        Footnote annotation = new Footnote(PersistenceContext.class);
        if (unitName != null) {
            annotation.with("unitName", unitName);
        }
        injectables.put(new InjectableKey(EntityManager.class, annotation), em);
    }

    /**
     * Add a default entity manager factory to the gloss. That is, one that will only match <code>@PersistenceUnit()
     * EntityManagerFactory</code> or appropriate setter method.
     *
     * @param emf <code>The EntityManagerFactory</code> to inject
     */
    public void addEMF(EntityManagerFactory emf) {
        addEMF(DEFAULT_UNIT_NAME, emf);
    }

    /**
     * Add a general entity manager factory to the gloss. That is, one that will match any <code>@PersistenceUnit</code>
     * annotated <code>EntityManagerFactory</code> or appropriate setter method.
     *
     * @param emf <code>The EntityManagerFactory</code> to inject
     */
    public void addGenericEMF(EntityManagerFactory emf) {
        addEMF(null, emf);
    }

    /**
     * Add an entity manager factory to the gloss. That is, one that will only match
     * <code>@PersistenceUnit(unitName="<i>unitName</i>") EntityManagerFactory</code> or appropriate setter method.
     *
     * @param unitName The <code>unitName</code> to inject for (if null => match any)
     * @param emf      <code>The EntityManagerFactory</code> to inject
     */
    public void addEMF(String unitName, EntityManagerFactory emf) {
        Footnote annotation = new Footnote(PersistenceUnit.class);
        if (unitName != null) {
            annotation.with("unitName", unitName);
        }
        injectables.put(new InjectableKey(EntityManagerFactory.class,
                annotation), emf);
    }

    /**
     * Add a default EJB to the gloss. That is, one that will match any <code>@EJB()</code> annotated field, property or
     * setter method with which the beanImpl is assignment compatable.
     *
     * @param beanImpl The implementation to inject.
     */
    public void addEJB(Object beanImpl) {
        addEJB(DEFAULT_EJB_NAME, DEFAULT_EJB_BEAN_NAME,
                DEFAULT_EJB_BEAN_INTERFACE, beanImpl);
    }

    /**
     * Add a general EJB to the gloss. That is, one that will match any <code>@EJB()</code> annotated field, property or
     * setter method with which the beanImpl is assignment compatable.
     *
     * @param beanImpl The implementation to inject.
     */
    public void addGenericEJB(Object beanImpl) {
        addEJB(null, null, null, beanImpl);
    }

    /**
     * Add a specific EJB to the gloss. That is, one that will match any <code>@EJB(name="<i>name</i>")</code> annotated
     * field, property or setter method with which the beanImpl is assignment compatable.
     *
     * @param name     The name to match (if null => match any)
     * @param beanImpl The implementation to inject.
     */
    public void addEJB(String name, Object beanImpl) {
        addEJB(name, DEFAULT_EJB_BEAN_NAME, DEFAULT_EJB_BEAN_INTERFACE,
                beanImpl);
    }

    /**
     * Add a specific EJB to the gloss. That is, one that will match any <code>@EJB(name="<i>name</i>",
     * beanName="<i>beanName</i>", beanInterface=<i>beanInterface</i>)</code> annotated field, property or setter method
     * with which the beanImpl is assignment compatable.
     *
     * @param name          The name to match (if null => match any)
     * @param beanName      The beanName to match (if null => match any)
     * @param beanInterface The beanInterface to match (if null => match any)
     * @param beanImpl      The implementation to inject.
     */
    public void addEJB(String name, String beanName, Class beanInterface,
                       Object beanImpl) {
        Footnote annotation = new Footnote(EJB.class);
        if (name != null) {
            annotation.with("name", name);
        }
        if (beanName != null) {
            annotation.with("beanName", beanName);
        }
        if (beanInterface != null) {
            annotation.with("beanInterface", beanInterface);
            ;
        }
        injectables.put(new InjectableKey(beanImpl.getClass(), annotation),
                beanImpl);
    }

    /**
     * Add a default resource to the gloss. That is, one that will match any <code>@Resource()</code> annotated field,
     * property or setter method with which the resource is assignment compatable.
     *
     * @param resource The resource to inject.
     */
    public void addResource(Object resource) {
        addResource(DEFAULT_RESOURCE_NAME, DEFAULT_RESOURCE_TYPE, resource);
    }

    /**
     * Add a general resource to the gloss. That is, one that will match any <code>@Resource()</code> annotated field,
     * property or setter method with which the resource is assignment compatable.
     *
     * @param resource The resource to inject.
     */
    public void addGenericResource(Object resource) {
        addResource(null, null, resource);
    }

    /**
     * Add a specific resource to the gloss. That is, one that will match any <code>@Resource(name="<i>name</i>")</code>
     * annotated field, property or setter method with which the resource is assignment compatable.
     *
     * @param name     The name to match (if null => match any)
     * @param resource The resource to inject.
     */
    public void addResource(String name, Object resource) {
        addResource(name, DEFAULT_RESOURCE_TYPE, resource);
    }

    /**
     * Add a specific resource to the gloss. That is, one that will match any <code>@Resource(name="<i>name</i>",
     * type=<i>type</i>)</code> annotated field, property or setter method with which the resource is assignment
     * compatable.
     *
     * @param name     The name to match (if null => match any)
     * @param type     The type to match (if null => match any)
     * @param resource The resource to inject.
     */
    public void addResource(String name, Class type, Object resource) {
        Footnote annotation = new Footnote(Resource.class);
        if (name != null) {
            annotation.with("name", name);
        }
        if (type != null) {
            annotation.with("type", type);
        }
        injectables.put(new InjectableKey(resource.getClass(), annotation),
                resource);
    }

    /**
     * Apply the gloss to the instance.  The <code>@PostConstruct</code> annotated method (if present) will be called
     * after application of the gloss.
     *
     * @param instance The instance to apply the gloss to.
     */
    public void apply(Object instance) {
        for (Map.Entry<InjectableKey, Object> injectable : injectables.entrySet()) {
            apply(injectable.getKey().getAnnotation(), instance,
                    injectable.getValue());
        }
        Footnote annotation = new Footnote(PostConstruct.class);
        invoke(annotation, instance);
    }

    /**
     * Apply the gloss to the class
     *
     * @param instanceClass The class to which the gloss should be applied
     */
    public void applyStatic(Class instanceClass) {
        for (Map.Entry<InjectableKey, Object> injectable : injectables.entrySet()) {
            applyStatic(injectable.getKey().getAnnotation(), instanceClass,
                    injectable.getValue());
        }
        // cannot call a PostConstruct since we don't have an instance
    }

    /**
     * Invoke any <code>@PostActivate</code> annotated callbacks in the instance.
     *
     * @param instance The instance to invoke.
     */
    public void afterActivate(Object instance) {
        invoke(new Footnote(PostActivate.class), instance);
    }

    /**
     * Invoke any <code>@PostLoad</code> annotated callbacks in the instance.
     *
     * @param instance The instance to invoke.
     */
    public void afterLoad(Object instance) {
        invoke(new Footnote(PostLoad.class), instance);
    }

    /**
     * Invoke any <code>@PostPersist</code> annotated callbacks in the instance.
     *
     * @param instance The instance to invoke.
     */
    public void afterPersist(Object instance) {
        invoke(new Footnote(PostPersist.class), instance);
    }

    /**
     * Invoke any <code>@PostRemove</code> annotated callbacks in the instance.
     *
     * @param instance The instance to invoke.
     */
    public void afterRemove(Object instance) {
        invoke(new Footnote(PostRemove.class), instance);
    }

    /**
     * Invoke any <code>@PostUpdate</code> annotated callbacks in the instance.
     *
     * @param instance The instance to invoke.
     */
    public void afterUpdate(Object instance) {
        invoke(new Footnote(PostUpdate.class), instance);
    }

    /**
     * Invoke any <code>@PreDestroy</code> annotated callbacks in the instance.
     *
     * @param instance The instance to invoke.
     */
    public void beforeDestroy(Object instance) {
        invoke(new Footnote(PreDestroy.class), instance);
    }

    /**
     * Invoke any <code>@PrePassivate</code> annotated callbacks in the instance.
     *
     * @param instance The instance to invoke.
     */
    public void beforePassivate(Object instance) {
        invoke(new Footnote(PrePassivate.class), instance);
    }

    /**
     * Invoke any <code>@PrePersist</code> annotated callbacks in the instance.
     *
     * @param instance The instance to invoke.
     */
    public void beforePersist(Object instance) {
        invoke(new Footnote(PrePersist.class), instance);
    }

    /**
     * Invoke any <code>@PreRemove</code> annotated callbacks in the instance.
     *
     * @param instance The instance to invoke.
     */
    public void beforeRemove(Object instance) {
        invoke(new Footnote(PreRemove.class), instance);
    }

    /**
     * Invoke any <code>@PreUpdate</code> annotated callbacks in the instance.
     *
     * @param instance The instance to invoke.
     */
    public void beforeUpdate(Object instance) {
        invoke(new Footnote(PreUpdate.class), instance);
    }

    /**
     * Generate a string representation of this gloss.
     *
     * @returns The string representation of this gloss
     */
    public String toString() {
        StringBuilder buf = new StringBuilder(this.getClass().getName());
        buf.append("(\n");
        boolean first = true;
        for (Map.Entry<InjectableKey, Object> injectable : injectables.entrySet()) {
            if (first) {
                first = false;
            } else {
                buf.append(",\n");
            }
            buf.append("\t");
            buf.append(injectable.getKey());
            buf.append("\n\t\t<= ");
            buf.append(injectable.getValue());
        }
        buf.append("\n)");
        return buf.toString();
    }

    /**
     * Compare this instance with another instance.
     *
     * @param obj The instance to compare with.
     *
     * @returns <code>true</code> if the two instances are logically equivalent.
     */
    public boolean equals(Object obj) {
        final boolean retVal;

        if (this == obj) {
            retVal = true;
        } else if ((obj == null) || (obj.getClass() != this.getClass())) {
            retVal = false;
        } else {
            // object must be a JavaEEGloss at this point
            JavaEEGloss that = (JavaEEGloss) obj;
            // this.injectables cannot equal that.injectables as they are 
            // instance specific instances
            retVal = this.injectables.equals(that.injectables);
        }
        return retVal;
    }

    /**
     * Calculate the hashCode for this instance.
     *
     * @returns the hash code.
     */
    public int hashCode() {
        int hash = 7;
        hash = 31 * hash + injectables.hashCode();
        return hash;
    }

}
