/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2008-2010 Oracle and/or its affiliates. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License.  You can
 * obtain a copy of the License at
 * https://glassfish.dev.java.net/public/CDDL+GPL_1_1.html
 * or packager/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 *
 * GPL Classpath Exception:
 * Oracle designates this particular file as subject to the "Classpath"
 * exception as provided by Oracle in the GPL Version 2 section of the License
 * file that accompanied this code.
 *
 * Modifications:
 * If applicable, add the following below the License Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyright [year] [name of copyright owner]"
 *
 * Contributor(s):
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package com.sun.pkg.client;

import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * An <code>Fmri</code> is an identifier for a package. 
 * 
 * The acronym stands for Fault Management Resource Identifier (even though 
 * IPS has little to do with 
 * fault management). The Fmri includes the name and version for the package.
 * For more information about the format and behavior of the version part of the
 * Fmri, see the {@link Version} class.
 * @see Version
 * @author trm
 */
public class Fmri implements Comparable<Fmri>, Cloneable
{
    public static final String PREF_AUTH_PFX = "_PRE";
    public static final Fmri nullFmri = new Fmri();

    String name = "";
    Version version = Version.nullVersion;
    String authname = "";
    
    /** 
     * Create an empty Fmri with a blank package name and a version of 0.
     */
    private Fmri() { }
    
    /**
     * Create an Fmri from a string representation.
     */
    public Fmri(String p) {
         int veridx = p.lastIndexOf('@');
         int nameidx;
         if (p.startsWith("pkg://")) {
             nameidx = p.indexOf('/', 6) + 1;
             authname = p.substring(6, nameidx - 1);
         }
         else if (p.startsWith("pkg:/")) {
             nameidx = 5;
         }
         else {
             nameidx = 0;
         }
         if (veridx != -1) {
             version = new Version(p.substring(veridx + 1));
             name = p.substring(nameidx, veridx);
         }
         else {
             version = Version.nullVersion;
             name = p.substring(nameidx);
         }
    }
    
    /**
     * Create an Fmri from a package name and version.
     */
    public Fmri(String name, String version) {
        this.name = name;
        this.version = new Version(version);
    }
    
    /** 
     * Create an Fmri from the meta directory for the package
     */
    Fmri(File dir) {
        try {
            String verStr = URLDecoder.decode(dir.getName(), "UTF-8");
            version = new Version(verStr);
        } catch (UnsupportedEncodingException ex) {
            Logger.getLogger(Fmri.class.getName()).log(Level.SEVERE, null, ex);
            version = new Version("");
        }
        name = dir.getParentFile().getName();
   }
    
    @Override
    public Fmri clone() {
        try {
            return (Fmri) super.clone();
        } catch (CloneNotSupportedException ex) {
            return null;
        }
    }
    
    /**
     * Obtain the authority from the Fmri.
     * 
     * @return The name of the authority. If the authority hasn't been set, 
     * return the empty string.
     */
    public String getAuthority() {
        return authname;
    }
    
    /**
     * Set the authority for the Fmri.
     * @param name - the name of the authority
     */
    void setAuthority(String name) {
        authname = name;
    }
    
    /**
     * Obtain the name of the package
     * 
     * @return The name of the package
     */
    public String getName() {
        return name;
    }
    
    /**
     * Obtain the version of the package
     * 
     * @return The version of the package
     */
    public Version getVersion() {
        return version;
    }
    
    /**
     * Return the directory that is used to store information about all versions 
     * of this package
     * 
     * @param pkgdir The directory within the image for storing all package meta data
     * @return The directory used for storing information about Fmris having the 
     * same name as this Fmri
     */
    public File getPkgDir(File pkgdir) throws IOException {
        return new File(pkgdir, URLEncoder.encode(name, "UTF-8"));
    }
            
    
    /**
     * Return the directory that is used to store information about this particular
     * package version.
     * 
     * @param pkgdir The directory within the image for storing all package meta data
     * @return The directory where information about this Fmri is stored.
     */
    public File getPkgVersionDir(File pkgdir) throws IOException {
        return new File(new File(pkgdir, URLEncoder.encode(name, "UTF-8")), 
                URLEncoder.encode(version.toString(), "UTF-8"));
    }
    
    /**
     * Return the URL path that is used to look up information about this Fmri.
     * 
     * @return A String that is used in a URL to get information about this Fmri.
     * @throws java.io.IOException
     */
    public String getURLPath() throws IOException {
        return URLEncoder.encode(name, "UTF-8") + "@" 
                + URLEncoder.encode(version.toString(), "UTF-8");
    }

    /**
     * Return the path to use in the state directory for looking up information
     * about this package.
     * @return A String naming the link that is used for information about this Fmri
     */
    String getLinkFileName() throws IOException {
        return getURLPath();
    }
    
    /**
     * Return the String representation of the Fmri.
     * 
     * @return the String representation of the Fmri.
     */
    @Override
    public String toString() {
        return "pkg:" + (authname.length() == 0 ? "" : "//" + authname) + "/" + name + "@" + version;
    }

    /**
     * Return the String representation of the Fmri without the authority name.
     */
    String toStringWithoutAuthority() {
        return "pkg:/" + name + "@" + version;
    }

    /**
     * Compare to Fmris for equality.
     * @return True if the two Fmris are equal
     */
    @Override
    public boolean equals(Object o) {
        Fmri fmri = (Fmri)o;
        return name.equals(fmri.name) && version.equals(fmri.version);
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 19 * hash + (this.name != null ? this.name.hashCode() : 0);
        hash = 19 * hash + (this.version != null ? this.version.hashCode() : 0);
        return hash;
    }
    
    /**
     * Compare two Fmris.
     * @param fmri the Fmri to compare with this Fmri
     * @return -1 if this Fmri is less than fmri, 0 if the Fmris are equal, 1 if
     * this Fmri is greater than fmri. 
     */
    public int compareTo(Fmri fmri) {
        int i = name.compareTo(fmri.name);
        if (i != 0) return i;
        return version.compareTo(fmri.version);
    }
    
    /**
     * Check if the given Fmri matches this Fmri.
     * 
     * @return True if the given Fmri matches this Fmri
     */
    public boolean matches(Fmri f) {
        return name.equals(f.name) && version.matches(f.version);
    }

    /**
     * Determine whether the given Fmri is for the same package as this Fmri,
     * even if for different versions.
     * @param f a given Fmri
     * @return True if the Fmris are for the same package
     */
    public boolean isSamePackage(Fmri f) {
        return name.equals(f.name);
    }

    /**
     * Determine whether this Fmri is a successor to the given Fmri
     * @param f the given Fmri
     * @return True if the two Fmris are for the same package, and the version 
     * of this Fmri is greater than the version of the given Fmri
     */
    public boolean isSuccessor(Fmri f) {
        if (!isSamePackage(f)) return false;
        return version.compareTo(f.version) >= 0;
    }

}
