/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2008-2011 Oracle and/or its affiliates. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License.  You can
 * obtain a copy of the License at
 * https://glassfish.dev.java.net/public/CDDL+GPL_1_1.html
 * or packager/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 *
 * GPL Classpath Exception:
 * Oracle designates this particular file as subject to the "Classpath"
 * exception as provided by Oracle in the GPL Version 2 section of the License
 * file that accompanied this code.
 *
 * Modifications:
 * If applicable, add the following below the License Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyright [year] [name of copyright owner]"
 *
 * Contributor(s):
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package com.sun.pkg.client;

import com.sun.pkg.client.Constraint.ConstraintException;
import com.sun.pkg.util.PEMUtil;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.net.HttpURLConnection;
import java.net.Proxy;
import java.net.ProxySelector;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.HashSet;
import java.util.Collection;
import java.util.Stack;
import java.util.UUID;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSocketFactory;


/**
 * An <code>Image</code> is a directory tree containing the laid-down contents
 * of a self-consistent graph of Packages.
 * <p>
 * An Image has a root path.
 * 
 * @author trm
 */
public class Image
{
    static class Authority 
    {
        String prefix;
        String origin;
        // List<String> mirrors; // not currently handled
        String uuid;
        File certFile;
        File keyFile;
        SSLSocketFactory sslfactory = null; // filled in as needed
        
        Authority(String name)
        {
            prefix = name;
        }
    }
    
    /**
     * An <code>FmriState</code> represents the current state of an Fmri within
     * the image.
     */
    public static class FmriState 
    {
        public Fmri fmri;
        public boolean installed = false;
        public boolean upgradable = false;
        public boolean frozen = false;
        public boolean incorporated = false;
        public boolean excludes = false;
    }

    enum PlanState { UNEVALUATED, EVALUATED_OK, EVALUATED_ERROR, 
                     EXECUTED_OK, EXECUTED_ERROR }
    
    /* not used */
    static class Filter {}

    class CatalogCache
    {
        class SubCacheEntry {
            Fmri fmri;
            List<Authority> auths = new ArrayList<Authority>();
            SubCacheEntry(Fmri f, Authority a) {
                fmri = f;
                auths.add(a);
            }
        }
        class CacheEntry {
            SortedSet<Version> versions = new TreeSet<Version>();
            HashMap<Version, SubCacheEntry> versionInfo = new HashMap<Version, SubCacheEntry>();
            CacheEntry(Fmri f, Authority a) {
                Version v = f.getVersion();
                versions.add(v);
                versionInfo.put(v, new SubCacheEntry(f, a));
            }
        }
        HashMap<String, CacheEntry> pkgs = new HashMap<String, CacheEntry>();
        
        void cacheInstalledPackages() throws IOException {
            for (Fmri f : getInstalledFmrisFromImage()) {
                Authority a = getAuthorityForInstalledPkg(f);
                if (a == null) throw new RuntimeException("can't determine authority for installed package: " + f);
                cacheFmri(f, a);
            }
        }
        
        void addCatalog(Catalog c, Authority a) {
            for (Fmri f : c.getFmris()) {
                cacheFmri(f, a);
            }
        }
        
        private void cacheFmri(Fmri f, Authority a) {
            String pname = f.getName();
            Version v = f.getVersion();               
            CacheEntry ce = pkgs.get(pname);
            if (ce == null) {
                pkgs.put(pname, new CacheEntry(f, a));
            } else {
                SubCacheEntry sce = ce.versionInfo.get(v);
                if (sce == null) {
                    ce.versionInfo.put(v, new SubCacheEntry(f, a));
                    ce.versions.add(v);
                } else {
                    if (!sce.auths.contains(a)) sce.auths.add(a);
                }
            }              
            
        }
        
        List<Fmri> multimatch(String pname, Fmri pkgFmris[]) {
            List<Fmri> matches = new ArrayList<Fmri>();
            pname = "/" + pname;
            for (Fmri f : pkgFmris) {
                if (pname.endsWith("/" + f.getName())) matches.add(f);
            }
            return matches;
        }
        
        List<FmriState> getInventory(String patternNames[], boolean allKnown) {
            if (patternNames == null) patternNames = new String[0];
            Fmri patternFmris[] = new Fmri[patternNames.length];
            for (int i = 0; i < patternNames.length; i++) {
                patternFmris[i] = new Fmri(patternNames[i]);
            }
            List<FmriState> results = new ArrayList<FmriState>();
            String pauth = getPreferredAuthorityName();
            for (String name : pkgs.keySet()) {
                List<Fmri> matches = multimatch(name, patternFmris);
                if (patternFmris.length > 0 && matches.size() == 0) continue;
                Version newest = pkgs.get(name).versions.last();
                Version vlist[] = pkgs.get(name).versions.toArray(new Version[0]);
                HashMap<Version, SubCacheEntry> vinfo = pkgs.get(name).versionInfo;
                // traverse through the version list in reverse order (newest to oldest)
                for (int i = vlist.length - 1; i >= 0; i--) {
                    List<Fmri> vmatches = new ArrayList<Fmri>();
                    // If a pattern specified a version and that version
                    // isn't succeeeded by vlist[i], then remove it from 
                    // consideration
                    for (Fmri m : matches) {
                        Version mv = m.getVersion();
                        if (mv.isNull() || vlist[i].isSuccessor(mv)) {
                            vmatches.add(m);
                        }
                    }
                    // If no contenders remain and we had some to begin with,
                    // go to the next version
                    if (matches.size() > 0 && vmatches.size() == 0) continue;
                    
                    // Now do the same thing with authories
                    List<Authority> authlist = vinfo.get(vlist[i]).auths;
                    List<Fmri> amatches = new ArrayList<Fmri>();
                    for (Fmri m : vmatches) {
                        String mauth = m.getAuthority();
                        if (mauth.length() == 0 || authlist.contains(authorities.get(mauth))) {
                            amatches.add(m);
                        }
                    }
                    if (vmatches.size() > 0 && amatches.size() == 0) continue;
                    
                    // If no patterns were specified or any still-
                    // matching pattern specified no authority, we
                    // use the entire authlist for this version.
                    // Otherwise, we use the intersection of authlist
                    // and the auths in the patterns.
                    List<Authority> alist = null;
                    if (amatches.size() == 0) alist = authlist;
                    else {
                        for (Fmri m : amatches) {
                            if (m.getAuthority().length() == 0) {
                                alist = authlist;
                                break;
                            }
                        }
                        if (alist == null) {
                            alist = new ArrayList<Authority>();
                            for (Authority a : authlist) {
                                for (Fmri m : amatches) {
                                    if (a.prefix.equals(m.getAuthority())) {
                                        alist.add(a);
                                    }
                                }
                            }
                        }
                    }
                    
                    // Now we have an Fmri (vinfo.get(vlist[i]).fmri) and a list of 
                    // authorities (alist) that it comes from.
                    // Add them to the results.
                    Fmri vfmri = vinfo.get(vlist[i]).fmri;
                    for (Authority a : alist) {
                        FmriState fs = new FmriState();
                        fs.fmri = vfmri.clone();
                        fs.fmri.setAuthority(a.prefix);
                        fs.upgradable = !vlist[i].equals(newest);
                        fs.installed = isInstalled(fs.fmri);
                        if (!allKnown && !fs.installed) continue;
                        results.add(fs);
                    }
                }
            }
            return results;       
        }
        
        /**
         * Get the first package matching pattern, taking from the preferred
         * repository if no repository is specified in the pattern.
         * 
         * @return the first matching Fmri
         **/
        Fmri getFirst(String pattern) {
            String plist[] = { pattern };
            List<FmriState> lfs = getInventory(plist, true);
            String pauth = getPreferredAuthorityName();
            Fmri firstNonPref = null;
            for (FmriState fs : lfs) {
                if (fs.fmri.getAuthority().equals(pauth)) return fs.fmri;
                if (firstNonPref == null) firstNonPref = fs.fmri;
            }
            return firstNonPref;
        }
    }

    /**
     * An <code>ImagePlan</code> takes a list of requested packages, an Image (and its
     * policy restrictions), and returns the set of package operations needed
     * to transform the Image to the list of requested packages.
     * <p>
     * Use of an ImagePlan involves the identification of the Image, the
     * Catalogs (implicitly), and a set of complete or partial package FMRIs.
     * The Image's policy, which is derived from its type and configuration
     * will cause the formulation of the plan or an exception state.
     * <p>
     * XXX In the current formulation, an ImagePlan can handle [null ->
     * PkgFmri] and [PkgFmri@Version1 -> PkgFmri@Version2], for a set of
     * PkgFmri objects.  With a correct Action object definition, deletion
     * should be able to be represented as [PkgFmri@V1 -> null].
     * <p>
     * XXX Should we allow downgrades?  There's an "arrow of time" associated
     * with the smf(5) configuration method, so it's better to direct
     * manipulators to snapshot-based rollback, but if people are going to do
     * "pkg delete fmri; pkg install fmri@v(n - 1)", then we'd better have a
     * plan to identify when this operation is safe or unsafe.
     */
    public class ImagePlan
    {
        private PlanState state;
        private boolean recursiveRemoval;
        private List<Fmri> targetFmris = new ArrayList<Fmri>();
        private List<Fmri> targetRemFmris = new ArrayList<Fmri>();
        private List<PackagePlan> pkgPlans = new ArrayList<PackagePlan>();
        private List<String> directories = null;
        
        ImagePlan(boolean recursiveRemoval, Filter filters[]) {
            state = PlanState.UNEVALUATED;
            this.recursiveRemoval = recursiveRemoval;

            // XXX add filter support (join image filters with filters arg)
        }
        
        /**
         * Generate a string representation for the ImagePlan.
         * @return a String representing the ImagePlan
         */
        @Override
        public String toString() {
            StringBuffer s = new StringBuffer();
            if (state == PlanState.UNEVALUATED) {
                s.append("UNEVALUTED:\n");
                for (Fmri t : targetFmris) s.append("+" + t + "\n");
                for (Fmri t : targetRemFmris) s.append("-" + t + "\n");
            } else {
                for (PackagePlan pp : pkgPlans) s.append(pp + "\n");
            }
            return s.toString();
        }
        
        /**
         * Return the Fmris that are proposed for installation as part of 
         * this plan.
         * 
         * @return the list of proposed Fmris
         */
        public Fmri[] getProposedFmris() {
            return targetFmris.toArray(new Fmri[targetFmris.size()]);
        }
        
        boolean isProposedFmri(Fmri fmri) {
            for (Fmri pf : targetFmris) {
                if (fmriIsSamePackage(fmri, pf)) {
                    return !fmriIsSuccessor(fmri, pf);
                }
            }
            return false;
        }

        boolean isProposedRemFmri(Fmri fmri) {
            for (Fmri pf : targetRemFmris) {
                if (fmriIsSamePackage(fmri, pf)) return true;
            }
            return false;
        }
        
        /*
         * Return version of fmri already proposed, or null if not proposed yet.
         */
        Fmri getProposedVersion(Fmri fmri)
        {
            for (Fmri p : targetFmris) {
                if (fmri.getName().equals(p.getName())) return p;
            }
            return null;
        }

        void proposeFmri(Fmri fmri) throws IOException, ConstraintException {
            if (hasVersionInstalled(fmri)) return;
            fmri = constraints.applyConstraintsToFmri(fmri);
            setFmriDefaultAuthority(fmri);
            // Add fmri to target list only if it (or a successor) isn't
            // there already.
            for (Fmri p : targetFmris) {
                if (fmriIsSuccessor(fmri, p)) {
                    targetFmris.remove(p);
                    break;
                }
                if (fmriIsSuccessor(p, fmri)) return;
            }
            targetFmris.add(fmri);
        }

        boolean olderVersionProposed(Fmri fmri) {
            // returns true if older version of this fmri has been proposed already
            for (Fmri p : targetFmris) {
                if (fmriIsSuccessor(fmri, p)) return true;
            }
            return false;
        }
        
        // XXX Need to make sure that the same package isn't being added and
        // removed in the same imageplan.
        void proposeFmriRemoval(Fmri fmri) throws IOException {
            if (!hasVersionInstalled(fmri)) return;
            for (Fmri p : targetRemFmris) {
                if (fmriIsSuccessor(fmri, p)) {
                    targetRemFmris.remove(p);
                    break;
                }
            }
            targetRemFmris.add(fmri);
        }

        List<Fmri> genNewInstalledPackages() {
            if (state != PlanState.EVALUATED_OK) {
                throw new RuntimeException("invalid image plan state");
            }
            List<Fmri> fmriset = getInstalledFmrisFromCache();
            for (PackagePlan p : pkgPlans) {
                p.updatePackageSet(fmriset);
            }
            return fmriset;
        }

        Collection<String> getDirectories() throws IOException {
            // return list of all directories in target image
            if (directories == null) {
                directories = new ArrayList<String>();
                directories.add("var/pkg");
                directories.add("var/sadm/install");
                for (Fmri fmri : genNewInstalledPackages()) {
                    for (Action act : getManifest(fmri).getActions()) {
                        List<String> dl = act.getReferencedDirectories();
                        if (dl != null) directories.addAll(dl);
                    }
                }
            }
            return directories;
        }
        
        void evaluateFmri(Fmri pfmri) throws IOException, ConstraintException {
            pushTarget(pfmri, "process");
            setFmriDefaultAuthority(pfmri);
            Manifest m = getManifest(pfmri);
            // examine manifest for dependencies
            List<DependAction> lda = m.getActionsByType(DependAction.class);
            if (constraints.startLoading(pfmri)) {
                for (DependAction da : lda) {
                    constraints.updateConstraints(da.getConstraint());
                }
                constraints.finishLoading(pfmri);
            }
            for (DependAction da : lda) {
                // discover if we have an installed or proposed version of 
                // this pkg already; proposed fmris will always be newer
                Fmri refFmri = getProposedVersion(da.getTargetFmri());
                if (refFmri == null) {
                    refFmri = getVersionInstalled(da.getTargetFmri());
                }
                // check if new constraint requires us to make any changes
                // to already proposed pkgs or existing ones
                if (da.getConstraint().checkForWork(refFmri) == null) continue;
             
                Fmri daf = constraints.applyConstraintsToFmri(da.getTargetFmri());
                // This will be the newest version of the specified
                // dependency package, coming from the preferred
                // authority, if it's available there.
                Fmri cf = catalogCache.getFirst(daf.toString());
                if (cf == null) {
                    throw new IOException("Unable to resolve dependency on package:" + da.keyValue());
                }
                proposeFmri(cf);
                evaluateFmri(cf);
            }
            popTarget();
        }
        
        void addPackagePlan(Fmri pfmri) throws IOException {
            Manifest m = getManifest(pfmri);
            PackagePlan pp = new PackagePlan();
            pp.proposeDestination(pfmri, m);
            pp.evaluate();
            pkgPlans.add(pp);
        }
        
        void evaluateFmriRemoval(Fmri pfmri) throws IOException {
            List<Fmri> dependents = getDependents(pfmri);
            // iterate on a copy
            for (Fmri d : new ArrayList<Fmri>(dependents)) {
                if (targetRemFmris.contains(d)) {
                    dependents.remove(d);
                }
            }
            if (dependents.size() > 0 && !recursiveRemoval) {
                StringBuffer msg = new StringBuffer();
                msg.append("Cannot remove '" + pfmri + 
                        "' due to the following packages that directly " +
                        "depend on it:\n");
                for (Fmri d : dependents) {
                    msg.append(d.toString());
                    msg.append("\n");
                }
                throw new IllegalArgumentException(msg.toString());
            }
            pushTarget(pfmri, "process");
            Manifest m = getManifest(pfmri);
            PackagePlan pp = new PackagePlan();
            pp.proposeRemoval(pfmri, m);
            pp.evaluate();
            
            for (Fmri d : dependents) {
                if (isProposedRemFmri(d)) continue;
                if (hasVersionInstalled(d)) continue;
                targetRemFmris.add(d);
                evaluateFmriRemoval(d);
            }
            // Post-order append will ensure topological sorting for acyclic
            // dependency graphs.  Cycles need to be arbitrarily broken, and
            // are done so in the loop above.
            pkgPlans.add(pp);
            popTarget();
        }
        
        void evaluate() throws IOException, ConstraintException {
            // Operate on a copy, as it will be modified in flight
            List<Fmri> targetFmrisCopy = new ArrayList<Fmri>(targetFmris);
            for (Fmri f : targetFmrisCopy) evaluateFmri(f);
            for (Fmri f : targetFmris) addPackagePlan(f);
            for (Fmri f : targetRemFmris) evaluateFmriRemoval(f);
            state = PlanState.EVALUATED_OK;
        }
        
        boolean nothingToDo() {
            return pkgPlans.size() == 0;
        }

        /**
         * Computes the total number of bytes that will be downloaded during {@link #execute()}.
         */
        public long computeTransferSize() {
            long r = 0;
            for (PackagePlan p : pkgPlans) {
                r += p.computeTransferSize();
            }
            return r;
        }

        /**
         * Computes the total number of files that will be downloaded during {@link #execute()}.
         */
        public int computeTransferFiles() {
            int r = 0;
            for (PackagePlan p : pkgPlans) {
                r += p.computeTransferFiles();
            }
            return r;
        }

        /**
         * Executes the plan with the empty callback.
         *
         * <p>
         * See {@link #execute(ImagePlanProgressTracker)} for the exact semantics of the method.
         */
        public void execute() throws IOException {
            // pass in empty callback
            execute(true, true);
        }

        /**
         * Executes the plan with the empty callback.
         *
         * <p>
         * See {@link #execute(ImagePlanProgressTracker)} for the exact semantics of the method.
         *
         * @param download boolean variable to trigger download
         * @param apply boolean variable to trigger installtion for downloaded packages
         */
        public void execute(boolean download, boolean apply) throws IOException {
            // pass in empty callback
            execute(new ImagePlanProgressTracker(), download, apply);
        }

        /**
         * Execute the actions required to bring about the plan.
         * <p>
         * This results in the installation or uninstallation as specified by
         * the plan.
         *
         * @param tracker
         *      The object that receives the progress of the execution. Must not be null.
         *
         * @throws java.io.IOException
         */
        public void execute(ImagePlanProgressTracker tracker) throws IOException {
            execute(tracker, true, true);
        }
        /**
         * Execute the actions required to bring about the plan.
         * <p>
         * This results in the installation or uninstallation as specified by
         * the plan.
         *
         * @param tracker
         *      The object that receives the progress of the execution. Must not be null.
         * @param download boolean variable to trigger download
         * @param apply boolean variable to trigger installtion for downloaded packages
         *
         * @throws java.io.IOException
         */
        
        public void execute(ImagePlanProgressTracker tracker, boolean download, boolean apply) throws IOException {
            // Invoke the evaluated image plan
            // preexecute, execute, and postexecute
            // execute actions need to be sorted across packages
            if (nothingToDo()) {
                    state = PlanState.EXECUTED_OK;
                    return;
                }
            
            if(download) {

                // determine the number of packages that perform some downloads.
                int n=0;
                for (PackagePlan pp : pkgPlans) {
                    if(pp.flist.computeTransferFiles()>0)
                        n++;
                }
                tracker.startDownloadPhase(n);
                for (PackagePlan p : pkgPlans) p.preexecute(tracker);
                tracker.endDownloadPhase();
                File install_cache = new File(metadir, "install_cache");
                Set<String> downloadedPkgNames = new HashSet<String>();
                for (PackagePlan p : pkgPlans) {
                    downloadedPkgNames.add(p.destinationFmri());
                }
                downloadedPkgs.retainAll(downloadedPkgNames);
                writeInstallCache(install_cache, downloadedPkgs, true);
            }
            // now we're ready to start install.  At this point we
            // should do a merge between removals and installs so that
            // any actions moving from pkg to pkg are seen as updates rather
            // than removal and re-install, since these two have separate
            // semantics.
            //
            // General install method is removals, updates and then 
            // installs.  
            
            // generate list of removal actions
            if(apply) {
                List<Action> removals = new ArrayList<Action>();
                for (PackagePlan pp : pkgPlans) {
                    removals.addAll(pp.getRemovalActions());
                }

                // remove dir removals if dir is still in final image
                // operate on a copy so that we can modify the removals
                Collection<String> imagedirs = getDirectories();
                for (Action a : new ArrayList<Action>(removals)) {
                    if (a instanceof DirAction &&
                            imagedirs.contains(((DirAction)a).path)) {
                        removals.remove(a);
                    }
                }

                // generate list of update actions
                List<Manifest.Difference.ActionPair> updates = new ArrayList<Manifest.Difference.ActionPair>();
                for (PackagePlan pp : pkgPlans) {
                    updates.addAll(pp.getUpdateActions());
                }

                // move any user/group actions into modify list to
                // permit package to add user/group and change existing
                // files to that user/group in a single update
                List<Action> installs = new ArrayList<Action>();
                for (PackagePlan pp : pkgPlans) {
                    installs.addAll(pp.getInstallActions());
                }
                // iterate over copy since we're modifying installs
                for (Action a : new ArrayList<Action>(installs)) {
                    if (a instanceof UserAction || a instanceof GroupAction) {
                        updates.add(new Manifest.Difference.ActionPair(null, a));
                        installs.remove(a);
                    }
                }

                tracker.startActions(removals.size()+updates.size()+installs.size());

                // sort and execute removals
                Collections.sort(removals, Collections.reverseOrder());
                log.log(Level.FINE, "removalphase");
                tracker.startRemovalPhase(removals.size());
                for (Action a : removals) {
                    tracker.onRemovalAction(a);
                    a.remove();
                }
                tracker.endRemovalPhase();

                // sort and execute updates
                Collections.sort(updates, new Comparator<Manifest.Difference.ActionPair>() {
                    public int compare(Manifest.Difference.ActionPair o1,
                                       Manifest.Difference.ActionPair o2) {
                        return o1.to.compareTo(o2.to);
                    }
                    public boolean equals(Manifest.Difference.ActionPair o1,
                                       Manifest.Difference.ActionPair o2) {
                        return o1.to.equals(o2.to);
                    }
                });
                log.log(Level.FINE, "updatephase");
                tracker.startUpdatePhase(updates.size());
                for (Manifest.Difference.ActionPair ap : updates) {
                    tracker.onUpdateAction(ap.from,ap.to);
                    ap.to.install(ap.from);
                }
                tracker.endUpdatePhase();

                // sort and execute installs
                Collections.sort(installs);
                log.log(Level.FINE, "installphase");
                tracker.startInstallPhase(installs.size());
                for (Action a : installs) {
                    tracker.onInstallAction(a);
                    a.install(null);
                }
                tracker.endInstallPhase();

                for (PackagePlan p : pkgPlans) p.postExecute();

                state = PlanState.EXECUTED_OK;

                // remove index as it is now out of date
                // TODO - actually update the index instead
                deleteTree(new File(getMetaDirectory(), "index"));

                File f = new File(metadir, "install_cache");
                Set<String> pkgNames = new HashSet<String>();
                readInstallCache(f, pkgNames);
                Collection<String> pkgToBeRemoved = new ArrayList<String>();
                for (PackagePlan p : pkgPlans) {
                    pkgToBeRemoved.add(p.destinationFmri());
                }
                if(pkgNames.removeAll(pkgToBeRemoved)) {
                   writeInstallCache(f, pkgNames, false);
                }
              }
            }
        }
 
    
    /**
     * A package plan takes two package FMRIs and an Image, and produces the
     * set of actions required to take the Image from the origin FMRI to the
     * destination FMRI.
     * 
     * If the destination FMRI is None, the package is removed.
     */
    class PackagePlan
    {
        private Fmri originFmri = null;
        private Fmri destFmri = null;
        private Manifest originManifest = Manifest.nullManifest;
        private Manifest destManifest = Manifest.nullManifest;
        private Manifest.Difference diff;
        private FileList flist;
       
        PackagePlan() {}
        
        @Override
        public String toString() {
            return "" + originFmri + " -> " + destFmri;
        }

        public String destinationFmri() {
            return ""+destFmri;
        }

        void proposeDestination(Fmri pfmri, Manifest m) {
            destFmri = pfmri;
            destManifest = m;
            if (isInstalled(pfmri)) {
                throw new IllegalArgumentException("already installed: " + pfmri);
            }
        }

        void proposeRemoval(Fmri pfmri, Manifest m) {
            originFmri = pfmri;
            originManifest = m;
            if (!isInstalled(pfmri)) {
                throw new IllegalArgumentException("not installed: " + pfmri);
            }
        }

        void updatePackageSet(List<Fmri> fmriset) {
            if (originFmri != null) fmriset.remove(originFmri);
            if (destFmri != null) fmriset.add(destFmri);
        }

        void evaluate() throws IOException {
            // Determine the actions required to transition the package
            
            // if origin unset, determine if we're dealing with a previously
            // installed version or if we're dealing with the null package
            if (originFmri == null) {
                Fmri f = getInstalledOlderVersion(destFmri);
                if (f != null) {
                    originFmri = f;
                    originManifest = getManifest(f);
                }
            }
            // Assume that origin actions are unique, but make sure that
            // destination ones are.
            if (destManifest != null) {
                // First filter out any actions that don't match image variants
                destManifest.filterByVariants(variants);
                Action[] dup = destManifest.findDuplicates();
                if (dup != null)
                    throw new IllegalArgumentException("manifest of " + destFmri + " has duplicates: "
                            + dup[0] + " & " + dup[1]);
            }
            
            // Get the differences between the two manifests
            diff = destManifest.getDifference(originManifest);
            
            // TODO - need to implement the following
            // figure out how many implicit directories disappear in this
            // transition and add directory remove actions.  These won't
            // do anything unless no pkgs reference that directory in
            // new state....
            
            // TODO - need to implement the following (as part of implementation of hardlinks)
            // over the list of update actions, check for any that are the
            // target of hardlink actions, and add the renewal of those hardlinks
            // to the install set
            
            flist = new FileList(Image.this, destFmri);
            for (Action a : diff.added) {
                a.buildFileList(null, flist);
            }
            for (Manifest.Difference.ActionPair ap : diff.changed) {
                ap.to.buildFileList(ap.from, flist);
            }
        }

        /**
         * Perform actions required prior to installation or removal of a package.
         * 
         * This method executes each action's preremove() or preinstall()
         * methods, as well as any package-wide steps that need to be taken
         * at such a time.
         */
        void preexecute(ImagePlanProgressTracker tracker) throws IOException {
            if (destFmri == null) {
                removeInstallFile(originFmri);
            }
            for (Action a : diff.removed) {
                a.preremove();
            }
            for (Manifest.Difference.ActionPair ap : diff.changed) {
                ap.from.preremove();
            }

            // UC_DOWNLOAD_THREADS is used for testing
            // -1 use 2.3.2 batch download
            // 0 use 2.3.3 single threaded download
            // n>0 use multi-threaded with threadpool size of n
            // Defaults to n=3
            String s = System.getenv("UC_DOWNLOAD_THREADS");
            int nthreads = DEFAULT_DOWNLOAD_THREADS;
            if (s != null) {
                log.log(Level.FINE, "UC_DOWNLOAD_THREADS=" + s);
                try {
                    nthreads = Integer.parseInt(s);
                    if (nthreads > MAX_DOWNLOAD_THREADS) {
                        // Don't allow large values. We don't want any one client
                        // to load the repo with too many simultaneous connections.
                        nthreads = MAX_DOWNLOAD_THREADS;
                        log.log(Level.WARNING,
                            "Too large of an integer value for UC_DOWNLOAD_THREADS. Reducing to " + Integer.toString(nthreads));
                    }
                } catch (NumberFormatException e) {
                    nthreads = DEFAULT_DOWNLOAD_THREADS;
                    log.log(Level.WARNING,
                        "Bad integer value for UC_DOWNLOAD_THREADS. Defaulting to " + Integer.toString(nthreads));
                }
            }

            if (nthreads < 0) {
                log.log(Level.FINE, "Using batch file downloader.");
                flist.bdownload(tracker);
            } else if (nthreads == 0) {
                log.log(Level.FINE, "Using unthreaded file downloader.");
                flist.sdownload(tracker);
            } else {
                log.log(Level.FINE,
                    "Using threaded file downloader. Using threadpool of size " +
                    Integer.toString(nthreads));
                flist.setThreadPoolSize(nthreads);
                flist.download(tracker);
            }
        }

        List<Action> getInstallActions() {
            return diff.added;
        }
        
        List<Action> getRemovalActions() {
            return diff.removed;
        }
        
        List<Manifest.Difference.ActionPair> getUpdateActions() {
            return diff.changed;
        }

        long computeTransferSize() {
            return flist.computeTransferSize();
        }

        int computeTransferFiles() {
            return flist.computeTransferFiles();
        }

        /**
         * Perform actions required after installation or removal of a package.
         * 
         * This method executes each action's postremove() or postinstall()
         * methods, as well as any package-wide steps that need to be taken
         * at such a time.
         */
        void postExecute() throws IOException {
            for (Action a : diff.added) {
                a.postinstall(null);
            }
            for (Action a : diff.removed) {
                a.postremove();
            }
            for (Manifest.Difference.ActionPair ap : diff.changed) {
                ap.from.postremove();
                ap.to.postinstall(ap.from);
            }
            
            // In the case of an upgrade, remove the installation file from
            // the origin's directory.
            // XXX should this just go in preexecute?
            if (originFmri != null && destFmri != null) {
                removeInstallFile(originFmri);
                // TODO - need to remove filters
            }
            if (destFmri != null) {
                addInstallFile(destFmri);
                // TODO - need to save the filters used to install the package
            }
            if (flist != null) flist.cleanupDownload();
        }
    }
    
    public final static int IMG_ENTIRE = 0;
    public final static int IMG_PARTIAL = 1;
    public final static int IMG_USER = 2;

    final static String img_user_prefix = ".org.opensolaris,pkg";
    final static String img_root_prefix = "var/pkg";
    final static String os = System.getProperty("os.name").toLowerCase();
    final static String os_version = getOSVersion();
    final static String arch = System.getProperty("os.arch").toLowerCase();

    // meta data directory names
    private final static String PKG_DIR = "pkg";
    private final static String INSTALLED_FILE = "installed";
    private final static String ISTATE_DIR = "state" + File.separator + "installed";
    private final static String ISTATE_DIR_TMP = "state" + File.separator + "installed.build";

    private final static String PKG_CLIENT_CONNECT_TIMEOUT = "PKG_CLIENT_CONNECT_TIMEOUT";
    private final static String PKG_CLIENT_READ_TIMEOUT = "PKG_CLIENT_READ_TIMEOUT";
    // Default number of threads used to download files
    static final int DEFAULT_DOWNLOAD_THREADS = 4;
    // Max number of threads we allow to be set by user
    static final int MAX_DOWNLOAD_THREADS = 6;

    static final String typeStr = "USER";
    private static String version = null;
    private static String defaultClientName = "pkg-java";
    static Logger log = Logger.getLogger("com.sun.pkg.client", "com/sun/pkg/client/messages");

    // instance fields
    int type = IMG_USER;
    String clientname = defaultClientName;
    Map<String, Authority> authorities = new HashMap<String, Authority>();
    List<String> lines = new ArrayList<String>();
    private Map<String, String> properties = new HashMap<String, String>();
    private Map<String, String> variants = new HashMap<String, String>();
    String preferred_authority_name = null;
    File cfgfile;
    File imgdir;    // the root directory of the image
    File metadir;   // the root directory of the image meta data
    Map<Authority, Catalog> catalogs = new HashMap<Authority, Catalog>();
    CatalogCache catalogCache = null;
    ProxySelector proxySel = SystemInfo.getProxySelector();
    Constraint.Set constraints = new Constraint.Set();
    String currentOperation = "unknown";
    Stack<Fmri> targetFmris = new Stack<Fmri>();
    Stack<String> targetReasons = new Stack<String>();
    Map<Fmri, List<Fmri>> reqDependents = null;
    HashMap<String, String> metaData = null;

    private int urlConnectTimeout_ms = 0;
    private int urlReadTimeout_ms = 0;

    Set<String> downloadedPkgs = new HashSet<String>();
    /**
     * Create an Image object for an existing user image based on a String
     * 
     * @param path The path for a file or directory within the image.
     */
    public Image(String path) throws Exception
    {
        this(new File(path));
    }
    
    /**
     * Open an Image object for an existing user image based on a File.
     *
     * @param path A File within the image.
     *
     * @see #create(File, String, URL)
     */ 
    public Image(File path) throws Exception
    {
        setDefaultURLTimeouts();
        // only IMG_USER images are supported for now
        imgdir = path;
        if (!imgdir.isDirectory()) {
            throw new Exception("image directory (" + path + ") doesn't exist");
        }

        metadir = new File(imgdir, img_user_prefix);
        if (!metadir.isDirectory()) throw new Exception("invalid USER image directory");

        loadConfig(new File(metadir, "cfg_cache"));
        
        // load catalogs
        for (Authority a : authorities.values()) {
            log.log(Level.CONFIG, "softwarerepo", a.origin);
            Catalog c = new Catalog(this, a.prefix);
            catalogs.put(a, c);
        }
        rebuildCatalogCache();

        // In case changes were made to the config as part of loading the
        // config - e.g. setting UUIDs, attempt to save the config. Doing so
        // ensures that modifications will be persisted even if the caller
        // does not explicitly invoke image.saveConfig().
        saveConfig();
    }

    /**
     * Creates a new user image at the specified path. The equivalent of "pkg image-create".
     * This method creates a set of metadata necessary for a directory to function
     * as an IPS image.
     *
     * @param path
     *      Directory that will become an image. If no such directory exists,
     *      it'll be created.
     * @param authName
     *      The name of the initial authority to be added.
     * @param origin
     *      The URL of the initial authority.
     */
    public static Image create(File path, String authName, URL origin) throws Exception {
        path.mkdirs();
        if (!path.isDirectory()) {
            throw new Exception("unable to create image directory (" + path + ")");
        }

        File metadir = new File(path, img_user_prefix);
        String[] required_dirs = { "catalog", "file", PKG_DIR };
        for (String d : required_dirs) {
            new File(metadir, d).mkdirs();
        }

        // just copied what "pkg image-create" actually creates:
        FileWriter w = new FileWriter(new File(metadir, "cfg_cache"));
        try {
            w.write(
                    "[property]\n" +
                    "preferred-authority = " + authName + "\n" +
                    "send-uuid = True\n" +
                    "require-optional = False\n" +
                    "flush-content-cache-on-success = False\n" +
                    "display-copyrights = True\n" +
                    "pursue-latest = True\n" +
                    "\n" +
                    "[filter]\n" +
                    "\n"
            );
        } finally {
            w.close();
        }

        Image img = new Image(path);
        img.startOperation("image-create");
        img.setAuthority(authName, origin, null);
        img.saveConfig();
        img.refreshCatalogs();
        return img;
    }

    private void rebuildCatalogCache() throws IOException {
        catalogCache = new CatalogCache();
        for (Entry<Authority, Catalog> e : catalogs.entrySet()) {
            catalogCache.addCatalog(e.getValue(), e.getKey());
        }
        catalogCache.cacheInstalledPackages();
    }
    
    /**
     * Obtain the root directory for this image.
     * 
     * @return the root directory for this image
     */
    public File getRootDirectory()
    {
        return imgdir;
    }
       
    File getMetaDirectory()
    {
        return metadir;
    }
    
    Logger getLogger()
    {
        return log;
    }

    String getUserAgent()
    {
        return "pkg-java/" + getVersion() + " (" + os + " " + arch + "; "
            + os_version + " "
            + System.getProperty("java.version") + "; "
            + typeStr + "; " + clientname + ")";
    }

    /**
     * Get a connected URLConnection to a requested resource.
     * 
     * @return a connected URLConnection to the requested resource
     * @throws java.io.IOException
     */
    synchronized HttpURLConnection getRepositoryURLConnection(String resource, String authname) throws IOException
    {
        Authority a = authorities.get(authname);
        if (a == null) {
            throw new IllegalArgumentException("invalid authority name: " + authname);
        }
        String o = a.origin;
        if (o == null) {
            throw new IOException("undefined origin URL for publisher " + authname);
        }
        if (!o.endsWith("/")) o = o + "/";
        URL repo = new URL(o);
        URL u = new URL(repo, resource);
        try {
            List<Proxy> pl = proxySel.select(u.toURI());
            // For now, only try the first one in the list
            Proxy p = pl.size() > 0 ? pl.get(0) : null;
            URLConnection urlc = p == null ? u.openConnection() : u.openConnection(pl.get(0));
            if (!(urlc instanceof HttpURLConnection)) {
                throw new IOException("unrecognized repository URL type:" + a.origin);
            }
            if (urlc instanceof HttpsURLConnection && a.certFile != null && a.keyFile != null) {
                if (a.sslfactory == null) {
                    a.sslfactory = PEMUtil.getSSLSocketFactory(a.certFile, a.keyFile);
                }
                HttpsURLConnection sslurlc = (HttpsURLConnection) urlc;
                sslurlc.setSSLSocketFactory(a.sslfactory);
            }
            HttpURLConnection hurlc = (HttpURLConnection) urlc;
            hurlc.setRequestProperty("User-Agent", getUserAgent());
            hurlc.setRequestProperty("X-IPkg-UUID", a.uuid);
            hurlc.setReadTimeout(urlReadTimeout_ms);
            hurlc.setConnectTimeout(urlConnectTimeout_ms);
            return hurlc;
        } catch (URISyntaxException ex) {
            Logger.getLogger(Image.class.getName()).log(Level.SEVERE, null, ex);
            IOException ioe = new IOException("invalid repository URL: " + u);
            ioe.initCause(ioe);
            throw ioe;
        }
    }
    
    /**
     * Get a connected URLConnection to a requested resource for an Fmri.
     * 
     * @return a connected URLConnection to the requested resource
     * @throws java.io.IOException
     */
    HttpURLConnection getRepositoryURLConnection(String resource, Fmri f) throws IOException
    {
        String authname = f.getAuthority();
        if (authname.length() == 0) {
            throw new IOException(
                    "attempt to access a resource for an Fmri that has no authority: " +
                    f);
        }
        return getRepositoryURLConnection(resource, authname);
    }

    /**
     * Check the connection for an HTTP_OK response code.  Throw an appropriate 
     * IOException for invalid codes.
     * @param urlc the connection to check
     * @throws IOException
     */
    void checkRepositoryConnection(HttpURLConnection urlc) throws IOException {
        int rc = urlc.getResponseCode();
        switch (rc) {
            case HttpURLConnection.HTTP_OK:
                return;
            case HttpURLConnection.HTTP_GATEWAY_TIMEOUT:
                throw new IOException("" + rc + ": Connection through proxy timed out");
            default:
                throw new IOException("Connection failed for URL " + urlc.getURL() + ": " +
                        rc + ": " + urlc.getResponseMessage());
        }
    }

    /**
     * Change the permissions on the Image meta data directory so that the 
     * directory will be hidden.
     * 
     * @throws java.io.IOException
     */
    public void hideMetaDirectory() throws IOException
    {
        if (os.indexOf("windows") == -1) return;
                
        String cmd[] = {"ATTRIB", "+H", getMetaDirectory().getCanonicalPath()};
        try {
            Process p = Runtime.getRuntime().exec(cmd);
            p.waitFor();
            if (p.exitValue() != 0) {
                throw new Error("ATTRIB failed: cannot hide meta data folder");
            }
        } catch (InterruptedException ie) {
        }
    }
    
    /**
     * Set the proxy to be used by connenctions to repositories.
     * <p>
     * The proxy for an Image is initialized to be the return value from 
     * SystemInfo.getProxy.  This method can be used to set the proxy to some 
     * other value or to null if the desire is to use the system defaults.
     *  
     * @param p - the proxy to use
     */
    public void setProxy(Proxy p) {
        proxySel = new PkgProxySelector(p);
    }

    /**
     * Set the client name to be used in connections to repositories.
     * <p>
     * The client name is included in the User-Agent header following the image
     * type. The default value is "pkg-java" if setClientName is not called or
     * another value as set by the setDefaultClientName method.
     *
     * @param clientname the name of the client that is using this image object
     */
    public void setClientName(String clientname) {
        this.clientname = clientname;
    }

    /**
     * Set the default client name for all Image objects created in this VM.
     * <p>
     * See the setClientName method for information on how the client name value
     * is used.
     *
     * @param clientname the default value to use for client name
     */
    static public void setDefaultClientName(String clientname) {
        defaultClientName = clientname;
    }
    
    void loadConfig(File configFile) throws FileNotFoundException, IOException
    {
        /* Eventually, this needs to be implemented to support full parsing of
         * the cfg_cache file, possibly using the ini4j library: see 
         * http://ini4j.sourceforge.net/.  For now, we just read each line and 
         * do simple parsing. 
         */
        cfgfile = configFile;
        BufferedReader r = new BufferedReader(new FileReader(cfgfile));
        String line;
        String tokens[];
        Authority curAuth = null;
        boolean inPolicy = false;
        boolean inProps = false;
        boolean inVariants = false;

        // Default variants. They will be overwritten by what is in cfg_cache
        setVariant("variant.arch", SystemInfo.getCanonicalOSArch());
        setVariant("variant.os.bits", SystemInfo.getOSBitness());

        while ((line = r.readLine()) != null) {
            lines.add(line); // save the lines for if we need to write
            
            // check if this is a new section
            if (curAuth != null && line.startsWith("[")) {
                // Just encountered another section while processing
                // an authority section. Before completing the
                // authority section, ensure UUID is present.
                ensureUuidPresent(curAuth);
                authorities.put(curAuth.prefix, curAuth);
                curAuth = null;                
            }
            if (inPolicy && line.startsWith("[")) {
                inPolicy = false;
            }
            if (inProps && line.startsWith("[")) {
                inProps = false;
            }
            if (inVariants && line.startsWith("[")) {
                inVariants = false;
            }
            
            if (line.startsWith("[authority") && line.endsWith("]")) {
                // start of an authority section
                line = line.substring(1, line.length() - 1);
                tokens = line.split("_");
                if (tokens.length == 2) {
                    curAuth = new Authority(tokens[1]);
                }
            }
            else if (curAuth != null && line.startsWith("uuid")) {
                tokens = line.split("\\s*=\\s*", 2);
                if (tokens.length == 2) {
                    curAuth.uuid = tokens[1].trim();
                }
            }
            else if (curAuth != null && line.startsWith("origin-for-")) {
                tokens = line.split("\\s*=\\s*", 2);
                if (tokens.length == 2) {
                    String plat = tokens[0].trim().substring(11);
                    // override the origin with one that is specific to this platform
                    if (isThisPlatform(plat)) curAuth.origin = tokens[1].trim();
                }
            }
            else if (curAuth != null && line.startsWith("origin")) {
                tokens = line.split("\\s*=\\s*", 2);
                // do not overwrite the origin if it is already set
                if (curAuth.origin == null && tokens.length == 2) {
                    curAuth.origin = tokens[1].trim();
                }
            }
            else if (curAuth != null && line.startsWith("prefix")) {
                tokens = line.split("\\s*=\\s*", 2);
                if (tokens.length == 2) {
                    curAuth.prefix = tokens[1].trim();
                }
            }
            else if (curAuth != null && line.startsWith("ssl_key")) {
                tokens = line.split("\\s*=\\s*", 2);
                if (tokens.length == 2) {
                    String fn = tokens[1].trim();
                    if (fn.length() > 0 && !fn.equals("None")) {
                        curAuth.keyFile = new File(fn);
                    }
                }
            }
            else if (curAuth != null && line.startsWith("ssl_cert")) {
                tokens = line.split("\\s*=\\s*", 2);
                if (tokens.length == 2) {
                    String fn = tokens[1].trim();
                    if (fn.length() > 0 && !fn.equals("None")) {
                        curAuth.certFile = new File(fn);
                    }
                }
            }
            else if (line.startsWith("[policy]")) {
                inPolicy = true;
            }
            else if (line.startsWith("[property]")) {
                inProps = true;
            }
            else if (line.startsWith("[variant]")) {
                inVariants = true;
            }
            else if (inProps || inPolicy) {
                tokens = line.split("\\s*=\\s*", 2);
                if (tokens.length == 2) {
                    setProperty(tokens[0].trim(), tokens[1]);
                }               
            }
            else if (inVariants) {
                tokens = line.split("\\s*=\\s*", 2);
                if (tokens.length == 2) {
                    setVariant(tokens[0].trim(), tokens[1]);
                }               
            }
        }
        if (curAuth != null) {
            // Finished processing all lines while working on an authority
            // section.  Ensure UUID is present for that authority.
            // This situation occurs when an authority section is the last
            // section in the file.
            ensureUuidPresent(curAuth);
            authorities.put(curAuth.prefix, curAuth);
        }
        r.close();
    }

    /**
     * Check for existence of UUID on authority/publisher and set if not
     * present.
     */
    private void ensureUuidPresent(Authority a)
    {
        if (a.uuid == null || a.uuid.equals("None")) {
            a.uuid = UUID.randomUUID().toString();
        }
    }
    
    /**
     * Save the configuration for the image.
     * 
     * Adjust the origin in the cfgfile so that it matches that of the current
     * platform as determined by origin_for_* properties.  This method writes
     * a new version of the cfgfile. 
     * @throws java.io.IOException
     */
    public void saveConfig() throws IOException
    {
        PrintWriter w = new PrintWriter(new BufferedWriter(new FileWriter(cfgfile)));
        String line;
        String tokens[];
        Authority curAuth = null;
        boolean inProps = false;
        boolean wroteUuid = false;
        Set<Authority> written = new HashSet<Authority>();
        for (int i = 0; i < lines.size(); i++) {
            // TODO: this code doesn't handle a removal of authority

            line = lines.get(i);
            if (curAuth != null && line.startsWith("[")) {
                // Just finished an authority. Check whether UUID was
                // written. If not, write it before continuing.
                if (!wroteUuid && curAuth.uuid != null) {
                    w.println("uuid = " + curAuth.uuid);
                    w.println("");
                }
                curAuth = null;
                wroteUuid = false;
            }
            if (inProps && line.startsWith("[")) {
                inProps = false;
            }
            if (line.startsWith("[authority") && line.endsWith("]")) {
                // start of an authority section
                String line1 = line.substring(1, line.length() - 1);
                tokens = line1.split("_");
                if (tokens.length == 2) {
                    curAuth = authorities.get(tokens[1]);
                    if (curAuth != null)
                        written.add(curAuth);
                }
            }
            else if (curAuth != null && line.startsWith("uuid")) {
                // write the uuid from the Authority
                line = "uuid = " + curAuth.uuid;
                wroteUuid = true;
            }
            else if (curAuth != null && line.startsWith("origin-for-")) {
                // leave origin-for lines unmodified
            }
            else if (curAuth != null && line.startsWith("origin")) {
                // write the origin from the Authority
                line = "origin = " + curAuth.origin;
            }
            else if (line.startsWith("[property]")) {
                inProps = true;
            }
            if (inProps) {
                // ignore property lines from file as they are written below
                continue;
            }
            w.println(line);
        }

        // Address cases in which an authority section appears at the end of the
        // config file.
        if (curAuth != null) {
            if (!wroteUuid && curAuth.uuid != null) {
                w.println("uuid = " + curAuth.uuid);
                w.println("");
            }
        }

        // write out authorities that haven't been written yet
        for (Authority a : authorities.values()) {
            if (written.contains(a) || a.origin == null) continue;
            w.println("[authority_"+a.prefix+']');
            w.println("origin = "+a.origin);
            w.println("prefix = "+a.prefix);
            if (a.uuid != null) {
                w.println("uuid = "+a.uuid);
            }
            w.println("mirrors = []");
            // TODO: are other properties needed?
        }

        // write properties
        w.println("[property]");
        for (String p : properties.keySet()) {
            w.println(p + " = " + properties.get(p));
        }

        w.close();
    }
    
    public String getPreferredAuthorityName() {
        return preferred_authority_name;
    }
    
    /**
     * Return the authority names defined for this image.
     * @return an array of authority names
     */
    public String[] getAuthorityNames() {
        return authorities.keySet().toArray(new String[0]);
    }
     
    /**
     * Set image authority attributes.
     * 
     * If authname names an authority that currently doesn't exist, it is added
     * to the image. Otherwise, the parameters are used to modify an existing
     * authority.
     * <p>
     * When adding an authority, the origin URL is required while
     * uuid may be null.  If the uuid is passed as null, a new uuid will be
     * generated and assigned to the authority.
     * <p>
     * When modifying an existing authority, the origin and uuid parameters may
     * be passed as nulls. In this case:
     *   <ul>
     *   <li>The existing origin will not be modified.</li>
     *   <li>If a uuid is already present, the existing uuid will not be modified.</li>
     *   <li>If a uuid is not present, a new uuid will be generated and assigned
     *     to the authority.</li>
     *   </ul>
     * <p>
     * 
     * @param authname - the name of the authority to change
     * @param origin - the origin URL for the authority
     * @param uuid - the UUID for the authority
     * @throws IllegalArgumentException - if the authority is not currently in the
     * image and origin is null.
     */
    public void setAuthority(String authname, URL origin, String uuid) throws IOException
    {
        Authority a = authorities.get(authname);
        if (a == null && origin == null) {
            throw new IllegalArgumentException("Origin is required for a new authority");
        }
        if (a == null) {
            a = new Authority(authname);
            authorities.put(authname, a);
            catalogs.put(a,new Catalog(this,a.prefix));
        }
        if (origin != null) a.origin = origin.toString();
        if (uuid != null) {
            a.uuid = uuid;
        } else {
           ensureUuidPresent(a);
        }
    }

    /**
     * Set image property value.
     *
     * The saveConfig method must be called to make the property persist.
     *
     * @param pname - the name of the property to set
     * @param pvalue - the value of the property
     */
    public void setProperty(String pname, String pvalue)
    {
        properties.put(pname, pvalue);
        if (pname.equals("preferred-authority")) {
            preferred_authority_name = pvalue.trim();
        }
    }

    /**
     * Set image variant value.
     *
     * @param name - the name of the variant to set
     * @param value - the value of the variant
     */
    public void setVariant(String name, String value)
    {
        variants.put(name, value);
    }
       
    /**
     * Call refresh on all of the Catalogs for this image.
     * 
     * This also updates the internal cache used by the Image class to store 
     * catalog information.
     * 
     * @throws java.io.IOException
     */
    public void refreshCatalogs() throws IOException
    {
        for (Catalog c : catalogs.values()) c.refresh();
        rebuildCatalogCache();
    }
    
    /**
     * Call refresh on the Catalog for the given authority.
     * 
     * This also updates the internal cache used by the Image class to store 
     * catalog information.
     * 
     * @throws java.io.IOException
     */
    public void refreshCatalog(String authname) throws IOException
    {
        Authority a = authorities.get(authname);
        if (a == null) throw new IllegalArgumentException("unknown authority: " + authname);
    
        Catalog c = catalogs.get(a);
        if (c == null) throw new IllegalArgumentException("no catalog for authority: " + authname);

        c.refresh();
        rebuildCatalogCache();
    }
    
    /**
     * Get the inventory of packages and their associated state.
     * 
     * The inventory is returned in FmriState objects that consist of an Fmri and 
     * the four states for the Fmri.  These states are:
     * <ul>
     * <li><b>upgradable</b> A package with a newer version if available in the 
     * catalog
     * <li><b>frozen</b> TBD
     * <li><b>incorporated</b> The package is part of an incorporation
     * <li><b>excludes</b> TBD
     * </ul>     * 
     *
     * @param pkg_list Limit the inventory to packages that match entries in the
     * list of package names
     * @param all_known If true, return all package versions.  Otherwise only 
     * return the most recent version of each package.
     * @return A List of FmriState objects that identify the Fmris that are 
     * available for the image.
     */
    public List<FmriState> getInventory(String[] pkg_list, boolean all_known)
    {
        return catalogCache.getInventory(pkg_list, all_known);
    }
    
    /**
     * Check whether a given package is installed in this image.
     * 
     * @return True if the given package is installed
     */
    boolean isInstalled(Fmri p)
    {
        try {
            File ifile = new File(p.getPkgVersionDir(new File(getMetaDirectory(), PKG_DIR)),
                INSTALLED_FILE);
            if (!ifile.exists()) return false;
            String auth = p.getAuthority();
            if (auth.length() == 0) throw new IllegalArgumentException("Fmri must have authority set");
            return auth.equals(getAuthorityForInstalledPkg(p).prefix);
        }
        catch (IOException ioe) {
            return false;
        }
    }

    /**
     * Create an plan for installing packages and any required dependencies.
     * @param pkgNames names of packages to install
     * @return An ImagePlan that will install the packages when executed.
     * @throws java.io.IOException
     */
    public ImagePlan makeInstallPlan(Collection<String> pkgNames) throws IOException, ConstraintException {
        return makeInstallPlan(pkgNames.toArray(new String[pkgNames.size()]), "install");
    }

    /**
     * Create an plan for installing packages and any required dependencies.
     * @param pkgNames names of packages to install
     * @param operation identify the operation (install, list, uninstall)
     * @return An ImagePlan that will install the packages when executed.
     * @throws java.io.IOException
     */
    public ImagePlan makeInstallPlan(Collection<String> pkgNames, String operation) throws IOException, ConstraintException {
        return makeInstallPlan(pkgNames.toArray(new String[pkgNames.size()]), operation);
    }

    /**
     * Create an plan for installing packages and any required dependencies.
     * @param pkgNames names of packages to install
     * @return An ImagePlan that will install the packages when executed.
     * @throws java.io.IOException
     */
    public ImagePlan makeInstallPlan(String... pkgNames) throws IOException, ConstraintException {
        return makeInstallPlan(pkgNames, "install");
    }

    public ImagePlan makeInstallPlan(String[] pkgNames, String operation) throws IOException, ConstraintException {
        startOperation(operation);
        loadConstraints();
        
        // order package list so that any unbound incorporations are done first
        List<String> incList = getInstalledUnboundIncorporationList();
        List<String> head = new ArrayList<String>();
        List<String> tail = new ArrayList<String>();
        for (String pn : pkgNames) {
            if (incList.contains(pn)) head.add(pn);
            else tail.add(pn);
        }
        head.addAll(tail);
        List<String> pkgnl = head;

        List<Fmri> pkgs = new ArrayList<Fmri>();
        for (String p : pkgnl) {
            Fmri conp = constraints.applyConstraintsToFmri(new Fmri(p));
            Fmri f = catalogCache.getFirst(conp.toString());
            if (f == null) {
                throw new IllegalArgumentException("no matching package for: " + p);
            }
            pkgs.add(f);
        }
        return makeAndEvaluateImagePlan(pkgs);
    }
    
    /**
     * Create an plan for installing packages and any required dependencies.
     * @param pkgs List of valid Fmris from the image inventory for packages to 
     * install.
     * @return An ImagePlan that will install the packages when executed.
     * @throws java.io.IOException
     */
    public ImagePlan makeInstallPlan(List<Fmri> pkgs) throws IOException, ConstraintException
    {
        startOperation("install");
        return makeAndEvaluateImagePlan(pkgs);
    }

    private ImagePlan makeAndEvaluateImagePlan(List<Fmri> pkgs) throws IOException, ConstraintException
    {
        ImagePlan ip = new ImagePlan(false, null);
        
        for (Fmri pf : pkgs) ip.proposeFmri(pf);
        log.log(Level.FINER, "beforeeval", ip.toString());
        ip.evaluate();
        log.log(Level.FINER, "aftereval", ip.toString());
        return ip;
    }

    /**
     * Create a plan for removing packages.
     *
     * @param pkgs
     *      {@link Fmri}s to be removed.
     * @return An ImagePlan that will uninstall the packages when executed.
     * @throws java.io.IOException
     */
    public ImagePlan makeUninstallPlan(Collection<Fmri> pkgs) throws IOException
    {
        startOperation("uninstall");
        ImagePlan ip = new ImagePlan(false, null);

        for (Fmri pf : pkgs) ip.proposeFmriRemoval(pf);
        log.log(Level.FINER, "beforeeval", ip.toString());
        ip.evaluate();
        log.log(Level.FINER, "aftereval", ip.toString());
        return ip;
    }

    private void readInstallCache(File aFile, Set<String> aPkgNames) throws IOException {
            if(aFile.exists()) {
                BufferedReader br = null;
                try {
                    br = new BufferedReader(new FileReader(aFile));
                    String pkg = null;
                    while((pkg = br.readLine()) != null && pkg.length() > 0) {
                        aPkgNames.add(pkg);
                    }

                } finally {
                    br.close();
                }
            }
        }

    private void writeInstallCache(File aFile, Set<String> aPkgNames, boolean aAppend) throws IOException {
            FileWriter fw = null;
            String EOL = System.getProperty("line.separator");
            try {
                fw = new FileWriter(aFile, aAppend);
                for (String p : aPkgNames) {
                    if(p.length() > 0) {
                        fw.append(p);
                        fw.append(EOL);
                    }
                }
            } finally {
                fw.close();
            }
        }

    public boolean isTherePendingInstall() throws IOException {
        boolean result = false;
        File f = new File(metadir, "install_cache");
        if(f.exists()) {
            BufferedReader br = null;
            try {
                br = new BufferedReader(new FileReader(f));
                String pkg = null;
                while((pkg = br.readLine()) != null) {
                    result = true;
                }
            } finally {
                br.close();
            }
        }
        return result;
    }

    public void completeInstall() throws IOException, ConstraintException
    {
        File f = new File(metadir, "install_cache");
        Set<String> pkgNames = new HashSet<String>();
        readInstallCache(f, pkgNames);

        if(pkgNames.size() > 0) {
            installPackages(false, true, (String[])pkgNames.toArray(new String[pkgNames.size()]));
        } else {
            log.log(Level.FINE, "Nothing to install at start.");
        }
    }

    /**
     * Install packages and any required dependencies.
     * <p>
     * The catalogs are searched for a matching package for each entry in the 
     * pkgNames array. 
     * If multiple packages match an entry, the most recent package is installed.
     * If no matching package is found, an IllegalArgumentException is thrown.
     * <p>
     * This method is shorthand for:
     * <pre>
     *      makeInstallPlan(pkgNames).execute();
     * </pre>
     *  
     * @param pkgNames names of packages to install
     * @throws java.io.IOException
     * @throws java.lang.IllegalArgumentException if a matching package cannot be found
     */
    public void installPackages(String... pkgNames) throws IOException, ConstraintException
    {  
        installPackages(true, true, pkgNames);
    }
    /**
     * Install packages and any required dependencies.
     * <p>
     * The catalogs are searched for a matching package for each entry in the
     * pkgNames array.
     * If multiple packages match an entry, the most recent package is installed.
     * If no matching package is found, an IllegalArgumentException is thrown.
     * <p>
     * This method is shorthand for:
     * <pre>
     *      makeInstallPlan(pkgNames).execute();
     * </pre>
     *
     * @param pkgNames names of packages to install
     * @param download boolean variable to trigger download
     * @param apply boolean variable to trigger installtion for downloaded packages
     * @param tracker Progress tracker. May be null.
     * @throws java.io.IOException
     * @throws java.lang.IllegalArgumentException if a matching package cannot be found
     */
    public void installPackages(ImagePlanProgressTracker tracker, boolean download, boolean apply, String... pkgNames) throws IOException, ConstraintException
    {
        ImagePlan ip = makeInstallPlan(pkgNames);
        if (ip.nothingToDo()) return;
        log.log(Level.FINE, "installing", Arrays.toString(ip.getProposedFmris()));
        for(String p:pkgNames) {
            Fmri conp = constraints.applyConstraintsToFmri(new Fmri(p));
            Fmri f = catalogCache.getFirst(conp.toString());
            if (f == null) {
                throw new IllegalArgumentException("no matching package for: " + p);
            }
            downloadedPkgs.add(f.toString());
        }
        if (tracker == null) {
            ip.execute(download, apply);
        } else {
            ip.execute(tracker, download, apply);
        }
    }

    public void installPackages(boolean download, boolean apply, String... pkgNames) throws IOException, ConstraintException
    {
        installPackages(null, download, apply, pkgNames);
    }

    /**
     * Install packages and any required dependencies.
     * <p>
     * This method is shorthand for:
     * <pre>
     *      makeInstallPlan(pkgs).execute();
     * </pre>
     * 
     * @param pkgs List of valid Fmris from the image inventory for packages to 
     * install.
     * @throws java.io.IOException
     */
    public void installPackages(List<Fmri> pkgs) throws IOException, ConstraintException
    {
        installPackages(true, true, pkgs);
    }

    /**
     * Install packages and any required dependencies.
     * <p>
     * This method is shorthand for:
     * <pre>
     *      makeInstallPlan(pkgs).execute();
     * </pre>
     *
     * @param pkgs List of valid Fmris from the image inventory for packages to
     * install.
     * @param download boolean variable to trigger download
     * @param apply boolean variable to trigger installtion for downloaded packages
     * @throws java.io.IOException
     */
    public void installPackages(boolean download, boolean apply, List<Fmri> pkgs) throws IOException, ConstraintException
    {
        ImagePlan ip = makeInstallPlan(pkgs);
        if (ip.nothingToDo()) return;
        log.log(Level.FINE, "installing", pkgs.toString());
        for(Fmri f:pkgs) {
            downloadedPkgs.add(f.toString());
        }
        ip.execute(download, apply);
    }
    
    /**
     * Uninstall packages.
     * 
     * The installed package list is searched for a matching package for 
     * each entry in the pkgNames array. 
     * 
     * If no matching package is found, the entry is ignored. 
     * 
     * @param pkgNames names of packages to uninstall
     * @throws java.io.IOException
     */
    public void uninstallPackages(String... pkgNames) throws IOException
    {
        uninstallPackages(getVersionsInstalled(Arrays.asList(pkgNames)));
    }
    
    /**
     * Uninstall packages.
     * 
     * @param pkgs List of valid Fmris to uninstall.
     * @throws java.io.IOException
     */
    public void uninstallPackages(List<Fmri> pkgs) throws IOException
    {
        ImagePlan plan = makeUninstallPlan(pkgs);
        log.log(Level.FINE, "uninstalling", pkgs.toString());
        plan.execute();
    }
    
    /**
     * Get the manifest for a package.
     * @return a Manifest
     */
    public Manifest getManifest(Fmri fmri) throws IOException {
        return new Manifest(this, fmri);
    }
    
    private boolean fmriIsSamePackage(Fmri fmri, Fmri pf) {
        // TODO - need to handle catalog renames
        return fmri.isSamePackage(pf);
    }
    
    private boolean fmriIsSuccessor(Fmri fmri, Fmri pf) {
        // TODO - need to handle catalog renames
        return fmri.isSuccessor(pf);
    }

    private List<Fmri> getInstalledFmrisFromCache() {
        List<FmriState> lfs = catalogCache.getInventory(null, false);
        List<Fmri> installed = new ArrayList<Fmri>();
        for (FmriState fs : lfs) {
            installed.add(fs.fmri);
        }
        return installed;
    }
    
    private List<Fmri> getInstalledFmrisFromImage() throws IOException {
        File istatedir = new File(getMetaDirectory(), ISTATE_DIR);
        List<Fmri> installedPkgs = new ArrayList<Fmri>();
        
        // XXX This can be removed at some point in the future once we
        // think this link is available on all systems
        if (!istatedir.exists()) updateInstalledPackages();
        for (File f : istatedir.listFiles()) {
            String fmristr = URLDecoder.decode(f.getName(), "UTF-8");
            Fmri fmri = new Fmri(fmristr);
            installedPkgs.add(fmri);
        }
        return installedPkgs;
    }

    /**
     * Obtain a boolean policy from the image configuration file
     * @param pname The name of the policy
     * @return the value for the policy
     */
    boolean getPolicy(String pname) {
        String bstr = properties.get(pname);
        if (bstr != null) {
            Boolean b = Boolean.parseBoolean(bstr.trim());
            return b != null && b;
        }
        return false;
    }

    /**
     * Set the Fmri's authority to be the preferred authority for this Image
     * if it isn't already set.
     */
    void setFmriDefaultAuthority(Fmri fmri) {
        if (fmri.getAuthority().length() == 0) {
            fmri.setAuthority(getPreferredAuthorityName());
        }
    }

    /**
     * Given just the package name (e.g., "SUNWbash"), find the version currently installed
     * in this image, and returns its {@link Fmri} with the version number.
     *
     * @return
     *      null if the package isn't installed.
     */
    public Fmri getVersionInstalled(String pkgName) throws IOException {
        return getVersionInstalled(new Fmri(pkgName));
    }

    /**
     * Plural version of {@link #getVersionInstalled(String)}.
     *
     * <p>
     * If a package specified isn't installed, it'll be just ignored.
     *
     * @return
     *      can be empty but never null. Never contains null.
     */
    public List<Fmri> getVersionsInstalled(Collection<String> pkgNames) throws IOException {
        List<Fmri> r = new ArrayList<Fmri>();
        for (String pkgName : pkgNames) {
            Fmri f = getVersionInstalled(pkgName);
            if(f!=null) r.add(f);
        }
        return r;
    }

    private Fmri getVersionInstalled(Fmri fmri) throws IOException {
        File pd = fmri.getPkgDir(new File(getMetaDirectory(), PKG_DIR));
        File pkgDirs[] = pd.listFiles();
        if (pkgDirs == null) return null;
        List<File> installedPkgs = new ArrayList<File>();
        for (File f : pkgDirs) {
            File ifile = new File(f, INSTALLED_FILE);
            if (ifile.exists()) installedPkgs.add(f);
        }
        if (installedPkgs.size() == 0) return null;
        if (installedPkgs.size() > 1) {
            throw new RuntimeException("package installed more than once: " + fmri);
        }
        Fmri ifmri = new Fmri(installedPkgs.get(0));
        ifmri.setAuthority(getAuthorityForInstalledPkg(ifmri).prefix);
        return ifmri;
    }
    
    private Fmri getInstalledOlderVersion(Fmri fmri) throws IOException {
        // TODO need to deal with package renames in catalog
        return getVersionInstalled(fmri);
    }

    private boolean hasVersionInstalled(Fmri fmri) throws IOException {
        Fmri f = getVersionInstalled(fmri);
        return f != null && fmriIsSuccessor(f, fmri);
    }

    private void startOperation(String op) {
        currentOperation = op;
        targetFmris = new Stack<Fmri>();
        targetReasons = new Stack<String>();
    }

    private void endOperation() {
        currentOperation = "unknown";
    }

    private void pushTarget(Fmri fmri, String intent) {
        targetFmris.push(fmri);
        targetReasons.push(intent);
    }

    private void popTarget() {
        targetFmris.pop();
        targetReasons.pop();
    }

    /**
     * Returns a string representing the intent of the client in retrieving
     * information based on the operation information provided by the image
     * object.
    */
    static final int wopkg = "pkg:/".length();
    String getIntent(Fmri pi) throws IOException {
        String op = currentOperation;
        String reason = "info";
        String target_pkg = "";
        String initial_pkg = "";
        String needed_by_pkg = "";
        String current_auth = pi.getAuthority();

        if (!targetFmris.empty()) {
                // Attempt to determine why the client is retrieving the
                // manifest for this fmri and what its current target is.
                Fmri target = targetFmris.peek();
                reason = targetReasons.peek();

                // Compare the FMRIs with no publisher information embedded.
                String na_current = pi.toStringWithoutAuthority();
                String na_target = pi.toStringWithoutAuthority();
                if (na_target.equals(na_current)) {
                    // Only provide this information if the fmri for the
                    // manifest being retrieved matches the fmri of the
                    // target.  If they do not match, then the target fmri is
                    // being retrieved for information purposes only (e.g.
                    // dependency calculation, etc.).
                    target_pkg = (target.getAuthority().equals(current_auth)) ?
                        na_target.substring(wopkg) :
                        "unknown";

                    // The very first fmri should be the initial target that
                    // caused the current and needed_by fmris to be retrieved.
                    Fmri initial = targetFmris.firstElement();
                    String initial_auth = initial.getAuthority();
                    if (initial_auth.equals(current_auth)) {
                        // Prevent providing information across publishers.
                        initial_pkg = initial.toStringWithoutAuthority().substring(wopkg);
                        if (target_pkg.equals(initial_pkg)) {
                            // Don't bother sending the target information if it is the same
                            // as the initial target (i.e. the manifest for foo@1.0 is being
                            // retrieved because the user is installing foo@1.0).
                            target_pkg = "";
                        }
                    } else {
                        // If they didn't match, indicate that the needed_by_pkg
                        // was a dependency of another, but not which one.
                        initial_pkg = "unknown";
                    }

                    if (targetFmris.size() > 1) {
                        // The fmri responsible for the current one being
                        // processed should immediately precede the
                        // current one in the target list.
                        Fmri needed_by = targetFmris.get(targetFmris.size() - 2);
                        String needed_by_auth = needed_by.getAuthority();
                        if (needed_by_auth.equals(current_auth)) {
                            // To prevent dependency information being shared
                            // across publisher boundaries, publishers must match.
                            needed_by_pkg = needed_by.toStringWithoutAuthority().substring(wopkg);
                        } else {
                            // If they didn't match, indicate that the package
                            // is needed by another, but not which one.
                            needed_by_pkg = "unknown";
                        }
                    }
                }
        } else {
            // An operation is being performed that has not provided any
            // target information and is likely for informational purposes
            // only.  Assume the "initial target" is what is being retrieved.
            initial_pkg = pi.toStringWithoutAuthority().substring(wopkg);
        }

        String prior_version = "";
        if (!reason.equals("info")) {
            // Only provide version information for non-informational operations.
            Fmri prior = getVersionInstalled(pi);
            if (prior != null) {
                String prior_auth = prior.getAuthority();
                // Prevent providing information across publishers by indicating
                // that a prior version was installed, but not which one.
                prior_version = prior_auth.equals(current_auth) ?
                    prior.getVersion().toString() : "unknown";
            }
        }

        String info[][] = {
            { "operation", op },
            { "prior_version", prior_version },
            { "reason", reason },
            { "target", target_pkg },
            { "initial_target", initial_pkg },
            { "needed_by", needed_by_pkg }
        };
        
        StringBuffer infobuf = new StringBuffer();
        infobuf.append("(");
        boolean haveData = false;
        for (int i = 0; i < info.length; i++) {
            if (!(info[i][1].length() == 0)) {
                if (haveData) infobuf.append(';');
                haveData = true;
                infobuf.append(info[i][0]).append("=").append(info[i][1]);
            }
        }
        infobuf.append(")");
        return infobuf.toString();
    }

    /*
     * Load constraints for all installed pkgs
     */
    private void loadConstraints() throws IOException {
        for (Fmri fmri : getInstalledFmrisFromCache()) {
            // skip loading if already done
            if (constraints.startLoading(fmri)) {
                Manifest m = getManifest(fmri);
                for (DependAction da : m.getActionsByType(DependAction.class)) {
                    constraints.updateConstraints(da.getConstraint());
                }
                constraints.finishLoading(fmri);
            }
        }
    }
    
    /*
     * Returns list of packages containing incorporation dependencies on which 
     * no other pkgs depend.
     */
    private List<String> getInstalledUnboundIncorporationList() throws IOException {
        Set<String> dependents = new TreeSet<String>();
        List<String> inc = new ArrayList<String>();
        for (Fmri fmri : getInstalledFmrisFromCache()) {
            String fmriName = fmri.getName();
            Manifest m = getManifest(fmri);
            for (DependAction da : m.getActionsByType(DependAction.class)) {
                Fmri cfmri = da.getConstrainedFmri();
                if (cfmri != null) {
                    String cName = cfmri.getName();
                    dependents.add(cName);
                    inc.add(fmriName);
                }
            }
        }
        // remove those incorporations which are depended on by other incorporations.
        Set<String> rv = new TreeSet<String>();
        for (String p : inc) if (!dependents.contains(p)) rv.add(p);
        return new ArrayList<String>(rv);        
    }
    
    Map<Fmri, List<Fmri>> buildReqDependents() throws IOException {
        // Build a dictionary mapping packages to the list of packages
        // that have required dependencies on them."""
        Map<Fmri, List<Fmri>> deps = new HashMap<Fmri, List<Fmri>>();

        for (Fmri fmri : getInstalledFmrisFromCache()) {
            Manifest m = getManifest(fmri);
            for (DependAction d : m.getActionsByType(DependAction.class)) {
                if (d.getType() != DependAction.Type.REQUIRED) continue;
                Fmri dfmri = d.getTargetFmri();
                List<Fmri> ldep = deps.get(dfmri);
                if (ldep == null) {
                    ldep = new ArrayList<Fmri>();
                    deps.put(dfmri, ldep);
                }
                ldep.add(fmri);
            }
        }
        return deps;
    }

    /**
     * Return a list of the packages directly dependent on the given Fmri.
     */
    private List<Fmri> getDependents(Fmri pfmri) throws IOException {
        if (reqDependents == null) {
            reqDependents = buildReqDependents();
        }
        List<Fmri> dependents = new ArrayList<Fmri>();
        // We run through all the keys, in case a package is depended
        // upon under multiple versions.  That is, if pkgA depends on
        // libc@1 and pkgB depends on libc@2, we need to return both pkgA
        // and pkgB.  If we used package names as keys, this would be
        // simpler, but it wouldn't handle package rename.
        for (Fmri f : reqDependents.keySet()) {
            if (fmriIsSuccessor(pfmri, f)) {
                dependents.addAll(reqDependents.get(f));
            }
        }
        return dependents;
    }
    
    private Authority getAuthorityForInstalledPkg(Fmri fmri) throws IOException {
        String authname = null;
        File ifile = new File(fmri.getPkgVersionDir(new File(getMetaDirectory(), PKG_DIR)),
                INSTALLED_FILE);
        BufferedReader r = new BufferedReader(new FileReader(ifile));
        String line;
        line = r.readLine();
        if (line != null && !line.equals("VERSION_1")) {
            // old-style file, line is the authority name
            authname = line;
        } else {
            line = r.readLine();
            String pfx = Fmri.PREF_AUTH_PFX + "_";
            if (line.startsWith(pfx)) {
                authname = line.substring(pfx.length());
            } else {
                authname = line;
            }
        }
        r.close();
        Authority a = authorities.get(authname);
        if (a == null) {
            // A package was installed for an authority no longer in cfg_cache
            a = new Authority(authname);
            authorities.put(authname, a);
        }
        return a;
    }
    
    private void addInstallFile(Fmri fmri) throws IOException {
        File istatedir = new File(getMetaDirectory(), ISTATE_DIR);
        
        // XXX This can be removed at some point in the future once we
        // think this link is available on all systems
        if (!istatedir.exists()) updateInstalledPackages();

        // remove cache of dependencies
        reqDependents = null;

        String authname = fmri.getAuthority();
        String prefpfx = "";
        String prefauth = getPreferredAuthorityName();
        if (authname.length() == 0 || authname.equals(prefauth)) {
            authname = prefauth;
            prefpfx = Fmri.PREF_AUTH_PFX + "_";
        }
        File ifile = new File(fmri.getPkgVersionDir(new File(getMetaDirectory(), PKG_DIR)),
                INSTALLED_FILE);
        FileWriter fw = new FileWriter(ifile);
        fw.write("VERSION_1\n" + prefpfx + authname);
        fw.close();
            
        File isfile = new File(istatedir, fmri.getLinkFileName());
        isfile.createNewFile();               
    }
    
    private void removeInstallFile(Fmri fmri) throws IOException {
        File istatedir = new File(getMetaDirectory(), ISTATE_DIR);
        
        // XXX This can be removed at some point in the future once we
        // think this link is available on all systems
        if (!istatedir.exists()) updateInstalledPackages();

        // remove cache of dependencies
        reqDependents = null;

        File ifile = new File(fmri.getPkgVersionDir(new File(getMetaDirectory(), PKG_DIR)),
                INSTALLED_FILE);
        ifile.delete();

        File isfile = new File(istatedir, fmri.getLinkFileName());
        isfile.delete();
    }

    /**
     * Take the image's record of installed packages from the prototype layout,
     * with an installed file in each $META/pkg/stem/version directory, to the
     * $META/state/installed summary directory form
     */
    private void updateInstalledPackages() throws IOException {
        File istatedir = new File(getMetaDirectory(), ISTATE_DIR);
        File tmpdir = new File(getMetaDirectory(), ISTATE_DIR_TMP);

        // Create the link forest in a temporary directory.  
        if (tmpdir.mkdirs() == false) {
            throw new IOException("unable to create installed packages directory");
        }

        File proot = new File(getMetaDirectory(), PKG_DIR);
        for (File pd : proot.listFiles()) {
            for (File vd : pd.listFiles()) {
                File path = new File(vd, "installed");
                if (!path.exists()) continue;
                String fmristr = URLDecoder.decode(
                        pd.getName() + "@" + vd.getName(), "UTF-8");
                Fmri f = new Fmri(fmristr);
                File fi = new File(tmpdir, f.getLinkFileName());
                fi.createNewFile();
            }
        }

        // Someone may have already created this directory.  Junk the
        // directory we just populated if that's the case.
        if (tmpdir.renameTo(istatedir) == false) {
            for (File f : tmpdir.listFiles()) f.delete();
            tmpdir.delete();
        }
    }

    private void setDefaultURLTimeouts() {
        urlConnectTimeout_ms = 2 * 60 * 1000;
        urlReadTimeout_ms =  2 * 60 * 1000;

        int n;
        String s = System.getenv(PKG_CLIENT_CONNECT_TIMEOUT);
        if (s != null && s.length() > 0) {
            try {
                n = Integer.parseInt(s);
                urlConnectTimeout_ms = n * 1000;
            } catch (NumberFormatException e) {
            }
        }

        s = System.getenv(PKG_CLIENT_READ_TIMEOUT);
        if (s != null && s.length() > 0) {
            try {
                n = Integer.parseInt(s);
                urlReadTimeout_ms =  n * 1000;
            } catch (NumberFormatException e) {
            }
        }
    }

    public void setURLConnectTimeout(int timeout_ms) {
        urlConnectTimeout_ms = timeout_ms;
    }

    public int getURLConnectTimeout() {
        return urlConnectTimeout_ms;
    }

    public void setURLReadmTimeout(int timeout_ms) {
        urlReadTimeout_ms = timeout_ms;
    }

    public int getURLReadmTimeout() {
        return urlReadTimeout_ms;
    }

    /**
     * If provided this metadata is included in any catalog requests via
     * the X-JPkg-MetaData property on the HTTP GET request.
     */
    public void setMetaData(HashMap<String,String> attrs) {
        metaData = attrs;
    }

    public HashMap<String,String> getMetaData() {
        return metaData;
    }
    
    /**
     * Get the version of the program from the MANIFEST file
     */
    static String getVersion()
    {
        if (version != null) return version;
        version = log.getResourceBundle().getString("version");
        /* the following should work, but it doesn't when in an OSGi bundle 
         String version = Image.class.getPackage().getImplementationVersion();
         */
        if (version == null) version = "0";
        log.log(Level.FINE, "versionmsg", version);
        return version;
    }

    /**
     * Get the version of the operating system
     */
    static String getOSVersion()
    {
        String osv = System.getProperty("os.version");
        if (os.equals("sunos")) {
            try {
                // for Solaris, add some information from the /etc/release file
                // grab the 2nd last token from the first line
                File f = new File("/etc/release");
                BufferedReader in = new BufferedReader(new FileReader(f));
                String line = in.readLine();
                Pattern p = Pattern.compile("^.*\\s(\\S+)\\s+\\S+$");
                Matcher m = p.matcher(line);
                if (m.matches()) {
                    osv = osv + "-" + m.group(1);
                }
                in.close();
            }
            catch (IOException ioe) {}
        }
        return osv;
    }
   
    boolean isThisPlatform(String plat)
    {
        // because indexOf is used for matching below, a blank means match
        // anything for the 2nd and 3rd columns of this table
        String c[][] = { {"sunos-i386", "sunos", "x86"}, 
                         {"sunos-sparc", "sunos", "sparc"},
                         {"linux-i386", "linux", ""},
                         {"windows-i386", "windows", ""},
                         {"darwin-universal", "mac os x", ""},
                         {"hpux-ia64", "hp-ux", ""},
                         {"aix-powerpc", "aix", "ppc" }
        };
        
        for (int i = 0; i < c.length; i++) {
            if (plat.equals(c[i][0]) &&
                    os.indexOf(c[i][1]) != -1 &&
                    arch.indexOf(c[i][2]) != -1) {
                return true;
            }
        }
        return false;
    }

    static void deleteTree(File file) {
        if (file.isFile()) file.delete();
        else if (file.isDirectory()) {
            for (File f : file.listFiles()) {
                deleteTree(f);
            }
            file.delete();
        }
    }
}
