/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2008-2010 Oracle and/or its affiliates. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License.  You can
 * obtain a copy of the License at
 * https://glassfish.dev.java.net/public/CDDL+GPL_1_1.html
 * or packager/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 *
 * GPL Classpath Exception:
 * Oracle designates this particular file as subject to the "Classpath"
 * exception as provided by Oracle in the GPL Version 2 section of the License
 * file that accompanied this code.
 *
 * Modifications:
 * If applicable, add the following below the License Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyright [year] [name of copyright owner]"
 *
 * Contributor(s):
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package com.sun.pkg.client;

import com.sun.pkg.client.Image.ImagePlan;

/**
 * Callback handler for {@link ImagePlan#execute()} to monitor the progress
 * of package add/remove/update operations.
 *
 * <p>
 * Implemented as an abstract class instead of interface so that future additions
 * of new callback methods won't break existing clients.
 *
 * <h2>Callback Order</h2>
 * <p>
 * The callback methods are invoked in the following sequences:
 *
 * <pre>
 * startDownloadPhase
 * for(each fmri that requires download) {
 *   startPackageDownload
 *   for(each file in this fmri that requires download) {
 *     startFileDownload
 *     while(transfer is in progress)
 *       onFileDownloadProgress
 *     endFileDownload
 *   }
 *   endPackageDownload
 * }
 * endDownloadPhase
 * startActions
 * startRemovalPhase
 * for(each removal action) {
 *   onRemovalAction
 * }
 * endRemovalPhase
 * startUpdatePhase
 * for(each Update action) {
 *   onUpdateAction
 * }
 * endUpdatePhase
 * startInstallPhase
 * for(each Install action) {
 *   onInstallAction
 * }
 * endInstallPhase
 * </pre>
 *
 * <h2>Notes</h2>
 * <p>
 * During download, the actual data transfer is subject to the transport level compression
 * (currently it is tar+gzip, but you shouldn't rely on it.) Therefore, depending on the kind
 * of files you are transferring, the progress of transfer may not be uniform.
 *
 * @author Kohsuke Kawaguchi
 */
public class ImagePlanProgressTracker {
    /**
     * Called when the execution enters the download phase, which is the 1st of the four phases.
     *
     * <p>
     * To obtain the total number of files to be downloaded during this phase, use
     * {@link ImagePlan#computeTransferFiles()}. Similarly, to obtain the total number of bytes
     * to be downloaded, use {@link ImagePlan#computeTransferSize()}.
     *
     * @param totalPackages
     *      Number of packages to be affected. Can be 0.
     *      {@link #startPackageDownload(Fmri,int,long)} ... {@link #endPackageDownload(Fmri, int)}
     *      callbacks are invoked this many number of times. This maybe smaller than the number
     *      of packages being affected, because not all packages require downloads.
     */
    public void startDownloadPhase(int totalPackages) {}

    /**
     * Called when the execution starts downloading files for the given package.
     *
     * @param pkg
     *      {@link Fmri} that indicates the package being downloaded. Never null.
     * @param totalDownloadFiles
     *      Number of files to be downloaded in this package. Always greater than 0.
     *      {@link #startFileDownload(int,long)} ... {@link #endFileDownload(int,long)}
     * @param totalDownloadSize
     *      Combined total number of bytes of all the files to be downloaded. Can be 0.
     */
    public void startPackageDownload(Fmri pkg, int totalDownloadFiles, long totalDownloadSize) {}

    /**
     * Called when the execution starts downloading a particular file.
     *
     * @param index
     *      The index of the current file being downloaded. This goes from 0 to
     *      the {@code totalDownloadFiles} parameter passed to {@link #startPackageDownload(Fmri,int,long)}.
     * @param fileSize
     *      The number of bytes to be downloaded for this file. 
     */
    public void startFileDownload(int index, long fileSize) {}

    /**
     * Called periodically while a file is being downloaded.
     *
     * <p>
     * This method is called repeatedly for the same file if the file is large.
     * Also, no matter how small the file is, this method is called at least once,
     * when the download of the file is finished.
     *
     * @param   index
     *      The same as the {@code index} parameter of the {@link #onFileDownloadProgress(int, long)}.
     * @param   xferedBytes
     *      The number of bytes transferred thus far. This is within [0,fileSize] where the fileSize
     *      is given by the {@link #startFileDownload(int, long)} callback.
     */
    public void onFileDownloadProgress(int index, long xferedBytes) {}

    /**
     * Called when a file download is completed.
     *
     * The parameters are the same as {@link #startFileDownload(int, long)}
     */
    public void endFileDownload(int index, long fileSize) {}

    /**
     * Called when downloading for a whole package is completed.
     *
     * The parameters are the same as {@link #startPackageDownload(Fmri,int,long)}.
     */
    public void endPackageDownload(Fmri pkg, int totalDownloadFiles) {}

    /**
     * Called when all that have to be downloaded are downloaded.
     */
    public void endDownloadPhase() {}

    /**
     * Called before the three {@link Action}-based phases are started, to
     * notify the total number of actions combined between those three phases.
     *
     * <p>
     * This callback enables the implementation to use a single progress bar for the following three phases.
     *
     * @param totalActions
     *      Sum of {@code totalXyzActions} passed as parameters to {@link #startRemovalPhase(int)},
     *      {@link #startUpdatePhase(int)}, and {@link #startInstallPhase(int)}. Can be 0.
     */
    public void startActions(int totalActions) {}

    /**
     * Called when the execution gets to the removal phase,
     * where a bunch of files and directories are removed.
     *
     * @param totalRemovalActions
     *      Number of things to be removed. Can be 0.
     *      {@link #onRemovalAction(Action)} callback is called for this many times.
     */
    public void startRemovalPhase(int totalRemovalActions) {}

    /**
     * Called for each removal action, right before it's executed.
     *
     * @param a
     *      The removal action to be performed. Never null. Do not change the state of this object
     *      from the callback.
     */
    public void onRemovalAction(Action a) {}

    /**
     * Called when all that have to be removed are removed.
     */
    public void endRemovalPhase() {}

    /**
     * Called when the execution gets to the update phase,
     * where a bunch of files and directories are overwritten/modified.
     *
     * @param totalUpdateActions
     *      Number of things to be updated. Can be 0.
     *      {@link #onUpdateAction(Action, Action)} callback is called for this many times.
     */
    public void startUpdatePhase(int totalUpdateActions) {}

    /**
     * Called for each update action, right before it's executed.
     *
     * @param from
     *      Unused. Passed for future extension. Never null.
     * @param to
     *      Represents the target of the state transition. Never null.
     */
    public void onUpdateAction(Action from, Action to) {}

    /**
     * Called when all that have to be updated are updated.
     */
    public void endUpdatePhase() {}

    /**
     * Called when the execution gets to the install phase,
     * where a bunch of new files and directories are installed.
     *
     * @param totalInstallActions
     *      Number of things to be newly installed. Can be 0.
     *      {@link #onInstallAction(Action)} callback is called for this many times.
     */
    public void startInstallPhase(int totalInstallActions) {}

    /**
     * Called for each install action, right before it's executed.
     *
     * @param a
     *      The install action to be performed. Never null. Do not change the state of this object
     *      from the callback.
     */
    public void onInstallAction(Action a) {}

    /**
     * Called when all that have to be installed are installed.
     */
    public void endInstallPhase() {}
}
