/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2008-2010 Oracle and/or its affiliates. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License.  You can
 * obtain a copy of the License at
 * https://glassfish.dev.java.net/public/CDDL+GPL_1_1.html
 * or packager/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 *
 * GPL Classpath Exception:
 * Oracle designates this particular file as subject to the "Classpath"
 * exception as provided by Oracle in the GPL Version 2 section of the License
 * file that accompanied this code.
 *
 * Modifications:
 * If applicable, add the following below the License Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyright [year] [name of copyright owner]"
 *
 * Contributor(s):
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package com.sun.pkg.client;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Hashtable;

public class Variant {

    /**
     * Convert an array of tokens of the form "property=value" to a
     * hashtable of variants.
     *
     * @param tokens
     * @return
     */
    static public Map<String, String> tokensToVariantTable(String[] tokens) {
        Map<String, String> variantTable = new Hashtable<String, String>();
        String[] variantPair = null;
        for (String t : tokens) {
            if (t == null || t.length() == 0) {
                continue;
            }
            // Only save tags that start with "variant."
            if (t.startsWith("variant.")) {
                variantPair = t.split("\\s*=\\s*", 2);
                if (variantPair != null) {
                    variantTable.put(variantPair[0].trim(), variantPair[1]);
                }
            }
        }
        return variantTable;
    }

    /**
     * Returns true if the named Variant is selected by the list of Variants.
     * Note that the assumption is that a variant will occur only once
     * in the imageVariants. If a variant occurs more than once then
     * the behavior of which variant is used to determine selection
     * is not defined.
     *
     * If the variant is not in the imageVariants, then it is not selected
     * If the variant is in the imageVariants and its value matches
     *  then it is selected.
     * If the variant is in the imageVariants and its value does not
     *  match then it is not selected.
     */
    public static boolean isSelectedBy(String vname, String vvalue,
                                       Map<String, String> imageVariants) {

        // If there is no named variant, or it is not set to any specific
        // value, then consider it selected.
        if (vname == null || vname.length() == 0 ||
            vvalue == null || vvalue.length() == 0) {
            return true;
        }

        // If an image has no variants defined then an action with a
        // variant can't be selected.
        if (imageVariants == null || imageVariants.size() == 0) {
            return false;
        }

        String imageValue =  imageVariants.get(vname);

        if (imageValue == null) {
            // The named variant is not in the list of variants for the image
            // so it is not selected.
            return false;
        }

        if (vvalue.equals(imageValue)) {
            // Name and value match. Selected
            return true;
        } else {
            // Name matches, but value does not. Not selected
            return false;
        }
    }


    /**
     * Returns true if all of the passed action variant tags are selected
     * by the passed image variants.
     */
    public static boolean isSelectedBy(Map<String, String> actionVariantTags,
                                      Map<String, String> imageVariants) {

        // If an action has no variants then it is selected.
        if (actionVariantTags == null || actionVariantTags.size() == 0) {
            return true;
        }

        // If an action has variants, but the image does not, then
        // the action can't be selected.
        if (imageVariants == null || imageVariants.size() == 0) {
            return false;
        }

        // Check all action variant tags and make sure they are all
        // selected by the image variants.
        Set<String> actionVariantNames = actionVariantTags.keySet();
        for (String vname : actionVariantNames) {
            String vvalue = actionVariantTags.get(vname);
            if (! isSelectedBy(vname, vvalue, imageVariants)) {
                return false;
            }
        }

        // All action variant tags are selected by the image variants
        return true;
    }

    /**
     * Test method.
     */
    public static boolean runTests() {
        Map<String, String> imageVariants = null;
        Map<String, String> actionVariantTags = null;
        int nfailures = 0;
        int nrun = 0;

        // Variants set on an image
        String[] imageVariantTokens = {
            "variant.os.bits=32",
            "variant.opensolaris.zone=global",
            "variant.arch=i386"
        };

        // Variant tags set on actions for actions that should be selected using
        // above image variants
        String[][] actionVariantTokensSelect = {
            {"variant.os.bits=32", "variant.arch=i386"},
            {"variant.os.bits=32"},
            {"variant.arch=i386"},
            {"variant.opensolaris.zone=global"},
            {"variant.arch=i386", "random.property=foobar"},
            {"random.property=foobar", "anotherrandomproperty=dog"},
            {"random.property=foobar"},
            {""},
            {null},
        };

        // Variant tags set on actions for actions that should not be selected using
        // above image variants
        String[][] actionVariantTokensNoSelect = {
            {"variant.os.bits=64", "variant.arch=i386"},
            {"variant.os.bits=32", "variant.arch=sparc"},
            {"variant.os.bits=64", "variant.arch=sparc"},
            {"variant.os.bits=32", "variant.arch=i386", "variant.junk=morejunk"},
            {"variant.junk=morejunk"},
            {"variant.os.bits=64"},
            {"variant.os.bits=64", "random.property=foobar"},
        };

        imageVariants = tokensToVariantTable(imageVariantTokens);

        boolean match = false;

        for (String[] tokens : actionVariantTokensSelect) {
            actionVariantTags = tokensToVariantTable(tokens);
            match = Variant.isSelectedBy(actionVariantTags, imageVariants);
            nrun++;
            if (!match) {
                nfailures++;
                testError(actionVariantTags, imageVariants, true);
            }
        }

        for (String[] tokens : actionVariantTokensNoSelect) {
            actionVariantTags = tokensToVariantTable(tokens);
            match = Variant.isSelectedBy(actionVariantTags, imageVariants);
            nrun++;
            if (match) {
                nfailures++;
                testError(actionVariantTags, imageVariants, false);
            }
        }

        // No image variants. Should not select any action with a variant tag
        imageVariants.clear();
        for (String[] tokens : actionVariantTokensSelect) {
            actionVariantTags = tokensToVariantTable(tokens);
            if (actionVariantTags == null || actionVariantTags.isEmpty()) {
                // If no action tags then skip the test (since it will be selected)
                continue;
            }
            match = Variant.isSelectedBy(actionVariantTags, imageVariants);
            nrun++;
            if (match) {
                nfailures++;
                testError(actionVariantTags, imageVariants, false);
            }
        }

        int npass = nrun - nfailures;
        System.out.println(nrun + " tests run: " + npass + " passed, " +
                nfailures + " failed.");

        return nfailures == 0;
    }

    public static void testError(Map<String, String> actionVariantTags,
                        Map<String, String> imageVariants,
                        boolean expectedResult) {

        if (expectedResult == true) {
            System.out.println("FAIL: Action with tags " + actionVariantTags.toString() +
                " was not selected by image variants " + imageVariants.toString());
        } else {
            System.out.println("FAIL: Action with tags " + actionVariantTags.toString() +
                " was selected by image variants " + imageVariants.toString());
        }
        return;
    }

    public static void main(String args[]) {
        runTests();
    }
}
