/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2008-2011 Oracle and/or its affiliates. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License.  You can
 * obtain a copy of the License at
 * https://glassfish.dev.java.net/public/CDDL+GPL_1_1.html
 * or packager/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 *
 * GPL Classpath Exception:
 * Oracle designates this particular file as subject to the "Classpath"
 * exception as provided by Oracle in the GPL Version 2 section of the License
 * file that accompanied this code.
 *
 * Modifications:
 * If applicable, add the following below the License Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyright [year] [name of copyright owner]"
 *
 * Contributor(s):
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package com.sun.pkg.client;

import java.io.File;
import java.net.InetSocketAddress;
import java.net.Proxy;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Arrays;

/**
 * A simple CLI for pkg-java.
 * @author trm
 */
class Client {
    static final String usage = 
            "Usage:\n" +
            "    java -jar pkg-client.jar [options] command [cmd_options] [operands]\n" +
            "where command is one of:\n" +
            "    list [-a]                      - list packages, list all with -a\n" +
            "    install [-l] pkgs...           - install packages, list license(s) with -l\n" +
            "    uninstall pkgs...              - uninstall packages\n" +
            "    image-create -a prefix=url dir - create a new image with the specified initial authority\n" +
            "    image-update [-n]              - update each installed package to the most recent version\n" +
            "                                     just print with -n, do not actually update\n" +
            "    refresh                        - refresh the catalogs\n" +
            "    prefs                          - print information from preferences (proxy, update check)\n" +
            "    info                           - print information about the image\n" +
            "    set-property pname pvalue      - set a property value\n";
    
    public static void main(String args[]) throws Exception {
        Image.setDefaultClientName("com.sun.pkg.client.Client");
        File cwd = new File(System.getProperty("user.dir"));

        if (args.length==0) {
            System.err.println(usage);
            System.exit(1);
        }
        String command = args[0];

        if (command.equals("list")) {
            boolean all = false;
            if (args.length >= 2 && args[1].equals("-a")) all = true;
            Image img = new Image(cwd);
            List<Image.FmriState> list = img.getInventory(null, all);
            for (Image.FmriState fs : list) {
                StringBuffer sb = new StringBuffer();
                StringBuffer flags = new StringBuffer();
                sb.append(fs.fmri.toString());
                sb.append(" - " + fs.fmri.getAuthority());
                if (fs.installed) flags.append("I");
                if (fs.upgradable) flags.append("U");
                if (flags.length() > 0) {
                    sb.append(" (").append(flags).append(")");
                }
                System.out.println(sb);
            }
        }
        else if (command.equals("refresh")) {
            Image img = new Image(cwd);

            // Set metadata on the catalog request.
            // This is mainly here for testing purposes.
            HashMap<String,String> map = new HashMap<String,String>();
            map.put("product", "pkg-java");
            map.put("context", "pkg-java cli");
            img.setMetaData(map);
            img.refreshCatalogs();
        }
        else if (command.equals("install")) {
            boolean showLicenses = false;
            ArrayList<String> pnames = new ArrayList<String>();
            for (int i = 1; i < args.length; i++) {
                if (args[i].equals("-l")) showLicenses = true;
                else pnames.add(args[i]);
            }
            Image img = new Image(cwd);
            Image.ImagePlan ip = img.makeInstallPlan(pnames);
            if (showLicenses) {
                StringBuffer licenseText = new StringBuffer();
                for (Fmri f : ip.getProposedFmris()) {
                    Manifest m = img.getManifest(f);
                    List<LicenseAction> lla = m.getActionsByType(LicenseAction.class);
                    for (LicenseAction la : lla) {
                        licenseText.append("===== ").append(la.getName()).append(" =====\n");
                        licenseText.append("\n");
                        licenseText.append(la.getText());
                        licenseText.append("\n\n");
                    }
                }
                if (licenseText.length() > 0) {
                    System.out.println(licenseText);
                }
            }
            ip.execute(new ProgressTrackerImpl());
        }
        else if (command.equals("prepare-install")) {
            ArrayList<String> pnames = new ArrayList<String>();
            for (int i = 1; i < args.length; i++) {
                pnames.add(args[i]);
            }
            Image img = new Image(cwd);
            String sa[] = new String[pnames.size()];
            sa = pnames.toArray(sa);
            img.installPackages(new ProgressTrackerImpl(),
                true, false, sa);
        }
        else if (command.equals("complete-install")) {
            Image img = new Image(cwd);
            img.completeInstall();
        }
        else if (command.equals("uninstall")) {
            try {
            Image img = new Image(cwd);
            Image.ImagePlan ip = img.makeUninstallPlan(img.getVersionsInstalled(Arrays.asList(args).subList(1, args.length)));
            ip.execute(new ProgressTrackerImpl());
            }
            catch (IllegalArgumentException iae) {
                System.out.println("Invalid argument: " + iae.getMessage());
                System.exit(1);
            }
        }
        else if (command.equals("image-create")) {
            if(args.length==4 && args[1].equals("-a")) {
                String[] prefixAndURL = args[2].split("=");
                if(prefixAndURL.length==2) {
                    Image.create(new File(args[3]),prefixAndURL[0],new URL(prefixAndURL[1]));
                } else {
                    System.out.println("Malformed -a option: "+args[1]);
                    System.exit(1);
                }
            } else {
                System.out.println("image-create command requires the -a option and a directory name");
                System.exit(1);
            }
        }
        else if (command.equals("image-update")) {
            boolean noexec = args.length >= 2 && args[1].equals("-n");
            Image img = new Image(cwd);
            List<Image.FmriState> installed = img.getInventory(null, false);
            String ipkgs[] = new String[installed.size()];
            int i = 0;
            for (Image.FmriState fs : installed) ipkgs[i++] = fs.fmri.getName();
            Image.ImagePlan ip = img.makeInstallPlan(ipkgs);
            Fmri[] proposed = ip.getProposedFmris();
            if (proposed.length > 0) {
                System.out.println("The following installed packages will be updated:");
                for (Image.FmriState fs : installed) {
                    for (Fmri p : proposed) {
                        if (fs.fmri.getName().equals(p.getName())) {
                            System.out.println("" + fs.fmri + " -> " + p);
                        }
                    }
                }
                if (!noexec) {
                    System.out.println("Updating the image...");
                    ip.execute(new ProgressTrackerImpl());
                }
           } else {
                System.out.println("No packages need to be updated.");
           }
        }
        else if (command.equals("prefs")) {
            Proxy p = SystemInfo.getProxy();
            if (p == null) {
                System.out.println("no proxy");
            }
            else {
                InetSocketAddress isa = (InetSocketAddress)p.address();
                System.out.println("proxy: " + isa.getHostName() + ":" + isa.getPort());
            }
            System.out.println("update check frequency: " + SystemInfo.getUpdateCheckFrequency());
        }
        else if (command.equals("info")) {
            Image img = new Image(cwd);
            System.out.println("preferred authority: " + img.getPreferredAuthorityName());
            System.out.println("user-agent: " + img.getUserAgent());
        }
        else if (command.equals("set-property")) {
            if (args.length != 3) {
                System.err.println(usage);
                System.exit(1);
            }
            Image img = new Image(cwd);
            img.setProperty(args[1], args[2]);
            img.saveConfig();
        }
        else if (command.equals("test")) {
        }
        else {
            System.err.println(usage);
            System.exit(1);
        }
        System.exit(0);
    }

    /**
     * Sample progress tracker to show the status of image plan executoin in progress.
     */
    private static class ProgressTrackerImpl extends ImagePlanProgressTracker {
        @Override
        public void startDownloadPhase(int totalPackages) {
            System.out.println(totalPackages + " packages to be downloaded");
        }

        long bytes;
        int total;
        int totalForPhase;

        @Override
        public void startPackageDownload(Fmri pkg, int totalDownloadFiles, long l) {
            System.out.println("Downloading " + pkg + " of " + totalDownloadFiles + " files");
            bytes = 0;
            total = totalDownloadFiles;
        }

        @Override
        public void onFileDownloadProgress(int index, long xferedBytes) {
            bytes += xferedBytes;
            System.out.print("\rDownloading " + (bytes / 1024) + "KB for " + (index + 1) + "/" + total);
        }

        @Override
        public void endPackageDownload(Fmri pkg, int totalDownloadFiles) {
            System.out.println();
        }

        int index;

        @Override
        public void startActions(int totalActions) {
            total = totalActions;
            index = 0;
        }
        
        @Override
        public void startRemovalPhase(int totalRemovalActions) {
            totalForPhase = totalRemovalActions;
            System.out.println("Removal phase: " + totalRemovalActions);
        }

        @Override
        public void onRemovalAction(Action a) {
            index++;
            System.out.print("\rRemoving " + index + "/" + total);
        }

        @Override
        public void endRemovalPhase() {
            if (totalForPhase > 0) System.out.println();
        }

        @Override
        public void startUpdatePhase(int totalUpdateActions) {
            totalForPhase = totalUpdateActions;
            System.out.println("Update phase: " + totalUpdateActions);
        }

        @Override
        public void onUpdateAction(Action from, Action to) {
            index++;
            System.out.print("\rUpdating " + index + "/" + total);
        }

        @Override
        public void endUpdatePhase() {
            if (totalForPhase > 0) System.out.println();
        }

        @Override
        public void startInstallPhase(int totalInstallActions) {
            totalForPhase = totalInstallActions;
            System.out.println("Install phase: " + totalInstallActions);
        }

        @Override
        public void onInstallAction(Action a) {
            index++;
            System.out.print("\rInstalling " + index + "/" + total);
        }

        @Override
        public void endInstallPhase() {
            if (totalForPhase > 0) System.out.println();
        }
    }
}
