/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2008-2010 Oracle and/or its affiliates. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License.  You can
 * obtain a copy of the License at
 * https://glassfish.dev.java.net/public/CDDL+GPL_1_1.html
 * or packager/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 *
 * GPL Classpath Exception:
 * Oracle designates this particular file as subject to the "Classpath"
 * exception as provided by Oracle in the GPL Version 2 section of the License
 * file that accompanied this code.
 *
 * Modifications:
 * If applicable, add the following below the License Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyright [year] [name of copyright owner]"
 *
 * Contributor(s):
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package com.sun.pkg.client;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

/**
 * A <code>DependAction</code> is an action that represents a dependency from
 * one package to another. 
 * <p>
 * The fmri attribute is expected to be the pkg FMRI that this package depends 
 * on.  The type attribute is one of
 * <p>
 * <b>optional</b> - dependency if present activates additional functionality, but is 
 * not needed
 * <p>
 * <b>require</b> - dependency is needed for correct function
 * <p>
 * <b>transfer</b> - dependency on minimum version of other package that donated
 * components to this package at earlier version.  Other package need not
 * be installed, but if it is, it must be at the specified version.  Effect
 * is the same as optional, but semantics are different.
 * <p>
 * <b>incorporate</b> - optional freeze at specified version
 * <p>
 * <b>exclude</b> - package non-functional if dependent package is present 
 * (unimplemented)
 * 
 * @author trm
 */
public final class DependAction extends Action {
    /**
     * Type of dependency.
     */
    public static enum Type {
        /**
         * Dependency is needed for correct function.
         */
        REQUIRED,
        /**
         * Dependency if present activates additional functionality, but is
         * not needed.
         */
        OPTIONAL,
        /**
         * Dependency on minimum version of other package that donated
         * components to this package at earlier version.  Other package need not
         * be installed, but if it is, it must be at the specified version.  Effect
         * is the same as optional, but semantics are different.
         */
        TRANSFER,
        /**
         * Optional freeze at specified version.
         */
        INCORPORATE,
        /**
         * Package non-functional if dependent package is present
         * (unimplemented)
         */
        EXCLUDE
    }

    static final Map<String, Type> knownTypes = new HashMap<String, Type>();
    {
        knownTypes.put("require", Type.REQUIRED);
        knownTypes.put("optional", Type.OPTIONAL);
        knownTypes.put("transfer", Type.TRANSFER);
        knownTypes.put("incorporate", Type.INCORPORATE); 
        knownTypes.put("exclude", Type.EXCLUDE);
    }
    private Image img;
    private Type type;
    private String fmriStr;
    private Constraint constraint;
    
    DependAction(Image image, Fmri srcfmri, String tokens[])
    {
        super(1, tokens);
        img = image;
        for (int i = 1; i < tokens.length; i++) {
            if (tokens[i].startsWith("fmri=")) fmriStr = tokens[i].substring(5);
            else if (tokens[i].startsWith("type=")) {
                String typeStr = tokens[i].substring(5);
                if (knownTypes.containsKey(typeStr)) {
                    type = knownTypes.get(typeStr);
                } else {
                    throw new IllegalArgumentException("unknown depend type: "
                            + typeStr);
                }
            }
        }
        Fmri f = new Fmri(fmriStr);
        String name = f.getName();
        Version minVer = f.getVersion();
        Version maxVer = null;
        Constraint.Presence presence = null;
        
        if (type == Type.REQUIRED) presence = Constraint.Presence.ALWAYS;
        else if (type == Type.EXCLUDE) presence = Constraint.Presence.NEVER;
        else if (type == Type.INCORPORATE) {
            presence = Constraint.Presence.MAYBE;
            maxVer = minVer;
        }
        else if (type == Type.OPTIONAL) {
            // TODO need to check cfg_cache policy
            presence = Constraint.Presence.MAYBE;
        }
        else if (type == Type.TRANSFER) presence = Constraint.Presence.MAYBE;
        
        constraint = new Constraint(name, minVer, maxVer, presence, srcfmri.getName());
    }
    
    void install(Action orig) throws IOException
    {
    }

    /**
     * Gets the type of the dependency.
     *
     * TODO: can this ever be null? I don't think so, but check and document.
     */
    public Type getType()
    {
        return type;
    }
       
    /*
     * Returns fmri of incorporation pkg or null if not an incorporation
     */
    Fmri getConstrainedFmri() {
        if (type != Type.INCORPORATE) return null;
        Fmri f = new Fmri(fmriStr);
        img.setFmriDefaultAuthority(f);
        return f;
    }

    /**
     * The {@link Fmri} specified as a dependency.
     *
     * @return never null.
     */
    public Fmri getTargetFmri() {
        return new Fmri(fmriStr);
    }
    
    Constraint getConstraint() {
        return constraint;
    }
 
    @Override
    String keyValue() 
    { 
        return fmriStr; 
    }
    
    @Override
    public String toString()
    {
        return "depend: fmri=" + fmriStr + ", type=" + type + 
                ", constraint=" + constraint;
    }
    
    boolean isDifferent(Action a) {
        if (!(a instanceof DependAction)) return true;
        DependAction da = (DependAction)a;
        return !equals(a) || 
               !constraint.equals(da.constraint);
    }
}
