/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2008-2011 Oracle and/or its affiliates. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License.  You can
 * obtain a copy of the License at
 * https://glassfish.dev.java.net/public/CDDL+GPL_1_1.html
 * or packager/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 *
 * GPL Classpath Exception:
 * Oracle designates this particular file as subject to the "Classpath"
 * exception as provided by Oracle in the GPL Version 2 section of the License
 * file that accompanied this code.
 *
 * Modifications:
 * If applicable, add the following below the License Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyright [year] [name of copyright owner]"
 *
 * Contributor(s):
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package com.sun.pkg.client;

import java.io.BufferedInputStream;
import java.util.List;
import java.util.ArrayList;
import java.util.Map;
import java.util.HashMap;
import java.net.URLEncoder;
import java.io.IOException;
import java.io.File;
import java.io.FileOutputStream;
import java.io.OutputStream;
import java.lang.management.ManagementFactory;
import java.net.HttpURLConnection;
import java.util.logging.Level;
import java.util.LinkedList;
import java.util.concurrent.*;

import java.util.Iterator;
import org.apache.tools.tar.TarInputStream;
import org.apache.tools.tar.TarEntry;
import java.io.OutputStreamWriter;


/**
 * A <code>FileList</code> is a list of files that need to be downloaded from 
 * a repository. Each file maps to one or more DataSinks that need the file
 * (example DataSinks are FileActions and LicenseActions).
 * 
 * @author trm
 */
class FileList {
    interface DataSink {
        String getHash();
        int getSize();
        void setGzipFile(File f);
    }

    // Size of the thread pool used to handle file downloads. Default to 3.
    // 3 was chosen as a compromise that gives good performance without
    // burdening the server with many connections.
    int thread_pool_size = 3;
    ExecutorService thread_pool = null;

    Image img;
    Fmri fmri;
    Map<String, List<DataSink>> flist = new HashMap<String, List<DataSink>>();
    File download_dir;
    
    public FileList(Image i, Fmri f) {
        img = i;
        fmri = f;
        download_dir = new File(new File(img.getMetaDirectory(), "download"), getPID());
    }
    
    public void setThreadPoolSize(int n) {
        thread_pool_size = n;
    }

    public int getThreadPoolSize() {
        return thread_pool_size;
    }

    public void add(DataSink a) {
        String h = a.getHash();
        List<DataSink> l;
        if ((l = flist.get(h)) == null) {
            l = new ArrayList<DataSink>();
            flist.put(h, l);
        }
        l.add(a);
    }

    /**
     * Computes the total amount of bytes that need to be downloaded.
     */
    public long computeTransferSize() {
        long total = 0;
        for (List<DataSink> ds : flist.values()) {
            total += ds.get(0).getSize();
        }
        return total;
    }

    /**
     * Number of total files to be downloaded.
     */
    public int computeTransferFiles() {
        return flist.size();
    }

    /**
     * Download the set of files contained in this file list.
     *
     * This implementation uses a thread pool so that we perform multiple
     * downloadeds in parallel to improve performance (bug 2175).
     *
     * Each file in the file list results in one
     * HTTP request to get the file. That's one round trip to the HTTP 
     * server per file -- a lot of round trips. By using multiple threads
     * we mitigate the impact of the latency of each round trip. Also
     * since HTTP Keep-Alive is used (transparently by the JDK) we end
     * up with one connection per thread that is re-used for all the HTTP
     * requests made by the thread.
     */
    public void download(ImagePlanProgressTracker tracker) throws IOException {
        if (flist.isEmpty()) return;

        if (thread_pool == null) {
            thread_pool = Executors.newFixedThreadPool(thread_pool_size);
        }

        download_dir.mkdirs();

        int nfile = 0;

        img.getLogger().log(Level.FINE, "downloadfiles", new Object[] {fmri, flist.size()});
        tracker.startPackageDownload(fmri, flist.size(), computeTransferSize());

        LinkedList<Future<File>> futures = new LinkedList<Future<File>>();

        // Create a download task for each file in the file list and
        // submit it to the thread pool for execution. Save the returned
        // Future so we can make sure they jave all completed before we
        // continue.
        for (String hash : flist.keySet()) {
            List<DataSink> fal = flist.get(hash);
            DownloadFileTask dft = new DownloadFileTask(
                img, fal, fmri, download_dir, nfile, tracker);
            nfile++;
            Future<File> future = thread_pool.submit(dft);
            futures.add(future);
        }

        // Make sure all the tasks are completed before we continue.
        for (Future<File> f : futures) {
            try {
                // This returns the File object of the downloaded file,
                // but we don't need that here.
                f.get();
            } catch (ExecutionException ex) {
                // A download task couldn't execute because it threw an
                // exception. Abort the entire download. and re-throw the
                // IOException.
                thread_pool.shutdown();
                Throwable cause = ex.getCause();
                if (cause instanceof IOException) {
                    throw (IOException) cause;
                } else {
                   // An unexpected exception. Wrap it and re-throw
                   throw new IOException(cause);
                }
            } catch (InterruptedException ex) {
                // Task was interrupted. Shutdown download.
                thread_pool.shutdown();
                throw new IOException(ex);
            }
        }

        tracker.endPackageDownload(fmri, flist.size());
    }

    /**
     * The download function that was used in 2.3.3. Fixes the auto-tune
     * windows hang (bug 2797) and matches the python impelemntation, but
     * is slow (bug 2175). "s" for single threaded.
     */
    public void sdownload(ImagePlanProgressTracker tracker) throws IOException {
        if (flist.isEmpty()) return;

        download_dir.mkdirs();

        int nfile = 0;
        byte[] buf = new byte[32 * 1024];

        img.getLogger().log(Level.FINE, "downloadfiles", new Object[] {fmri, flist.size()});
        tracker.startPackageDownload(fmri, flist.size(), computeTransferSize());

        for (String hash : flist.keySet()) {
            String hv = URLEncoder.encode(hash, "UTF-8");
            HttpURLConnection uc = img.getRepositoryURLConnection("file/0/" + hv, fmri);
            img.checkRepositoryConnection(uc);
            BufferedInputStream tis = new BufferedInputStream(uc.getInputStream());
            DataSink fa = flist.get(hash).get(0);
            List<DataSink> fal = flist.get(hash);
            String outfilename = hash;
            img.getLogger().log(Level.FINER, "downloading", outfilename);
            tracker.startFileDownload(nfile, fa.getSize());

            File gzipfile = new File(download_dir, outfilename);
            fa.setGzipFile(gzipfile);
            OutputStream fos = new FileOutputStream(gzipfile);
            long xferedBytes = 0;
            while (true) {
                int numRead = tis.read(buf, 0, buf.length);
                if (numRead == -1) break;
                fos.write(buf, 0, numRead);
                xferedBytes += numRead;
                tracker.onFileDownloadProgress(nfile, xferedBytes);
            }
            tracker.onFileDownloadProgress(nfile, xferedBytes); // make sure this is called at least once

            fos.close();
            tracker.endFileDownload(nfile++, fa.getSize());
            for (int j = 1; j < fal.size(); j++) {
                DataSink nfa = fal.get(j);
                nfa.setGzipFile(gzipfile);
            }
        }

        tracker.endPackageDownload(fmri, flist.size());
    }


    /**
     * The original 2.3 download function. Requests batches of files at once.
     * Fast, but does not match python implementation and can trigger an 
     * autotune hang on Windows (see bug 2797). "b" for batching
     */
    public void bdownload(ImagePlanProgressTracker tracker) throws IOException {
        if (flist.isEmpty()) return;

        download_dir.mkdirs();

        StringBuffer reqbuf = new StringBuffer();
        int i = 0;
        int size = 0;
        int nfile = 0;
        byte[] buf = new byte[32 * 1024];

        img.getLogger().log(Level.FINE, "downloadfiles", new Object[] {fmri, flist.size()});
        tracker.startPackageDownload(fmri, flist.size(), computeTransferSize());

        for (Iterator<String> aiter = flist.keySet().iterator(); aiter.hasNext(); i++) {
            String hash = aiter.next();
            if (reqbuf.length() > 0) reqbuf.append("&");
            reqbuf.append("File-Name-" + i + "=" + URLEncoder.encode(hash, "UTF-8"));
            DataSink fa = flist.get(hash).get(0);
            size += fa.getSize();
            if (size > (1024 * 1024) || !aiter.hasNext()) {
                // do a download
                HttpURLConnection uc = img.getRepositoryURLConnection("filelist/0", fmri);
                uc.setDoOutput(true);
                OutputStreamWriter osw = new OutputStreamWriter(uc.getOutputStream());
                osw.write(reqbuf.toString());
                osw.close();
                img.checkRepositoryConnection(uc);

                TarInputStream tis = new TarInputStream(new BufferedInputStream(uc.getInputStream()));

                TarEntry te;
                while ((te = tis.getNextEntry()) != null) {
                    hash = te.getName();
                    List<DataSink> fal = flist.get(hash);
                    if (fal == null) {
                        throw new IOException("invalid response from server: unrequested file: " +
                                hash);
                    }
                    fa = fal.get(0);
                    String outfilename = hash;
                    img.getLogger().log(Level.FINER, "downloading", outfilename);
                    tracker.startFileDownload(nfile, fa.getSize());

                    File gzipfile = new File(download_dir, outfilename);
                    fa.setGzipFile(gzipfile);
                    OutputStream fos = new FileOutputStream(gzipfile);
                    long xferedBytes = 0;
                    while (true) {
                        int numRead = tis.read(buf, 0, buf.length);
                        if (numRead == -1) break;
                        fos.write(buf, 0, numRead);
                        xferedBytes += numRead;
                        tracker.onFileDownloadProgress(nfile, xferedBytes);
                    }
                    tracker.onFileDownloadProgress(nfile, xferedBytes); // make sure this is called at least once

                    fos.close();
                    tracker.endFileDownload(nfile++, fa.getSize());
                    for (int j = 1; j < fal.size(); j++) {
                        DataSink nfa = fal.get(j);
                        nfa.setGzipFile(gzipfile);
                    }
                }

                // reset for another batch of files
                size = 0;
                i = 0;
                reqbuf = new StringBuffer();
            }
        }

        tracker.endPackageDownload(fmri, flist.size());
    }

    /** 
     * Delete all of the temporary files created by this filelist
     * 
     * @throws java.io.IOException
     */
    public void cleanupDownload() throws IOException {
        img.getLogger().log(Level.FINEST, "deletingtemp");

        File files[] = download_dir.listFiles();
        if (files != null && files.length > 0) {
            for (File f : files) f.delete();
        }
        if (download_dir.exists()) download_dir.delete();
    }
    
    static String getPID() {
        String name = ManagementFactory.getRuntimeMXBean().getName();
        int i = 0; 
        while (i < name.length() && Character.isDigit(name.charAt(i))) i++;
        return i > 0 ? name.substring(0, i) : "0";
    }
}
