/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2008-2010 Oracle and/or its affiliates. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License.  You can
 * obtain a copy of the License at
 * https://glassfish.dev.java.net/public/CDDL+GPL_1_1.html
 * or packager/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 *
 * GPL Classpath Exception:
 * Oracle designates this particular file as subject to the "Classpath"
 * exception as provided by Oracle in the GPL Version 2 section of the License
 * file that accompanied this code.
 *
 * Modifications:
 * If applicable, add the following below the License Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyright [year] [name of copyright owner]"
 *
 * Contributor(s):
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package com.sun.pkg.client;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.CharBuffer;
import java.util.List;
import java.util.zip.GZIPInputStream;

/**
 * A <code>LicenseAction</code> is a file action that represents a license file 
 * within the image.
 * 
 * @author trm
 */
public final class LicenseAction extends Action implements FileList.DataSink {
    Image img;
    int size;
    String hash;
    String path;
    Fmri pi;
    String license = "";
    File gzipfile;
    
    LicenseAction(Image image, Fmri fmri, String tokens[]) 
    {
        super(0,tokens);
        img = image;
        pi = fmri;
        hash = tokens[1];

        for (int i = 2; i < tokens.length; i++) {
            if (tokens[i].startsWith("license=")) license = tokens[i].substring(8);
            else if (tokens[i].startsWith("pkg.size=")) {
                size = Integer.parseInt(tokens[i].substring(9));
            }
        }
        path = "license." + license;
    }

    @Override
    void buildFileList(Action orig, FileList flist) { 
        flist.add(this);
    }

    @Override
    void install(Action orig) throws IOException
    {
        if (gzipfile == null || license == null) return;
        File target = new File(pi.getPkgVersionDir(new File(img.getMetaDirectory(), "pkg")), path);

        // If there is an old copy of this license file installed then we
        // Need to delete it since it will be read-only and can't be
        // overwritten.
        target.delete();
        target.getParentFile().mkdirs();
        FileOutputStream fos = new FileOutputStream(target);
        GZIPInputStream gis = new GZIPInputStream(new FileInputStream(gzipfile));
        byte buf[] = new byte[4096];
        int n;
        while ((n = gis.read(buf)) > 0) {
            fos.write(buf, 0, n);
        }
        fos.close();
        gis.close();
        setPermissions(target, "0444");
    }
    
    @Override
    String keyValue() 
    { 
        return license; 
    }

    @Override
    public String toString()
    {
        return "license: license=" + license;
    }

    boolean isDifferent(Action a) {
        if (!(a instanceof LicenseAction)) return true;
        LicenseAction la = (LicenseAction)a;
        return !equals(a) ||
                !hash.equals(la.hash);
    }

    public String getHash() {
        return hash;
    }

    @Override
    public int getSize() {
        return size;
    }
    
    /**
     * Obtain the name of the license.
     * @return the name of the license
     */
    public String getName() {
        return license;
    }
    
    /**
     * Obtain the license text.
     * @return the text of the license
     */
    public String getText() throws IOException {
        File target = new File(pi.getPkgVersionDir(new File(img.getMetaDirectory(), "pkg")), path);
        InputStream is = target.exists() ? 
            new FileInputStream(target) : getRemoteData(img, pi, hash);
        BufferedReader r = new BufferedReader(new InputStreamReader(is, "ISO-8859-1"));
        String line;
        StringBuffer text = new StringBuffer();
        while ((line = r.readLine()) != null) {
            text.append(line).append("\n");
        }
        return text.toString();
    }

    public void setGzipFile(File f) {
        gzipfile = f;
    }
}
