/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2008-2010 Oracle and/or its affiliates. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License.  You can
 * obtain a copy of the License at
 * https://glassfish.dev.java.net/public/CDDL+GPL_1_1.html
 * or packager/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 *
 * GPL Classpath Exception:
 * Oracle designates this particular file as subject to the "Classpath"
 * exception as provided by Oracle in the GPL Version 2 section of the License
 * file that accompanied this code.
 *
 * Modifications:
 * If applicable, add the following below the License Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyright [year] [name of copyright owner]"
 *
 * Contributor(s):
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package com.sun.pkg.client;

import java.util.List;
import java.util.ArrayList;
import java.io.File;
import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.util.HashMap;
import java.util.Map;
import java.util.Iterator;
import java.util.logging.Level;

/**
 * A <code>Manifest</code> is a list of actions within a package.
 * 
 * @author trm
 */
public class Manifest 
{
    /** 
     * An object representing the difference between two manifests
     */
    static class Difference {
        static class ActionPair {
            Action from;
            Action to;
            
            ActionPair(Action f, Action t) {
                from = f;
                to = t;
            }
        }
        List<Action> added;
        List<Action> removed;
        List<ActionPair> changed;
    }
    static final Manifest nullManifest = new Manifest();
    
    List<Action> actions = new ArrayList<Action>();
    Map<String,String> attrs = new HashMap<String,String>();
    String mfilename = "";
    
    public Manifest(Image img, Fmri pi) throws IOException {
        File mfile = new File(pi.getPkgVersionDir(new File(img.getMetaDirectory(), "pkg")), "manifest");

        mfilename = mfile.getPath();
        
        if (!mfile.exists()) {
            // download the manifest
            img.getLogger().log(Level.FINE, "downloadmanifest", pi.toString());
            HttpURLConnection urlc = img.getRepositoryURLConnection("manifest/0/" + pi.getURLPath(), pi);
            urlc.setRequestProperty("X-IPkg-Intent", img.getIntent(pi));
            img.checkRepositoryConnection(urlc);
            BufferedReader ir = new BufferedReader(new InputStreamReader(urlc.getInputStream(), 
                    "ISO-8859-1"));
            new File(mfile.getParent()).mkdirs();
            BufferedWriter ow = new BufferedWriter(new FileWriter(mfile));
            ow.write("set name=fmri value=" + pi);
            ow.newLine();
            ow.write("set name=authority value=" + pi.getAuthority());
            ow.newLine();
            
            String line;
            while ((line = ir.readLine()) != null) {
                if (line.startsWith("#")) continue;
                ow.write(line);
                ow.newLine();
            }
            ir.close();
            ow.close();
        } else {
            try {
                HttpURLConnection urlc = img.getRepositoryURLConnection("manifest/0/" + pi.getURLPath(), pi);
                urlc.setRequestProperty("X-IPkg-Intent", img.getIntent(pi));
                urlc.setRequestMethod("HEAD");
                img.checkRepositoryConnection(urlc);
            }
            catch (IOException ioe) {} // ignore exceptions because this is for information only
        }
        
        // read the manifest from the file
        BufferedReader r = new BufferedReader(new FileReader(mfile));
        String line;
        String tokens[];
        while ((line = r.readLine()) != null) {
            tokens = line.split(" ");
            if (line.indexOf('"') != -1) tokens = rejoin(tokens);
            if (tokens.length >= 1) {
                String ttype = tokens[0];
                if (ttype.equals("file")) {
                    actions.add(new FileAction(img, pi, tokens));
                }
                else if (ttype.equals("depend")) {
                    actions.add(new DependAction(img, pi, tokens));
                }
                else if (ttype.equals("dir")) {
                    actions.add(new DirAction(img, tokens));
                }
                else if (ttype.equals("link")) {
                    actions.add(new LinkAction(img, tokens));
                }
                else if (ttype.equals("license")) {
                    actions.add(new LicenseAction(img, pi, tokens));
                }
                else if (ttype.equals("set")) {
                    SetAction sa = new SetAction(img, tokens);
                    actions.add(sa);
                    attrs.put(sa.name, sa.value);
                }
            }
        }
    }
    
    private Manifest() { }

    private String[] rejoin(String list[]) {
        // Simple state machine to reconnect the elements that shouldn't have
        // been split.  
        boolean inStr = false;
        List<String> nlist = new ArrayList<String>();
        String n = "";
        for (String i : list) {
            if (i.indexOf("=\"") != -1) {
                n = i.replaceAll("=\"", "=");
                if (n.endsWith("\"")) {
                    nlist.add(n.substring(0, n.length() - 1));
                    n = "";
                }
                else inStr = true;
            }
            else if (i.endsWith("\"")) {
                n = n + " " + i.substring(0, i.length() - 1);
                nlist.add(n);
                n = "";
                inStr = false;
            }
            else if (inStr) {
                n = n + " " + i;
            }
            else {
                nlist.add(i);
            }
        }
        if (n.length() != 0) {
            throw new IllegalArgumentException("Unmatched \" in action");
        }
        return nlist.toArray(new String[0]);
    }
    
    List<Action> getActions() {
        return actions;
    }
    
    /**
     * Obtain the actions of a given type.
     * 
     * @param actionClass - the desired type of operation
     * @return a list of the actions of the given type
     */
    public <T extends Action> List<T> getActionsByType(Class<T> actionClass) {
        List<T> alist = new ArrayList<T>();
        for (Action a : actions) {
            if (actionClass.isInstance(a)) {
                alist.add(actionClass.cast(a));
            }
        }
        return alist;
    }
        
    /**
     * Get the difference between the origin and this manifest
     * @param origin the original manifest or null if there is no origin
     * @return a Difference object representing the difference
     */
    Difference getDifference(Manifest origin) {
        if (origin == null) origin = nullManifest;
        
        Difference diff = new Difference();

        diff.added = new ArrayList<Action>(getActions());
        diff.added.removeAll(origin.getActions());

        diff.removed = new ArrayList<Action>(origin.getActions());
        diff.removed.removeAll(getActions());

        diff.changed = new ArrayList<Difference.ActionPair>();
        Map<Action, Action> sset = new HashMap<Action, Action>();
        for (Action a : getActions()) sset.put(a, a);
        Map<Action, Action> oset = new HashMap<Action, Action>();
        for (Action a : origin.getActions()) oset.put(a, a);
        List<Action> isect = new ArrayList<Action>(origin.getActions());
        isect.retainAll(getActions());
        for (Action a : isect) {
            Action sa = sset.get(a);
            Action oa = oset.get(a);
            if (sa.isDifferent(oa) || sa instanceof LicenseAction) {
                diff.changed.add(new Difference.ActionPair(oa, sa));
            }
        }
        return diff;
    }
    
    /** 
     * Determine whether there are duplicate actions in this manifest.
     * 
     * Actions are considered duplicates if they are equal based on the 
     * Action.equal method, not if they are identical.
     * 
     * @return True if there are duplicates
     */
    boolean hasDuplicates() {
        return findDuplicates()!=null;
    }

    /**
     * If this package has a duplicate action, return one such pair.
     *
     * <p>
     * Actions are considered duplicates if they are equal based on the
     * Action.equal method, but if they are {@linkplain Action#isDifferent(Action) different}.
     * The idea here is to catch broken manifests that are inconsistent within itself.
     */
    public Action[] findDuplicates() {
        Map<Action,Action> uniq = new HashMap<Action,Action>(getActions().size());
        for (Action a : getActions()) {
            Action o = uniq.put(a, a);
            /*
                While looking at http://pkg.opensolaris.org/release/manifest/0/SUNWlxml%402.6.31%2C5.11-0.101%3A20081119T224404Z
                I noticed that even official Solaris packages contain duplicate "dir" actions. So the duplicate check
                needs to take this into account, hence the introduction of "isDifferent".

                # SUNWlxml@2.6.31-0.101, client release 5.11
                dir group=sys mode=0755 owner=root path=usr
                dir group=bin mode=0755 owner=root path=usr/bin
                dir group=bin mode=0755 owner=root path=usr/lib
                dir group=bin mode=0755 owner=root path=usr/lib/amd64
                link path=usr/lib/amd64/libxml2.so target=libxml2.so.2
                link path=usr/lib/amd64/libxml2.so.2 target=../../../lib/amd64/libxml2.so.2
                link path=usr/lib/libxml2.so target=./libxml2.so.2
                link path=usr/lib/libxml2.so.2 target=../../lib/libxml2.so.2
                dir group=sys mode=0755 owner=root path=usr
                dir group=bin mode=0755 owner=root path=usr/bin
                ...
             */
            if (o != null && o.isDifferent(a))
                return new Action[]{o, a};
        }
        return null;
    }

    /**
     * Remove all actions that are not selected by the image Variants
     * Returns # of actions removed
     */
    public int filterByVariants(Map<String, String> imageVariants) {
        int count = 0;

        Image.log.log(Level.FINER,
              "filterByVariants(imageVariants={0}): manifest={1}",
              new Object[] {(imageVariants == null ? "null" : imageVariants.toString()), mfilename} );

        List<Action> actions = getActions();
        Iterator<Action> it = actions.iterator();
        while (it.hasNext()) {
            Action a = it.next();
            if (Variant.isSelectedBy(a.variants, imageVariants)) {
                continue;
            } else {
                // Action not selected by image variants. Remove it
                Image.log.log(Level.FINER,
                    "    Removing action {0} with variant tags {1}",
                    new Object[] {a.toString(), a.variants.toString()} );
                it.remove();
                count++;
            }
        }

        Image.log.log(Level.FINER,
            "    Removed {0} actions from manifest {1}",
             new Object[] { Integer.toString(count) , mfilename} );

        return count;
    }
    
    /**
     * Returns a package attribute defined by a set action.
     * @return a package attribute for the given name, or null if there is no 
     * such attribute
     */
    public String getAttribute(String name) {
        return attrs.get(name);
    }
    
    /**
     * Returns the size of the package.
     * 
     * The size of the package is defined as the sum of the pkg.size attributes
     * of the actions within the Manifest.
     * 
     * @return the size of the package
     */
    public int getPackageSize() {
        int size = 0;
        for (Action a : getActions()) size += a.getSize();
        return size;
    }
}
