/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright (c) 2008-2011 Oracle and/or its affiliates. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License.  You can
 * obtain a copy of the License at
 * https://glassfish.dev.java.net/public/CDDL+GPL_1_1.html
 * or packager/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 *
 * GPL Classpath Exception:
 * Oracle designates this particular file as subject to the "Classpath"
 * exception as provided by Oracle in the GPL Version 2 section of the License
 * file that accompanied this code.
 *
 * Modifications:
 * If applicable, add the following below the License Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyright [year] [name of copyright owner]"
 *
 * Contributor(s):
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package com.sun.pkg.client;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.InetSocketAddress;
import java.net.Proxy;
import java.net.ProxySelector;
import java.net.SocketAddress;
import java.net.URL;
import java.util.Date;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.Hashtable;
import java.lang.reflect.InvocationTargetException;

/**
 * <code>SystemInfo</code> provides access to system information related to the
 * pkg(5) Java API.  This includes information related to the updatetool GUI and
 * bootstrap programs that are built on top of this API.  This class can be used
 * to read and write information.
 */
public class SystemInfo {

    public enum UpdateCheckFrequency {

        NEVER, DAILY, WEEKLY, MONTHLY
    }

    static class ArchInfo {
        String arch;
        String bits;

        public ArchInfo(String arch, String bits) {
            this.arch = arch;
            this.bits = bits;
        }
    }

    private static final String APPNAME = "updatetool";
    private static final String INITFILE = "init.cfg";
    private static final String DEFAULTSFILE = "defaults.cfg";
    private static final String WIN_APP_DATA = "Application Data";
    private static final String MAC_APP_DATA = "/Library/Application Support/";
    private static final String eol = System.getProperty("line.separator");
    private static Proxy httpProxy = null;
    private static Proxy httpsProxy = null;
    private static File proxyInfoFile = null;
    private static long proxyInfoLastModified = 0;
    private static final String OS_ARCH_PROP = "os.arch";
    private static final Hashtable<String, ArchInfo> os_arch_table =
                new Hashtable<String, ArchInfo>();

    static {
        // This table maps os.arch values to a canonicalized arch
        // value and a bitness. Unfortunately os.arch is based on the
        // build of the JDK -- not necessarily the OS
        os_arch_table.put("ia64",       new ArchInfo("ia64", "64"));
        os_arch_table.put("ia64n",      new ArchInfo("ia64", "64"));
        os_arch_table.put("sparc",      new ArchInfo("sparc", "64"));
        os_arch_table.put("sparcv9",    new ArchInfo("sparc", "64"));
        os_arch_table.put("amd64",      new ArchInfo("i386", "64"));
        os_arch_table.put("x64",        new ArchInfo("i386", "64"));
        os_arch_table.put("x86",        new ArchInfo("i386", "32"));
        os_arch_table.put("x86_64",     new ArchInfo("i386", "64"));
        os_arch_table.put("i386",       new ArchInfo("i386", "32"));
        os_arch_table.put("i686",       new ArchInfo("i386", "32"));
        os_arch_table.put("i686_64",    new ArchInfo("i386", "64"));
        os_arch_table.put("pa-risc2.0", new ArchInfo("pa-risc", "64"));
        os_arch_table.put("pa-risc",    new ArchInfo("pa-risc", "32"));
        os_arch_table.put("power",      new ArchInfo("powerpc", "32"));
        os_arch_table.put("powerpc",    new ArchInfo("powerpc", "32"));
        os_arch_table.put("ppc",        new ArchInfo("powerpc", "32"));
        os_arch_table.put("ppc64",      new ArchInfo("powerpc", "64"));
    }

    /**
     * Get the bitness of the OS. Note that Java is unable to do this
     * accurately -- so we are really getting the bitness of the JVM.
     */
    public static String getOSBitness() {
        ArchInfo info  = os_arch_table.get(System.getProperty(OS_ARCH_PROP).toLowerCase());
        if (info.bits == null) {
            return "32";
        } else {
            return info.bits;
        }
    }

    /**
     * Get the bitness of the OS. Get the OS architecture as a
     * canonicalized string. I.e. this will return "i386" if the os.arch
     * is "x86", "i686", "i686_64", etc.
     */
    public static String getCanonicalOSArch() {
        ArchInfo info  = os_arch_table.get(System.getProperty(OS_ARCH_PROP).toLowerCase());
        if (info.bits == null) {
            return "i386";
        } else {
            return info.arch;
        }
    }

    /**
     * Obtain a ProxySelector based on the updatetool preferences.
     * <p>
     * This method returns a <code>ProxySelector</code> based on the information  
     * stored in the updatetool preferences. Proxies returned form this ProxySelector
     * may be null if updatetool is using system proxy settings that are not stored 
     * in the updatetool preferences file. To mimic the behavior of 
     * using the system proxy settings in Java, use the java.net.useSystemProxies
     * system property.
     * 
     * If unable to read the proxy information or if the proxy information is 
     * invalid, a ProxySelector is returned that will yield null Proxies and 
     * error information is logged.
     * 
     * @return a ProxySelector object
     */
    public static ProxySelector getProxySelector() {
        loadProxyInfo();
        return new PkgProxySelector(httpProxy, httpsProxy);
    }

    /**
     * Obtain proxy based on the updatetool preferences.
     * <p>
     * This method returns a <code>Proxy</code> based on the information  
     * stored in the updatetool preferences. This method may return null if
     * updatetool is using system proxy settings that are not stored in the 
     * updatetool preferences file. To mimic the behavior of 
     * using the system proxy settings in Java, use the java.net.useSystemProxies
     * system property.
     * 
     * If unable to read the proxy information or if the proxy information is 
     * invalid, null is returned and error information is logged.
     * 
     * @return a Proxy object
     */
    public static Proxy getProxy() {
        loadProxyInfo();
        return httpProxy;
    }
    
    private static void loadProxyInfo() {
        Logger log = Logger.getLogger(SystemInfo.class.getName());
        try {

            // read values from the environment first, if either is set, then
            // ignore the configuration file
            String hp = System.getenv("http_proxy");
            String hsp = System.getenv("https_proxy");
            if (hp != null || hsp != null) {
                httpProxy = (hp == null || hp.length() == 0) ? null : AuthProxy.newProxy(new URL(hp));
                httpsProxy = (hsp == null || hsp.length() == 0) ? null : AuthProxy.newProxy(new URL(hsp));
                return;
            }

            // If the init.cfg file is there, use the values from that.  Otherwise
            // use the values from defaults.cfg.
            File f = getInitFile();
            if (!f.exists()) {
                f = getDefaultsFile();
                if (!f.exists()) {
                    // no information to read
                    return;
                }
            }
            if (proxyInfoFile != null && proxyInfoFile.equals(f) &&
                    proxyInfoLastModified >= f.lastModified()) {
                // information hasn't changed, no need to read
                return;
            }
            proxyInfoFile = f;
            proxyInfoLastModified = f.lastModified();
            httpProxy = httpsProxy = null;

            // This is a poor man's implementation of a configuration file reader
            // It just looks for particular strings at the beginning of lines,
            // ignoring section boundaries.  This is sufficient to get the proxy
            // information.
            // Fortunately, the names that are used to store the proxy information
            // are the same in both files, so the same loop can be used no matter
            // which file we are reading.
            BufferedReader r = getFileReader(f);
            String line;
            String[] tokens;
            String proxy_host = null;
            int proxy_port = 80;
            String proxy_https_host = null;
            int proxy_https_port = 443;
            String proxy_username = "";
            String proxy_password = "";
            boolean useSystem = false;
            boolean proxy_required = false;
            boolean proxy_ssl_use_http = true;
            boolean proxy_auth = false;
            while ((line = r.readLine()) != null) {
                if (!line.startsWith("proxy.")) {
                    continue;
                }
                tokens = line.split("\\s*[:=]\\s*", 2);
                if (tokens.length != 2) {
                    // ignore lines that start with just an attribute name
                    // e.g., proxy.host, with no = or :
                    continue;
                }
                if (tokens[0].equals("proxy.host")) {
                    proxy_host = tokens[1].trim();
                } else if (tokens[0].equals("proxy.port") &&
                           tokens[1].trim().length() > 0) {
                    try {
                        proxy_port = Integer.parseInt(tokens[1].trim());
                    }
                    catch (NumberFormatException nfe) {
                        log.log(Level.SEVERE, "invalidproxyport",
                            new Object[] {"HTTP", proxyInfoFile, tokens[1]} );
                    }
                } else if (tokens[0].equals("proxy.https.host")) {
                    proxy_https_host = tokens[1].trim();
                } else if (tokens[0].equals("proxy.https.port") &&
                           tokens[1].trim().length() > 0) {
                    try {
                        proxy_https_port = Integer.parseInt(tokens[1].trim());
                    }
                    catch (NumberFormatException nfe) {
                        log.log(Level.SEVERE, "invalidproxyport",
                            new Object[] {"HTTPS", proxyInfoFile, tokens[1]} );
                    }
                } else if (tokens[0].equals("proxy.required")) {
                    proxy_required = Boolean.parseBoolean(tokens[1].trim());
                } else if (tokens[0].equals("proxy.auth")) {
                    proxy_auth = Boolean.parseBoolean(tokens[1].trim());
                } else if (tokens[0].equals("proxy.ssl_use_http")) {
                    proxy_ssl_use_http = Boolean.parseBoolean(tokens[1].trim());
                } else if (tokens[0].equals("proxy.username")) {
                    proxy_username = tokens[1];
                } else if (tokens[0].equals("proxy.password")) {
                    proxy_password = tokens[1];
                } else if (tokens[0].equals("proxy.use.system")) {
                    useSystem = Boolean.parseBoolean(tokens[1].trim());
                }
            }

            if (proxy_required) {
                if (proxy_host == null || proxy_host.length() == 0) {
                    log.log(Level.SEVERE, "invalidproxy",
                            new Object[] {"HTTP", proxyInfoFile} );
                } else {
                    SocketAddress addr = new InetSocketAddress(proxy_host, proxy_port);
                    httpProxy = proxy_auth ?
                        new AuthProxy(Proxy.Type.HTTP, addr, proxy_username, proxy_password) :
                        new AuthProxy(Proxy.Type.HTTP, addr);
                }

                if (proxy_ssl_use_http) {
                    httpsProxy = httpProxy;
                } else {
                    if (proxy_https_host == null || proxy_https_host.length() == 0) {
                        log.log(Level.SEVERE, "invalidproxy", 
                                new Object[] {"HTTPS", proxyInfoFile});
                    } else {
                        SocketAddress addr = new InetSocketAddress(proxy_https_host,
                                proxy_https_port);
                        httpsProxy = proxy_auth ?
                            new AuthProxy(Proxy.Type.HTTP, addr, proxy_username, proxy_password) :
                            new AuthProxy(Proxy.Type.HTTP, addr);
                    }
                }
            } else if (useSystem) {
                System.setProperty("java.net.useSystemProxies", "true");
            }
        } catch (Exception ex) {
            log.log(Level.SEVERE, "badproxy", proxyInfoFile);
            ex.printStackTrace();
        }
    }

    /**
     * Obtain the update check frequency based on the updatetool preferences.
     * <p>
     * If the user has opted out of update notifications, NEVER is returned. Otherwise 
     * the value for the update check frequence is returned. If there is no setting, 
     * the default value is WEEKLY.
     * 
     * @return an UpdateCheckFrequency
     */
    public static UpdateCheckFrequency getUpdateCheckFrequency() {
        UpdateCheckFrequency rv = UpdateCheckFrequency.WEEKLY;
        boolean optin = true;
        try {
            File dfile = getDefaultsFile();
            if (dfile.exists()) {
                // This is a poor man's implementation of a configuration file reader
                // It just looks for particular strings at the beginning of lines,
                // ignoring section boundaries.  This is sufficient to get the information.
                BufferedReader r = getFileReader(dfile);
                String line;
                String[] tokens;
                while ((line = r.readLine()) != null) {
                    if (line.startsWith("check_frequency")) {
                        tokens = line.split("\\s*=\\s*", 2);
                        if (tokens.length == 2) {
                            if (tokens[1].equals("daily")) {
                                rv = UpdateCheckFrequency.DAILY;
                            } else if (tokens[1].equals("weekly")) {
                                rv = UpdateCheckFrequency.WEEKLY;
                            } else if (tokens[1].equals("monthly")) {
                                rv = UpdateCheckFrequency.MONTHLY;
                            } else if (tokens[1].equals("never")) {
                                rv = UpdateCheckFrequency.NEVER;
                            }
                        }
                        break;
                    }
                }
                r.close();
            }

            // If the init file is there, and it has optin set to false, then use
            // that value since it is newer
            File ifile = getInitFile();
            if (ifile.exists()) {
                BufferedReader r = getFileReader(ifile);
                String line;
                String[] tokens;
                while ((line = r.readLine()) != null) {
                    if (line.startsWith("optin.update.notification")) {
                        tokens = line.split("\\s*:\\s*", 2);
                        if (tokens.length == 2 && !tokens[1].equals("true")) {
                            optin = false;
                        }
                        break;
                    }
                }
                r.close();
            }
        } catch (IOException ex) {
            Logger.getLogger(SystemInfo.class.getName()).log(Level.SEVERE, null, ex);
        }
        if (!optin) {
            return UpdateCheckFrequency.NEVER;
        }
        return rv;
    }

    /**
     * Initialize the updatetool preferences from properties.
     * <p>
     * This method passes preferences information to the updatetool program by 
     * writing values to an updatetool initialization configuration file. When
     * updatetool next runs, it merges the initialization information into its 
     * preferences file. 
     * <p>
     * The properties recognized from the props object are:
     * <ul>
     * <li>optin.update.notification</li>
     * <li>image.path</li>
     * <li>proxy.URL</li>
     * </ul>
     * 
     * If the value for a property is missing, no change is made for that property.
     * To make updatetool use no proxy, pass in the empty string ("") for the 
     * proxy.URL value. 
     * 
     * @param props - values to use for writing the file
     * @throws java.lang.Exception
     */
    public static void initUpdateToolProps(Properties props) throws Exception {
        // Write the props to the init.cfg file.
        try {
            String v;
            File f = getInitFile();

            // Check to see if the directory exists.  If it doesn't we create it.
            try {
                File configdir = f.getParentFile();
                if (!configdir.exists()) {
                    boolean success = configdir.mkdirs();
                    if (!success) {
                        throw new IOException("Unable to initialize properties: mkdirs() failed: " +
                                configdir);
                    }
                }

                if (!configdir.isDirectory()) {
                    throw new IOException("Unable to initialize properties: file exists but is not a directory: " +
                            configdir);
                }
            } catch (SecurityException e) {
                IOException ioe = new IOException("Unable to initialize updatetool preferences");
                ioe.initCause(e);
                throw ioe;
            }

            BufferedWriter out = getFileWriter(f);

            out.write("[main]" + eol);
            out.write("date: " + new Date().getTime() + eol);

            v = props.getProperty("optin.update.notification");
            if (v != null) {
                out.write("optin.update.notification: " + v + eol);
            }

            v = props.getProperty("image.path");
            if (v != null) {
                out.write("image_list: " +
                        new File(v.trim()).getAbsolutePath() + eol);
            }

            out.write("[network]" + eol);
            v = props.getProperty("proxy.use.system");
            if (v != null) {
                out.write("proxy.use.system: " + v.trim() + eol);
            }

            v = props.getProperty("proxy.URL");
            if (v != null) {
                if (v.trim().length() > 0) {
                    URL proxy = new URL(v.trim());
                    out.write("proxy.required: true" + eol);
                    out.write("proxy.host: " + proxy.getHost() + eol);
                    if (proxy.getPort() != -1) {
                        out.write("proxy.port: " + proxy.getPort() + eol);
                    }
                    String userinfo = proxy.getUserInfo();
                    if (userinfo != null && userinfo.length() != 0) {
                        String[] tokens = userinfo.split(":", 2);
                        String user = tokens[0];
                        String password = (tokens.length > 1) ? tokens[1] : null;
                        if (user != null) {
                            out.write("proxy.username: " + user + eol);
                        }
                        if (password != null) {
                            out.write("proxy.password: " + password + eol);
                        }
                        out.write("proxy.auth: true" + eol);
                    } else {
                        out.write("proxy.auth: false" + eol);
                    }
                } else {
                    out.write("proxy.required: false" + eol);
                }
            }
            out.close();
            // Set the permission to allow only the owner to write the file.
            // The file may contain a username and password.
            setPermissions(f, "0600");
        } catch (IOException e) {
            throw new Exception("Unable to initialize properties", e);
        }

    // XXX: When we support storing proxy auth info in the init file
    // XXX: we must change the perms on the init file to allow only
    // XXX: the user to read/write the values.
    }

    private static File getConfigDir() throws IOException {
        String configpath;

        // Compute the path to the config file
        String homedir = System.getProperty("user.home");
        String osname = System.getProperty("os.name");
        String username = System.getProperty("user.name");
        if (osname.startsWith("Windows")) {
            configpath = new String(homedir + File.separator +
                    WIN_APP_DATA + File.separator + APPNAME);
        } else if (osname.startsWith("Mac OS")) {

            if (username.equals("root")) {
                try {
                    homedir = getMacHomeDir();
                } catch (Exception e) {
                    throw new IOException("Unable to determine $HOME for: " +
                            username);
                }
            }
            configpath = new String(homedir + MAC_APP_DATA + APPNAME);
        } else {
            configpath = new String(homedir + File.separator + "." + APPNAME);
        }

        return new File(configpath);
    }

    /*
     * We need to determine the $HOME for the root user.  When Java is run
     * in a sudo environement on OS X the user.home does not represent root's
     * $HOME but rather the user who ran sudo.  So we need to look
     * up root's $HOME from /etc/passwd.
     */
    private static String getMacHomeDir() throws Exception {

        String homedir = null;
        String[] cmdarray = {"/usr/bin/awk", "-F:", "{ if ( $1 == \"root\" ) print $6 }", "/etc/passwd"};

        Runtime rt = Runtime.getRuntime();
        Process proc = rt.exec(cmdarray);

        int exitVal = proc.waitFor();

        // Read the output streams... (stdin)
        InputStream stdin = proc.getInputStream();
        InputStreamReader isr = new InputStreamReader(stdin);
        BufferedReader br = new BufferedReader(isr);
        String line = null;
        // There should be only one line.  
        // If there is more than one the last will have to do.
        while ((line = br.readLine()) != null) {
            homedir = line;
        }

        stdin.close();

        // Clear the output streams... (stderr)
        InputStream stderr = proc.getErrorStream();
        isr = new InputStreamReader(stderr);
        br = new BufferedReader(isr);
        while ((line = br.readLine()) != null) {
        } // We ignore the output.

        stderr.close();

        return homedir;
    }

    /**
     * This method takes a mode in IPS format (aka UNIX format) and sets the
     * permissions on the file as best as possible given the current platform
     * and Java version
     * @param file - the file to change
     * @param mode - the mode, e.g., 0644
     */  
    static private void setPermissions(File file, String mode) throws IOException {
        int m = Integer.parseInt(mode, 8);

        file = new File(file.getCanonicalPath());

        // First we try to use the methods from Java SE 6.
        // NOTE: this method ignores the group permission settings
        if (Action.haveJavaMethods) {
            try {
                boolean setOwnerExe = (m & 0100) != 0;
                boolean setOtherExe = (m & 0001) != 0;
                boolean setOwnerWrite = (m & 0200) != 0;
                boolean setOtherWrite = (m & 0002) != 0;
                boolean setOwnerRead = (m & 0400) != 0;
                boolean setOtherRead = (m & 0004) != 0;
                Action.setExecutable2.invoke(file, setOtherExe, false);
                Action.setExecutable.invoke(file, setOwnerExe);
                Action.setWritable2.invoke(file, setOtherWrite, false);
                Action.setWritable.invoke(file, setOwnerWrite);
                Action.setReadable2.invoke(file, setOtherRead, false);
                Action.setReadable.invoke(file, setOwnerRead);
                return;
            }
            catch (InvocationTargetException ite) {
            }
            catch (IllegalAccessException iae) {
            }
        }
      
        // Next, if we are on Unix, we try the chmod command.
        if (Action.isPosix) {
            if (file.isFile()) {
                String cmd[] = {"/bin/chmod", mode, file.getCanonicalPath() };
                try {
                    Process p = Runtime.getRuntime().exec(cmd);
                    p.waitFor();
                    if (p.exitValue() != 0) throw new Error("/bin/chmod failed");     
                }
                catch (InterruptedException ie) {}
                return;
            }
        }            
    }

    private static File getInitFile() throws IOException {
        return new File(getConfigDir(), INITFILE);
    }

    private static File getDefaultsFile() throws IOException {
        return new File(getConfigDir(), DEFAULTSFILE);
    }

    private static BufferedReader getFileReader(File f) throws IOException {
        return new BufferedReader(new InputStreamReader(new FileInputStream(f), "UTF-8"));
    }

    private static BufferedWriter getFileWriter(File f) throws IOException {
        return new BufferedWriter(new OutputStreamWriter(new FileOutputStream(f), "UTF-8"));
    }
}
