/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2008 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or updatetool/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 *
 * Contributor(s):
 *
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package com.sun.pkg.client;

import java.io.BufferedInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.HttpURLConnection;
import java.util.List;
import java.util.zip.GZIPInputStream;

/**
 * An <code>Action</code> is the base class for all actions that are contained within 
 * packages. 
 * 
 * @author trm
 */
abstract class Action implements Comparable<Action> {
    static boolean haveJavaMethods = false;
    static Method setExecutable;
    static Method setWriteable;
    static Method setReadable;
    static Method setExecutable2;
    static Method setWriteable2;
    static Method setReadable2;
    static protected boolean isPosix = false;
    static String posixNames[] = { "sunos", "linux", "mac os x", "solaris" };
    static {
        try {
            setExecutable = File.class.getMethod("setExecutable", boolean.class);
            setWriteable = File.class.getMethod("setWriteable",boolean.class);
            setReadable = File.class.getMethod("setReadable", boolean.class);
            setExecutable2 = File.class.getMethod("setExecutable", boolean.class, boolean.class);
            setWriteable2 = File.class.getMethod("setWriteable", boolean.class, boolean.class);
            setReadable2 = File.class.getMethod("setReadable", boolean.class, boolean.class);
            haveJavaMethods = true;
        }
        catch (NoSuchMethodException e) {
            // this will be thrown if the JVM is prior to Java SE 6. Fall through 
            // to leave haveJavaMethods false.
        }
        
        String os = System.getProperty("os.name").toLowerCase();
        for (int i = 0; i < posixNames.length; i++) {
            if (os.indexOf(posixNames[i]) != -1) {
                isPosix = true;
                break;
            }
        }
    }
    
    protected int key;
    
    public Action(int k) {
        key = k;
    }
    
    void preinstall(Action orig, FileList flist) { }
    
    abstract void install(Action orig) throws IOException;

    void postinstall(Action orig) { }
    
    void preremove() { }
    
    void remove() throws IOException { }
    
    void postremove() { }
    
    /**
     * Return the size of the installed bits for this action.
     * @return the size of the action
     */
    public int getSize() {
        return 0;
    }
    
    public int compareTo(Action a) {
        int diff = key - a.key;
        if (diff != 0) return diff;
        return keyValue().compareTo(a.keyValue());
    }
    
    /**
     * Determine if two Actions are the "same".
     * 
     * Actions are the same if they are the same type and if the key value for 
     * the actions are the same.  The actions may vary in other details.
     * @param a the Action to be compared
     * @return True if the actions are the same.
     */
    @Override
    public boolean equals(Object a) {
        return compareTo((Action)a) == 0;
    }
    
    @Override
    public int hashCode() {
        return keyValue().hashCode();
    }

    String keyValue() { return ""; }
    
    /**
     * Get the list of directories referenced by this action.
     * @return a List of Files representing referenced directories or null if
     * there aren't any
     */
    List<File> getReferencedDirectories() {
        return null;
    }
    
    /**
     * This method takes a mode in IPS format (aka UNIX format) and sets the 
     * permissions on the file as best as possible given the current platform
     * and Java version
     * @param file - the file to change
     * @param mode - the mode, e.g., 0644
     */
    void setPermissions(File file, String mode) throws IOException { 
        int m = Integer.parseInt(mode, 8);
        // First we try to use the methods from Java SE 6.  
        // NOTE: this method ignores the group permission settings
        if (haveJavaMethods) {
            try {
                boolean setOwnerExe = (m & 0100) != 0;
                boolean setOtherExe = (m & 0001) != 0;
                boolean setOwnerWrite = (m & 0200) != 0;
                boolean setOtherWrite = (m & 0002) != 0;
                boolean setOwnerRead = (m & 0400) != 0;
                boolean setOtherRead = (m & 0004) != 0;
                setExecutable2.invoke(file, setOtherExe, true);
                setExecutable.invoke(file, setOwnerExe);
                setWriteable2.invoke(file, setOtherWrite, true);
                setWriteable.invoke(file, setOwnerWrite);
                setReadable2.invoke(file, setOtherRead, true);
                setReadable.invoke(file, setOwnerRead);
                return;
            }
            catch (InvocationTargetException ite) {
            }
            catch (IllegalAccessException iae) {
            }
        }
        
        // Next, if we are on Unix, we try the chmod command, but only for 
        // changing a file (non-directory) to be executable.
        if (isPosix) {
            boolean setOwnerExe = (m & 0100) != 0;
            boolean setGroupExe = (m & 0010) != 0;
            boolean setOtherExe = (m & 0001) != 0;
            if (file.isFile() && (setOwnerExe || setGroupExe || setOtherExe)) {
                String cmd[] = {"/bin/chmod", mode, file.getCanonicalPath() };
                try {
                    Process p = Runtime.getRuntime().exec(cmd);
                    p.waitFor();
                    if (p.exitValue() != 0) throw new Error("/bin/chmod failed");
                }
                catch (InterruptedException ie) {}
                return;
            }
        }
        
        // At this point, the only thing left is to set the file as read-only 
        // (non-writeable), since files are created as read/write.
        boolean setOwnerWrite = (m & 0200) != 0;
        if (!setOwnerWrite) {
            file.setReadOnly();
        }
    }

    /**
     * Returns True if other represents a non-ignorable change from self.
     * 
     * By default, this means two actions are different if any of their
     * attributes are different.  Subclasses should override this
     * behavior when appropriate.         
     * @param a The Action to compare against.
     * @return True if the actions are different
     */
    abstract boolean isDifferent(Action a);

    /**
     * Get the (uncompressed) data for the action.
     * @return an InputStream that provides the data for the action
     */
    static InputStream getRemoteData(Image img, Fmri fmri, String hashval) throws IOException {
        HttpURLConnection urlc = img.getRepositoryURLConnection("file/0/" + hashval, fmri);
        img.checkRepositoryConnection(urlc);
        return new GZIPInputStream(new BufferedInputStream(urlc.getInputStream()));
    }
}
