/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2008 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or updatetool/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 *
 * Contributor(s):
 *
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package com.sun.pkg.client;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

/**
 * Basic constraint object; describes constraints on fmris and provides a 
 * method of computing the intersection of two constraints.
 * @author trm
 */
class Constraint {

    static class ConstraintException extends IllegalArgumentException {
        Constraint newConstraint;
        Constraint oldConstraint;
        ConstraintException(Constraint newc, Constraint oldc) {
            newConstraint = newc;
            oldConstraint = oldc;
        }
    }
    
    static class PresenceConflict extends ConstraintException {
        PresenceConflict(Constraint newc, Constraint oldc) {
            super(newc, oldc);
        }
        
        @Override
        public String getMessage() {
            return "Package presence is both required and prohibited:\n\t" +
                    newConstraint + "\n\t" + oldConstraint + "\n";
        }
    }

    static class VersionConflict extends ConstraintException {
        VersionConflict(Constraint newc, Constraint oldc) {
            super(newc, oldc);
        }
        
        @Override
        public String getMessage() {
            return "Package " + newConstraint.source + 
                   " contains constraint incompatible with constraint in installed package " +
                   oldConstraint.source + "\nproposed: " + newConstraint +
                   "\ninstalled: " + oldConstraint;
        }
    }
    
    static class DowngradeConflict extends ConstraintException {
        Fmri fmri;
        DowngradeConflict(Constraint c, Fmri f) {
            super(c, null);
            fmri = f;
        }
        
        @Override
        public String getMessage() {
            return "Package " + newConstraint.source + 
                   " contains constraint that requires downgrade of installed pkg " +
                   fmri + ": " + newConstraint;
        }
    }
    
    static class FmriConflict extends ConstraintException {
        Fmri fmri;
        FmriConflict(Constraint c, Fmri f) {
            super(c, null);
            fmri = f;
        }
        
        @Override
        public String getMessage() {
            return "Package " + fmri + " conflicts with constraint in installed pkg:/" +
                   newConstraint.source + ": " + newConstraint;
        }
    }
    
    static enum Presence { 
        ERROR (0, "ERROR"), 
        ALWAYS (1, "Required"),
        MAYBE (2, "Optional"), 
        NEVER (3, "Excluded");
        
        static final Presence compat[][] = {
            { ALWAYS, ALWAYS, ALWAYS },
            { ALWAYS, MAYBE, ALWAYS },
            { ALWAYS, NEVER, ERROR },
            { MAYBE, MAYBE, MAYBE },
            { MAYBE, NEVER, NEVER },
            { NEVER, NEVER, NEVER }
        };
        private final int index;
        private final String str;
        
        Presence(int index, String str) {
            this.index = index;
            this.str = str;
        }
        
        @Override
        public String toString() {
            return str;
        }

        private Presence combine(Presence p) {
            Presence left = this;
            Presence right = p;
            if (index > p.index) {
                left = p;
                right = this;
            }
            for (Presence pa[] : compat) {
                if (pa[0].equals(left) && pa[1].equals(right)) return pa[2];
            }
            throw new IllegalArgumentException("invalid Presence: this=" + this +
                    ", p=" + p);
        }
    }

    static class Set {
        class ConstraintEntry {
            Version v;
            List<String> pkglist;
        }
        Fmri activeFmri = null;
        HashMap<String, List<Constraint>> constraints = new HashMap<String, List<Constraint>>();
        // dict of version, constrained pkgs by pkg name
        HashMap<String, ConstraintEntry> loadedFmriVersions = new HashMap<String, ConstraintEntry>();

        Set() {}

        /*
         * Declare that we're done loading constraints from this fmri
         */
        void finishLoading(Fmri fmri) {
            if (activeFmri != fmri) {
                throw new IllegalArgumentException("finishing for wrong fmri (" +
                        activeFmri + " != " + fmri + ")");
            }
            activeFmri = null;
        }

        /*
         * Load a new set of constraints from fmri, deleting any constraints defined 
         * by previous versions of this fmri... skip if we're reloading the same
         * one by returning False, otherwise True.
         */
        boolean startLoading(Fmri fmri) {
            if (activeFmri != null) {
                throw new IllegalArgumentException("Already loading!");
            }
            activeFmri = fmri;
            String fmri_name = fmri.getName();
            ConstraintEntry ce = loadedFmriVersions.get(fmri_name);
            if (ce != null) {
                if (ce.v.equals(fmri.getVersion())) {
                    activeFmri = null;
                    return false; // already loaded this pkg once
                }
                // remove constraints established by previous version
                for (String p : ce.pkglist) {
                    List<Constraint> cl = constraints.get(p);
                    for (Constraint c : new ArrayList<Constraint>(cl)) {
                        if (c.source.equals(fmri_name)) cl.remove(c);
                    }
                }
            }
            else {
                ce = new ConstraintEntry();
            }
            ce.v = fmri.getVersion();
            ce.pkglist = new ArrayList<String>();
            loadedFmriVersions.put(fmri_name, ce);
            return true;
        }

        /*
         * Add a constraint from the active fmri to the set of system constraints
         */
        void updateConstraints(Constraint c) throws ConstraintException {
            String activeFmriName = activeFmri.getName();
            ConstraintEntry ce = loadedFmriVersions.get(activeFmriName);
            if (!activeFmriName.equals(c.source)) {
                throw new IllegalArgumentException("invalid constraint argument");
            }
            if (c.presence == Presence.ALWAYS) {
                // don't record these here for now
                return;
            }
            // find existing constraint list for this package 
            List<Constraint> cl = constraints.get(c.pkgName);
            if (cl != null) {
                /* check to make sure new constraint is
                 * compatible w/ existing constraints
                 * compatiblity is such that if
                 * A will combine w/ any item in list B,
                 * A will combine with combination of all of B
                 */
                for (Constraint cc : cl) cc.combine(c);
                cl.add(c);
            }
            else {
                cl = new ArrayList<Constraint>();
                cl.add(c);
                constraints.put(c.pkgName, cl);
            }
            if (!ce.pkglist.contains(c.pkgName)) {
                ce.pkglist.add(c.pkgName);
            }
        }

        /*
         * If constraints exist for this package, apply them.  Apply the new one 
         * last so that exception contains proper error message... error message
         * generation will be unclear if multiple constraints exist
         */
        Constraint applyConstraints(Constraint c) throws ConstraintException {
            List<Constraint> cl = constraints.get(c.pkgName);
            if (cl != null) {
                Iterator<Constraint> ic = cl.iterator();
                Constraint ca = ic.next();
                while (ic.hasNext()) {
                    ca = ca.combine(ic.next());
                }
                return ca.combine(c);
            }
            return null;
        }

        /*
         * Treats fmri as min required version; apply any constraints and if fmri 
         * is more specific, return original fmri, otherwise return more constrained
         * version... remap exception for better error handling
         */
        Fmri applyConstraintsToFmri(Fmri fmri) throws ConstraintException {
            Constraint ic = Constraint.makeRequiredConstraint(fmri, "");
            Constraint nc;
            try {
                nc = applyConstraints(ic);
            }
            catch (ConstraintException ce) {
                throw new FmriConflict(ce.oldConstraint, fmri);
            }
            if (nc == null || ic.equals(nc)) return fmri;
            Fmri nfmri = fmri.clone();
            nfmri.version = nc.minVer;
            return nfmri;
        }
    }

    String pkgName;
    Version minVer;
    Version maxVer;
    Presence presence;
    String source;
    
    Constraint(String pkgName, Version minVer, Version maxVer, Presence presence, String source) {
        if (pkgName == null || minVer == null || presence == null || source == null) {
            throw new IllegalArgumentException("invalid Constraint args");
        }
        this.pkgName = pkgName;
        this.minVer = minVer;
        this.maxVer = maxVer;
        this.presence = presence;
        this.source = source;
                
    }

    @Override
    public String toString() {
        return "Pkg " + pkgName + ": " + presence + " min_version: " + minVer 
                + " max_version: " + maxVer + " defined by: pkg:/" + source;
    }

    @Override
    public boolean equals(Object o) {
        Constraint v = (Constraint)o;
        return pkgName.equals(v.pkgName) &&
                presence.equals(v.presence) &&
                minVer.equals(v.minVer) && 
                ((maxVer == null && v.maxVer == null) || 
                 (maxVer != null && v.maxVer != null && maxVer.equals(v.maxVer)));
    }

    @Override
    public int hashCode() {
        int hash = 7;
        hash = 59 * hash + (this.pkgName != null ? this.pkgName.hashCode() : 0);
        hash = 59 * hash + (this.minVer != null ? this.minVer.hashCode() : 0);
        hash = 59 * hash + (this.maxVer != null ? this.maxVer.hashCode() : 0);
        hash = 59 * hash + (this.presence != null ? this.presence.hashCode() : 0);
        return hash;
    }

    static Constraint makeRequiredConstraint(Fmri fmri, String src) {
        return new Constraint(fmri.getName(), fmri.getVersion(), null, 
                Presence.ALWAYS, src);
    }

    static Constraint makeOptionalConstraint(Fmri fmri, String src) {
        return new Constraint(fmri.getName(), fmri.getVersion(), null, 
                Presence.MAYBE, src);
    }

    static Constraint makeIncorporationConstraint(Fmri fmri, String src) {
        return new Constraint(fmri.getName(), fmri.getVersion(), fmri.getVersion(), 
                Presence.MAYBE, src);
    }

    /*
     * Evaluate work needed to meet new constraint if fmriPresent
     * is the fmri installed (null if none is installed).  Returns
     * null if no work to do, otherwise version to be installed.
     * Raises ConstraintException in case of uninstall or downgrade
     * required.
     */
    Version checkForWork(Fmri fmriPresent) throws ConstraintException {
        if (fmriPresent == null) {
            if (presence == Presence.MAYBE || presence == Presence.NEVER) return null;
            return minVer;
        }
        if (presence == Presence.NEVER) {
            throw new PresenceConflict(this, makeRequiredConstraint(fmriPresent, ""));
        }
        Version vp = fmriPresent.getVersion();
        if (vp.compareTo(minVer) < 0) return minVer;
        if (maxVer != null && vp.compareTo(maxVer) > 0 && !vp.isSuccessor(maxVer)) {
            throw new DowngradeConflict(this, fmriPresent);
        }
        return null;
    }

    Constraint combine(Constraint proposed) throws ConstraintException {
        if (!pkgName.equals(proposed.pkgName)) {
            throw new IllegalArgumentException("constraint package names do not match");
        }
        Presence p = presence.combine(proposed.presence);
        if (p == Presence.ERROR) {
            throw new PresenceConflict(proposed, this);
        }
        Version maxv = null;
        if (maxVer == null) maxv = proposed.maxVer;
        else if (proposed.maxVer == null) maxv = maxVer;
        else if (maxVer.compareTo(proposed.maxVer) < 0) maxv = maxVer;
        else maxv = proposed.maxVer;
        
        Version minv = minVer.compareTo(proposed.minVer) < 0 ?
            proposed.minVer : minVer;
        
        if (maxv != null && maxv.compareTo(minv) < 0 &&
            !minv.isSuccessor(maxv)) {
            throw new VersionConflict(proposed, this);
        }
        return new Constraint(pkgName, minv, maxv, p, source);
    }
}
