/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2008 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or updatetool/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 *
 * Contributor(s):
 *
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package com.sun.pkg.client;

import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.StringTokenizer;

/** 
 * A <code>Version</code> represents the version part of an IPS package FMRI.
 * 
 * The version format is release[,build_release]-branch:publishtime, which is
 * decomposed into three DotSequences and the publish time.  The text
 * representation of the publish time is in the ISO8601-compliant form "YYYYMMDDTHHMMSSZ",
 * referring to the UTC time associated with the version.  The release and
 * branch DotSequences are interpreted normally, where v1 &lt; v2 implies that
 * v2 is a later release or branch.  The build_release DotSequence records
 * the system on which the package binaries were constructed.
 * Interpretation of the build_release by the client is that, in the case
 * b1 &lt; b2, a b1 package can be run on either b1 or b2 systems, while a b2
 * package can only be run on a b2 system.
 */
public class Version implements Comparable<Version> {
    
    /**
     * The <code>DotSequence</code> class represents the x.y.z parts of a Version object.
     */
    static public class DotSequence implements Comparable<DotSequence> {
        int values[];
        int hc = 0;
        
        DotSequence(String s) {
            if (s.length() == 0) {
                values = new int[0];
                return;
            }
            StringTokenizer st = new StringTokenizer(s, ".");
            values = new int[st.countTokens()];
            for (int i = 0; i < values.length; i++)
            {
                values[i] = Integer.parseInt(st.nextToken());
                hc += values[i];
            }
        }
        
        /**
         * Compare two DotSequences for equality.
         * @return True if the two DotSequences are equal
         */
        @Override
        public boolean equals(Object v) {
            return compareTo((DotSequence)v) == 0;
        }

        @Override
        public int hashCode() {
            return hc;
        }
        
        public int compareTo(DotSequence d) {
            int i;
            for (i = 0; i < values.length && i < d.values.length; i++) 
            {
                if (values[i] < d.values[i]) return -1;
                if (values[i] > d.values[i]) return 1;
            }
            if (values.length == d.values.length) return 0;
            if (i < values.length) {
                for (; i < values.length; i++)
                {
                    if (values[i] > 0) return 1;
                }
                return 0;
            }
            for (; i < d.values.length; i++)
            {
                if (d.values[i] > 0) return -1;
            }
            return 0;
        }
        
        public boolean matches(DotSequence d) {
            for (int i = 0; i < values.length && i < d.values.length; i++)
            {
                if (values[i] != d.values[i]) return false;
            }
            if (values.length < d.values.length) return false;
            return true;
        }
        
        /**
         * Determine if this DotSequence is a subsequence of another.
         * 
         * A Dotsequence is a subsequence of another if the other and this one 
         * have identical components, up to the length of this sequence.
         * @param d the Dotsequence to compare
         * @return true if this DotSequence is a subsequence of d
         */
        public boolean isSubsequence(DotSequence d) {
            if (values.length > d.values.length) return false;
            
            for (int i = 0; i < values.length; i++) {
                if (values[i] != d.values[i]) return false;
            }
            return true;
        }
        
        /**
         * Returns a string form of the DotSequence.
         * @return a string representing the DotSequence.  This are the numbers of 
         * the Dotsequence separated by periods.
         */
        @Override
        public String toString() {
            if (values.length == 0) return "";
            StringBuffer buf = new StringBuffer();
            buf.append(values[0]);
            for (int i = 1; i < values.length; i++)
            {
                buf.append(".");
                buf.append(values[i]);
            }
            return buf.toString();
        }
    }
    
    static final DotSequence zero = new DotSequence("");
    static final Date zerotime = new Date(0);
    static final Version nullVersion = new Version("");
    DotSequence release = zero;
    DotSequence build = zero;
    DotSequence branch = zero;
    String versionstr = "";
    Date timestamp = zerotime;
    static DateFormat df = new SimpleDateFormat("yyyyMMdd'T'HHmmss'Z'");
    
    /**
     * Create a <code>Version</code> based on the given string representation.
     * @param s the string representation for the version
     */
    public Version(String s) {
        versionstr = s;
        int timeidx = s.indexOf(":");
        String timestr = "";
        if (timeidx != -1) {
            timestr = s.substring(timeidx + 1);
        }
        else timeidx = s.length();

        int branchidx = s.indexOf("-");
        String branchstr = "";
        if (branchidx != -1) {
            branchstr = s.substring(branchidx + 1, timeidx);
        }
        else branchidx = timeidx;
        
        int buildidx = s.indexOf(",");
        String buildstr = "";
        if (buildidx != -1) {
            buildstr = s.substring(buildidx + 1, branchidx);
        }
        else buildidx = branchidx;
        
        String releasestr = s.substring(0, buildidx);

        if (releasestr.length() > 0) release = new DotSequence(releasestr);
        if (buildstr.length() > 0) build = new DotSequence(buildstr);
        if (branchstr.length() > 0) branch = new DotSequence(branchstr);

        if (timestr.length() > 0) {
            if (timestr.endsWith("Z") && timestr.indexOf("T") != -1) {
                try {
                    timestamp = df.parse(timestr);
                } catch (ParseException ex) {
                    throw new IllegalArgumentException("Invalid timestamp in version: " + timestr,
                            ex);
                }
            }
            else {
                throw new IllegalArgumentException("Unrecognized timestamp in version: " + timestr);
            }
        }
    }
    
    /**
     * Determine if a version matches this version
     * 
     * A version v matches this version if it is all of the version information
     * that is specified matches the corresponding version information in this
     * version. For example version 1 matches 1.1 and 1.2 but not 2.1.
     *
     * @param v the version to compare
     * @return true if version v matches this version
     */
    public boolean matches(Version v) {
        boolean q = release.matches(v.release) &&
                build.matches(v.build) &&
                branch.matches(v.branch);
        if (v.timestamp == zerotime) return q;
        if (timestamp.equals(v.timestamp)) return q;
        return false;
    }
    /**
     * Compare two versions for equality.
     * @return True if the two versions are equal
     */
    @Override
    public boolean equals(Object v) {
        return compareTo((Version)v) == 0;
    }

    /**
     * Returns a hash code value for the object. 
     * 
     * See the description of Object.hashcode for more details.
     * @return a hash code value for this object.
     */
    @Override
    public int hashCode() {
        int hash = 7;
        hash = 37 * hash + (this.release != null ? this.release.hashCode() : 0);
        hash = 37 * hash + (this.branch != null ? this.branch.hashCode() : 0);
        hash = 37 * hash + (this.timestamp != null ? this.timestamp.hashCode() : 0);
        return hash;
    }
    
    /** 
     * Compare two versions.
     * @param v
     * @return -1 if this version is less than v, 0 if they are equal, 1 if this 
     * version is greater than v
     */
    public int compareTo(Version v) {
        int i = release.compareTo(v.release);
        if (i != 0) return i;
        i = branch.compareTo(v.branch);
        if (i != 0) return i;
        return timestamp.compareTo(v.timestamp);
    }
    
    /**
     * Determine if a version is a successor to this version.
     * 
     * The definition of "successor" here is that of the CONSTRAINT_AUTO that is 
     * implemented in the pkg(5) version module. See the Python code in version.py
     * for details.
     * 
     */
    public boolean isSuccessor(Version v) {
        boolean release_match = false;
        boolean branch_match = false;
        boolean date_match = false;
        
        if (v.release != zero && release != zero) {
            if (v.release.isSubsequence(release)) release_match = true;
        } else if (v.release == zero) release_match = true;
        
        if (v.branch != zero && branch != zero) {
            if (v.branch.isSubsequence(branch)) branch_match = true;
        } else if (v.branch == zero) branch_match = true;
        
        if (v.timestamp != zerotime && timestamp != zerotime) {
            if (v.timestamp.equals(timestamp)) date_match = true;
        } else if (v.timestamp == zerotime) date_match = true;
        
        return release_match && branch_match && date_match;
    }
    
    /**
     * Returns a string representation for the <code>Version</code>.
     * @return a string representation for the object.
     */
    @Override
    public String toString() {
        return versionstr;
    }
    
    /**
     * Tests whether this object is equal to the null or empty Version, 
     * i.e., Version("")
     * @return true if this is the nullVersion
     */
    public boolean isNull() {
        return equals(nullVersion);
    }
    
    /**
     * Returns the release part of the <code>Version</code>.
     * 
     * @return the release part of the object.
     */
    public DotSequence getRelease() {
        return release;
    }

    /**
     * Returns the branch part of the <code>Version</code>.
     * 
     * @return the branch part of the object.
     */
    public DotSequence getBranch() {
        return branch;
    }

    /**
     * Returns the publish time part of the <code>Version</code>.
     * 
     * If no publish time was provided when the object was created, the publish
     * time returned is a Date created with 0.
     * 
     * @return the publish time part of the object.
     */
    public Date getPublishDate() {
        return timestamp;
    }
}
