/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2008 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or updatetool/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 *
 * Contributor(s):
 *
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.sun.pkg.client;

import com.sun.pkg.util.*;
import java.net.InetSocketAddress;
import java.net.Proxy;
import java.net.SocketAddress;
import java.net.URL;
import java.net.URLConnection;
import java.net.HttpURLConnection;
import java.io.BufferedReader;
import java.io.File;
import java.io.InputStreamReader;
import java.net.Authenticator;
import java.net.PasswordAuthentication;
import javax.net.ssl.HttpsURLConnection;

/**
 * An AuthProxy is a Proxy that supports user authentication
 */
class AuthProxy extends Proxy {

    String  user = null;
    String  password = null;

    AuthProxy(Proxy.Type type, SocketAddress sa) {
        super(type, sa);
    }

    /**
     * Create a Proxy with user authentication information.
     * username and password may be null if no user authentication.
     */
    AuthProxy(Proxy.Type type, SocketAddress sa, String username, String passwd) {

        super(type, sa);

        this.user     = username;
        this.password = passwd;

        if (user != null) {
            /* If we have user authentication info set the Authenticator */
            Authenticator.setDefault(this.new MyAuthenticator());
        }
    }

    /**
     * Factory method for creating a AuthProxy from a URL. The URL
     * may include proxy authentication user information. For 
     * example: "http://user:passwd@webcache:8080"
     */
    private static AuthProxy newProxy(URL proxyURL) {

        String pHost = proxyURL.getHost();
        int    pPort = proxyURL.getPort();
        String userInfo = proxyURL.getUserInfo();
        String user     = null;
        String password = null;

        if (userInfo != null && userInfo.length() != 0) {
            String[] tokens = userInfo.split(":", 2);
            user = tokens[0];
            password = tokens[1];
        }

        SocketAddress addr = new InetSocketAddress(pHost, pPort);

        return new AuthProxy(Proxy.Type.HTTP, addr, user, password);
    }

    @Override
    public String toString() {
        String s = "";
        InetSocketAddress ia = (InetSocketAddress)this.address();

        switch (this.type()) {
            case DIRECT:
                s = "Direct";
                break;
            case HTTP:
                s = "http://";
                break;
            case SOCKS:
                s = "socks:";
                break;
            default:
                s = "";
                break;
        }

        if (user == null) {
            return s + ia.getHostName() + ":" + ia.getPort();
        } else {
            StringBuffer sb = new StringBuffer("");
            if (password != null) {
                for (int n = password.length(); n > 0; n--) {
                    sb.append('*');
                }
            } 
            return s + user + ":" + sb + "@" + ia.getHostName() + ":" +
                                        ia.getPort();
        }
    }

    /**
     * Handle proxy authentication
     */
    private class MyAuthenticator extends Authenticator {

        @Override
        public PasswordAuthentication getPasswordAuthentication() {

            if (getRequestorType() != Authenticator.RequestorType.PROXY) {
                return null;
            }

            String protocol = getRequestingProtocol();
            if (! protocol.equalsIgnoreCase("http")) {
                return null;
            }

            if (user == null) {
                return null;
            }

            return new PasswordAuthentication(user,
                        (password == null ? null : password.toCharArray()));
        }
    }

    public static void main (String args[]) throws Exception {

        Proxy proxy = null;
        
        // Get http_proxy environment variable setting
        String pString = System.getenv("http_proxy");
        if (pString == null) {
            pString = System.getenv("HTTP_PROXY");
        }

        if (pString != null) {
            try {
                URL url = new URL(pString);
                proxy = AuthProxy.newProxy(url);
            } catch (Exception e) {
                System.out.println(e);
                System.exit(1);
            }
        } 

        if (args.length == 0) {
            System.exit(0);
        }
       
        URL url = new URL(args[0]);
        URLConnection urlc = null;
        if (proxy == null) {
            urlc = url.openConnection();
        } else {
            System.out.println("Proxy: " + proxy.toString());
            urlc = url.openConnection(proxy);
        }
        if (urlc instanceof HttpsURLConnection) {
            HttpsURLConnection sslurlc = (HttpsURLConnection)urlc;
            sslurlc.setSSLSocketFactory(PEMUtil.getSSLSocketFactory(
                    new File("certificate.pem"), 
                    new File("key.pem")));
        }
        HttpURLConnection hurlc = (HttpURLConnection)urlc;
        hurlc.setRequestProperty("User-Agent", Image.userAgent);

        BufferedReader br = new BufferedReader(new InputStreamReader(hurlc.getInputStream()));
        String line = null;
        while ((line = br.readLine()) != null) {
            System.out.println(line);
        }
    }
}
