/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2008 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or updatetool/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 *
 * Contributor(s):
 *
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */

package com.sun.pkg.client;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.FileWriter;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.util.List;
import java.util.ArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * A <code>Catalog</code> is the list of packages (Fmris) that are available 
 * from an authority.
 * 
 * @author trm
 */
class Catalog {
    Image image;
    String authname;
    File cfile;
    File afile;
    List<Fmri> pkg_fmris = new ArrayList<Fmri>();

    public Catalog(Image img, String authname) throws IOException {
        image = img;
        this.authname = authname;
        File catroot = new File(getCatalogDir(), authname);
        cfile = new File(catroot, "catalog");
        afile = new File(catroot, "attrs");
        if (cfile.exists()) {
            load();
        }
    }
    
    /** 
     * Load the catalog from the local meta data.
     * 
     * @throws java.io.IOException
     */
    private void load() throws IOException {
        BufferedReader r = new BufferedReader(new FileReader(cfile));
        String line;
        String tokens[];
        pkg_fmris.clear();
        while ((line = r.readLine()) != null) {
            tokens = line.split(" ");
            if (tokens.length == 4 && tokens[0].equals("V")) {
                try {
                    Fmri p = new Fmri(tokens[2], tokens[3]);
                    pkg_fmris.add(p);
                } catch (Exception ex) {
                    Logger.getLogger(Catalog.class.getName()).log(Level.SEVERE, null, ex);
                }
            }
        }
        r.close();
    }

    /**
     * Refresh the local content of the catalog by downloading it from the repository.
     * 
     * @throws java.io.IOException
     */
    public void refresh() throws IOException {
        HttpURLConnection urlc = image.getRepositoryURLConnection("catalog/0", authname);
        image.checkRepositoryConnection(urlc);
        byte buf[] = new byte[4096];
        BufferedReader ir = new BufferedReader(new InputStreamReader(urlc.getInputStream()));
        new File(cfile.getParent()).mkdirs();
        BufferedWriter cw = new BufferedWriter(new FileWriter(cfile));
        BufferedWriter aw = new BufferedWriter(new FileWriter(afile));
        String line;
        while ((line = ir.readLine()) != null) 
        {
            if (!Character.isWhitespace(line.charAt(1))) 
            {
                continue;
            }
            if ("CSVR".indexOf(line.charAt(0)) == -1 ||
                    line.startsWith("R ")) {
                cw.write(line);
                cw.newLine();
            }
            else if (line.startsWith("S ")) {
                aw.write(line);
                aw.newLine();
            }
            else {
                // XXX Need to be able to handle old and new format catalogs.
                Fmri f = new Fmri(line.substring(2));
                cw.write(line.substring(0, 1) + " pkg " + f.name + " " + f.version);
                cw.newLine();
            }
        }
        ir.close();
        cw.close();
        aw.close();
        load();
        
        // Remove the catalog cache that the python API uses.  Eventually, 
        // the python API should detect if any catalog is more recent than the
        // cache. TODO - file IPS issue on that.
        new File(getCatalogDir(), "catalog.pkl").delete();
    }

    /**
     * Get the size of the catalog, i.e., the number of FMRIs contained in it.
     * 
     * @return the number of packages in this catalog
     */
    public int size() {
        return pkg_fmris.size();
    }
    
    /**
     * Get the Fmris that are int his catalog.
     * 
     * @return List of Fmris in this catalog
     */
    public List<Fmri> getFmris() {
        return pkg_fmris;
    }
    
    File getCatalogDir() {
        return new File(image.getMetaDirectory(), "catalog");
    }
}
