/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2008 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or updatetool/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 *
 * Contributor(s):
 *
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.sun.pkg.client;

import com.sun.pkg.client.Constraint.ConstraintException;
import com.sun.pkg.util.PEMUtil;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.HttpURLConnection;
import java.net.Proxy;
import java.net.ProxySelector;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLDecoder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSocketFactory;


/**
 * An <code>Image</code> is a directory tree containing the laid-down contents
 * of a self-consistent graph of Packages.
 * <p>
 * An Image has a root path.
 * 
 * @author trm
 */
public class Image
{
    static class Authority 
    {
        String prefix;
        String origin;
        // List<String> mirrors; // not currently handled
        String uuid = "None";
        File certFile;
        File keyFile;
        SSLSocketFactory sslfactory = null; // filled in as needed
        
        Authority(String name)
        {
            prefix = name;
        }
    }
    
    /**
     * An <code>FmriState</code> represents the current state of an Fmri within
     * the image.
     */
    public static class FmriState 
    {
        public Fmri fmri;
        public boolean installed = false;
        public boolean upgradable = false;
        public boolean frozen = false;
        public boolean incorporated = false;
        public boolean excludes = false;
    }

    enum PlanState { UNEVALUATED, EVALUATED_OK, EVALUATED_ERROR, 
                     EXECUTED_OK, EXECUTED_ERROR }
    
    /* not used */
    static class Filter {}

    class CatalogCache
    {
        class SubCacheEntry {
            Fmri fmri;
            List<Authority> auths = new ArrayList<Authority>();
            SubCacheEntry(Fmri f, Authority a) {
                fmri = f;
                auths.add(a);
            }
        }
        class CacheEntry {
            SortedSet<Version> versions = new TreeSet<Version>();
            HashMap<Version, SubCacheEntry> versionInfo = new HashMap<Version, SubCacheEntry>();
            CacheEntry(Fmri f, Authority a) {
                Version v = f.getVersion();
                versions.add(v);
                versionInfo.put(v, new SubCacheEntry(f, a));
            }
        }
        HashMap<String, CacheEntry> pkgs = new HashMap<String, CacheEntry>();
        
        void cacheInstalledPackages() throws IOException {
            for (Fmri f : getInstalledFmrisFromImage()) {
                Authority a = getAuthorityForInstalledPkg(f);
                if (a == null) throw new RuntimeException("can't determine authority for installed package: " + f);
                cacheFmri(f, a);
            }
        }
        
        void addCatalog(Catalog c, Authority a) {
            for (Fmri f : c.getFmris()) {
                cacheFmri(f, a);
            }
        }
        
        private void cacheFmri(Fmri f, Authority a) {
            String pname = f.getName();
            Version v = f.getVersion();               
            CacheEntry ce = pkgs.get(pname);
            if (ce == null) {
                pkgs.put(pname, new CacheEntry(f, a));
            } else {
                SubCacheEntry sce = ce.versionInfo.get(v);
                if (sce == null) {
                    ce.versionInfo.put(v, new SubCacheEntry(f, a));
                    ce.versions.add(v);
                } else {
                    if (!sce.auths.contains(a)) sce.auths.add(a);
                }
            }              
            
        }
        
        List<Fmri> multimatch(String pname, Fmri pkgFmris[]) {
            List<Fmri> matches = new ArrayList<Fmri>();
            pname = "/" + pname;
            for (Fmri f : pkgFmris) {
                if (pname.endsWith("/" + f.getName())) matches.add(f);
            }
            return matches;
        }
        
        List<FmriState> getInventory(String patternNames[], boolean allKnown) {
            if (patternNames == null) patternNames = new String[0];
            Fmri patternFmris[] = new Fmri[patternNames.length];
            for (int i = 0; i < patternNames.length; i++) {
                patternFmris[i] = new Fmri(patternNames[i]);
            }
            List<FmriState> results = new ArrayList<FmriState>();
            String pauth = getPreferredAuthorityName();
            for (String name : pkgs.keySet()) {
                List<Fmri> matches = multimatch(name, patternFmris);
                if (patternFmris.length > 0 && matches.size() == 0) continue;
                Version newest = pkgs.get(name).versions.last();
                Version vlist[] = pkgs.get(name).versions.toArray(new Version[0]);
                HashMap<Version, SubCacheEntry> vinfo = pkgs.get(name).versionInfo;
                // traverse through the version list in reverse order (newest to oldest)
                for (int i = vlist.length - 1; i >= 0; i--) {
                    List<Fmri> vmatches = new ArrayList<Fmri>();
                    // If a pattern specified a version and that version
                    // isn't succeeeded by vlist[i], then remove it from 
                    // consideration
                    for (Fmri m : matches) {
                        Version mv = m.getVersion();
                        if (mv.isNull() || vlist[i].isSuccessor(mv)) {
                            vmatches.add(m);
                        }
                    }
                    // If no contenders remain and we had some to begin with,
                    // go to the next version
                    if (matches.size() > 0 && vmatches.size() == 0) continue;
                    
                    // Now do the same thing with authories
                    List<Authority> authlist = vinfo.get(vlist[i]).auths;
                    List<Fmri> amatches = new ArrayList<Fmri>();
                    for (Fmri m : vmatches) {
                        String mauth = m.getAuthority();
                        if (mauth == null || authlist.contains(authorities.get(mauth))) {
                            amatches.add(m);
                        }
                    }
                    if (vmatches.size() > 0 && amatches.size() == 0) continue;
                    
                    // If no patterns were specified or any still-
                    // matching pattern specified no authority, we
                    // use the entire authlist for this version.
                    // Otherwise, we use the intersection of authlist
                    // and the auths in the patterns.
                    List<Authority> alist = null;
                    if (amatches.size() == 0) alist = authlist;
                    else {
                        for (Fmri m : amatches) {
                            if (m.getAuthority() == null) {
                                alist = authlist;
                                break;
                            }
                        }
                        if (alist == null) {
                            alist = new ArrayList<Authority>();
                            for (Authority a : authlist) {
                                for (Fmri m : amatches) {
                                    if (a.prefix.equals(m.getAuthority())) {
                                        alist.add(a);
                                    }
                                }
                            }
                        }
                    }
                    
                    // Now we have an Fmri (vinfo.get(vlist[i]).fmri) and a list of 
                    // authorities (alist) that it comes from.
                    // Add them to the results.
                    Fmri vfmri = vinfo.get(vlist[i]).fmri;
                    for (Authority a : alist) {
                        FmriState fs = new FmriState();
                        fs.fmri = vfmri.clone();
                        fs.fmri.setAuthority(a.prefix);
                        fs.upgradable = !vlist[i].equals(newest);
                        fs.installed = isInstalled(fs.fmri);
                        if (!allKnown && !fs.installed) continue;
                        results.add(fs);
                    }
                }
            }
            return results;       
        }
        
        /**
         * Get the first package matching pattern, taking from the preferred
         * repository if no repository is specified in the pattern.
         * 
         * @return the first matching Fmri
         **/
        Fmri getFirst(String pattern) {
            String plist[] = { pattern };
            List<FmriState> lfs = getInventory(plist, true);
            String pauth = getPreferredAuthorityName();
            Fmri firstNonPref = null;
            for (FmriState fs : lfs) {
                if (fs.fmri.getAuthority().equals(pauth)) return fs.fmri;
                if (firstNonPref == null) firstNonPref = fs.fmri;
            }
            return firstNonPref;
        }
    }
    
    /**
     * An <code>ImagePlan</code> takes a list of requested packages, an Image (and its
     * policy restrictions), and returns the set of package operations needed
     * to transform the Image to the list of requested packages.
     * <p>
     * Use of an ImagePlan involves the identification of the Image, the
     * Catalogs (implicitly), and a set of complete or partial package FMRIs.
     * The Image's policy, which is derived from its type and configuration
     * will cause the formulation of the plan or an exception state.
     * <p>
     * XXX In the current formulation, an ImagePlan can handle [null ->
     * PkgFmri] and [PkgFmri@Version1 -> PkgFmri@Version2], for a set of
     * PkgFmri objects.  With a correct Action object definition, deletion
     * should be able to be represented as [PkgFmri@V1 -> null].
     * <p>
     * XXX Should we allow downgrades?  There's an "arrow of time" associated
     * with the smf(5) configuration method, so it's better to direct
     * manipulators to snapshot-based rollback, but if people are going to do
     * "pkg delete fmri; pkg install fmri@v(n - 1)", then we'd better have a
     * plan to identify when this operation is safe or unsafe.
     */
    public class ImagePlan
    {
        private PlanState state;
        private boolean recursiveRemoval;
        private List<Fmri> targetFmris = new ArrayList<Fmri>();
        private List<Fmri> targetRemFmris = new ArrayList<Fmri>();
        private List<PackagePlan> pkgPlans = new ArrayList<PackagePlan>();
        private List<File> directories = null;
        
        ImagePlan(boolean recursiveRemoval, Filter filters[]) {
            state = PlanState.UNEVALUATED;
            this.recursiveRemoval = recursiveRemoval;

            // XXX add filter support (join image filters with filters arg)
        }
        
        /**
         * Generate a string representation for the ImagePlan.
         * @return a String representing the ImagePlan
         */
        @Override
        public String toString() {
            StringBuffer s = new StringBuffer();
            if (state == PlanState.UNEVALUATED) {
                s.append("UNEVALUTED:\n");
                for (Fmri t : targetFmris) s.append("+" + t + "\n");
                for (Fmri t : targetRemFmris) s.append("-" + t + "\n");
            } else {
                for (PackagePlan pp : pkgPlans) s.append(pp + "\n");
            }
            return s.toString();
        }
        
        /**
         * Return the Fmris that are proposed for installation as part of 
         * this plan.
         * 
         * @return the list of proposed Fmris
         */
        public Fmri[] getProposedFmris() {
            return targetFmris.toArray(new Fmri[targetFmris.size()]);
        }
        
        boolean isProposedFmri(Fmri fmri) {
            for (Fmri pf : targetFmris) {
                if (fmriIsSamePackage(fmri, pf)) {
                    return !fmriIsSuccessor(fmri, pf);
                }
            }
            return false;
        }

        boolean isProposedRemFmri(Fmri fmri) {
            for (Fmri pf : targetRemFmris) {
                if (fmriIsSamePackage(fmri, pf)) return true;
            }
            return false;
        }
        
        /*
         * Return version of fmri already proposed, or null if not proposed yet.
         */
        Fmri getProposedVersion(Fmri fmri)
        {
            for (Fmri p : targetFmris) {
                if (fmri.getName().equals(p.getName())) return p;
            }
            return null;
        }

        void proposeFmri(Fmri fmri) throws IOException, ConstraintException {
            if (hasVersionInstalled(fmri)) return;
            fmri = constraints.applyConstraintsToFmri(fmri);
            setFmriDefaultAuthority(fmri);
            // Add fmri to target list only if it (or a successor) isn't
            // there already.
            for (Fmri p : targetFmris) {
                if (fmriIsSuccessor(fmri, p)) {
                    targetFmris.remove(p);
                    break;
                }
                if (fmriIsSuccessor(p, fmri)) return;
            }
            targetFmris.add(fmri);
        }

        boolean olderVersionProposed(Fmri fmri) {
            // returns true if older version of this fmri has been proposed already
            for (Fmri p : targetFmris) {
                if (fmriIsSuccessor(fmri, p)) return true;
            }
            return false;
        }
        
        // XXX Need to make sure that the same package isn't being added and
        // removed in the same imageplan.
        void proposeFmriRemoval(Fmri fmri) throws IOException {
            if (!hasVersionInstalled(fmri)) return;
            for (Fmri p : targetRemFmris) {
                if (fmriIsSuccessor(fmri, p)) {
                    targetRemFmris.remove(p);
                    break;
                }
            }
            targetRemFmris.add(fmri);
        }

        List<Fmri> genNewInstalledPackages() {
            if (state != PlanState.EVALUATED_OK) {
                throw new RuntimeException("invalid image plan state");
            }
            List<Fmri> fmriset = getInstalledFmrisFromCache();
            for (PackagePlan p : pkgPlans) {
                p.updatePackageSet(fmriset);
            }
            return fmriset;
        }

        List<File> getDirectories() throws IOException {
            // return list of all directories in target image
            if (directories == null) {
                directories = new ArrayList<File>();
                directories.add(new File("var/pkg"));
                directories.add(new File("var/sadm/install"));
                for (Fmri fmri : genNewInstalledPackages()) {
                    for (Action act : getManifest(fmri).getActions()) {
                        List<File> dl = act.getReferencedDirectories();
                        if (dl != null) directories.addAll(dl);
                    }
                }
            }
            return directories;
        }
        
        void evaluateFmri(Fmri pfmri) throws IOException, ConstraintException {
            setFmriDefaultAuthority(pfmri);
            Manifest m = getManifest(pfmri);
            // examine manifest for dependencies
            List<DependAction> lda = m.getActionsByType(DependAction.class);
            if (constraints.startLoading(pfmri)) {
                for (DependAction da : lda) {
                    constraints.updateConstraints(da.getConstraint());
                }
                constraints.finishLoading(pfmri);
            }
            for (DependAction da : lda) {
                // discover if we have an installed or proposed version of 
                // this pkg already; proposed fmris will always be newer
                Fmri refFmri = getProposedVersion(da.getTargetFmri());
                if (refFmri == null) {
                    refFmri = getVersionInstalled(da.getTargetFmri());
                }
                // check if new constraint requires us to make any changes
                // to already proposed pkgs or existing ones
                if (da.getConstraint().checkForWork(refFmri) == null) continue;
             
                Fmri daf = constraints.applyConstraintsToFmri(da.getTargetFmri());
                // This will be the newest version of the specified
                // dependency package, coming from the preferred
                // authority, if it's available there.
                Fmri cf = catalogCache.getFirst(daf.toString());
                if (cf == null) {
                    throw new IOException("Unable to resolve dependency on package:" + da.keyValue());
                }
                proposeFmri(cf);
                evaluateFmri(cf);
            }
        }
        
        void addPackagePlan(Fmri pfmri) throws IOException {
            Manifest m = getManifest(pfmri);
            PackagePlan pp = new PackagePlan();
            pp.proposeDestination(pfmri, m);
            pp.evaluate();
            pkgPlans.add(pp);
        }
        
        void evaluateFmriRemoval(Fmri pfmri) throws IOException {
            List<Fmri> dependents = getDependents(pfmri);
            for (Fmri d : dependents) {
                if (targetRemFmris.contains(d)) {
                    dependents.remove(d);
                }
            }
            if (dependents.size() > 0 && !recursiveRemoval) {
                StringBuffer msg = new StringBuffer();
                msg.append("Cannot remove '" + pfmri + 
                        "' due to the following packages that directly " +
                        "depend on it:\n");
                for (Fmri d : dependents) {
                    msg.append(d.toString());
                    msg.append("\n");
                }
                throw new IllegalArgumentException(msg.toString());
            }
            Manifest m = getManifest(pfmri);
            PackagePlan pp = new PackagePlan();
            pp.proposeRemoval(pfmri, m);
            pp.evaluate();
            
            for (Fmri d : dependents) {
                if (isProposedRemFmri(d)) continue;
                if (hasVersionInstalled(d)) continue;
                targetRemFmris.add(d);
                evaluateFmriRemoval(d);
            }
            // Post-order append will ensure topological sorting for acyclic
            // dependency graphs.  Cycles need to be arbitrarily broken, and
            // are done so in the loop above.
            pkgPlans.add(pp);
        }
        
        void evaluate() throws IOException, ConstraintException {
            String outstring = "";
            // Operate on a copy, as it will be modified in flight
            List<Fmri> targetFmrisCopy = new ArrayList<Fmri>(targetFmris);
            for (Fmri f : targetFmrisCopy) evaluateFmri(f);
            for (Fmri f : targetFmris) addPackagePlan(f);
            for (Fmri f : targetRemFmris) evaluateFmriRemoval(f);
            state = PlanState.EVALUATED_OK;
        }
        
        boolean nothingToDo() {
            return pkgPlans.size() == 0;
        }
        
        /**
         * Execute the actions required to bring about the plan.
         * <p>
         * This results in the installation or uninstallation as specified by
         * the plan.
         * @throws java.io.IOException
         */
        public void execute() throws IOException {
            // Invoke the evaluated image plan
            // preexecute, execute, and postexecute
            // execute actions need to be sorted across packages
            if (nothingToDo()) {
                state = PlanState.EXECUTED_OK;
                return;
            }

            for (PackagePlan p : pkgPlans) p.preexecute();
        
            // now we're ready to start install.  At this point we
            // should do a merge between removals and installs so that
            // any actions moving from pkg to pkg are seen as updates rather
            // than removal and re-install, since these two have separate
            // semantics.
            //
            // General install method is removals, updates and then 
            // installs.  
            
            // generate list of removal actions, sort and execute

            List<Action> removals = new ArrayList<Action>();
            for (PackagePlan pp : pkgPlans) {
                removals.addAll(pp.getRemovalActions());
            }
            Collections.sort(removals, Collections.reverseOrder());
            
            log.log(Level.FINE, "removalphase");
            for (Action a : removals) a.remove();

            // generate list of update actions, sort and execute
            List<Manifest.Difference.ActionPair> updates = new ArrayList<Manifest.Difference.ActionPair>();
            for (PackagePlan pp : pkgPlans) {
                updates.addAll(pp.getUpdateActions());
            }

            // move any user/group actions into modify list to 
            // permit package to add user/group and change existing
            // files to that user/group in a single update
            List<Action> installs = new ArrayList<Action>();
            for (PackagePlan pp : pkgPlans) {
                installs.addAll(pp.getInstallActions());
            }
            // iterate over copy since we're modifying installs
            for (Action a : new ArrayList<Action>(installs)) {
                if (a instanceof UserAction || a instanceof GroupAction) {
                    updates.add(new Manifest.Difference.ActionPair(null, a));
                    installs.remove(a);
                }
            }
                           
            Collections.sort(updates, new Comparator<Manifest.Difference.ActionPair>() {
                public int compare(Manifest.Difference.ActionPair o1, 
                                   Manifest.Difference.ActionPair o2) {
                    return o1.to.compareTo(o2.to);
                }
                public boolean equals(Manifest.Difference.ActionPair o1, 
                                   Manifest.Difference.ActionPair o2) {
                    return o1.to.equals(o2.to);
                }
            });
            log.log(Level.FINE, "updatephase");
            for (Manifest.Difference.ActionPair ap : updates) ap.to.install(ap.from);

            // generate list of install , sort and execute
            Collections.sort(installs);
            log.log(Level.FINE, "installphase");
            for (Action a : installs) a.install(null);

            for (PackagePlan p : pkgPlans) p.postExecute();

            state = PlanState.EXECUTED_OK;

            // remove index as it is now out of date
            // TODO - actually update the index instead
            deleteTree(new File(getMetaDirectory(), "index"));
        }
    }
    
    /**
     * A package plan takes two package FMRIs and an Image, and produces the
     * set of actions required to take the Image from the origin FMRI to the
     * destination FMRI.
     * 
     * If the destination FMRI is None, the package is removed.
     */
    class PackagePlan
    {
        private Fmri originFmri = null;
        private Fmri destFmri = null;
        private Manifest originManifest = Manifest.nullManifest;
        private Manifest destManifest = Manifest.nullManifest;
        private Manifest.Difference diff;
        private FileList flist;
       
        PackagePlan() {}
        
        @Override
        public String toString() {
            return "" + originFmri + " -> " + destFmri;
        }

        void proposeDestination(Fmri pfmri, Manifest m) {
            destFmri = pfmri;
            destManifest = m;
            if (isInstalled(pfmri)) {
                throw new IllegalArgumentException("already installed: " + pfmri);
            }
        }

        void proposeRemoval(Fmri pfmri, Manifest m) {
            originFmri = pfmri;
            originManifest = m;
            if (!isInstalled(pfmri)) {
                throw new IllegalArgumentException("not installed: " + pfmri);
            }
        }

        void updatePackageSet(List<Fmri> fmriset) {
            if (originFmri != null) fmriset.remove(originFmri);
            if (destFmri != null) fmriset.add(destFmri);
        }

        void evaluate() throws IOException {
            // Determine the actions required to transition the package
            
            // if origin unset, determine if we're dealing with a previously
            // installed version or if we're dealing with the null package
            if (originFmri == null) {
                Fmri f = getInstalledOlderVersion(destFmri);
                if (f != null) {
                    originFmri = f;
                    originManifest = getManifest(f);
                }
            }
            // Assume that origin actions are unique, but make sure that
            // destination ones are.
            if (destManifest != null && destManifest.hasDuplicates()) {
                throw new IllegalArgumentException("dest manifest has duplicates: "
                        + destFmri);
            }
            
            // Get the differences between the two manifests
            diff = destManifest.getDifference(originManifest);
            
            // TODO - need to implement the following
            // figure out how many implicit directories disappear in this
            // transition and add directory remove actions.  These won't
            // do anything unless no pkgs reference that directory in
            // new state....
            
            // TODO - need to implement the following (as part of implementation of hardlinks)
            // over the list of update actions, check for any that are the
            // target of hardlink actions, and add the renewal of those hardlinks
            // to the install set
            
        }

        /**
         * Perform actions required prior to installation or removal of a package.
         * 
         * This method executes each action's preremove() or preinstall()
         * methods, as well as any package-wide steps that need to be taken
         * at such a time.
         */
        void preexecute() throws IOException {
            flist = new FileList(Image.this, destFmri);
            if (destFmri == null) {
                removeInstallFile(originFmri);
            }
            for (Action a : diff.added) {
                a.preinstall(null, flist);
            }
            for (Action a : diff.removed) {
                a.preremove();
            }
            for (Manifest.Difference.ActionPair ap : diff.changed) {
                ap.from.preremove();
                ap.to.preinstall(ap.from, flist);
            }
            flist.download();
        }

        List<Action> getInstallActions() {
            return diff.added;
        }
        
        List<Action> getRemovalActions() {
            return diff.removed;
        }
        
        List<Manifest.Difference.ActionPair> getUpdateActions() {
            return diff.changed;
        }
        

        /**
         * Perform actions required after installation or removal of a package.
         * 
         * This method executes each action's postremove() or postinstall()
         * methods, as well as any package-wide steps that need to be taken
         * at such a time.
         */
        void postExecute() throws IOException {
            for (Action a : diff.added) {
                a.postinstall(null);
            }
            for (Action a : diff.removed) {
                a.postremove();
            }
            for (Manifest.Difference.ActionPair ap : diff.changed) {
                ap.from.postremove();
                ap.to.postinstall(ap.from);
            }
            
            // In the case of an upgrade, remove the installation file from
            // the origin's directory.
            // XXX should this just go in preexecute?
            if (originFmri != null && destFmri != null) {
                removeInstallFile(originFmri);
                // TODO - need to remove filters
            }
            if (destFmri != null) {
                addInstallFile(destFmri);
                // TODO - need to save the filters used to install the package
            }
            if (flist != null) flist.cleanupDownload();
        }        
    }
    
    public final static int IMG_ENTIRE = 0;
    public final static int IMG_PARTIAL = 1;
    public final static int IMG_USER = 2;

    final static String img_user_prefix = ".org.opensolaris,pkg";
    final static String img_root_prefix = "var/pkg";
    final static String os = System.getProperty("os.name").toLowerCase();
    final static String arch = System.getProperty("os.arch").toLowerCase();

    // meta data directory names
    private final static String PKG_DIR = "pkg";
    private final static String INSTALLED_FILE = "installed";
    private final static String ISTATE_DIR = "state" + File.separator + "installed";
    private final static String ISTATE_DIR_TMP = "state" + File.separator + "installed.build";
    
    static final String typeStr = "USER";
    static Logger log = Logger.getLogger("com.sun.pkg.client", "com/sun/pkg/client/messages");
    static final String userAgent = "pkg-java/" + getVersion()
            + " (" + os + " " + arch + "; "
            + System.getProperty("os.version") + " " 
            + System.getProperty("java.version") + "; "
            + typeStr + ")"; 

    // instance fields
    int type = IMG_USER;
    Map<String, Authority> authorities = new HashMap<String, Authority>();
    List<String> lines = new ArrayList<String>();
    Map<String, Boolean> policies = new HashMap<String, Boolean>();
    String preferred_authority_name = null;
    File cfgfile;
    File imgdir;    // the root directory of the image
    File metadir;   // the root directory of the image meta data
    Map<Authority, Catalog> catalogs = new HashMap<Authority, Catalog>();
    CatalogCache catalogCache = null;
    ProxySelector proxySel = SystemInfo.getProxySelector();
    Constraint.Set constraints = new Constraint.Set();
    
    /**
     * Create an Image object for an existing user image based on a String
     * 
     * @param path The path for a file or directory within the image.
     */
    public Image(String path) throws Exception
    {
        this(new File(path));
    }
    
    /**
     * Create an Image object for an existing user image based on a File.
     * 
     * @param path A File within the image.
     */ 
    public Image(File path) throws Exception
    { 
        // only IMG_USER images are supported for now
        imgdir = path;
        if (!imgdir.isDirectory()) {
            throw new Exception("image directory (" + path + ") doesn't exist");
        }

        metadir = new File(imgdir, img_user_prefix);
        if (!imgdir.isDirectory()) throw new Exception("invalid USER image directory");

        loadConfig(new File(metadir, "cfg_cache"));
        
        // load catalogs
        for (Authority a : authorities.values()) {
            log.log(Level.CONFIG, "softwarerepo", a.origin);
            Catalog c = new Catalog(this, a.prefix);
            catalogs.put(a, c);
        }
        rebuildCatalogCache();
    }
    
    private void rebuildCatalogCache() throws IOException {
        catalogCache = new CatalogCache();
        for (Entry<Authority, Catalog> e : catalogs.entrySet()) {
            catalogCache.addCatalog(e.getValue(), e.getKey());
        }
        catalogCache.cacheInstalledPackages();
    }
    
    /**
     * Obtain the root directory for this image.
     * 
     * @return the root directory for this image
     */
    public File getRootDirectory()
    {
        return imgdir;
    }
       
    File getMetaDirectory()
    {
        return metadir;
    }
    
    Logger getLogger()
    {
        return log;
    }
    
    /**
     * Get a connected URLConnection to a requested resource.
     * 
     * @return a connected URLConnection to the requested resource
     * @throws java.io.IOException
     */
    HttpURLConnection getRepositoryURLConnection(String resource, String authname) throws IOException
    {
        Authority a = authorities.get(authname);
        if (a == null) {
            throw new IllegalArgumentException("invalid authority name: " + authname);
        }
        URL repo = new URL(a.origin);
        URL u = new URL(repo, resource);
        try {
            List<Proxy> pl = proxySel.select(u.toURI());
            // For now, only try the first one in the list
            Proxy p = pl.size() > 0 ? pl.get(0) : null;
            URLConnection urlc = p == null ? u.openConnection() : u.openConnection(pl.get(0));
            if (!(urlc instanceof HttpURLConnection)) {
                throw new IOException("unrecognized repository URL type:" + a.origin);
            }
            if (urlc instanceof HttpsURLConnection && a.certFile != null && a.keyFile != null) {
                if (a.sslfactory == null) {
                    a.sslfactory = PEMUtil.getSSLSocketFactory(a.certFile, a.keyFile);
                }
                HttpsURLConnection sslurlc = (HttpsURLConnection) urlc;
                sslurlc.setSSLSocketFactory(a.sslfactory);
            }
            HttpURLConnection hurlc = (HttpURLConnection) urlc;
            hurlc.setRequestProperty("User-Agent", userAgent);
            hurlc.setRequestProperty("X-IPkg-UUID", a.uuid);
            return hurlc;
        } catch (URISyntaxException ex) {
            Logger.getLogger(Image.class.getName()).log(Level.SEVERE, null, ex);
            IOException ioe = new IOException("invalid repository URL: " + u);
            ioe.initCause(ioe);
            throw ioe;
        }
    }
    
    /**
     * Get a connected URLConnection to a requested resource for an Fmri.
     * 
     * @return a connected URLConnection to the requested resource
     * @throws java.io.IOException
     */
    HttpURLConnection getRepositoryURLConnection(String resource, Fmri f) throws IOException
    {
        String authname = f.getAuthority();
        if (authname == null) {
            throw new IllegalArgumentException(
                    "attempt to access a resource for an Fmri that has no authority: " +
                    f);
        }
        return getRepositoryURLConnection(resource, authname);
    }

    /**
     * Check the connection for an HTTP_OK response code.  Throw an appropriate 
     * IOException for invalid codes.
     * @param urlc the connection to check
     * @throws IOException
     */
    void checkRepositoryConnection(HttpURLConnection urlc) throws IOException {
        int rc = urlc.getResponseCode();
        switch (rc) {
            case HttpURLConnection.HTTP_OK:
                return;
            case HttpURLConnection.HTTP_GATEWAY_TIMEOUT:
                throw new IOException("" + rc + ": Connection through proxy timed out");
            default:
                throw new IOException("Connection failed: " + 
                        rc + ": " + urlc.getResponseMessage());
        }
    }

    /**
     * Change the permissions on the Image meta data directory so that the 
     * directory will be hidden.
     * 
     * @throws java.io.IOException
     */
    public void hideMetaDirectory() throws IOException
    {
        if (os.indexOf("windows") == -1) return;
                
        String cmd[] = {"ATTRIB", "+H", getMetaDirectory().getCanonicalPath()};
        try {
            Process p = Runtime.getRuntime().exec(cmd);
            p.waitFor();
            if (p.exitValue() != 0) {
                throw new Error("ATTRIB failed: cannot hide meta data folder");
            }
        } catch (InterruptedException ie) {
        }
    }
    
    /**
     * Set the proxy to be used by connenctions to repositories.
     * <p>
     * The proxy for an Image is initialized to be the return value from 
     * SystemInfo.getProxy.  This method can be used to set the proxy to some 
     * other value or to null if the desire is to use the system defaults.
     *  
     * @param p - the proxy to use
     */
    public void setProxy(Proxy p)
    {
        proxySel = new PkgProxySelector(p);
    }
    
    void loadConfig(File configFile) throws FileNotFoundException, IOException
    {
        /* Eventually, this needs to be implemented to support full parsing of
         * the cfg_cache file, possibly using the ini4j library: see 
         * http://ini4j.sourceforge.net/.  For now, we just read each line and 
         * do simple parsing. 
         */
        cfgfile = configFile;
        BufferedReader r = new BufferedReader(new FileReader(cfgfile));
        String line;
        String tokens[];
        Authority curAuth = null;
        boolean inPolicy = false;
        while ((line = r.readLine()) != null) {
            lines.add(line); // save the lines for if we need to write
            
            // check if this is a new section
            if (curAuth != null && line.startsWith("[")) {
                authorities.put(curAuth.prefix, curAuth);
                curAuth = null;                
            }
            if (inPolicy && line.startsWith("[")) {
                inPolicy = false;
            }
            
            if (line.startsWith("preferred-authority")) {
                tokens = line.split("\\s*=\\s*", 2);
                if (tokens.length == 2) {
                    preferred_authority_name = tokens[1].trim();
                }
            }
            else if (line.startsWith("[authority") && line.endsWith("]")) {
                // start of an authority section
                line = line.substring(1, line.length() - 1);
                tokens = line.split("_");
                if (tokens.length == 2) {
                    curAuth = new Authority(tokens[1]);
                }
            }
            else if (curAuth != null && line.startsWith("uuid")) {
                tokens = line.split("\\s*=\\s*", 2);
                if (tokens.length == 2) {
                    curAuth.uuid = tokens[1].trim();
                }
            }
            else if (curAuth != null && line.startsWith("origin-for-")) {
                tokens = line.split("\\s*=\\s*", 2);
                if (tokens.length == 2) {
                    String plat = tokens[0].trim().substring(11);
                    // override the origin with one that is specific to this platform
                    if (isThisPlatform(plat)) curAuth.origin = tokens[1].trim();
                }
            }
            else if (curAuth != null && line.startsWith("origin")) {
                tokens = line.split("\\s*=\\s*", 2);
                // do not overwrite the origin if it is already set
                if (curAuth.origin == null && tokens.length == 2) {
                    curAuth.origin = tokens[1].trim();
                }
            }
            else if (curAuth != null && line.startsWith("prefix")) {
                tokens = line.split("\\s*=\\s*", 2);
                if (tokens.length == 2) {
                    curAuth.prefix = tokens[1].trim();
                }
            }
            else if (curAuth != null && line.startsWith("ssl_key")) {
                tokens = line.split("\\s*=\\s*", 2);
                if (tokens.length == 2) {
                    String fn = tokens[1].trim();
                    if (fn.length() > 0 && !fn.equals("None")) {
                        curAuth.keyFile = new File(fn);
                    }
                }
            }
            else if (curAuth != null && line.startsWith("ssl_cert")) {
                tokens = line.split("\\s*=\\s*", 2);
                if (tokens.length == 2) {
                    String fn = tokens[1].trim();
                    if (fn.length() > 0 && !fn.equals("None")) {
                        curAuth.certFile = new File(fn);
                    }
                }
            }
            else if (line.startsWith("[policy]")) {
                inPolicy = true;
            }
            else if (inPolicy) {
                tokens = line.split("\\s*=\\s*", 2);
                if (tokens.length == 2) {
                    policies.put(tokens[0].trim(), 
                                 Boolean.parseBoolean(tokens[1].trim()));
                }
            }
        }
        if (curAuth != null) {
            authorities.put(curAuth.prefix, curAuth);
        }
        r.close();
    }
    
    /**
     * Save the configuration for the image.
     * 
     * Adjust the origin in the cfgfile so that it matches that of the current
     * platform as determined by origin_for_* properties.  This method writes
     * a new version of the cfgfile. 
     * @throws java.io.IOException
     */
    public void saveConfig() throws IOException
    {
        BufferedWriter w = new BufferedWriter(new FileWriter(cfgfile));
        String line;
        String tokens[];
        Authority curAuth = null;
        for (int i = 0; i < lines.size(); i++) {
            line = lines.get(i);
            if (line.startsWith("[authority") && line.endsWith("]")) {
                // start of an authority section
                String line1 = line.substring(1, line.length() - 1);
                tokens = line1.split("_");
                if (tokens.length == 2) {
                    curAuth = authorities.get(tokens[1]);
                }
            }
            else if (curAuth != null && line.startsWith("uuid")) {
                // write the uuid from the Authority
                line = "uuid = " + curAuth.uuid;
            }
            else if (curAuth != null && line.startsWith("origin-for-")) {
                // leave origin-for lines unmodified
            }
            else if (curAuth != null && line.startsWith("origin")) {
                // write the origin from the Authority
                line = "origin = " + curAuth.origin;
            }            
            w.write(line);
            w.newLine();
        }
        w.close();
    }
    
    public String getPreferredAuthorityName() {
        return preferred_authority_name;
    }
    
    /**
     * Return the authority names defined for this image.
     * @return an array of authority names
     */
    public String[] getAuthorityNames() {
        return authorities.keySet().toArray(new String[0]);
    }
     
    /**
     * Set image authority attributes.
     * 
     * If authname names an authority that currently doesn't exist, it is added
     * to the image. For adding an authority, the origin URL is required. 
     * For an existing authority, the origin or uuid parameters may be null 
     * and in that case the value for the attribute is not changed.
     * <p>
     * NOTE: setting the uuid is effective only if a uuid attribute is already there
     * <br>
     * NOTE: adding an authority doesn't yet save it to disk
     * 
     * @param authname - the name of the authority to change
     * @param origin - the origin URL for the authority
     * @param uuid - the UUID for the authority
     * @throws IllegalArgumentException - if the authority is not currently in the
     * image and origin is null.
     */
    public void setAuthority(String authname, URL origin, String uuid)
    {
        Authority a = authorities.get(authname);
        if (a == null && origin == null) {
            throw new IllegalArgumentException("Origin is required for a new authority");
        }
        if (a == null) {
            a = new Authority(authname);
            authorities.put(authname, a);
        }
        if (origin != null) a.origin = origin.toString();
        if (uuid != null) a.uuid = uuid;
    }
       
    /**
     * Call refresh on all of the Catalogs for this image.
     * 
     * This also updates the internal cache used by the Image class to store 
     * catalog information.
     * 
     * @throws java.io.IOException
     */
    public void refreshCatalogs() throws IOException
    {
        for (Catalog c : catalogs.values()) c.refresh();
        rebuildCatalogCache();
    }
    
    /**
     * Call refresh on the Catalog for the given authority.
     * 
     * This also updates the internal cache used by the Image class to store 
     * catalog information.
     * 
     * @throws java.io.IOException
     */
    public void refreshCatalog(String authname) throws IOException
    {
        Authority a = authorities.get(authname);
        if (a == null) throw new IllegalArgumentException("unknown authority: " + authname);
    
        Catalog c = catalogs.get(a);
        if (c == null) throw new IllegalArgumentException("no catalog for authority: " + authname);

        c.refresh();
        rebuildCatalogCache();
    }
    
    /**
     * Get the inventory of packages and their associated state.
     * 
     * The inventory is returned in FmriState objects that consist of an Fmri and 
     * the four states for the Fmri.  These states are:
     * <ul>
     * <li><b>upgradable</b> A package with a newer version if available in the 
     * catalog
     * <li><b>frozen</b> TBD
     * <li><b>incorporated</b> The package is part of an incorporation
     * <li><b>excludes</b> TBD
     * </ul>     * 
     *
     * @param pkg_list Limit the inventory to packages that match entries in the
     * list of package names
     * @param all_known If true, return all package versions.  Otherwise only 
     * return the most recent version of each package.
     * @return A List of FmriState objects that identify the Fmris that are 
     * available for the image.
     */
    public List<FmriState> getInventory(String[] pkg_list, boolean all_known)
    {
        return catalogCache.getInventory(pkg_list, all_known);
    }
    
    /**
     * Check whether a given package is installed in this image.
     * 
     * @return True if the given package is installed
     */
    boolean isInstalled(Fmri p)
    {
        try {
            File ifile = new File(p.getPkgVersionDir(new File(getMetaDirectory(), PKG_DIR)),
                INSTALLED_FILE);
            if (!ifile.exists()) return false;
            String auth = p.getAuthority();
            if (auth == null) throw new IllegalArgumentException("Fmri must have authority set");
            return auth.equals(getAuthorityForInstalledPkg(p).prefix);
        }
        catch (IOException ioe) {
            return false;
        }
    }

    /**
     * Create an plan for installing packages and any required dependencies.
     * @param pkgNames names of packages to install
     * @return An ImagePlan that will install the packages when executed.
     * @throws java.io.IOException
     */
    public ImagePlan makeInstallPlan(String[] pkgNames) throws IOException, ConstraintException
    {
        loadConstraints();
        
        // order package list so that any unbound incorporations are done first
        List<String> incList = getInstalledUnboundIncorporationList();
        List<String> head = new ArrayList<String>();
        List<String> tail = new ArrayList<String>();
        for (String pn : pkgNames) {
            if (incList.contains(pn)) head.add(pn);
            else tail.add(pn);
        }
        head.addAll(tail);
        List<String> pkgnl = head;

        List<Fmri> pkgs = new ArrayList<Fmri>();
        for (String p : pkgnl) {
            Fmri conp = constraints.applyConstraintsToFmri(new Fmri(p));
            Fmri f = catalogCache.getFirst(conp.toString());
            if (f == null) {
                throw new IllegalArgumentException("no matching package for: " + p);
            }
            pkgs.add(f);
        }
        return makeInstallPlan(pkgs);
    }
    
    /**
     * Create an plan for installing packages and any required dependencies.
     * @param pkgs List of valid Fmris from the image inventory for packages to 
     * install.
     * @return An ImagePlan that will install the packages when executed.
     * @throws java.io.IOException
     */
    public ImagePlan makeInstallPlan(List<Fmri> pkgs) throws IOException, ConstraintException
    {
        ImagePlan ip = new ImagePlan(false, null);
        
        for (Fmri pf : pkgs) ip.proposeFmri(pf);
        log.log(Level.FINER, "beforeeval", ip.toString());
        ip.evaluate();
        log.log(Level.FINER, "aftereval", ip.toString());
        return ip;
    }

    /**
     * Install packages and any required dependencies.
     * <p>
     * The catalogs are searched for a matching package for each entry in the 
     * pkgNames array. 
     * If multiple packages match an entry, the most recent package is installed.
     * If no matching package is found, an IllegalArgumentException is thrown.
     * <p>
     * This method is shorthand for:
     * <pre>
     *      makeInstallPlan(pkgNames).execute();
     * </pre>
     *  
     * @param pkgNames names of packages to install
     * @throws java.io.IOException
     * @throws java.lang.IllegalArgumentException if a matching package cannot be found
     */
    public void installPackages(String[] pkgNames) throws IOException, ConstraintException
    {  
        ImagePlan ip = makeInstallPlan(pkgNames);
        if (ip.nothingToDo()) return;
        log.log(Level.FINE, "installing", Arrays.toString(ip.getProposedFmris()));
        ip.execute();
    }
    
    /**
     * Install packages and any required dependencies.
     * <p>
     * This method is shorthand for:
     * <pre>
     *      makeInstallPlan(pkgs).execute();
     * </pre>
     * 
     * @param pkgs List of valid Fmris from the image inventory for packages to 
     * install.
     * @throws java.io.IOException
     */
    public void installPackages(List<Fmri> pkgs) throws IOException, ConstraintException
    {
        ImagePlan ip = makeInstallPlan(pkgs);
        if (ip.nothingToDo()) return;
        log.log(Level.FINE, "installing", pkgs.toString());
        ip.execute();
    }
    
    /**
     * Uninstall packages.
     * 
     * The installed package list is searched for a matching package for 
     * each entry in the pkgNames array. 
     * 
     * If no matching package is found, the entry is ignored. 
     * 
     * @param pkgNames names of packages to uninstall
     * @throws java.io.IOException
     */
    public void uninstallPackages(String[] pkgNames) throws IOException
    {
        ArrayList<Fmri> list = new ArrayList<Fmri>(pkgNames.length);
        for (int i = 0; i < pkgNames.length; i++)
        {
            Fmri f = getVersionInstalled(new Fmri(pkgNames[i]));
            if (f != null) list.add(f);
        }
        uninstallPackages(list);
    }
    
    /**
     * Uninstall packages.
     * 
     * @param pkgs List of valid Fmris to uninstall.
     * @throws java.io.IOException
     */
    public void uninstallPackages(List<Fmri> pkgs) throws IOException
    {
        ImagePlan ip = new ImagePlan(false, null);
        
        for (Fmri pf : pkgs) ip.proposeFmriRemoval(pf);
        log.log(Level.FINER, "beforeeval", ip.toString());
        ip.evaluate();
        log.log(Level.FINER, "aftereval", ip.toString());
        if (ip.nothingToDo()) return;
        log.log(Level.FINE, "uninstalling", pkgs.toString());
        ip.execute();
    }
    
    /**
     * Get the manifest for a package.
     * @return a Manifest
     */
    public Manifest getManifest(Fmri fmri) throws IOException {
        return new Manifest(this, fmri);
    }
    
    private boolean fmriIsSamePackage(Fmri fmri, Fmri pf) {
        // TODO - need to handle catalog renames
        return fmri.isSamePackage(pf);
    }
    
    private boolean fmriIsSuccessor(Fmri fmri, Fmri pf) {
        // TODO - need to handle catalog renames
        return fmri.isSuccessor(pf);
    }

    private List<Fmri> getInstalledFmrisFromCache() {
        List<FmriState> lfs = catalogCache.getInventory(null, false);
        List<Fmri> installed = new ArrayList<Fmri>();
        for (FmriState fs : lfs) {
            installed.add(fs.fmri);
        }
        return installed;
    }
    
    private List<Fmri> getInstalledFmrisFromImage() throws IOException {
        File istatedir = new File(getMetaDirectory(), ISTATE_DIR);
        List<Fmri> installedPkgs = new ArrayList<Fmri>();
        
        // XXX This can be removed at some point in the future once we
        // think this link is available on all systems
        if (!istatedir.exists()) updateInstalledPackages();
        for (File f : istatedir.listFiles()) {
            String fmristr = URLDecoder.decode(f.getName(), "UTF-8");
            Fmri fmri = new Fmri(fmristr);
            installedPkgs.add(fmri);
        }
        return installedPkgs;
    }

    /**
     * Obtain a boolean policy from the image configuration file
     * @param pname The name of the policy
     * @return the value for the policy
     */
    boolean getPolicy(String pname) {
        Boolean b = policies.get(pname);
        return b != null && b.booleanValue();
    }

    /**
     * Set the Fmri's authority to be the preferred authority for this Image
     * if it isn't already set.
     */
    void setFmriDefaultAuthority(Fmri fmri) {
        if (fmri.getAuthority() == null) {
            fmri.setAuthority(getPreferredAuthorityName());
        }
    }

    private Fmri getVersionInstalled(Fmri fmri) throws IOException {
        File pd = fmri.getPkgDir(new File(getMetaDirectory(), PKG_DIR));
        File pkgDirs[] = pd.listFiles();
        if (pkgDirs == null) return null;
        List<File> installedPkgs = new ArrayList<File>();
        for (File f : pkgDirs) {
            File ifile = new File(f, INSTALLED_FILE);
            if (ifile.exists()) installedPkgs.add(f);
        }
        if (installedPkgs.size() == 0) return null;
        if (installedPkgs.size() > 1) {
            throw new RuntimeException("package installed more than once: " + fmri);
        }
        Fmri ifmri = new Fmri(installedPkgs.get(0));
        ifmri.setAuthority(getAuthorityForInstalledPkg(ifmri).prefix);
        return ifmri;
    }
    
    private Fmri getInstalledOlderVersion(Fmri fmri) throws IOException {
        // TODO need to deal with package renames in catalog
        return getVersionInstalled(fmri);
    }

    private boolean hasVersionInstalled(Fmri fmri) throws IOException {
        Fmri f = getVersionInstalled(fmri);
        return f != null && fmriIsSuccessor(f, fmri);
    }

    /*
     * Load constraints for all installed pkgs
     */
    private void loadConstraints() throws IOException {
        for (Fmri fmri : getInstalledFmrisFromCache()) {
            // skip loading if already done
            if (constraints.startLoading(fmri)) {
                Manifest m = getManifest(fmri);
                for (DependAction da : m.getActionsByType(DependAction.class)) {
                    constraints.updateConstraints(da.getConstraint());
                }
                constraints.finishLoading(fmri);
            }
        }
    }
    
    /*
     * Returns list of packages containing incorporation dependencies on which 
     * no other pkgs depend.
     */
    private List<String> getInstalledUnboundIncorporationList() throws IOException {
        Set<String> dependents = new TreeSet<String>();
        List<String> inc = new ArrayList<String>();
        for (Fmri fmri : getInstalledFmrisFromCache()) {
            String fmriName = fmri.getName();
            Manifest m = getManifest(fmri);
            for (DependAction da : m.getActionsByType(DependAction.class)) {
                Fmri cfmri = da.getConstrainedFmri();
                if (cfmri != null) {
                    String cName = cfmri.getName();
                    dependents.add(cName);
                    inc.add(fmriName);
                }
            }
        }
        // remove those incorporations which are depended on by other incorporations.
        Set<String> rv = new TreeSet<String>();
        for (String p : inc) if (!dependents.contains(p)) rv.add(p);
        return new ArrayList<String>(rv);        
    }
    
    /**
     * Return a list of the packages directly dependent on the given Fmri.
     */
    private List<Fmri> getDependents(Fmri pfmri) {
        File thedir = new File(new File(new File(this.getMetaDirectory(), "index"), 
                "depend"), pfmri.getName());
        if (!thedir.isDirectory()) return Collections.emptyList();
        
        for (File f : thedir.listFiles()) {
            Fmri fmri = new Fmri(f);
            if (fmriIsSuccessor(pfmri, fmri)) {
                List<Fmri> dependents = new ArrayList<Fmri>();
                for (File d : f.listFiles()) {
                    File ifile = new File(d, INSTALLED_FILE);
                    if (ifile.exists()) {
                        try {
                            String dep = URLDecoder.decode(d.getName(), "UTF-8");
                            dependents.add(new Fmri(dep));
                        } catch (UnsupportedEncodingException ex) {
                            Logger.getLogger(Image.class.getName()).log(Level.SEVERE, null, ex);
                        }
                    }
                }
                return dependents;
            }
        }
        return Collections.emptyList();
    }
    
    private Authority getAuthorityForInstalledPkg(Fmri fmri) throws IOException {
        String authname = null;
        File ifile = new File(fmri.getPkgVersionDir(new File(getMetaDirectory(), PKG_DIR)),
                INSTALLED_FILE);
        BufferedReader r = new BufferedReader(new FileReader(ifile));
        String line;
        line = r.readLine();
        if (line != null && !line.equals("VERSION_1")) {
            // old-style file, line is the authority name
            authname = line;
        } else {
            line = r.readLine();
            String pfx = Fmri.PREF_AUTH_PFX + "_";
            if (line.startsWith(pfx)) {
                authname = line.substring(pfx.length());
            } else {
                authname = line;
            }
        }
        r.close();
        Authority a = authorities.get(authname);
        if (a == null) {
            // A package was installed for an authority no longer in cfg_cache
            a = new Authority(authname);
            authorities.put(authname, a);
        }
        return a;
    }
    
    private void addInstallFile(Fmri fmri) throws IOException {
        File istatedir = new File(getMetaDirectory(), ISTATE_DIR);
        
        // XXX This can be removed at some point in the future once we
        // think this link is available on all systems
        if (!istatedir.exists()) updateInstalledPackages();

        String authname = fmri.getAuthority();
        String prefpfx = "";
        String prefauth = getPreferredAuthorityName();
        if (authname == null || authname.equals(prefauth)) {
            authname = prefauth;
            prefpfx = Fmri.PREF_AUTH_PFX + "_";
        }
        File ifile = new File(fmri.getPkgVersionDir(new File(getMetaDirectory(), PKG_DIR)),
                INSTALLED_FILE);
        FileWriter fw = new FileWriter(ifile);
        fw.write("VERSION_1\n" + prefpfx + authname);
        fw.close();
            
        File isfile = new File(istatedir, fmri.getLinkFileName());
        isfile.createNewFile();               
    }
    
    private void removeInstallFile(Fmri fmri) throws IOException {
        File istatedir = new File(getMetaDirectory(), ISTATE_DIR);
        
        // XXX This can be removed at some point in the future once we
        // think this link is available on all systems
        if (!istatedir.exists()) updateInstalledPackages();
                
        File ifile = new File(fmri.getPkgVersionDir(new File(getMetaDirectory(), PKG_DIR)),
                INSTALLED_FILE);
        ifile.delete();

        File isfile = new File(istatedir, fmri.getLinkFileName());
        isfile.delete();
    }

    /**
     * Take the image's record of installed packages from the prototype layout,
     * with an installed file in each $META/pkg/stem/version directory, to the
     * $META/state/installed summary directory form
     */
    private void updateInstalledPackages() throws IOException {
        File istatedir = new File(getMetaDirectory(), ISTATE_DIR);
        File tmpdir = new File(getMetaDirectory(), ISTATE_DIR_TMP);

        // Create the link forest in a temporary directory.  
        if (tmpdir.mkdirs() == false) {
            throw new IOException("unable to create installed packages directory");
        }

        File proot = new File(getMetaDirectory(), PKG_DIR);
        for (File pd : proot.listFiles()) {
            for (File vd : pd.listFiles()) {
                File path = new File(vd, "installed");
                if (!path.exists()) continue;
                String fmristr = URLDecoder.decode(
                        pd.getName() + "@" + vd.getName(), "UTF-8");
                Fmri f = new Fmri(fmristr);
                File fi = new File(tmpdir, f.getLinkFileName());
                fi.createNewFile();
            }
        }

        // Someone may have already created this directory.  Junk the
        // directory we just populated if that's the case.
        if (tmpdir.renameTo(istatedir) == false) {
            for (File f : tmpdir.listFiles()) f.delete();
            tmpdir.delete();
        }
    }
    
    /**
     * Get the version of the program from the MANIFEST file
     */
    static String getVersion()
    {
        String version = log.getResourceBundle().getString("version");
        /* the following should work, but it doesn't when in an OSGi bundle 
         String version = Image.class.getPackage().getImplementationVersion();
         */
        if (version == null) version = "0";
        log.log(Level.FINE, "versionmsg", version);
        return version;
    }
   
    boolean isThisPlatform(String plat)
    {
        // because indexOf is used for matching below, a blank means match
        // anything for the 2nd and 3rd columns of this table
        String c[][] = { {"sunos-i386", "sunos", "x86"}, 
                         {"sunos-sparc", "sunos", "sparc"},
                         {"linux-i386", "linux", ""},
                         {"windows-i386", "windows", ""},
                         {"darwin-universal", "mac os x", ""}
        };
        
        for (int i = 0; i < c.length; i++) {
            if (plat.equals(c[i][0]) &&
                    os.indexOf(c[i][1]) != -1 &&
                    arch.indexOf(c[i][2]) != -1) {
                return true;
            }
        }
        return false;
    }

    static void deleteTree(File file) {
        if (file.isFile()) file.delete();
        else if (file.isDirectory()) {
            for (File f : file.listFiles()) {
                deleteTree(f);
            }
            file.delete();
        }
    }
}
