/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 2008 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or updatetool/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 *
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 *
 * Contributor(s):
 *
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package com.sun.pkg.client;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.InetSocketAddress;
import java.net.Proxy;
import java.net.ProxySelector;
import java.net.SocketAddress;
import java.net.URL;
import java.util.Date;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * <code>SystemInfo</code> provides access to system information related to the
 * pkg(5) Java API.  This includes information related to the updatetool GUI and
 * bootstrap programs that are built on top of this API.  This class can be used
 * to read and write information.
 */
public class SystemInfo {

    public enum UpdateCheckFrequency {

        NEVER, DAILY, WEEKLY, MONTHLY
    }
    private static final String APPNAME = "updatetool";
    private static final String INITFILE = "init.cfg";
    private static final String DEFAULTSFILE = "defaults.cfg";
    private static final String WIN_APP_DATA = "Application Data";
    private static final String MAC_APP_DATA = "/Library/Application Support/";
    private static final String eol = System.getProperty("line.separator");
    private static Proxy httpProxy = null;
    private static Proxy httpsProxy = null;

    /**
     * Obtain a ProxySelector based on the updatetool preferences.
     * <p>
     * This method returns a <code>ProxySelector</code> based on the information  
     * stored in the updatetool preferences. Proxies returned form this ProxySelector
     * may be null if updatetool is using system proxy settings that are not stored 
     * in the updatetool preferences file. To mimic the behavior of 
     * using the system proxy settings in Java, use the java.net.useSystemProxies
     * system property.
     * 
     * If unable to read the proxy information or if the proxy information is 
     * invalid, a ProxySelector is returned that will yield null Proxies and 
     * error information is logged.
     * 
     * @return a ProxySelector object
     */
    public static ProxySelector getProxySelector() {
        loadProxyInfo();
        return new PkgProxySelector(httpProxy, httpsProxy);
    }

    /**
     * Obtain proxy based on the updatetool preferences.
     * <p>
     * This method returns a <code>Proxy</code> based on the information  
     * stored in the updatetool preferences. This method may return null if
     * updatetool is using system proxy settings that are not stored in the 
     * updatetool preferences file. To mimic the behavior of 
     * using the system proxy settings in Java, use the java.net.useSystemProxies
     * system property.
     * 
     * If unable to read the proxy information or if the proxy information is 
     * invalid, null is returned and error information is logged.
     * 
     * @return a Proxy object
     */
    public static Proxy getProxy() {
        loadProxyInfo();
        return httpProxy;
    }
    
    private static void loadProxyInfo() {
        httpProxy = httpsProxy = null;
        try {
            // If the init.cfg file is there, use the values from that.  Otherwise
            // use the values from defaults.cfg.
            File f = getInitFile();
            if (!f.exists()) {
                f = getDefaultsFile();
                if (!f.exists()) {
                    // no information to read
                    return;
                }
            }
            // This is a poor man's implementation of a configuration file reader
            // It just looks for particular strings at the beginning of lines,
            // ignoring section boundaries.  This is sufficient to get the proxy
            // information.
            // Fortunately, the names that are used to store the proxy information
            // are the same in both files, so the same loop can be used no matter
            // which file we are reading.
            BufferedReader r = getFileReader(f);
            String line;
            String[] tokens;
            String proxy_host = null;
            String proxy_port = null;
            String proxy_https_host = null;
            String proxy_https_port = null;
            String proxy_username = null;
            String proxy_password = null;
            boolean useSystem = false;
            boolean proxy_required = false;
            boolean proxy_ssl_use_http = true;
            boolean proxy_auth = false;
            while ((line = r.readLine()) != null) {
                if (!line.startsWith("proxy.")) {
                    continue;
                }
                tokens = line.split("\\s*[:=]\\s*", 2);
                if (tokens[0].equals("proxy.host")) {
                    proxy_host = tokens[1].trim();
                } else if (tokens[0].equals("proxy.port")) {
                    proxy_port = tokens[1];
                } else if (tokens[0].equals("proxy.https.host")) {
                    proxy_https_host = tokens[1].trim();
                } else if (tokens[0].equals("proxy.https.port")) {
                    proxy_https_port = tokens[1];
                } else if (tokens[0].equals("proxy.required")) {
                    proxy_required = Boolean.parseBoolean(tokens[1].trim());
                } else if (tokens[0].equals("proxy.auth")) {
                    proxy_auth = Boolean.parseBoolean(tokens[1].trim());
                } else if (tokens[0].equals("proxy.ssl_use_http")) {
                    proxy_ssl_use_http = Boolean.parseBoolean(tokens[1].trim());
                } else if (tokens[0].equals("proxy.username")) {
                    proxy_username = tokens[1];
                } else if (tokens[0].equals("proxy.password")) {
                    proxy_password = tokens[1];
                } else if (tokens[0].equals("proxy.use.system")) {
                    useSystem = Boolean.parseBoolean(tokens[1].trim());
                }
            }

            if (proxy_required) {
                if (proxy_host == null || proxy_host.length() == 0) {
                    throw new Exception("invalid proxy setting: empty host");
                }
                SocketAddress addr = new InetSocketAddress(proxy_host, 
                        Integer.parseInt(proxy_port));
                httpProxy = proxy_auth ? 
                    new AuthProxy(Proxy.Type.HTTP, addr, proxy_username, proxy_password) :
                    new AuthProxy(Proxy.Type.HTTP, addr);
                if (proxy_ssl_use_http) {
                    httpsProxy = httpProxy;
                } else {
                    addr = new InetSocketAddress(
                            proxy_https_host, Integer.parseInt(proxy_https_port));
                    httpsProxy = proxy_auth ? 
                        new AuthProxy(Proxy.Type.HTTP, addr, proxy_username, proxy_password) :
                        new AuthProxy(Proxy.Type.HTTP, addr);
                }
            } else if (useSystem) {
                System.setProperty("java.net.useSystemProxies", "true");
            }
        } catch (Exception ex) {
            Logger.getLogger(SystemInfo.class.getName()).log(Level.SEVERE, "unable to get proxy", ex);
        }
    }

    /**
     * Obtain the update check frequency based on the updatetool preferences.
     * <p>
     * If the user has opted out of update notifications, NEVER is returned. Otherwise 
     * the value for the update check frequence is returned. If there is no setting, 
     * the default value is DAILY.
     * 
     * @return an UpdateCheckFrequency
     */
    public static UpdateCheckFrequency getUpdateCheckFrequency() {
        UpdateCheckFrequency rv = UpdateCheckFrequency.DAILY;
        boolean optin = true;
        try {
            File dfile = getDefaultsFile();
            if (dfile.exists()) {
                // This is a poor man's implementation of a configuration file reader
                // It just looks for particular strings at the beginning of lines,
                // ignoring section boundaries.  This is sufficient to get the information.
                BufferedReader r = getFileReader(dfile);
                String line;
                String[] tokens;
                while ((line = r.readLine()) != null) {
                    if (line.startsWith("check_frequency")) {
                        tokens = line.split("\\s*=\\s*", 2);
                        if (tokens.length == 2) {
                            if (tokens[1].equals("daily")) {
                                rv = UpdateCheckFrequency.DAILY;
                            } else if (tokens[1].equals("weekly")) {
                                rv = UpdateCheckFrequency.WEEKLY;
                            } else if (tokens[1].equals("monthly")) {
                                rv = UpdateCheckFrequency.MONTHLY;
                            } else if (tokens[1].equals("never")) {
                                rv = UpdateCheckFrequency.NEVER;
                            }
                        }
                        break;
                    }
                }
                r.close();
            }

            // If the init file is there, and it has optin set to false, then use
            // that value since it is newer
            File ifile = getInitFile();
            if (ifile.exists()) {
                BufferedReader r = getFileReader(ifile);
                String line;
                String[] tokens;
                while ((line = r.readLine()) != null) {
                    if (line.startsWith("optin.update.notification")) {
                        tokens = line.split("\\s*:\\s*", 2);
                        if (tokens.length == 2 && !tokens[1].equals("true")) {
                            optin = false;
                        }
                        break;
                    }
                }
                r.close();
            }
        } catch (IOException ex) {
            Logger.getLogger(SystemInfo.class.getName()).log(Level.SEVERE, null, ex);
        }
        if (!optin) {
            return UpdateCheckFrequency.NEVER;
        }
        return rv;
    }

    /**
     * Initialize the updatetool preferences from properties.
     * <p>
     * This method passes preferences information to the updatetool program by 
     * writing values to an updatetool initialization configuration file. When
     * updatetool next runs, it merges the initialization information into its 
     * preferences file. 
     * <p>
     * The properties recognized from the props object are:
     * <ul>
     * <li>optin.update.notification</li>
     * <li>optin.usage.reporting</li>
     * <li>image.path</li>
     * <li>proxy.URL</li>
     * </ul>
     * 
     * If the value for a property is missing, no change is made for that property.
     * To make updatetool use no proxy, pass in the empty string ("") for the 
     * proxy.URL value. 
     * 
     * @param props - values to use for writing the file
     * @throws java.lang.Exception
     */
    public static void initUpdateToolProps(Properties props) throws Exception {
        // Write the props to the init.cfg file.
        try {
            String v;
            BufferedWriter out = getFileWriter(getInitFile());

            out.write("[main]" + eol);
            out.write("date: " + new Date().getTime() + eol);

            v = props.getProperty("optin.update.notification");
            if (v != null) {
                out.write("optin.update.notification: " + v + eol);
            }

            v = props.getProperty("optin.usage.reporting");
            if (v != null) {
                out.write("optin.usage.reporting: " + v + eol);
            }

            v = props.getProperty("image.path");
            if (v != null) {
                out.write("image_list: " +
                        new File(v.trim()).getAbsolutePath() + eol);
            }

            out.write("[network]" + eol);
            v = props.getProperty("proxy.use.system");
            if (v != null) {
                out.write("proxy.use.system: " + v.trim() + eol);
            }

            v = props.getProperty("proxy.URL");
            if (v != null) {
                if (v.trim().length() > 0) {
                    URL proxy = new URL(v.trim());
                    out.write("proxy.required: true" + eol);
                    out.write("proxy.host: " + proxy.getHost() + eol);
                    out.write("proxy.port: " + proxy.getPort() + eol);
                    String userinfo = proxy.getUserInfo();
                    if (userinfo != null && userinfo.length() != 0) {
                        String[] tokens = userinfo.split(":", 2);
                        String user = tokens[0];
                        String password = tokens[1];
                        if (user != null) {
                            out.write("proxy.username: " + user + eol);
                        }
                        if (password != null) {
                            out.write("proxy.password: " + password + eol);
                        }
                        out.write("proxy.auth: true" + eol);
                    } else {
                        out.write("proxy.auth: false" + eol);
                    }
                } else {
                    out.write("proxy.required: false" + eol);
                }
            }
            out.close();
        } catch (IOException e) {
            throw new Exception("Unable to initialize properties", e);
        }

    // XXX: When we support storing proxy auth info in the init file
    // XXX: we must change the perms on the init file to allow only
    // XXX: the user to read/write the values.
    }

    private static File getConfigDir() throws IOException {
        String configpath;

        // Compute the path to the config file
        String homedir = System.getProperty("user.home");
        String osname = System.getProperty("os.name");
        String username = System.getProperty("user.name");
        if (osname.startsWith("Windows")) {
            configpath = new String(homedir + File.separator +
                    WIN_APP_DATA + File.separator + APPNAME);
        } else if (osname.startsWith("Mac OS")) {

            if (username.equals("root")) {
                try {
                    homedir = getMacHomeDir();
                } catch (Exception e) {
                    throw new IOException("Unable to determine $HOME for: " +
                            username);
                }
            }
            configpath = new String(homedir + MAC_APP_DATA + APPNAME);
        } else {
            configpath = new String(homedir + File.separator + "." + APPNAME);
        }

        // Check to see if the directory exists.  If it doesn't we create it.
        try {
            File configdir = new File(configpath);
            if (!configdir.exists()) {
                boolean success = configdir.mkdirs();
                if (!success) {
                    throw new IOException("Unable to access properties: mkdirs() failed: " +
                            configpath);
                }
            }

            if (!configdir.isDirectory()) {
                throw new IOException("Unable to access properties: file exists but is not a directory: " +
                        configpath);
            }
            return configdir;
        } catch (SecurityException e) {
            IOException ioe = new IOException("Unable to access updatetool preferences");
            ioe.initCause(e);
            throw ioe;
        }
    }

    /*
     * We need to determine the $HOME for the root user.  When Java is run
     * in a sudo environement on OS X the user.home does not represent root's
     * $HOME but rather the user who ran sudo.  So we need to look
     * up root's $HOME from /etc/passwd.
     */
    private static String getMacHomeDir() throws Exception {

        String homedir = null;
        String[] cmdarray = {"/usr/bin/awk", "-F:", "{ if ( $1 == \"root\" ) print $6 }", "/etc/passwd"};

        Runtime rt = Runtime.getRuntime();
        Process proc = rt.exec(cmdarray);

        int exitVal = proc.waitFor();

        // Read the output streams... (stdin)
        InputStream stdin = proc.getInputStream();
        InputStreamReader isr = new InputStreamReader(stdin);
        BufferedReader br = new BufferedReader(isr);
        String line = null;
        // There should be only one line.  
        // If there is more than one the last will have to do.
        while ((line = br.readLine()) != null) {
            homedir = line;
        }

        stdin.close();

        // Clear the output streams... (stderr)
        InputStream stderr = proc.getErrorStream();
        isr = new InputStreamReader(stderr);
        br = new BufferedReader(isr);
        while ((line = br.readLine()) != null) {
        } // We ignore the output.

        stderr.close();

        return homedir;
    }

    private static File getInitFile() throws IOException {
        return new File(getConfigDir(), INITFILE);
    }

    private static File getDefaultsFile() throws IOException {
        return new File(getConfigDir(), DEFAULTSFILE);
    }

    private static BufferedReader getFileReader(File f) throws IOException {
        return new BufferedReader(new InputStreamReader(new FileInputStream(f), "UTF-8"));
    }

    private static BufferedWriter getFileWriter(File f) throws IOException {
        return new BufferedWriter(new OutputStreamWriter(new FileOutputStream(f), "UTF-8"));
    }
}
