/*
 * Copyright (c) 1997, 2018 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0, which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the
 * Eclipse Public License v. 2.0 are satisfied: GNU General Public License,
 * version 2 with the GNU Classpath Exception, which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 */

package com.sun.xml.rpc.streaming;

import javax.xml.namespace.QName;

import com.sun.xml.rpc.util.xml.CDATA;

/**
 * <p> The XMLWriter interface is used to write XML documents. </p>
 *
 * <p> Concrete XMLWriters can be created using a XMLWriterFactory. </p>
 *
 * @see XMLWriterFactory
 *
 * @author JAX-RPC Development Team
 */
public interface XMLWriter {

    /**
     * Write the start tag for an element.
     */
    public void startElement(QName name);

    /**
     * Write the start tag for an element.
     */
    public void startElement(QName name, String prefix);

    /**
     * Write the start tag for an element.
     */
    public void startElement(String localName);

    /**
     * Write the start tag for an element.
     */
    public void startElement(String localName, String uri);

    /**
     * Write the start tag for an element.
     */
    public void startElement(String localName, String uri, String prefix);

    /**
     * Write an attribute of the current element.
     */
    public void writeAttribute(QName name, String value);

    /**
     * Write an attribute of the current element.
     */
    public void writeAttribute(String localName, String value);

    /**
     * Write an attribute of the current element.
     */
    public void writeAttribute(String localName, String uri, String value);

    /**
     * Write an attribute (unquoted) of the current element.
     */
    public void writeAttributeUnquoted(QName name, String value);

    /**
     * Write an attribute (unquoted) of the current element.
     */
    public void writeAttributeUnquoted(String localName, String value);

    /**
     * Write an attribute (unquoted) of the current element.
     */
    public void writeAttributeUnquoted(
        String localName,
        String uri,
        String value);

    /**
     * Write a namespace declaration of the current element.
     */
    public void writeNamespaceDeclaration(String prefix, String uri);

    /**
     * Write a namespace declaration of the current element. The prefix name
     * will be generated by the PrefixFactory currently configured for
     * this writer.
     */
    public void writeNamespaceDeclaration(String uri);

    /**
     * Write character data within an element.
     */
    public void writeChars(String chars);

    /**
     * Write character data within an element.
     */
    public void writeChars(CDATA chars);

    /**
     * Write character data within an element, skipping quoting.
     */
    public void writeCharsUnquoted(String chars);

    /**
     * Write character data within an element, skipping quoting.
     */
    public void writeCharsUnquoted(char[] buf, int offset, int len);

    /**
     * Write a comment within an element.
     */
    public void writeComment(String comment);

    /**
     * Write the end tag for the current element.
     */
    public void endElement();

    /**
     * Return the prefix factory in use by this writer.
     */
    public PrefixFactory getPrefixFactory();

    /**
     * Set the prefix factory to be used by this writer.
     */
    public void setPrefixFactory(PrefixFactory factory);

    /**
     * Return the URI for a given prefix.
     *
     * If the prefix is undeclared, return null.
     */
    public String getURI(String prefix);

    /**
     * Return a prefix for the given URI.
     *
     * <p> If no prefix for the given URI is in scope, return null. </p>
     */
    public String getPrefix(String uri);

    /**
     * Flush the writer and its underlying stream.
     */
    public void flush();

    /**
     * Close the writer and its underlying stream.
     */
    public void close();
}
