/*
 * Decompiled with CFR 0.152.
 */
package com.swirlds.common.signingtool;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.swirlds.common.CommonUtils;
import com.swirlds.common.constructable.ConstructableRegistry;
import com.swirlds.common.constructable.ConstructableRegistryException;
import com.swirlds.common.crypto.Hash;
import com.swirlds.common.crypto.SignatureType;
import com.swirlds.common.internal.SettingsCommon;
import com.swirlds.common.stream.EventStreamType;
import com.swirlds.common.stream.InvalidStreamFileException;
import com.swirlds.common.stream.LinkedObjectStreamUtilities;
import com.swirlds.common.stream.StreamType;
import com.swirlds.common.stream.StreamTypeFromJson;
import com.swirlds.common.stream.TimestampStreamFileWriter;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.security.InvalidKeyException;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.util.ArrayList;
import java.util.Arrays;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.Marker;
import org.apache.logging.log4j.MarkerManager;
import org.apache.logging.log4j.core.LoggerContext;

public class FileSignTool {
    public static final String CSV_EXTENSION = ".csv";
    public static final String ACCOUNT_BALANCE_EXTENSION = ".pb";
    public static final String SIG_FILE_NAME_END = "_sig";
    private static final Logger LOGGER = LogManager.getLogger();
    private static final Marker MARKER = MarkerManager.getMarker((String)"FILE_SIGN");
    private static final int BYTES_COUNT_IN_INT = 4;
    public static final byte TYPE_SIGNATURE = 3;
    public static final byte TYPE_FILE_HASH = 4;
    private static final String DEFAULT_LOG_CONFIG = "log4j2.xml";
    private static final int VERSION_5 = 5;
    private static final String KEYSTORE_TYPE = "pkcs12";
    private static final String RECORD_STREAM_EXTENSION = "rcd";
    public static final String STREAM_TYPE_JSON_PROPERTY = "streamTypeJson";
    public static final String LOG_CONFIG_PROPERTY = "logConfig";
    public static final String FILE_NAME_PROPERTY = "fileName";
    public static final String KEY_PROPERTY = "key";
    public static final String DEST_DIR_PROPERTY = "destDir";
    public static final String ALIAS_PROPERTY = "alias";
    public static final String PASSWORD_PROPERTY = "password";
    public static final String DIR_PROPERTY = "dir";

    public static byte[] sign(byte[] data, KeyPair sigKeyPair) throws NoSuchAlgorithmException, NoSuchProviderException, InvalidKeyException, SignatureException {
        Signature signature = Signature.getInstance(SignatureType.RSA.signingAlgorithm(), SignatureType.RSA.provider());
        signature.initSign(sigKeyPair.getPrivate());
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug(MARKER, "data is being signed, publicKey={}", (Object)CommonUtils.hex(sigKeyPair.getPublic().getEncoded()));
        }
        signature.update(data);
        return signature.sign();
    }

    public static boolean verifySignature(byte[] data, byte[] signature, PublicKey publicKey, String sigFilePath) {
        try {
            Signature sig = Signature.getInstance(SignatureType.RSA.signingAlgorithm(), SignatureType.RSA.provider());
            sig.initVerify(publicKey);
            sig.update(data);
            return sig.verify(signature);
        }
        catch (InvalidKeyException | NoSuchAlgorithmException | NoSuchProviderException | SignatureException e) {
            LOGGER.error(MARKER, "Failed to verify Signature: {}, PublicKey: {}, File: {}", (Object)CommonUtils.hex(signature), (Object)CommonUtils.hex(publicKey.getEncoded()), (Object)sigFilePath, (Object)e);
            return false;
        }
    }

    public static KeyPair loadPfxKey(String keyFileName, String password, String alias) {
        KeyPair sigKeyPair = null;
        try (FileInputStream fis = new FileInputStream(keyFileName);){
            KeyStore keyStore = KeyStore.getInstance(KEYSTORE_TYPE);
            keyStore.load(fis, password.toCharArray());
            sigKeyPair = new KeyPair(keyStore.getCertificate(alias).getPublicKey(), (PrivateKey)keyStore.getKey(alias, password.toCharArray()));
            LOGGER.info(MARKER, "keypair has loaded successfully from file {}", (Object)keyFileName);
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableKeyException | CertificateException e) {
            LOGGER.error(MARKER, "loadPfxKey :: ERROR ", (Throwable)e);
        }
        return sigKeyPair;
    }

    public static String buildDestSigFilePath(File destDir, File streamFile) {
        String sigFileName = streamFile.getName() + SIG_FILE_NAME_END;
        return new File(destDir, sigFileName).getPath();
    }

    public static void generateSigFileOldVersion(String filePath, byte[] signature, byte[] fileHash) {
        try (FileOutputStream output = new FileOutputStream(filePath, false);){
            output.write(4);
            output.write(fileHash);
            output.write(3);
            output.write(FileSignTool.integerToBytes(signature.length));
            output.write(signature);
        }
        catch (IOException e) {
            LOGGER.error(MARKER, "generateSigFile :: Fail to generate signature file for {}. Exception: {}", (Object)filePath, (Object)e);
        }
        System.out.println("generate sig file: " + filePath);
    }

    public static byte[] integerToBytes(int number) {
        ByteBuffer b = ByteBuffer.allocate(4);
        b.putInt(number);
        return b.array();
    }

    public static void signSingleFile(KeyPair sigKeyPair, File streamFile, File destDir, StreamType streamType) {
        String destSigFilePath = FileSignTool.buildDestSigFilePath(destDir, streamFile);
        try {
            if (streamType.isStreamFile(streamFile)) {
                int version = LinkedObjectStreamUtilities.readFirstIntFromFile(streamFile);
                if (version != 5) {
                    LOGGER.error(MARKER, "Failed to sign file {} with unsupported version {} ", (Object)streamFile.getName(), (Object)version);
                    return;
                }
                Hash entireHash = LinkedObjectStreamUtilities.computeEntireHash(streamFile);
                Hash metaHash = LinkedObjectStreamUtilities.computeMetaHash(streamFile, streamType);
                com.swirlds.common.crypto.Signature entireSignature = new com.swirlds.common.crypto.Signature(SignatureType.RSA, FileSignTool.sign(entireHash.getValue(), sigKeyPair));
                com.swirlds.common.crypto.Signature metaSignature = new com.swirlds.common.crypto.Signature(SignatureType.RSA, FileSignTool.sign(metaHash.getValue(), sigKeyPair));
                TimestampStreamFileWriter.writeSignatureFile(entireHash, entireSignature, metaHash, metaSignature, destSigFilePath, streamType);
            } else {
                FileSignTool.signSingleFileOldVersion(sigKeyPair, streamFile, destSigFilePath);
            }
        }
        catch (InvalidStreamFileException | IOException | InvalidKeyException | NoSuchAlgorithmException | NoSuchProviderException | SignatureException e) {
            LOGGER.error(MARKER, "Failed to sign file {} ", (Object)streamFile.getName(), (Object)e);
        }
        LOGGER.info(MARKER, "Finish generating signature file {}", (Object)destSigFilePath);
    }

    public static void signSingleFileOldVersion(KeyPair sigKeyPair, File streamFile, String destSigFilePath) throws IOException, NoSuchAlgorithmException, NoSuchProviderException, InvalidKeyException, SignatureException {
        byte[] fileHash = LinkedObjectStreamUtilities.computeEntireHash(streamFile).getValue();
        byte[] signature = FileSignTool.sign(fileHash, sigKeyPair);
        FileSignTool.generateSigFileOldVersion(destSigFilePath, signature, fileHash);
    }

    public static StreamType loadStreamTypeFromJson(String jsonPath) throws IOException {
        ObjectMapper objectMapper = new ObjectMapper();
        File file = new File(jsonPath);
        return (StreamType)objectMapper.readValue(file, StreamTypeFromJson.class);
    }

    public static void prepare(StreamType streamType) throws ConstructableRegistryException {
        ConstructableRegistry.registerConstructables("com.swirlds.common");
        if (streamType.getExtension().equalsIgnoreCase(RECORD_STREAM_EXTENSION)) {
            LOGGER.info(MARKER, "registering Constructables for parsing record stream files");
            ConstructableRegistry.registerConstructables("com.hedera.services.stream");
        }
        SettingsCommon.maxTransactionCountPerEvent = 245760;
        SettingsCommon.maxTransactionBytesPerEvent = 245760;
        SettingsCommon.transactionMaxBytes = 6144;
        SettingsCommon.maxAddressSizeAllowed = 1024;
    }

    public static void main(String[] args) {
        File logConfigFile;
        String streamTypeJsonPath = System.getProperty(STREAM_TYPE_JSON_PROPERTY);
        StreamType streamType = EventStreamType.EVENT;
        if (streamTypeJsonPath != null) {
            try {
                streamType = FileSignTool.loadStreamTypeFromJson(streamTypeJsonPath);
            }
            catch (IOException e) {
                LOGGER.error(MARKER, "fail to load StreamType from {}.", (Object)streamTypeJsonPath, (Object)e);
                return;
            }
        }
        try {
            FileSignTool.prepare(streamType);
        }
        catch (ConstructableRegistryException e) {
            LOGGER.error(MARKER, "fail to register constructables.", (Throwable)e);
            return;
        }
        String logConfigPath = System.getProperty(LOG_CONFIG_PROPERTY);
        File file = logConfigFile = logConfigPath == null ? CommonUtils.canonicalFile(".", DEFAULT_LOG_CONFIG) : new File(logConfigPath);
        if (logConfigFile.exists()) {
            LoggerContext context = (LoggerContext)LogManager.getContext((boolean)false);
            context.setConfigLocation(logConfigFile.toURI());
        }
        String fileName = System.getProperty(FILE_NAME_PROPERTY);
        String keyFileName = System.getProperty(KEY_PROPERTY);
        String destDirName = System.getProperty(DEST_DIR_PROPERTY);
        String alias = System.getProperty(ALIAS_PROPERTY);
        String password = System.getProperty(PASSWORD_PROPERTY);
        KeyPair sigKeyPair = FileSignTool.loadPfxKey(keyFileName, password, alias);
        String fileDirName = System.getProperty(DIR_PROPERTY);
        try {
            File destDir = new File(Files.createDirectories(Paths.get(destDirName, new String[0]), new FileAttribute[0]).toUri());
            if (fileDirName != null) {
                File folder = new File(fileDirName);
                StreamType finalStreamType = streamType;
                File[] streamFiles = folder.listFiles((dir, name) -> finalStreamType.isStreamFile(name));
                File[] accountBalanceFiles = folder.listFiles((dir, name) -> {
                    String lowerCaseName = name.toLowerCase();
                    return lowerCaseName.endsWith(CSV_EXTENSION) || lowerCaseName.endsWith(ACCOUNT_BALANCE_EXTENSION);
                });
                ArrayList<File> totalList = new ArrayList<File>();
                totalList.addAll(Arrays.asList(streamFiles));
                totalList.addAll(Arrays.asList(accountBalanceFiles));
                for (File item : totalList) {
                    FileSignTool.signSingleFile(sigKeyPair, item, destDir, streamType);
                }
            } else {
                FileSignTool.signSingleFile(sigKeyPair, new File(fileName), destDir, streamType);
            }
        }
        catch (IOException e) {
            LOGGER.error(MARKER, "Got IOException", (Throwable)e);
        }
    }
}

