/*
 * Decompiled with CFR 0.152.
 */
package com.swirlds.common.threading.internal;

import com.swirlds.common.threading.InterruptableRunnable;
import com.swirlds.common.threading.StoppableThreadConfiguration;
import com.swirlds.common.threading.TypedStoppableThread;
import com.swirlds.common.utility.CompareTo;
import com.swirlds.logging.LogMarker;
import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class StoppableThreadImpl<T extends InterruptableRunnable>
implements TypedStoppableThread<T> {
    private static final Logger log = LogManager.getLogger();
    private volatile boolean alive = true;
    private final boolean interruptable;
    private final boolean pausable;
    private final int joinWaitMs;
    private final Semaphore pauseSemaphore;
    private final Duration minimumPeriod;
    private Instant previousCycleStart;
    private final T work;
    private final InterruptableRunnable finalCycleWork;
    private final Thread thread;
    private final Duration hangingThreadDuration;
    private volatile boolean hanging;

    public StoppableThreadImpl(StoppableThreadConfiguration<T> configuration) {
        this.interruptable = configuration.isInterruptable();
        this.pausable = configuration.isPausable();
        this.pauseSemaphore = this.pausable ? new Semaphore(1, true) : null;
        this.joinWaitMs = configuration.getJoinWaitMs();
        this.hangingThreadDuration = configuration.getHangingThreadPeriod();
        this.work = configuration.getWork();
        this.finalCycleWork = configuration.getFinalCycleWork();
        this.minimumPeriod = configuration.getMinimumPeriod();
        this.thread = configuration.getThreadConfiguration().setRunnable(this::run).build();
    }

    private void run() {
        while (this.alive) {
            try {
                if (this.pausable) {
                    this.pauseSemaphore.acquire();
                }
                this.enforceMinimumPeriod();
                this.doWork();
                if (!this.pausable) continue;
                this.pauseSemaphore.release();
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
    }

    private void enforceMinimumPeriod() throws InterruptedException {
        if (this.minimumPeriod == null) {
            return;
        }
        Instant now = Instant.now();
        if (this.previousCycleStart == null) {
            this.previousCycleStart = now;
            return;
        }
        Duration previousDuration = Duration.between(this.previousCycleStart, now);
        Duration remainingCycleTime = this.minimumPeriod.minus(previousDuration);
        if (CompareTo.isGreaterThan(remainingCycleTime, Duration.ZERO)) {
            TimeUnit.NANOSECONDS.sleep(remainingCycleTime.toNanos());
            this.previousCycleStart = now.plus(remainingCycleTime);
        } else {
            this.previousCycleStart = now;
        }
    }

    @Override
    public void start() {
        this.thread.start();
    }

    @Override
    public void pause() throws InterruptedException {
        if (!this.pausable) {
            throw new IllegalStateException("this thread is not pausable");
        }
        this.pauseSemaphore.acquire();
    }

    @Override
    public void resume() {
        if (!this.pausable) {
            throw new IllegalStateException("this thread is not pausable");
        }
        this.pauseSemaphore.release();
    }

    @Override
    public void join() throws InterruptedException {
        this.thread.join();
    }

    @Override
    public void join(long millis) throws InterruptedException {
        this.thread.join(millis);
    }

    @Override
    public void join(long millis, int nanos) throws InterruptedException {
        this.thread.join(millis, nanos);
    }

    @Override
    public void stop() {
        try {
            if (this.interruptable) {
                this.interruptClose();
            } else {
                this.blockingClose();
            }
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
        }
    }

    @Override
    public void interrupt() {
        this.thread.interrupt();
    }

    @Override
    public boolean isAlive() {
        return this.thread.isAlive();
    }

    private void interruptClose() throws InterruptedException {
        if (this.alive) {
            this.alive = false;
            this.thread.join(this.joinWaitMs);
            this.thread.interrupt();
            this.joinInternal();
        }
    }

    private void blockingClose() throws InterruptedException {
        if (this.alive) {
            this.alive = false;
            this.joinInternal();
            this.doFinalCycleWork();
        }
    }

    private void joinInternal() throws InterruptedException {
        if (!this.hangingThreadDuration.isZero()) {
            this.thread.join(this.hangingThreadDuration.toMillis());
            if (this.thread.isAlive()) {
                this.logHangingThread();
                this.thread.join();
            }
        } else {
            this.thread.join();
        }
    }

    private void doWork() throws InterruptedException {
        this.work.run();
    }

    private void doFinalCycleWork() throws InterruptedException {
        if (this.finalCycleWork != null) {
            this.finalCycleWork.run();
        }
    }

    private void logHangingThread() {
        this.hanging = true;
        StringBuilder sb = new StringBuilder();
        sb.append("hanging thread detected: ").append(this.thread.getName()).append(" was requested to stop but is still alive after ").append(this.hangingThreadDuration).append("ms. Interrupt enabled = ").append(this.interruptable);
        log.error(LogMarker.EXCEPTION.getMarker(), (CharSequence)sb);
        sb = new StringBuilder("stack trace for hanging thread ").append(this.thread.getName()).append(":\n");
        for (StackTraceElement element : this.thread.getStackTrace()) {
            sb.append("   ").append(element).append("\n");
        }
        log.error(LogMarker.EXCEPTION.getMarker(), (CharSequence)sb);
    }

    @Override
    public boolean isHanging() {
        return this.thread.isAlive() && this.hanging;
    }

    @Override
    public String getName() {
        return this.thread.getName();
    }

    @Override
    public T getWork() {
        return this.work;
    }
}

