/*
 * Decompiled with CFR 0.152.
 */
package com.swirlds.common.merkle.hash;

import com.swirlds.common.crypto.Cryptography;
import com.swirlds.common.crypto.CryptographyHolder;
import com.swirlds.common.crypto.Hash;
import com.swirlds.common.crypto.SerializableHashable;
import com.swirlds.common.merkle.MerkleInternal;
import com.swirlds.common.merkle.MerkleNode;
import com.swirlds.common.merkle.crypto.MerkleCryptoFactory;
import com.swirlds.common.merkle.crypto.MerkleCryptography;
import com.swirlds.common.merkle.iterators.MerkleIterationOrder;
import com.swirlds.common.merkle.iterators.MerkleIterator;
import com.swirlds.common.merkle.utility.DebugIterationEndpoint;
import com.swirlds.common.utility.ValueReference;
import com.swirlds.logging.LogMarker;
import com.swirlds.logging.LoggingUtils;
import java.util.LinkedList;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Predicate;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public final class MerkleHashChecker {
    private static final Logger logger = LogManager.getLogger(MerkleHashChecker.class);

    private MerkleHashChecker() {
    }

    public static void findInvalidHashes(MerkleNode root, Consumer<MerkleNode> mismatchCallback) {
        if (root == null) {
            return;
        }
        root.forEachNode(node -> MerkleHashChecker.findInvalidHash(node, mismatchCallback));
    }

    private static void findInvalidHash(MerkleNode node, Consumer<MerkleNode> mismatchCallback) {
        Hash recalculated;
        MerkleCryptography cryptography = MerkleCryptoFactory.getInstance();
        if (node == null || node.isSelfHashing()) {
            return;
        }
        Hash old = node.getHash();
        if (old == null) {
            mismatchCallback.accept(node);
            return;
        }
        if (node.isLeaf()) {
            recalculated = CryptographyHolder.get().digestSync((SerializableHashable)((Object)node), Cryptography.DEFAULT_DIGEST_TYPE, false);
        } else {
            MerkleInternal internal = node.asInternal();
            for (int childIndex = 0; childIndex < internal.getNumberOfChildren(); ++childIndex) {
                Object child = internal.getChild(childIndex);
                if (child == null || child.getHash() != null) continue;
                return;
            }
            recalculated = cryptography.digestSync(internal, Cryptography.DEFAULT_DIGEST_TYPE, false);
        }
        if (!old.equals(recalculated)) {
            mismatchCallback.accept(node);
        }
    }

    public static List<MerkleNode> getNodesWithInvalidHashes(MerkleNode root) {
        LinkedList<MerkleNode> nodesWithInvalidHashes = new LinkedList<MerkleNode>();
        MerkleHashChecker.findInvalidHashes(root, nodesWithInvalidHashes::add);
        return nodesWithInvalidHashes;
    }

    public static boolean checkHashAndLog(MerkleNode root, String context, int limit) {
        List<MerkleNode> nodesWithInvalidHashes = MerkleHashChecker.getNodesWithInvalidHashes(root);
        if (nodesWithInvalidHashes.isEmpty()) {
            return true;
        }
        StringBuilder sb = new StringBuilder();
        sb.append("Invalid merkle hashes detected in tree. Context = ").append(context).append("\n");
        int nodesWithNullHash = 0;
        int nodesWithInvalidHash = 0;
        for (MerkleNode node : nodesWithInvalidHashes) {
            if (node.getHash() == null) {
                ++nodesWithNullHash;
                continue;
            }
            ++nodesWithInvalidHash;
        }
        sb.append(nodesWithNullHash).append(" ").append(LoggingUtils.plural((long)nodesWithNullHash, (String)"node")).append(" ").append(LoggingUtils.plural((long)nodesWithNullHash, (String)"has a null hash", (String)"have null hashes")).append(". ");
        sb.append(nodesWithInvalidHash).append(" ").append(LoggingUtils.plural((long)nodesWithInvalidHash, (String)"node")).append(" ").append(LoggingUtils.plural((long)nodesWithInvalidHash, (String)"has an invalid hash", (String)"have invalid hashes")).append(".\n");
        if (nodesWithInvalidHashes.size() > limit) {
            sb.append("Number of nodes exceeds maximum limit, only logging the first ").append(limit).append(" ").append(LoggingUtils.plural((long)limit, (String)"node")).append(".\n");
        }
        int count = 0;
        for (MerkleNode node : nodesWithInvalidHashes) {
            if (++count > limit) break;
            sb.append("   - ").append(node.getClass().getSimpleName()).append(" @ ").append(node.getRoute()).append(" ");
            if (node.getHash() == null) {
                sb.append("has a null hash");
            } else {
                sb.append("has an invalid hash");
            }
            sb.append("\n");
        }
        logger.error(LogMarker.EXCEPTION.getMarker(), (CharSequence)sb);
        return false;
    }

    public static String generateHashDebugString(MerkleNode root, int maxDepth) {
        StringBuilder sb = new StringBuilder();
        Predicate<MerkleInternal> filter = parent -> !parent.getClass().isAnnotationPresent(DebugIterationEndpoint.class) && parent.getRoute().size() < maxDepth;
        MerkleIterator iterator = new MerkleIterator(root).setOrder(MerkleIterationOrder.PRE_ORDERED_DEPTH_FIRST).setDescendantFilter(filter);
        iterator.forEachRemaining(node -> {
            String indexString;
            int depth = node.getRoute().size();
            for (int indent = 0; indent < depth; ++indent) {
                sb.append("   ");
            }
            if (depth == 0) {
                indexString = "(root)";
            } else {
                ValueReference<Integer> step = new ValueReference<Integer>(-1);
                node.getRoute().iterator().forEachRemaining(step::setValue);
                indexString = Integer.toString(step.getValue());
            }
            sb.append(indexString).append(" ").append(node.getClass().getSimpleName()).append(" ").append(node.getHash());
            if (iterator.hasNext()) {
                sb.append("\n");
            }
        });
        return sb.toString();
    }
}

