/*
 * Decompiled with CFR 0.152.
 */
package com.swirlds.common.threading.framework.internal;

import com.swirlds.common.threading.framework.Stoppable;
import com.swirlds.common.threading.framework.StoppableThread;
import com.swirlds.common.threading.framework.ThreadSeed;
import com.swirlds.common.threading.framework.TypedStoppableThread;
import com.swirlds.common.threading.framework.internal.AbstractStoppableThreadConfiguration;
import com.swirlds.common.threading.interrupt.InterruptableRunnable;
import com.swirlds.common.threading.interrupt.Uninterruptable;
import com.swirlds.common.utility.CompareTo;
import com.swirlds.common.utility.DurationUtils;
import com.swirlds.common.utility.StackTrace;
import com.swirlds.logging.LogMarker;
import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.util.Supplier;

public class StoppableThreadImpl<T extends InterruptableRunnable>
implements TypedStoppableThread<T> {
    private static final Logger logger = LogManager.getLogger(StoppableThreadImpl.class);
    private static final Duration MINIMUM_PAUSE_AWAIT = Duration.ofMillis(1L);
    private final AtomicReference<StoppableThread.Status> status = new AtomicReference<StoppableThread.Status>(StoppableThread.Status.NOT_STARTED);
    private final Stoppable.StopBehavior stopBehavior;
    private final int joinWaitMs;
    private final Duration logStackTracePauseDuration;
    private final AtomicReference<CountDownLatch> pauseStartedLatch = new AtomicReference<CountDownLatch>(new CountDownLatch(1));
    private final AtomicReference<CountDownLatch> pauseCompletedLatch = new AtomicReference<CountDownLatch>(new CountDownLatch(1));
    private final Duration minimumPeriod;
    private Instant previousCycleStart;
    private final T work;
    private final InterruptableRunnable finalCycleWork;
    private final AtomicReference<Thread> thread = new AtomicReference();
    private volatile boolean injected;
    private final CountDownLatch started = new CountDownLatch(1);
    private final CountDownLatch finished = new CountDownLatch(1);
    private final Duration hangingThreadDuration;
    private volatile boolean hanging;
    private final AbstractStoppableThreadConfiguration<?, T> configuration;

    public StoppableThreadImpl(AbstractStoppableThreadConfiguration<?, T> configuration) {
        this.configuration = configuration;
        this.stopBehavior = configuration.getStopBehavior();
        this.joinWaitMs = configuration.getJoinWaitMs();
        this.hangingThreadDuration = configuration.getHangingThreadPeriod();
        this.work = configuration.getWork();
        this.finalCycleWork = configuration.getFinalCycleWork();
        this.minimumPeriod = configuration.getMinimumPeriod();
        this.logStackTracePauseDuration = configuration.getLogAfterPauseDuration();
        configuration.setRunnable(this::run);
    }

    @Override
    public synchronized ThreadSeed buildSeed() {
        if (this.injected) {
            throw new IllegalStateException("this StoppableThread has already built a seed");
        }
        if (this.thread.get() != null) {
            throw new IllegalStateException("can not build seed after thread is started");
        }
        this.injected = true;
        return this.configuration.buildStoppableThreadSeed(this);
    }

    protected void setInjected() {
        this.injected = true;
    }

    protected boolean hasBeenStartedOrInjected() {
        return this.injected || this.thread.get() != null;
    }

    private void run() {
        Thread currentThread = Thread.currentThread();
        try {
            StoppableThread.Status currentStatus = this.status.get();
            while (currentStatus != StoppableThread.Status.DYING && !currentThread.isInterrupted()) {
                if (currentStatus == StoppableThread.Status.PAUSED) {
                    this.waitUntilUnpaused();
                } else {
                    this.enforceMinimumPeriod();
                    this.doWork();
                }
                currentStatus = this.status.get();
            }
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        finally {
            this.status.set(StoppableThread.Status.DEAD);
            this.finished.countDown();
        }
    }

    private void enforceMinimumPeriod() throws InterruptedException {
        if (this.minimumPeriod == null) {
            return;
        }
        Instant now = Instant.now();
        if (this.previousCycleStart == null) {
            this.previousCycleStart = now;
            return;
        }
        Duration previousDuration = Duration.between(this.previousCycleStart, now);
        Duration remainingCycleTime = this.minimumPeriod.minus(previousDuration);
        if (CompareTo.isGreaterThan(remainingCycleTime, Duration.ZERO)) {
            TimeUnit.NANOSECONDS.sleep(remainingCycleTime.toNanos());
            this.previousCycleStart = now.plus(remainingCycleTime);
        } else {
            this.previousCycleStart = now;
        }
    }

    @Override
    public synchronized void start() {
        if (this.injected) {
            throw new IllegalStateException("Thread can not be started if it has built a seed");
        }
        StoppableThread.Status originalStatus = this.status.get();
        if (originalStatus != StoppableThread.Status.NOT_STARTED) {
            throw new IllegalStateException("can not start thread " + this.getName() + " when it is in the state " + originalStatus.name());
        }
        Thread t = this.configuration.buildThread(false);
        this.markAsStarted(t);
        t.start();
    }

    private Thread uninterruptableGetThread() {
        Thread t = this.thread.get();
        while (t == null) {
            Uninterruptable.retryIfInterrupted(() -> TimeUnit.MILLISECONDS.sleep(1L));
            t = this.thread.get();
        }
        return t;
    }

    private Thread getThread() throws InterruptedException {
        Thread t = this.thread.get();
        while (t == null) {
            TimeUnit.MILLISECONDS.sleep(1L);
            t = this.thread.get();
        }
        return t;
    }

    private void waitUntilUnpaused() throws InterruptedException {
        CountDownLatch pauseCompleted = this.pauseCompletedLatch.get();
        this.pauseStartedLatch.get().countDown();
        pauseCompleted.await();
    }

    @Override
    public synchronized boolean pause() {
        StoppableThread.Status originalStatus = this.status.get();
        if (originalStatus != StoppableThread.Status.ALIVE) {
            Supplier[] supplierArray = new Supplier[2];
            supplierArray[0] = this::getName;
            supplierArray[1] = originalStatus::name;
            logger.error(LogMarker.EXCEPTION.getMarker(), "can not pause thread {} when it is in the state {}", supplierArray);
            return false;
        }
        Thread t = this.thread.get();
        this.status.set(StoppableThread.Status.PAUSED);
        Uninterruptable.retryIfInterrupted(() -> {
            while (!t.isInterrupted() && !this.waitForThreadToPause()) {
                if (!this.pauseLogStackTrace()) continue;
                Supplier[] supplierArray = new Supplier[2];
                supplierArray[0] = this::getName;
                supplierArray[1] = this.logStackTracePauseDuration::toString;
                logger.error(LogMarker.EXCEPTION.getMarker(), "pausing thread {} is taking longer than {}", supplierArray);
                logger.error(LogMarker.EXCEPTION.getMarker(), "stack trace of {}:\n{}", new Supplier[]{this::getName, () -> new StackTrace(t.getStackTrace()).toString()});
            }
        });
        return true;
    }

    private boolean waitForThreadToPause() throws InterruptedException {
        return this.pauseStartedLatch.get().await(DurationUtils.max(this.logStackTracePauseDuration, MINIMUM_PAUSE_AWAIT).toMillis(), TimeUnit.MILLISECONDS);
    }

    private boolean pauseLogStackTrace() {
        return DurationUtils.isLonger(this.logStackTracePauseDuration, Duration.ZERO);
    }

    @Override
    public synchronized boolean resume() {
        StoppableThread.Status originalStatus = this.status.get();
        if (originalStatus != StoppableThread.Status.PAUSED) {
            Supplier[] supplierArray = new Supplier[2];
            supplierArray[0] = this::getName;
            supplierArray[1] = originalStatus::name;
            logger.error(LogMarker.EXCEPTION.getMarker(), "can not resume thread {} when it is in the state {}", supplierArray);
            return false;
        }
        this.status.set(StoppableThread.Status.ALIVE);
        this.unblockPausedThread();
        return true;
    }

    private void unblockPausedThread() {
        this.pauseCompletedLatch.get().countDown();
        this.pauseStartedLatch.set(new CountDownLatch(1));
        this.pauseCompletedLatch.set(new CountDownLatch(1));
    }

    @Override
    public void join() throws InterruptedException {
        while (this.status.get() == StoppableThread.Status.NOT_STARTED) {
            TimeUnit.MILLISECONDS.sleep(1L);
        }
        if (this.injected) {
            this.finished.await();
        } else {
            this.thread.get().join();
        }
    }

    @Override
    public void join(long millis) throws InterruptedException {
        while (this.status.get() == StoppableThread.Status.NOT_STARTED) {
            TimeUnit.MILLISECONDS.sleep(1L);
        }
        if (this.injected) {
            this.finished.await(millis, TimeUnit.MILLISECONDS);
        } else {
            this.getThread().join(millis);
        }
    }

    @Override
    public void join(long millis, int nanos) throws InterruptedException {
        while (this.status.get() == StoppableThread.Status.NOT_STARTED) {
            TimeUnit.MILLISECONDS.sleep(1L);
        }
        if (this.injected) {
            this.finished.await((long)((double)millis + (double)nanos * 1.0E-6), TimeUnit.MILLISECONDS);
        } else {
            this.getThread().join(millis, nanos);
        }
    }

    @Override
    public boolean stop() {
        return this.stop(this.stopBehavior);
    }

    @Override
    public synchronized boolean stop(Stoppable.StopBehavior behavior) {
        StoppableThread.Status originalStatus = this.status.get();
        if (originalStatus != StoppableThread.Status.ALIVE && originalStatus != StoppableThread.Status.PAUSED) {
            Thread t = this.thread.get();
            String name = t == null ? "null" : t.getName();
            String message = "can not stop thread {} when it is in the state {}";
            if (originalStatus == StoppableThread.Status.DEAD) {
                Supplier[] supplierArray = new Supplier[2];
                supplierArray[0] = () -> name;
                supplierArray[1] = originalStatus::name;
                logger.warn(LogMarker.THREADS.getMarker(), "can not stop thread {} when it is in the state {}", supplierArray);
            } else {
                Supplier[] supplierArray = new Supplier[2];
                supplierArray[0] = () -> name;
                supplierArray[1] = originalStatus::name;
                logger.error(LogMarker.EXCEPTION.getMarker(), "can not stop thread {} when it is in the state {}", supplierArray);
            }
            return false;
        }
        this.status.set(StoppableThread.Status.DYING);
        if (originalStatus == StoppableThread.Status.PAUSED) {
            this.unblockPausedThread();
        }
        try {
            this.close(behavior);
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
        }
        return true;
    }

    @Override
    public boolean interrupt() {
        Thread t = this.thread.get();
        if (t == null) {
            return false;
        }
        t.interrupt();
        return true;
    }

    @Override
    public boolean isAlive() {
        return this.status.get() != StoppableThread.Status.DEAD;
    }

    @Override
    public StoppableThread.Status getStatus() {
        return this.status.get();
    }

    protected void markAsStarted(Thread thread) {
        this.thread.set(thread);
        this.status.set(StoppableThread.Status.ALIVE);
        this.started.countDown();
    }

    private void close(Stoppable.StopBehavior stopBehavior) throws InterruptedException {
        if (stopBehavior == Stoppable.StopBehavior.BLOCKING) {
            this.blockingClose();
        } else if (stopBehavior == Stoppable.StopBehavior.INTERRUPTABLE) {
            this.interruptClose();
        } else {
            throw new IllegalArgumentException();
        }
    }

    private void interruptClose() throws InterruptedException {
        this.join(this.joinWaitMs);
        if (this.isAlive()) {
            this.interrupt();
            this.joinInternal();
        }
    }

    private void blockingClose() throws InterruptedException {
        this.joinInternal();
        this.doFinalCycleWork();
    }

    private void joinInternal() throws InterruptedException {
        if (!this.hangingThreadDuration.isZero()) {
            this.join(this.hangingThreadDuration.toMillis());
            if (this.isAlive()) {
                this.logHangingThread();
                this.join();
            }
        } else {
            this.join();
        }
    }

    private void doWork() throws InterruptedException {
        this.work.run();
    }

    private void doFinalCycleWork() throws InterruptedException {
        if (this.finalCycleWork != null) {
            this.finalCycleWork.run();
        }
    }

    private void logHangingThread() {
        this.hanging = true;
        StringBuilder sb = new StringBuilder();
        sb.append("hanging thread detected: ").append(this.getName()).append(" was requested to stop but is still alive after ").append(this.hangingThreadDuration).append("ms. Stop behavior = ").append(this.stopBehavior.toString());
        logger.error(LogMarker.EXCEPTION.getMarker(), (CharSequence)sb);
        sb = new StringBuilder("stack trace for hanging thread ").append(this.getName()).append(":\n").append(StackTrace.getStackTrace(this.uninterruptableGetThread()));
        logger.error(LogMarker.EXCEPTION.getMarker(), (CharSequence)sb);
    }

    @Override
    public boolean isHanging() {
        return this.uninterruptableGetThread().isAlive() && this.hanging;
    }

    @Override
    public String getName() {
        return this.configuration.getThreadName();
    }

    @Override
    public T getWork() {
        return this.work;
    }

    public String toString() {
        return "StoppableThread(" + this.getName() + ")";
    }
}

