/*
 * Decompiled with CFR 0.152.
 */
package com.swirlds.common.scratchpad.internal;

import com.swirlds.common.config.StateConfig;
import com.swirlds.common.context.PlatformContext;
import com.swirlds.common.formatting.TextTable;
import com.swirlds.common.io.SelfSerializable;
import com.swirlds.common.io.streams.SerializableDataInputStream;
import com.swirlds.common.io.streams.SerializableDataOutputStream;
import com.swirlds.common.io.utility.FileUtils;
import com.swirlds.common.io.utility.TemporaryFileBuilder;
import com.swirlds.common.scratchpad.Scratchpad;
import com.swirlds.common.scratchpad.ScratchpadType;
import com.swirlds.common.system.NodeId;
import com.swirlds.common.threading.locks.AutoClosableLock;
import com.swirlds.common.threading.locks.Locks;
import com.swirlds.common.threading.locks.locked.Locked;
import com.swirlds.logging.LogMarker;
import edu.umd.cs.findbugs.annotations.NonNull;
import edu.umd.cs.findbugs.annotations.Nullable;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class StandardScratchpad<K extends Enum<K>>
implements Scratchpad<K> {
    private static final Logger logger = LogManager.getLogger(StandardScratchpad.class);
    public static final String SCRATCHPAD_DIRECTORY_NAME = "scratchpad";
    public static final String SCRATCHPAD_FILE_EXTENSION = ".scratchpad";
    private final Set<K> fields;
    private final String id;
    private final Map<K, SelfSerializable> data = new HashMap<K, SelfSerializable>();
    private final AutoClosableLock lock = Locks.createAutoLock();
    private final Path scratchpadDirectory;
    private long nextScratchpadIndex;
    private final int fileVersion = 1;

    public StandardScratchpad(@NonNull PlatformContext platformContext, @NonNull NodeId selfId, @NonNull Class<K> clazz, @NonNull String id) {
        StateConfig stateConfig = (StateConfig)platformContext.getConfiguration().getConfigData(StateConfig.class);
        this.scratchpadDirectory = stateConfig.savedStateDirectory().resolve(SCRATCHPAD_DIRECTORY_NAME).resolve(Long.toString(selfId.id())).resolve(id);
        if (id.isEmpty()) {
            throw new IllegalArgumentException("scratchpad ID must not be empty");
        }
        if (!id.matches("[a-zA-Z0-9_.-]+")) {
            throw new IllegalArgumentException("scratchpad ID must only contain alphanumeric characters, '_', '-', and '.'");
        }
        this.id = id;
        this.fields = EnumSet.allOf(clazz);
        HashMap<Integer, Enum> indexToFieldMap = new HashMap<Integer, Enum>();
        for (Enum key : this.fields) {
            Enum previous = indexToFieldMap.put(((ScratchpadType)((Object)key)).getFieldId(), key);
            if (previous == null) continue;
            throw new RuntimeException("duplicate scratchpad field ID: " + ((ScratchpadType)((Object)key)).getFieldId());
        }
        this.loadFromDisk(indexToFieldMap);
    }

    @Override
    public void logContents() {
        TextTable table = new TextTable().setBordersEnabled(false);
        try (Locked ignored = this.lock.lock();){
            for (Enum field : this.fields) {
                SelfSerializable value = this.data.get(field);
                if (value == null) {
                    table.addToRow(field.name(), "null");
                    continue;
                }
                table.addRow(field.name(), value.toString());
            }
        }
        logger.info(LogMarker.STARTUP.getMarker(), "Scratchpad {} contents:\n{}", (Object)this.id, (Object)table.render());
    }

    @Override
    @Nullable
    public <V extends SelfSerializable> V get(@NonNull K key) {
        try (Locked ignored = this.lock.lock();){
            SelfSerializable selfSerializable = this.data.get(key);
            return (V)selfSerializable;
        }
    }

    @Override
    @Nullable
    public <V extends SelfSerializable> V set(@NonNull K key, @Nullable V value) {
        logger.info(LogMarker.STARTUP.getMarker(), "Setting scratchpad field {}:{} to {}", (Object)this.id, key, value);
        try (Locked ignored = this.lock.lock();){
            SelfSerializable previous = this.data.put(key, value);
            this.flush();
            SelfSerializable selfSerializable = previous;
            return (V)selfSerializable;
        }
    }

    @Override
    public void atomicOperation(@NonNull Consumer<Map<K, SelfSerializable>> operation) {
        try (Locked ignored = this.lock.lock();){
            operation.accept(this.data);
            this.flush();
        }
    }

    @Override
    public void atomicOperation(@NonNull Function<Map<K, SelfSerializable>, Boolean> operation) {
        try (Locked ignored = this.lock.lock();){
            boolean modified = operation.apply(this.data);
            if (modified) {
                this.flush();
            }
        }
    }

    @Override
    public void clear() {
        logger.info(LogMarker.STARTUP.getMarker(), "Clearing scratchpad {}", (Object)this.id);
        try (Locked ignored = this.lock.lock();){
            this.data.clear();
            FileUtils.deleteDirectory(this.scratchpadDirectory);
        }
        catch (IOException e) {
            throw new UncheckedIOException("unable to clear scratchpad", e);
        }
    }

    private void loadFromDisk(Map<Integer, K> indexToFieldMap) {
        try {
            List<Path> files = this.getScratchpadFiles();
            if (files.isEmpty()) {
                return;
            }
            for (int index = 0; index < files.size() - 1; ++index) {
                Files.delete(files.get(index));
            }
            Path scratchpadFile = files.get(files.size() - 1);
            this.nextScratchpadIndex = this.getFileIndex(scratchpadFile) + 1L;
            try (SerializableDataInputStream in = new SerializableDataInputStream(new BufferedInputStream(new FileInputStream(scratchpadFile.toFile())));){
                int fileVersion = in.readInt();
                if (fileVersion != this.fileVersion) {
                    throw new RuntimeException("scratchpad file version mismatch");
                }
                int fieldCount = in.readInt();
                for (int index = 0; index < fieldCount; ++index) {
                    int fieldId = in.readInt();
                    Enum key = (Enum)indexToFieldMap.get(fieldId);
                    if (key == null) {
                        throw new IOException("scratchpad file contains unknown field " + fieldId);
                    }
                    Object value = in.readSerializable();
                    this.data.put(key, (SelfSerializable)value);
                }
            }
        }
        catch (IOException e) {
            throw new RuntimeException("unable to load scratchpad", e);
        }
    }

    @NonNull
    private Path flushToTemporaryFile() throws IOException {
        Path temporaryFile = TemporaryFileBuilder.buildTemporaryFile();
        try (SerializableDataOutputStream out = new SerializableDataOutputStream(new BufferedOutputStream(new FileOutputStream(temporaryFile.toFile(), false)));){
            out.writeInt(1);
            int fieldCount = 0;
            for (Enum keys : this.fields) {
                if (this.data.get(keys) == null) continue;
                ++fieldCount;
            }
            out.writeInt(fieldCount);
            for (Enum key : this.fields) {
                SelfSerializable value = this.data.get(key);
                if (value == null) continue;
                out.writeInt(((ScratchpadType)((Object)key)).getFieldId());
                out.writeSerializable(value, true);
            }
        }
        return temporaryFile;
    }

    @NonNull
    private Path generateNextFilePath() {
        return this.scratchpadDirectory.resolve(this.nextScratchpadIndex++ + SCRATCHPAD_FILE_EXTENSION);
    }

    private long getFileIndex(@NonNull Path path) {
        String fileName = path.getFileName().toString();
        return Long.parseLong(fileName.substring(0, fileName.indexOf(46)));
    }

    @NonNull
    private List<Path> getScratchpadFiles() {
        if (!Files.exists(this.scratchpadDirectory, new LinkOption[0])) {
            return List.of();
        }
        ArrayList<Path> files = new ArrayList<Path>();
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(this.scratchpadDirectory);){
            for (Path path : stream) {
                if (!path.toString().endsWith(SCRATCHPAD_FILE_EXTENSION)) continue;
                files.add(path);
            }
        }
        catch (IOException e) {
            throw new RuntimeException("unable to list scratchpad files", e);
        }
        files.sort((a, b) -> Long.compare(this.getFileIndex((Path)a), this.getFileIndex((Path)b)));
        return files;
    }

    private void flush() {
        try {
            List<Path> scratchpadFiles = this.getScratchpadFiles();
            Path temporaryFile = this.flushToTemporaryFile();
            if (!Files.exists(this.scratchpadDirectory, new LinkOption[0])) {
                Files.createDirectories(this.scratchpadDirectory, new FileAttribute[0]);
            }
            Files.move(temporaryFile, this.generateNextFilePath(), StandardCopyOption.ATOMIC_MOVE);
            for (Path scratchpadFile : scratchpadFiles) {
                Files.delete(scratchpadFile);
            }
        }
        catch (IOException e) {
            throw new RuntimeException("unable to flush scratchpad", e);
        }
    }
}

