/*
 * Decompiled with CFR 0.152.
 */
package com.swirlds.virtualmap.internal.pipeline;

import com.swirlds.common.threading.ThreadConfiguration;
import com.swirlds.common.utility.CompareTo;
import com.swirlds.logging.LogMarker;
import com.swirlds.virtualmap.VirtualMapSettingsFactory;
import com.swirlds.virtualmap.internal.pipeline.PipelineList;
import com.swirlds.virtualmap.internal.pipeline.PipelineListNode;
import com.swirlds.virtualmap.internal.pipeline.VirtualRoot;
import java.nio.file.Path;
import java.time.Duration;
import java.util.Iterator;
import java.util.Objects;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class VirtualPipeline {
    private static final String PIPELINE_COMPONENT = "virtual-pipeline";
    private static final String PIPELINE_THREAD_NAME = "lifecycle";
    private static final Logger LOG = LogManager.getLogger(VirtualPipeline.class);
    private final PipelineList<VirtualRoot> copies;
    private final AtomicInteger unreleasedCopies = new AtomicInteger();
    private final ConcurrentLinkedDeque<VirtualRoot> unhashedCopies;
    private final AtomicReference<VirtualRoot> mostRecentCopy = new AtomicReference();
    private volatile boolean alive = true;
    private final ExecutorService executorService;
    private final AtomicInteger flushBacklog = new AtomicInteger(0);
    private final Lock hashLock;

    public VirtualPipeline() {
        this.copies = new PipelineList();
        this.unhashedCopies = new ConcurrentLinkedDeque();
        this.hashLock = new ReentrantLock();
        this.executorService = Executors.newSingleThreadExecutor(new ThreadConfiguration().setComponent(PIPELINE_COMPONENT).setThreadName(PIPELINE_THREAD_NAME).buildFactory());
    }

    private void validatePipelineRegistration(VirtualRoot copy) {
        if (!copy.isRegisteredToPipeline(this)) {
            throw new IllegalStateException("copy is not registered with this pipeline");
        }
    }

    public int getFlushBacklogSize() {
        return this.flushBacklog.get();
    }

    private void applyFlushBackpressure() {
        int backlogExcess = this.flushBacklog.get() - VirtualMapSettingsFactory.get().getPreferredFlushQueueSize();
        if (backlogExcess <= 0) {
            return;
        }
        Duration computedSleepTime = VirtualMapSettingsFactory.get().getFlushThrottleStepSize().multipliedBy((long)backlogExcess * (long)backlogExcess);
        Duration maxSleepTime = VirtualMapSettingsFactory.get().getMaximumFlushThrottlePeriod();
        Duration sleepTime = (Duration)CompareTo.min((Comparable)computedSleepTime, (Comparable)maxSleepTime);
        try {
            TimeUnit.MILLISECONDS.sleep(sleepTime.toMillis());
        }
        catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void registerCopy(VirtualRoot copy) {
        Objects.requireNonNull(copy);
        if (copy.isImmutable()) {
            throw new IllegalStateException("Only mutable copies may be registered");
        }
        if (copy.shouldBeFlushed()) {
            this.flushBacklog.getAndIncrement();
        }
        this.unreleasedCopies.getAndIncrement();
        this.copies.add(copy);
        this.unhashedCopies.add(copy);
        this.mostRecentCopy.set(copy);
        VirtualPipeline virtualPipeline = this;
        synchronized (virtualPipeline) {
            if (this.alive) {
                this.executorService.submit(this::doWork);
            }
        }
        this.applyFlushBackpressure();
    }

    public synchronized void terminate() {
        if (!this.alive) {
            return;
        }
        this.pausePipelineAndExecute("terminate", () -> this.shutdown(false));
    }

    public synchronized void releaseCopy() {
        if (!this.alive) {
            return;
        }
        int remainingCopies = this.unreleasedCopies.decrementAndGet();
        if (remainingCopies < 0) {
            throw new IllegalStateException("copies released too many times");
        }
        if (remainingCopies == 0) {
            this.shutdown(true);
        } else {
            this.executorService.submit(this::doWork);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void hashCopy(VirtualRoot copy) {
        this.validatePipelineRegistration(copy);
        this.hashLock.lock();
        try {
            if (copy.isHashed()) {
                return;
            }
            Iterator<VirtualRoot> iterator = this.unhashedCopies.iterator();
            while (iterator.hasNext()) {
                VirtualRoot unhashedCopy = iterator.next();
                iterator.remove();
                unhashedCopy.computeHash();
                if (unhashedCopy != copy) continue;
                break;
            }
            if (!copy.isHashed()) {
                throw new IllegalStateException("failed to hash copy");
            }
        }
        finally {
            this.hashLock.unlock();
        }
    }

    public <T> T detachCopy(VirtualRoot copy, boolean withDbCompactionEnabled) {
        return this.detachCopy(copy, null, null, true, withDbCompactionEnabled);
    }

    public <T> T detachCopy(VirtualRoot copy, String label, Path targetDirectory, boolean reopen, boolean withDbCompactionEnabled) {
        this.validatePipelineRegistration(copy);
        AtomicReference ret = new AtomicReference();
        this.pausePipelineAndExecute("detach", () -> ret.set(copy.detach(label, targetDirectory, reopen, withDbCompactionEnabled)));
        if (this.alive) {
            this.executorService.submit(this::doWork);
        }
        return (T)ret.get();
    }

    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
        return this.executorService.awaitTermination(timeout, unit);
    }

    private static boolean shouldFlush(VirtualRoot copy) {
        return copy.shouldBeFlushed() && copy.isImmutable() && !copy.isFlushed();
    }

    private void flush(VirtualRoot copy) {
        if (copy.isFlushed()) {
            throw new IllegalStateException("copy is already flushed");
        }
        if (!copy.isHashed()) {
            this.hashCopy(copy);
        }
        copy.flush();
        this.flushBacklog.getAndDecrement();
    }

    private static boolean shouldMerge(PipelineListNode<VirtualRoot> mergeCandidate) {
        VirtualRoot copy = mergeCandidate.getValue();
        PipelineListNode<VirtualRoot> mergeTarget = mergeCandidate.getNext();
        return copy.shouldBeMerged() && (copy.isReleased() || copy.isDetached()) && !copy.isMerged() && mergeTarget != null && mergeTarget.getValue().isImmutable();
    }

    private void merge(PipelineListNode<VirtualRoot> node) {
        VirtualRoot next;
        VirtualRoot copy = node.getValue();
        if (copy.isMerged()) {
            throw new IllegalStateException("copy is already merged");
        }
        if (!copy.isHashed()) {
            this.hashCopy(copy);
        }
        if (!(next = node.getNext().getValue()).isHashed()) {
            this.hashCopy(next);
        }
        copy.merge();
    }

    private static boolean shouldBeRemovedFromPipeline(VirtualRoot copy) {
        return copy.isReleased() && (copy.isFlushed() || copy.isMerged());
    }

    private static boolean shouldBlockFlushes(VirtualRoot copy) {
        return !copy.isReleased() && !copy.isDetached() || copy.shouldBeMerged() && !copy.isMerged() || copy.shouldBeFlushed() && !copy.isFlushed();
    }

    private void hashFlushMerge() {
        boolean flushBlocked = false;
        for (PipelineListNode<VirtualRoot> next = this.copies.getFirst(); next != null; next = next.getNext()) {
            VirtualRoot copy = next.getValue();
            if (VirtualPipeline.shouldFlush(copy)) {
                if (!flushBlocked) {
                    this.flush(copy);
                }
            } else if (VirtualPipeline.shouldMerge(next)) {
                this.merge(next);
            }
            if (VirtualPipeline.shouldBeRemovedFromPipeline(copy)) {
                this.copies.remove(next);
            }
            flushBlocked |= VirtualPipeline.shouldBlockFlushes(copy);
        }
    }

    private void doWork() {
        try {
            this.hashFlushMerge();
        }
        catch (Exception e) {
            LOG.error(LogMarker.EXCEPTION.getMarker(), "exception on virtual pipeline thread", (Throwable)e);
            this.shutdown(true);
        }
    }

    private synchronized void shutdown(boolean immediately) {
        this.alive = false;
        if (!this.executorService.isShutdown()) {
            if (immediately) {
                this.executorService.shutdownNow();
                this.fireOnShutdown(immediately);
            } else {
                this.executorService.submit(() -> this.fireOnShutdown(false));
                this.executorService.shutdown();
            }
        }
    }

    private void pausePipelineAndExecute(String label, Runnable runnable) {
        Objects.requireNonNull(runnable);
        CountDownLatch waitForBackgroundThreadToStart = new CountDownLatch(1);
        CountDownLatch waitForRunnableToFinish = new CountDownLatch(1);
        this.executorService.execute(() -> {
            waitForBackgroundThreadToStart.countDown();
            try {
                waitForRunnableToFinish.await();
            }
            catch (InterruptedException ie) {
                Thread.currentThread().interrupt();
                throw new RuntimeException("Fatal error: interrupted while waiting for runnable " + label + " to finish");
            }
        });
        try {
            waitForBackgroundThreadToStart.await();
        }
        catch (InterruptedException ie) {
            Thread.currentThread().interrupt();
            throw new RuntimeException("Fatal error: failed to start " + label);
        }
        runnable.run();
        waitForRunnableToFinish.countDown();
    }

    public boolean isTerminated() {
        return !this.alive;
    }

    private void fireOnShutdown(boolean immediately) {
        VirtualRoot copy = this.mostRecentCopy.get();
        if (copy != null) {
            copy.onShutdown(immediately);
        }
    }

    private static String uppercaseBoolean(boolean value) {
        return value ? "TRUE" : "FALSE";
    }

    public void logDebugInfo() {
        StringBuilder sb = new StringBuilder();
        sb.append("Virtual pipeline dump, ");
        sb.append("  size = ").append(this.copies.getSize()).append("\n");
        sb.append("Copies listed oldest to newest:\n");
        int index = 0;
        for (PipelineListNode<VirtualRoot> next = this.copies.getFirst(); next != null; next = next.getNext()) {
            VirtualRoot copy = next.getValue();
            sb.append(index).append(" should be flushed = ").append(VirtualPipeline.uppercaseBoolean(copy.shouldBeFlushed()));
            sb.append(", ready to be flushed = ").append(VirtualPipeline.uppercaseBoolean(VirtualPipeline.shouldFlush(copy)));
            sb.append(", ready to be merged = ").append(VirtualPipeline.uppercaseBoolean(VirtualPipeline.shouldMerge(next)));
            sb.append(", flushed = ").append(VirtualPipeline.uppercaseBoolean(copy.isFlushed()));
            sb.append(", released = ").append(VirtualPipeline.uppercaseBoolean(copy.isReleased()));
            sb.append(", hashed = ").append(VirtualPipeline.uppercaseBoolean(copy.isHashed()));
            sb.append(", detached = ").append(VirtualPipeline.uppercaseBoolean(copy.isDetached()));
            sb.append("\n");
            ++index;
        }
        sb.append("There is no problem if this has happened during a freeze.\n");
        LOG.info(LogMarker.VIRTUAL_MERKLE_STATS.getMarker(), "{}", (Object)sb);
    }
}

