package com.teambytes.cloudwatch.appenders.log4j

import java.util.Date
import java.util.concurrent.atomic.{AtomicBoolean, AtomicReference}

import com.amazonaws.services.logs.AWSLogsAsyncClient
import com.amazonaws.services.logs.model._
import com.fasterxml.jackson.databind.{SerializationFeature, ObjectMapper}
import org.apache.log4j.AppenderSkeleton
import org.apache.log4j.spi.LoggingEvent

import scala.collection.JavaConverters._
import scala.beans.BeanProperty

class CloudwatchAppender extends AppenderSkeleton {

  private val awsClient = new AWSLogsAsyncClient()
  private val om = {
    val omap = new ObjectMapper()
    omap.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false)
    omap
  }
  @BeanProperty
  var logGroupName: String = null
  @BeanProperty
  var logStreamName: String = null
  private val activated = new AtomicBoolean(false)
  private val lastSequenceToken = new AtomicReference[String](null)

  override def activateOptions(): Unit = {
    if (getLayout == null) {
      println("Layout was not defined, will only log the message, no stack traces or custom layout")
    }

    val createLogGroupRequest = new CreateLogGroupRequest(logGroupName)
    try {
      awsClient.createLogGroup(createLogGroupRequest)
    } catch {
      case e: ResourceAlreadyExistsException =>
        println(s"Log group '$logGroupName' already exists")
    }

    val createLogStreamRequest = new CreateLogStreamRequest(logGroupName, logStreamName)
    try {
      awsClient.createLogStream(createLogStreamRequest)
    } catch {
      case e: ResourceAlreadyExistsException =>
        println(s"Log stream '$logStreamName' in group '$logGroupName' already exists")
    }

    try {
      sendEvent("Getting the next expected sequenceToken for CloudwatchAppender")
    } catch {
      case e: InvalidSequenceTokenException =>
        lastSequenceToken.set(e.getExpectedSequenceToken)
    }

    super.activateOptions()
    activated.set(true)
  }

  def append(event: LoggingEvent): Unit = {
    if(!activated.get()) {
      activateOptions()
    }
    try {
      sendEvent(om.writeValueAsString(loggingEvent2Map(event)))
    } catch {
      case e: Exception =>
        e.printStackTrace()
    }
  }

  private def sendEvent(message: String) = {
    val logEvents = Seq(new InputLogEvent().
      withTimestamp(new Date().getTime).
      withMessage(message))
    val putLogEventsRequest = new PutLogEventsRequest(logGroupName, logStreamName, logEvents.asJava)
    putLogEventsRequest.setSequenceToken(lastSequenceToken.get())
    val result = awsClient.putLogEvents(putLogEventsRequest)
    lastSequenceToken.set(result.getNextSequenceToken)
  }

  private def loggingEvent2Map(event: LoggingEvent) = {
    val formattedEvent = layout.format( event)

    Map(
      "threadName" -> event.getThreadName,
      "level" -> event.getLevel,
      "message" -> event.getMessage,
      "formattedMessage" -> formattedEvent,
      "loggerName" -> event.getLoggerName,
      "timeStamp" -> event.getTimeStamp
    )
  }

  def requiresLayout(): Boolean = true

  def close(): Unit = {

  }
}
