/*
 * Decompiled with CFR 0.152.
 */
package org.conqat.lib.commons.filesystem;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.EOFException;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.IllegalFormatException;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.JarOutputStream;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.zip.GZIPInputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.conqat.lib.commons.assertion.CCSMAssert;
import org.conqat.lib.commons.collections.CaseInsensitiveStringSet;
import org.conqat.lib.commons.collections.CollectionUtils;
import org.conqat.lib.commons.filesystem.EByteOrderMark;
import org.conqat.lib.commons.filesystem.FilenameComparator;
import org.conqat.lib.commons.filesystem.PathBasedContentProviderBase;
import org.conqat.lib.commons.filesystem.ZipFile;
import org.conqat.lib.commons.resources.Resource;
import org.conqat.lib.commons.string.StringUtils;

public class FileSystemUtils {
    public static final String UTF8_ENCODING = StandardCharsets.UTF_8.name();
    public static final String TEMP_DIR_PATH = System.getProperty("java.io.tmpdir");
    public static final char UNIX_SEPARATOR = '/';
    public static final char WINDOWS_SEPARATOR = '\\';
    public static final String METRIC_SYSTEM_UNITS = "KMGTPEZY";
    private static final Pattern DATA_SIZE_UNIT_START_PATTERN = Pattern.compile("[^\\d\\s.,]");
    private static final Set<String> RESERVED_PATH_SEGMENT_NAMES = new CaseInsensitiveStringSet(Arrays.asList("CON", "PRN", "AUX", "CLOCK$", "NUL", "COM1", "COM2", "COM3", "COM4", "COM5", "COM6", "COM7", "COM8", "COM9", "LPT1", "LPT2", "LPT3", "LPT4", "LPT5", "LPT6", "LPT7", "LPT8", "LPT9"));

    public static int copy(InputStream input, OutputStream output) throws IOException {
        int len;
        byte[] buffer = new byte[1024];
        int size = 0;
        while ((len = input.read(buffer)) > 0) {
            output.write(buffer, 0, len);
            size += len;
        }
        return size;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void copyFile(File sourceFile, File targetFile) throws IOException {
        if (sourceFile.getAbsoluteFile().equals(targetFile.getAbsoluteFile())) {
            throw new IOException("Can not copy file onto itself: " + sourceFile);
        }
        FileSystemUtils.ensureParentDirectoryExists(targetFile);
        FileChannel sourceChannel = null;
        FileChannel targetChannel = null;
        try {
            sourceChannel = new FileInputStream(sourceFile).getChannel();
            targetChannel = new FileOutputStream(targetFile).getChannel();
            sourceChannel.transferTo(0L, sourceChannel.size(), targetChannel);
        }
        catch (Throwable throwable) {
            FileSystemUtils.close(sourceChannel);
            FileSystemUtils.close(targetChannel);
            throw throwable;
        }
        FileSystemUtils.close(sourceChannel);
        FileSystemUtils.close(targetChannel);
    }

    public static void copyFile(String sourceFilename, String targetFilename) throws IOException {
        FileSystemUtils.copyFile(new File(sourceFilename), new File(targetFilename));
    }

    public static int copyFiles(File sourceDirectory, File targetDirectory, FileFilter fileFilter) throws IOException {
        List<File> files = FileSystemUtils.listFilesRecursively(sourceDirectory, fileFilter);
        int fileCount = 0;
        for (File sourceFile : files) {
            if (!sourceFile.isFile()) continue;
            String path = sourceFile.getAbsolutePath();
            int index = sourceDirectory.getAbsolutePath().length();
            String newPath = path.substring(index);
            File targetFile = new File(targetDirectory, newPath);
            FileSystemUtils.copyFile(sourceFile, targetFile);
            ++fileCount;
        }
        return fileCount;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static int createJARFile(File jarFile, File sourceDirectory, FileFilter filter) throws IOException {
        JarOutputStream out = null;
        int fileCount = 0;
        try {
            out = new JarOutputStream(new FileOutputStream(jarFile));
            for (File file : FileSystemUtils.listFilesRecursively(sourceDirectory, filter)) {
                if (!file.isFile()) continue;
                FileInputStream in = null;
                ++fileCount;
                try {
                    String entryName = FileSystemUtils.normalizeSeparators(file.getAbsolutePath().substring(sourceDirectory.getAbsolutePath().length() + 1));
                    out.putNextEntry(new ZipEntry(entryName));
                    in = new FileInputStream(file);
                    FileSystemUtils.copy(in, out);
                    out.closeEntry();
                }
                catch (Throwable throwable) {
                    FileSystemUtils.close(in);
                    throw throwable;
                }
                FileSystemUtils.close(in);
            }
        }
        catch (Throwable throwable) {
            FileSystemUtils.close(out);
            throw throwable;
        }
        FileSystemUtils.close(out);
        return fileCount;
    }

    public static String createRelativePath(File path, File relativeTo) throws IOException {
        File root;
        CCSMAssert.isNotNull((Object)path, "Path must not be null!");
        CCSMAssert.isNotNull((Object)relativeTo, "relativeTo must not be null!");
        if (!path.isDirectory() || !relativeTo.isDirectory()) {
            throw new IllegalArgumentException("Both arguments must be existing directories!");
        }
        path = path.getCanonicalFile();
        relativeTo = relativeTo.getCanonicalFile();
        HashSet<File> parents = new HashSet<File>();
        for (File f = path; f != null; f = f.getParentFile()) {
            parents.add(f);
        }
        for (root = relativeTo; root != null && !parents.contains(root); root = root.getParentFile()) {
        }
        if (root == null) {
            return path.getAbsolutePath();
        }
        String result = "";
        while (!path.equals(root)) {
            result = path.getName() + "/" + result;
            path = path.getParentFile();
        }
        while (!relativeTo.equals(root)) {
            result = "../" + result;
            relativeTo = relativeTo.getParentFile();
        }
        return result;
    }

    public static void deleteRecursively(File directory) {
        if (directory == null) {
            throw new IllegalArgumentException("Directory may not be null.");
        }
        File[] filesInDirectory = directory.listFiles();
        if (filesInDirectory == null) {
            throw new IllegalArgumentException(directory.getAbsolutePath() + " is not a valid directory.");
        }
        for (File entry : filesInDirectory) {
            if (entry.isDirectory()) {
                FileSystemUtils.deleteRecursively(entry);
            }
            entry.delete();
        }
        directory.delete();
    }

    public static void deleteFile(File file) throws IOException {
        if (file.exists() && !file.delete()) {
            throw new IOException("Could not delete " + file);
        }
    }

    public static void renameFileTo(File file, File dest) throws IOException {
        if (!file.renameTo(dest)) {
            throw new IOException("Could not rename " + file + " to " + dest);
        }
    }

    public static void mkdir(File dir) throws IOException {
        if (!dir.mkdir()) {
            throw new IOException("Could not create directory " + dir);
        }
    }

    public static void mkdirs(File dir) throws IOException {
        if (dir.exists() && dir.isDirectory()) {
            return;
        }
        if (!dir.mkdirs()) {
            throw new IOException("Could not create directory " + dir);
        }
    }

    public static boolean isEmptyDirectory(File dir) throws IOException {
        return FileSystemUtils.isEmptyDirectory(dir.toPath());
    }

    public static boolean isEmptyDirectory(Path dir) throws IOException {
        if (Files.isDirectory(dir, new LinkOption[0])) {
            try (DirectoryStream<Path> directoryStream = Files.newDirectoryStream(dir);){
                boolean bl = !directoryStream.iterator().hasNext();
                return bl;
            }
        }
        return false;
    }

    public static void ensureDirectoryExists(File directory) throws IOException {
        if (!directory.exists() && !directory.mkdirs()) {
            throw new IOException("Couldn't create directory: " + directory);
        }
    }

    public static void ensureParentDirectoryExists(File file) throws IOException {
        FileSystemUtils.ensureDirectoryExists(file.getCanonicalFile().getParentFile());
    }

    public static List<File> listFilesRecursively(File directory) {
        return FileSystemUtils.listFilesRecursively(directory, null);
    }

    public static List<File> listFilesRecursively(File directory, FileFilter filter) {
        if (directory == null || !directory.isDirectory()) {
            return CollectionUtils.emptyList();
        }
        ArrayList<File> result = new ArrayList<File>();
        FileSystemUtils.listFilesRecursively(directory, result, filter);
        return result;
    }

    public static List<String> listFilesInSameLocationForURL(URL baseUrl) throws IOException {
        return FileSystemUtils.listFilesInSameLocationForURL(baseUrl, false);
    }

    public static List<String> listFilesInSameLocationForURL(URL baseUrl, boolean includeSubfolders) throws IOException {
        String protocol = baseUrl.getProtocol();
        if ("file".equals(protocol)) {
            return FileSystemUtils.listFilesForFileURL(baseUrl, includeSubfolders);
        }
        if ("jar".equals(protocol)) {
            return FileSystemUtils.listFilesForJarURL(baseUrl, includeSubfolders);
        }
        throw new IOException("Unsupported protocol: " + protocol);
    }

    private static String getJarUrlParentDirectoryPrefix(URL baseUrl) {
        String parentPath = StringUtils.getLastPart(baseUrl.getPath(), '!');
        parentPath = (parentPath = StringUtils.stripPrefix(parentPath, "/")).endsWith(".class") ? StringUtils.stripSuffix(parentPath, StringUtils.getLastPart(parentPath, '/')) : StringUtils.ensureEndsWith(parentPath, String.valueOf('/'));
        return parentPath;
    }

    private static List<String> listFilesForJarURL(URL baseUrl, boolean recursive) throws IOException {
        try (JarFile jarFile = new JarFile(FileSystemUtils.extractJarFileFromJarURL(baseUrl));){
            String parentPath = FileSystemUtils.getJarUrlParentDirectoryPrefix(baseUrl);
            List<String> list = jarFile.stream().filter(entry -> FileSystemUtils.shouldBeContainedInResult(entry, parentPath, recursive)).map(entry -> StringUtils.stripPrefix(entry.getName(), parentPath)).collect(Collectors.toList());
            return list;
        }
    }

    private static boolean shouldBeContainedInResult(JarEntry entry, String path, boolean recursive) {
        if (entry.isDirectory()) {
            return false;
        }
        String simpleName = StringUtils.getLastPart(entry.getName(), '/');
        String entryPath = StringUtils.stripSuffix(entry.getName(), simpleName);
        return !recursive && entryPath.equals(path) || recursive && entryPath.startsWith(path);
    }

    private static List<String> listFilesForFileURL(URL baseUrl, boolean includeSubfolders) throws IOException {
        try {
            File directory = new File(baseUrl.toURI());
            if (!directory.isDirectory()) {
                directory = directory.getParentFile();
            }
            if (!directory.isDirectory()) {
                throw new IOException("Parent directory does not exist or is not readable for " + baseUrl);
            }
            if (includeSubfolders) {
                File finalDirectory = directory;
                return CollectionUtils.filterAndMap(FileSystemUtils.listFilesRecursively(directory), File::isFile, file -> {
                    String relativeFilePath = StringUtils.stripPrefix(file.getAbsolutePath(), finalDirectory.getAbsolutePath());
                    relativeFilePath = FileSystemUtils.normalizeSeparators(relativeFilePath);
                    return StringUtils.stripPrefix(relativeFilePath, String.valueOf('/'));
                });
            }
            ArrayList<String> names = new ArrayList<String>();
            for (File file2 : directory.listFiles()) {
                if (!file2.isFile()) continue;
                names.add(file2.getName());
            }
            return names;
        }
        catch (URISyntaxException e) {
            throw new IOException("Could not convert URL to valid file: " + baseUrl, e);
        }
    }

    public static List<String> listTopLevelClassesInJarFile(File jarFile) throws IOException {
        ArrayList<String> result = new ArrayList<String>();
        PathBasedContentProviderBase provider = PathBasedContentProviderBase.createProvider(jarFile);
        Collection<String> paths = provider.getPaths();
        for (String path : paths) {
            if (!path.endsWith(".class") || path.contains("$")) continue;
            String fqn = StringUtils.removeLastPart(path, '.');
            fqn = fqn.replace('/', '.');
            result.add(fqn);
        }
        return result;
    }

    public static String getFileExtension(File file) {
        return FileSystemUtils.getFileExtension(file.getName());
    }

    public static String getFileExtension(String path) {
        int posLastDot = path.lastIndexOf(46);
        if (posLastDot < 0) {
            return null;
        }
        return path.substring(posLastDot + 1);
    }

    public static String getFilenameWithoutExtension(File file) {
        return FileSystemUtils.getFilenameWithoutExtension(file.getName());
    }

    public static String getFilenameWithoutExtension(String fileName) {
        return StringUtils.removeLastPart(fileName, '.');
    }

    public static String getLastPathSegment(String filePath) {
        String[] split = FileSystemUtils.getPathSegments(filePath);
        return split[split.length - 1];
    }

    public static String[] getPathSegments(String filePath) {
        return FileSystemUtils.normalizeSeparators(filePath).split(String.valueOf('/'));
    }

    public static boolean isValidPath(String path) {
        try {
            Paths.get(path, new String[0]);
        }
        catch (InvalidPathException ex) {
            return false;
        }
        return Arrays.stream(path.split(Pattern.quote(File.separator))).noneMatch(pathSegment -> pathSegment.contains(String.valueOf('\\')) || pathSegment.contains(String.valueOf('/')));
    }

    public static boolean isPathWriteable(File file) {
        File folderToCheck = file;
        if (!file.isDirectory()) {
            folderToCheck = file.getAbsoluteFile().getParentFile();
        }
        while (folderToCheck != null && !folderToCheck.isDirectory()) {
            folderToCheck = folderToCheck.getParentFile();
        }
        return folderToCheck != null && folderToCheck.canWrite();
    }

    public static File newFile(File parentFile, String ... pathElements) {
        if (pathElements.length == 0) {
            return parentFile;
        }
        File child = new File(parentFile, pathElements[0]);
        String[] remainingElements = new String[pathElements.length - 1];
        System.arraycopy(pathElements, 1, remainingElements, 0, pathElements.length - 1);
        return FileSystemUtils.newFile(child, remainingElements);
    }

    public static String readFile(File file) throws IOException {
        return FileSystemUtils.readFile(file, Charset.defaultCharset());
    }

    public static String readFileUTF8(File file) throws IOException {
        return FileSystemUtils.readFile(file, StandardCharsets.UTF_8);
    }

    public static String readFileUTF8WithLineBreakNormalization(File file) throws IOException {
        return StringUtils.normalizeLineSeparatorsPlatformSpecific(FileSystemUtils.readFile(file, StandardCharsets.UTF_8));
    }

    public static String readFile(File file, Charset encoding) throws IOException {
        byte[] buffer = FileSystemUtils.readFileBinary(file);
        return StringUtils.bytesToString(buffer, encoding);
    }

    public static List<String> readLines(File file, Charset encoding) throws IOException {
        return StringUtils.splitLinesAsList(FileSystemUtils.readFile(file, encoding));
    }

    public static List<String> readLinesUTF8(String filePath) throws IOException {
        return FileSystemUtils.readLinesUTF8(new File(filePath));
    }

    public static List<String> readLinesUTF8(File file) throws IOException {
        return FileSystemUtils.readLines(file, StandardCharsets.UTF_8);
    }

    public static byte[] readFileBinary(String filePath) throws IOException {
        return FileSystemUtils.readFileBinary(new File(filePath));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static byte[] readFileBinary(File file) throws IOException {
        FileInputStream in = new FileInputStream(file);
        byte[] buffer = new byte[(int)file.length()];
        ByteBuffer byteBuffer = ByteBuffer.wrap(buffer);
        FileChannel channel = in.getChannel();
        try {
            int read;
            for (int readSum = 0; readSum < buffer.length; readSum += read) {
                read = channel.read(byteBuffer);
                if (read >= 0) continue;
                throw new IOException("Reached EOF before entire file could be read!");
            }
        }
        finally {
            FileSystemUtils.close(channel);
            FileSystemUtils.close(in);
        }
        return buffer;
    }

    public static void unjar(File jarFile, File targetDirectory) throws IOException {
        FileSystemUtils.unzip(jarFile, targetDirectory);
    }

    public static void unzip(File zipFile, File targetDirectory) throws IOException {
        FileSystemUtils.unzip(zipFile, targetDirectory, null, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static List<String> unzip(File zipFile, File targetDirectory, String prefix, Charset charset) throws IOException {
        List<String> list;
        ZipFile zip = null;
        try {
            zip = charset == null ? new ZipFile(zipFile) : new ZipFile(zipFile, charset);
            list = FileSystemUtils.unzip(zip, targetDirectory, prefix);
        }
        catch (Throwable throwable) {
            FileSystemUtils.close(zip);
            throw throwable;
        }
        FileSystemUtils.close(zip);
        return list;
    }

    public static List<String> unzip(ZipFile zip, File targetDirectory, String prefix) throws IOException {
        Enumeration entries = zip.getEntries();
        ArrayList<String> extractedPaths = new ArrayList<String>();
        while (entries.hasMoreElements()) {
            ZipArchiveEntry entry = (ZipArchiveEntry)entries.nextElement();
            if (entry.isDirectory()) continue;
            String fileName = entry.getName();
            if (!StringUtils.isEmpty(prefix)) {
                if (!fileName.startsWith(prefix)) continue;
                fileName = StringUtils.stripPrefix(fileName, prefix);
            }
            try (InputStream entryStream = zip.getInputStream(entry);){
                File file = new File(targetDirectory, fileName);
                FileSystemUtils.ensureParentDirectoryExists(file);
                try (FileOutputStream outputStream = new FileOutputStream(file);){
                    FileSystemUtils.copy(entryStream, outputStream);
                }
            }
            extractedPaths.add(fileName);
        }
        return extractedPaths;
    }

    public static List<String> unzip(InputStream inputStream, File targetDirectory) throws IOException {
        ArrayList<String> extractedPaths = new ArrayList<String>();
        try (ZipInputStream zipStream = new ZipInputStream(inputStream);){
            ZipEntry entry;
            while ((entry = zipStream.getNextEntry()) != null) {
                if (entry.isDirectory()) continue;
                String fileName = entry.getName();
                File file = new File(targetDirectory, fileName);
                FileSystemUtils.ensureParentDirectoryExists(file);
                try (FileOutputStream targetStream = new FileOutputStream(file);){
                    FileSystemUtils.copy(zipStream, targetStream);
                }
                extractedPaths.add(fileName);
            }
        }
        return extractedPaths;
    }

    public static void writeFile(File file, String content) throws IOException {
        FileSystemUtils.writeFile(file, content, Charset.defaultCharset().name());
    }

    public static void writeLines(File file, Collection<String> lines) throws IOException {
        FileSystemUtils.writeFile(file, StringUtils.concat(lines, "\n"));
    }

    public static void writeFileUTF8(File file, String content) throws IOException {
        FileSystemUtils.writeFile(file, content, UTF8_ENCODING);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void writeFile(File file, String content, String encoding) throws IOException {
        FileSystemUtils.ensureParentDirectoryExists(file);
        OutputStreamWriter writer = null;
        try {
            writer = new OutputStreamWriter((OutputStream)new FileOutputStream(file), encoding);
            writer.write(content);
        }
        catch (Throwable throwable) {
            FileSystemUtils.close(writer);
            throw throwable;
        }
        FileSystemUtils.close(writer);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void writeFileWithBOM(File file, String content, EByteOrderMark bom) throws IOException {
        FileSystemUtils.ensureParentDirectoryExists(file);
        FileOutputStream out = null;
        OutputStreamWriter writer = null;
        try {
            out = new FileOutputStream(file);
            out.write(bom.getBOM());
            writer = new OutputStreamWriter((OutputStream)out, bom.getEncoding());
            writer.write(content);
            writer.flush();
        }
        catch (Throwable throwable) {
            FileSystemUtils.close(out);
            FileSystemUtils.close(writer);
            throw throwable;
        }
        FileSystemUtils.close(out);
        FileSystemUtils.close(writer);
    }

    public static void writeFileBinary(File file, byte[] bytes) throws IOException {
        FileSystemUtils.ensureParentDirectoryExists(file);
        FileOutputStream out = null;
        try {
            out = new FileOutputStream(file);
            out.write(bytes);
        }
        catch (Throwable throwable) {
            FileSystemUtils.close(out);
            throw throwable;
        }
        FileSystemUtils.close(out);
    }

    private static void listFilesRecursively(File directory, Collection<File> result, FileFilter filter) {
        for (File file : directory.listFiles()) {
            if (file.isDirectory()) {
                FileSystemUtils.listFilesRecursively(file, result, filter);
            }
            if (filter != null && !filter.accept(file)) continue;
            result.add(file);
        }
    }

    public static void mergeTemplate(File templateFile, File outFile, Object ... arguments) throws IOException {
        String output;
        String template = FileSystemUtils.readFile(templateFile);
        try {
            output = String.format(template, arguments);
        }
        catch (IllegalFormatException e) {
            throw new IOException("Illegal format: " + e.getMessage(), e);
        }
        FileSystemUtils.writeFile(outFile, output);
    }

    public static InputStream mergeTemplate(InputStream inStream, Object ... arguments) throws IOException {
        String output;
        String template = FileSystemUtils.readStream(inStream);
        try {
            output = String.format(template, arguments);
        }
        catch (IllegalFormatException e) {
            throw new IOException("Illegal format: " + e.getMessage(), e);
        }
        return new ByteArrayInputStream(output.getBytes());
    }

    public static String readStream(InputStream input) throws IOException {
        return FileSystemUtils.readStream(input, Charset.defaultCharset());
    }

    public static String readStreamUTF8(InputStream input) throws IOException {
        return FileSystemUtils.readStream(input, StandardCharsets.UTF_8);
    }

    public static String readStream(InputStream input, Charset encoding) throws IOException {
        int n;
        StringBuilder out = new StringBuilder();
        Reader r = FileSystemUtils.streamReader(input, encoding);
        char[] b = new char[4096];
        while ((n = r.read(b)) != -1) {
            out.append(b, 0, n);
        }
        return out.toString();
    }

    public static byte[] readStreamBinary(InputStream input) throws IOException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        FileSystemUtils.copy(input, out);
        return out.toByteArray();
    }

    public static Reader streamReader(InputStream in, Charset encoding) throws IOException {
        if (!in.markSupported()) {
            in = new BufferedInputStream(in);
        }
        in.mark(4);
        byte[] prefix = new byte[4];
        EByteOrderMark bom = null;
        try {
            FileSystemUtils.safeRead(in, prefix);
            bom = EByteOrderMark.determineBOM(prefix).orElse(null);
        }
        catch (IOException iOException) {
            // empty catch block
        }
        in.reset();
        if (bom != null) {
            encoding = bom.getEncoding();
            for (int i = 0; i < bom.getBOMLength(); ++i) {
                in.read();
            }
        }
        return new InputStreamReader(in, encoding);
    }

    public static Properties readProperties(File propertiesFile) throws IOException {
        return FileSystemUtils.readProperties(() -> new FileInputStream(propertiesFile));
    }

    public static Properties readProperties(Resource resource) throws IOException {
        return FileSystemUtils.readProperties(resource::getAsStream);
    }

    public static Properties readProperties(CollectionUtils.SupplierWithException<? extends InputStream, IOException> streamSupplier) throws IOException {
        try (InputStream stream = streamSupplier.get();){
            Properties props = new Properties();
            props.load(stream);
            Properties properties = props;
            return properties;
        }
    }

    public static File commonRoot(Iterable<? extends File> files) throws IOException {
        HashSet<String> absolutePaths = new HashSet<String>();
        for (File file : files) {
            absolutePaths.add(file.getCanonicalPath());
        }
        CCSMAssert.isTrue(absolutePaths.size() >= 2, "Expected are at least 2 files");
        String longestCommonPrefix = StringUtils.longestCommonPrefix(absolutePaths);
        int n = longestCommonPrefix.lastIndexOf(File.separator);
        if (n > -1) {
            longestCommonPrefix = longestCommonPrefix.substring(0, n);
        }
        if (StringUtils.isEmpty(longestCommonPrefix)) {
            return null;
        }
        return new File(longestCommonPrefix);
    }

    public static InputStream autoDecompressStream(InputStream in) throws IOException {
        if (!in.markSupported()) {
            in = new BufferedInputStream(in);
        }
        in.mark(2);
        boolean isGZIP = (in.read() & 0xFF | (in.read() & 0xFF) << 8) == 35615;
        in.reset();
        if (isGZIP) {
            return new GZIPInputStream(in);
        }
        return in;
    }

    public static void close(ZipFile zipFile) {
        if (zipFile == null) {
            return;
        }
        try {
            zipFile.close();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public static void close(Closeable closeable) {
        if (closeable == null) {
            return;
        }
        try {
            closeable.close();
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public static void sort(List<File> files) {
        files.sort(new FilenameComparator());
    }

    public static String normalizeSeparators(String path) {
        return path.replace(File.separatorChar, '/');
    }

    public static String normalizeSeparatorsPlatformIndependently(String path) {
        if (path.contains(String.valueOf('\\')) && !path.contains(String.valueOf('/'))) {
            return path.replace('\\', '/');
        }
        return path;
    }

    public static File extractJarFileFromJarURL(URL url) {
        CCSMAssert.isTrue("jar".equals(url.getProtocol()), "May only be used with 'jar' URLs!");
        String path = url.getPath();
        CCSMAssert.isTrue(path.startsWith("file:"), "May only be used for URLs pointing to files");
        int index = path.indexOf(33);
        CCSMAssert.isTrue(index >= 0, "Unknown format for jar URLs");
        path = path.substring(0, index);
        return FileSystemUtils.fromURL(path);
    }

    private static File fromURL(String url) {
        url = url.replace(" ", "%20");
        try {
            return new File(new URI(url));
        }
        catch (URISyntaxException e) {
            throw new AssertionError("The assumption is that this method is capable of working with non-standard-compliant URLs, too. Apparently it is not. Invalid URL: " + url + ". Ex: " + e.getMessage(), e);
        }
    }

    public static boolean isAbsolutePath(String filename) {
        if (filename.startsWith("/") || filename.startsWith("~")) {
            return true;
        }
        if (filename.length() > 2 && Character.isLetter(filename.charAt(0)) && filename.charAt(1) == ':') {
            return true;
        }
        return filename.startsWith("\\\\");
    }

    public static void safeRead(InputStream in, byte[] data) throws IOException, EOFException {
        FileSystemUtils.safeRead(in, data, 0, data.length);
    }

    public static void safeRead(InputStream in, byte[] data, int offset, int length) throws IOException, EOFException {
        while (length > 0) {
            int read = in.read(data, offset, length);
            if (read < 0) {
                throw new EOFException("Reached end of file before completing read.");
            }
            offset += read;
            length -= read;
        }
    }

    public static File getTmpDir() {
        return new File(TEMP_DIR_PATH);
    }

    public static File getUserHomeDir() {
        return new File(System.getProperty("user.home"));
    }

    public static File getJvmWorkingDirOrTempForDevMode() {
        if (Boolean.getBoolean("com.teamscale.dev-mode") || FileSystemUtils.isJUnitTest()) {
            return FileSystemUtils.getTmpDir();
        }
        return new File(System.getProperty("user.dir"));
    }

    private static boolean isJUnitTest() {
        StackTraceElement[] stackTrace;
        for (StackTraceElement element : stackTrace = Thread.currentThread().getStackTrace()) {
            if (!element.getClassName().startsWith("org.junit.")) continue;
            return true;
        }
        return false;
    }

    public static boolean contentsEqual(File file1, File file2) throws IOException {
        byte[] content1 = FileSystemUtils.readFileBinary(file1);
        byte[] content2 = FileSystemUtils.readFileBinary(file2);
        return Arrays.equals(content1, content2);
    }

    public static InputStream openJarFileEntry(JarFile jarFile, String entryName) throws IOException {
        JarEntry entry = jarFile.getJarEntry(entryName);
        if (entry == null) {
            throw new IOException("No entry '" + entryName + "' found in JAR file '" + jarFile + "'");
        }
        return jarFile.getInputStream(entry);
    }

    public static boolean isReadableFile(File ... files) {
        for (File file : files) {
            if (file != null && file.exists() && file.isFile() && file.canRead()) continue;
            return false;
        }
        return true;
    }

    public static String concatenatePaths(String firstParent, String ... paths) {
        return FileSystemUtils.normalizeSeparators(Paths.get(firstParent, paths).toString());
    }

    public static void recursivelyRemoveDirectoryIfEmpty(File path) throws IOException {
        String[] children = path.list();
        if (children == null) {
            if (path.exists()) {
                return;
            }
        } else if (children.length == 0) {
            FileSystemUtils.deleteFile(path);
        } else {
            return;
        }
        FileSystemUtils.recursivelyRemoveDirectoryIfEmpty(path.getParentFile());
    }

    public static File getNonExistingFile(File file) {
        if (!file.exists()) {
            return file;
        }
        String extensionlessName = FileSystemUtils.getFilenameWithoutExtension(file);
        int suffix = 0;
        String extension = FileSystemUtils.getFileExtension(file);
        while ((file = new File(file.getParentFile(), extensionlessName + "_" + ++suffix + "." + extension)).exists()) {
        }
        return file;
    }

    public static long parseDataSize(String dataSize) {
        String dataSizeWithoutComma = dataSize.replaceAll(",", "");
        int unitBeginIndex = StringUtils.indexOfMatch(dataSizeWithoutComma, DATA_SIZE_UNIT_START_PATTERN);
        if (unitBeginIndex == -1) {
            return Long.parseLong(dataSizeWithoutComma);
        }
        double rawDataSize = Double.parseDouble(dataSizeWithoutComma.substring(0, unitBeginIndex));
        String unitString = dataSizeWithoutComma.substring(unitBeginIndex);
        char unitChar = unitString.charAt(0);
        int power = METRIC_SYSTEM_UNITS.indexOf(unitChar) + 1;
        boolean isSi = unitBeginIndex != -1 && unitString.length() >= 2 && unitString.charAt(1) == 'i';
        int factor = 1024;
        if (isSi) {
            factor = 1000;
            if (StringUtils.stripSuffix(unitString, "B").length() != 2) {
                throw new NumberFormatException("Malformed data size: " + dataSizeWithoutComma);
            }
        } else if (power == 0 ? StringUtils.stripSuffix(unitString, "B").length() != 0 : StringUtils.stripSuffix(unitString, "B").length() != 1) {
            throw new NumberFormatException("Malformed data size: " + dataSizeWithoutComma);
        }
        return (long)(rawDataSize * Math.pow(factor, power));
    }

    public static long getLastModifiedTimestamp(File file) throws IOException {
        return Files.getLastModifiedTime(Paths.get(file.toURI()), new LinkOption[0]).toMillis();
    }

    public static String toSafeFilename(String name) {
        name = name.replaceAll("\\W+", "-");
        name = name.replaceAll("[-_]+", "-");
        return name;
    }

    public static String toValidFileName(String name) {
        return name.replaceAll("[:\\\\/*\"?|<>']", "-");
    }

    public static File escapeReservedFileNames(File file) {
        Object[] parts = file.getPath().split(Pattern.quote(File.separator));
        for (int i = 0; i < parts.length; ++i) {
            if (!RESERVED_PATH_SEGMENT_NAMES.contains(parts[i])) continue;
            parts[i] = "_" + (String)parts[i];
        }
        return new File(StringUtils.concat(parts, File.separator));
    }

    public static String readFileSystemIndependent(File file) throws IOException {
        return StringUtils.normalizeLineSeparatorsPlatformIndependent(FileSystemUtils.readFileUTF8(file));
    }

    public static String replaceFilePathFilenameWith(String uniformPath, String newFileName) {
        int folderSepIndex = uniformPath.lastIndexOf(47);
        if (uniformPath.endsWith("/")) {
            return uniformPath + newFileName;
        }
        if (folderSepIndex == -1) {
            return newFileName;
        }
        return uniformPath.substring(0, folderSepIndex) + "/" + newFileName;
    }
}

