package com.tencent.imsdk.common;

import android.text.TextUtils;
import android.util.Log;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Authenticator;
import java.net.HttpURLConnection;
import java.net.InetSocketAddress;
import java.net.PasswordAuthentication;
import java.net.Proxy;
import java.net.URL;
import java.net.UnknownHostException;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSession;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Executor;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

public class HttpClient {
    private static final String TAG = HttpClient.class.getSimpleName();

    public static final int HTTP_ACTION_REQUEST = 0;
    public static final int HTTP_ACTION_RESPONSE = 1;

    //获取当前设备的CPU数
    private static final int CPU_COUNT = Runtime.getRuntime().availableProcessors();
    //核心池大小设为CPU数加1
    private static final int CORE_POOL_SIZE = CPU_COUNT + 1;
    //设置线程池的最大大小
    private static final int MAX_POOL_SIZE = 2 * CPU_COUNT + 1;
    //存活时间
    private static final long KEEP_ALIVE = 5L;
    //Http 代理
    private static final int PROXY_TYPE_HTTP = 1;
    //Socks5 代理
    private static final int PROXY_TYPE_SOCKS5 = 2;
    // 切换 https 为 http 的配置
    private static String mRollbackHttps2Http = "";
    // 是否切换 https 为 http
    private static boolean mNeedRollbackHttps2Http = false;

    //创建线程池对象
    private static final Executor mThreadPoolExecutor = new ThreadPoolExecutor(CORE_POOL_SIZE, MAX_POOL_SIZE, KEEP_ALIVE, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>());

    private static native void nativeProgressCallback(int action, int currentSize, int totalSize, long callback);

    private static native void nativeResponseCallback(int code, String[] headerKeys, String[] headerValues, byte[] response, long responseCallback);

    public interface HttpRequestListener {
        void onProgress(final int action, final int currentSize, final int totalSize);

        void onCompleted(final int code, final Map<String, String> headers, final byte[] response);
    }

    // 代理的密码验证类
    static class BasicAuthenticator extends Authenticator {
        private String userName = "";
        private String password = "";

        public BasicAuthenticator(String userName, String password) {
            this.userName = userName;
            this.password = password;
        }

        @Override
        protected PasswordAuthentication getPasswordAuthentication() {
            return new PasswordAuthentication(userName, password.toCharArray());
        }
    }

    private static void httpRequest(final String method, final String url, final Boolean use_origin_ip,
                                    final Map<String, String> headers,
                                    final byte[] content, final String uploadFile,
                                    final String downloadFile,
                                    final HttpRequestListener listener,
                                    final int proxyType,
                                    final String proxyHost, final int proxyPort,
                                    final String proxyUserName, final String proxyPassword,
                                    final int connectTimeout, final int recvTimeout,
                                    final String rollbackHttps2Http) {
        //创建一个新的请求任务
        Runnable requestRunnable = new Runnable() {
            @Override
            public void run() {
                HttpURLConnection conn = null;
                InputStream responseInputStream = null;
                int code = HttpURLConnection.HTTP_OK;
                Map<String, String> rspHeaders = null;
                byte[] response = null;
                String requestUrl = url;

                if (url.startsWith("https")) {
                    if (!mRollbackHttps2Http.equals(rollbackHttps2Http)) {
                        mRollbackHttps2Http = rollbackHttps2Http;
                        mNeedRollbackHttps2Http = needRollbackHttps2Http(rollbackHttps2Http);
                    }
                    
                    if (mNeedRollbackHttps2Http) {
                        requestUrl = url.replaceFirst("https", "http");
                    }
                }

                try {
                    if (PROXY_TYPE_HTTP == proxyType && !proxyHost.isEmpty() && proxyPort != 0) {
                        Proxy proxy = new Proxy(Proxy.Type.HTTP, new InetSocketAddress(proxyHost, proxyPort));
                        conn = (HttpURLConnection) (new URL(requestUrl).openConnection(proxy));
                        if (!proxyUserName.isEmpty() && !proxyPassword.isEmpty()) {
                            Authenticator.setDefault(new BasicAuthenticator(proxyUserName, proxyPassword));
                        }
                    } else if (PROXY_TYPE_SOCKS5 == proxyType && !proxyHost.isEmpty() && proxyPort != 0) {
                        Proxy proxy = new Proxy(Proxy.Type.SOCKS, new InetSocketAddress(proxyHost, proxyPort));
                        conn = (HttpURLConnection) (new URL(requestUrl).openConnection(proxy));
                        if (!proxyUserName.isEmpty() && !proxyPassword.isEmpty()) {
                            Authenticator.setDefault(new BasicAuthenticator(proxyUserName, proxyPassword));
                        }
                    } else {
                        conn = (HttpURLConnection) (new URL(requestUrl).openConnection());
                    }

                    conn.setRequestMethod(method);
                    conn.setConnectTimeout(connectTimeout);
                    conn.setReadTimeout(recvTimeout);
                    conn.setUseCaches(false);
                    conn.setDoInput(true);

                    if (headers != null) {
                        for (Map.Entry<String, String> entry : headers.entrySet()) {
                            conn.setRequestProperty(entry.getKey(), entry.getValue());
                        }
                    }

                    if (use_origin_ip && conn instanceof HttpsURLConnection) {
                        final HttpsURLConnection httpsConn = (HttpsURLConnection) conn;
                        httpsConn.setHostnameVerifier(new HostnameVerifier() {
                            @Override
                            public boolean verify(String hostname, SSLSession session) {
                                String host = httpsConn.getRequestProperty("Host");
                                return HttpsURLConnection.getDefaultHostnameVerifier().verify(host, session);
                            }
                        });
                    }

                    //写入请求数据
                    boolean hasContent = content != null && content.length > 0;
                    boolean hasUploadFile = uploadFile != null && uploadFile.length() != 0;
                    boolean hasOutput = hasContent || hasUploadFile;
                    if (!method.equalsIgnoreCase("GET") && hasOutput) {
                        conn.setDoOutput(true);
                        InputStream inputStream = null;
                        if (hasUploadFile) {
                            inputStream = new FileInputStream(uploadFile);
                        } else {
                            inputStream = new ByteArrayInputStream(content);
                        }
                        if (inputStream != null) {
                            int totalSize = inputStream.available(); //TODO alderzhang：此处大于2G的文件会出现问题
                            conn.setFixedLengthStreamingMode(totalSize);
                            int bytesTransferred = 0;
                            byte[] buf = new byte[4096];
                            while (true) {
                                int len = inputStream.read(buf);
                                if (len < 0) break;
                                bytesTransferred += len;
                                conn.getOutputStream().write(buf, 0, len);
                                if (listener != null) {
                                    listener.onProgress(HTTP_ACTION_REQUEST, bytesTransferred, totalSize);
                                }
                            }
                            inputStream.close();
                        }
                    }

                    code = conn.getResponseCode();

                    int headerCount = conn.getHeaderFields().size();
                    if (headerCount > 0) {
                        rspHeaders = new HashMap<String, String>();
                        for (int i = 0; i < headerCount; ++i) {
                            rspHeaders.put(conn.getHeaderFieldKey(i), conn.getHeaderField(i));
                        }
                    }

                    int totalSize = conn.getContentLength();
                    if (totalSize == -1) {
                        totalSize = 0;
                    }

                    //读取返回数据
                    boolean hasDownloadFile = downloadFile != null && downloadFile.length() != 0;
                    if (code == HttpURLConnection.HTTP_OK) {
                        responseInputStream = new BufferedInputStream(conn.getInputStream());
                        OutputStream outputStream = null;
                        if (hasDownloadFile) {
                            outputStream = new FileOutputStream(downloadFile);
                        } else {
                            outputStream = new ByteArrayOutputStream();
                        }
                        if (outputStream != null) {
                            int bytesReceived = 0;
                            byte[] buf = new byte[4096];
                            while (true) {
                                int len = responseInputStream.read(buf);
                                if (len < 0) break;
                                bytesReceived += len;
                                outputStream.write(buf, 0, len);
                                if (listener != null) {
                                    listener.onProgress(HTTP_ACTION_RESPONSE, bytesReceived, totalSize);
                                }
                            }
                            if (hasDownloadFile) {
                                response = new byte[]{};
                            } else {
                                response = ((ByteArrayOutputStream) outputStream).toByteArray();
                            }
                            outputStream.close();
                        }
                    }

                } catch (UnknownHostException e) {
                    IMLog.e(TAG, "http request failed." + e.getLocalizedMessage());
                    code = HttpURLConnection.HTTP_NOT_FOUND;
                } catch (Exception e) {
                    IMLog.e(TAG, "http request failed." + e.getLocalizedMessage());
                    code = -1;
                    response = Log.getStackTraceString(e).getBytes();
                } finally {
                    //清理资源
                    if (responseInputStream != null) {
                        try {
                            responseInputStream.close();
                        } catch (IOException e) {
                            e.printStackTrace();
                        }
                    }

                    if (conn != null) {
                        conn.disconnect();
                    }

                    if (listener != null) {
                        listener.onCompleted(code, rspHeaders, response);
                    }
                }
            }
        };
        mThreadPoolExecutor.execute(requestRunnable);
    }

    private static void httpRequest(final String method, final String url, final boolean use_origin_ip,
                                    final String[] headerKeys, final String[] headerValues,
                                    final byte[] content, final String uploadFile,
                                    final String downloadFile,
                                    final long progressCallback, final long responseCallback,
                                    final int proxyType,
                                    final String proxyHost, final int proxyPort,
                                    final String proxyUserName, final String proxyPassword,
                                    final int connectTimeout, final int recvTimeout,
                                    final String rollbackHttps2Http) {
        Map<String, String> headers = null;
        if (headerKeys != null && headerValues != null && headerKeys.length == headerValues.length) {
            headers = new HashMap<String, String>();
            for (int i = 0; i < headerKeys.length; ++i) {
                headers.put(headerKeys[i], headerValues[i]);
            }
        }
        httpRequest(method, url, use_origin_ip, headers, content, uploadFile, downloadFile, new HttpRequestListener() {
            @Override
            public void onProgress(int action, int currentSize, int totalSize) {
                if (progressCallback != 0) {
                    nativeProgressCallback(action, currentSize, totalSize, progressCallback);
                }
            }

            @Override
            public void onCompleted(int code, Map<String, String> headers, byte[] response) {
                if (responseCallback != 0) {
                    String[] rspHeaderKeys = null;
                    String[] rspHeaderValues = null;
                    if (headers != null) {
                        rspHeaderKeys = new String[headers.size()];
                        rspHeaderValues = new String[headers.size()];
                        int i = 0;
                        for (Map.Entry<String, String> entry : headers.entrySet()) {
                            rspHeaderKeys[i] = entry.getKey();
                            rspHeaderValues[i] = entry.getValue();
                            i++;
                        }
                    }
                    nativeResponseCallback(code, rspHeaderKeys, rspHeaderValues, response, responseCallback);
                }
            }
        }, proxyType, proxyHost, proxyPort, proxyUserName, proxyPassword, connectTimeout, recvTimeout, rollbackHttps2Http);
    }

    private static boolean needRollbackHttps2Http(String rollbackHttps2Http) {
        boolean result = false;

        if (TextUtils.isEmpty(rollbackHttps2Http)) {
            return result;
        }

        try {
            JSONArray rollbackHttps2HttpJsonArray = new JSONArray(rollbackHttps2Http);
            // 厂商名均未小写，与云控保持一致
            String brand = "";
            if (SystemUtil.isBrandOppo()) {
                brand = "oppo";
            } else if (SystemUtil.isBrandVivo()) {
                brand = "vivo";
            } else if (SystemUtil.isBrandHuawei()) {
                brand = "huawei";
            } else if (SystemUtil.isBrandXiaoMi()) {
                brand = "xiaomi";
            } else if (SystemUtil.isBrandMeizu()) {
                brand = "meizu";
            }

            for (int i = 0; i < rollbackHttps2HttpJsonArray.length(); i++) {
                JSONObject brandObject = rollbackHttps2HttpJsonArray.getJSONObject(i);
                String brandConfig = brandObject.getString("brand");
                int below_version = brandObject.getInt("below_version");
                if (brand.equals(brandConfig)) {
                    int androidVersion = SystemUtil.getSDKVersion();
                    if (androidVersion <= below_version) {
                        result = true;
                    }

                    break;
                }
            }
        } catch (JSONException e) {
            e.printStackTrace();
        }

        return result;
    }
}
