package com.tencent.imsdk.common;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.net.ConnectivityManager;
import android.net.LinkAddress;
import android.net.LinkProperties;
import android.net.Network;
import android.net.NetworkCapabilities;
import android.net.NetworkInfo;
import android.net.wifi.WifiManager;
import android.os.Build;
import android.os.SystemClock;
import android.telephony.TelephonyManager;
import android.util.Log;

import com.tencent.imsdk.BuildConfig;

import java.lang.ref.WeakReference;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.util.List;

public class NetworkInfoCenter extends BroadcastReceiver {
    private static final String TAG = NetworkInfoCenter.class.getSimpleName();

    public interface INetworkChangeListener {
        void onNetworkChange(boolean connected, int networkType, int ipType, String networkID,
                             long wifiNetworkHandle, long xgNetworkHandle, long initializeCostTime);
    }

    // 网络类型
    // 未知
    private static final int NETWORK_UNKNOWN = 0;
    // WIFI
    private static final int NETWORK_WIFI = 1;
    // 2G
    private static final int NETWORK_2G3G = 2;  // 较早的ios版本会上报
    private static final int NETWORK_GPRS = 101;
    private static final int NETWORK_EDGE = 102;
    private static final int NETWORK_CDMA = 104;
    private static final int NETWORK_1xRTT = 107;
    private static final int NETWORK_IDEN = 111;
    // 3G
    private static final int NETWORK_CDMA1X = 98;
    private static final int NETWORK_WCDMA = 99;
    private static final int NETWORK_UMTS = 103;
    private static final int NETWORK_EVDO_0 = 105;
    private static final int NETWORK_EVDO_A = 106;
    private static final int NETWORK_HSDPA = 108;
    private static final int NETWORK_HSUPA = 109;
    private static final int NETWORK_HSPA = 110;
    private static final int NETWORK_EVDO_B = 112;
    private static final int NETWORK_EHRPD = 114;
    private static final int NETWORK_HSPAP = 115;
    // 4G
    private static final int NETWORK_LTE = 113;

    // IPv4/v6
    final int IP_TYPE_UNKNOWN = 0x0;
    final int IP_TYPE_IPV4_ONLY = 0x1;
    final int IP_TYPE_IPV6_ONLY = 0x2;
    final int IP_TYPE_IPV6_DUAL = 0x3;

    private Context mContext;

    private WeakReference<INetworkChangeListener> mListener;

    private ConnectivityManager mConnectivityManager;
    private WifiManager mWifiManager;
    private TelephonyManager mTelephonyManager;

    private boolean mNetworkConnected = false;
    private int mNetworkType = NETWORK_UNKNOWN;
    private int mIPType = IP_TYPE_UNKNOWN;
    private String mNetworkSSID = "";
    private String mServiceProvider = "";
    private long mInitializeCostTime = 0;

    private boolean isRegisterReceiver = false;

    private boolean mCheckNetworkInterface = false;
    private long mWifiNetworkHandle = 0;
    private long mXgNetworkHandle = 0;
    
    private static class Holder {
        static NetworkInfoCenter instance = new NetworkInfoCenter();
    }

    public static NetworkInfoCenter getInstance() {
        return Holder.instance;
    }

    public void init(Context context, INetworkChangeListener listener) {
        long begin = SystemClock.uptimeMillis();

        mContext = context.getApplicationContext();
        mListener = new WeakReference<>(listener);

        IntentFilter intentFilter = new IntentFilter();
        intentFilter.addAction(ConnectivityManager.CONNECTIVITY_ACTION);
        intentFilter.addAction(WifiManager.NETWORK_STATE_CHANGED_ACTION);
        intentFilter.addAction(WifiManager.WIFI_STATE_CHANGED_ACTION);
        if (mContext != null) {
            mContext.getApplicationContext().registerReceiver(this, intentFilter);
            isRegisterReceiver = true;
        }

        if ("pro".equals(BuildConfig.SDK_FLAVOR)) {
            mCheckNetworkInterface = true;
        }

        mConnectivityManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        if (mConnectivityManager != null) {
            NetworkInfo activeNetworkInfo = null;
            try {
                activeNetworkInfo = mConnectivityManager.getActiveNetworkInfo();
            } catch (Exception e) {
                e.printStackTrace();
            }

            updateNetworkType(activeNetworkInfo);
        }

        updateNetworkSSID();
        updateProviderName();
        updateIPType();
        updateNetworkInterface();

        mInitializeCostTime = SystemClock.uptimeMillis() - begin;
    }

    public void uninit() {
        if (isRegisterReceiver && mContext != null) {
            mContext.getApplicationContext().unregisterReceiver(this);
            isRegisterReceiver = false;
        }
    }

    @Override
    public void onReceive(Context context, Intent intent) {
        if (intent == null) {
            Log.e(TAG, "receive broadcast intent == null");
            return;
        }

        String action = intent.getAction();
        if (action == null) {
            Log.e(TAG, "receive broadcast intent.getAction == null");
            return;
        }

        if (action.equals(ConnectivityManager.CONNECTIVITY_ACTION) ||
            action.equals(WifiManager.NETWORK_STATE_CHANGED_ACTION) ||
            action.equals(WifiManager.WIFI_STATE_CHANGED_ACTION)) {
            onNetworkChanged(context, intent);
        }
    }

    private void onNetworkChanged(Context context, Intent intent) {
        Log.i(TAG, "network changed, action: " + intent.getAction());

        if (ConnectivityManager.CONNECTIVITY_ACTION.equals(intent.getAction())) {
            if (mConnectivityManager == null) {
                mConnectivityManager = (ConnectivityManager) mContext.getSystemService(Context.CONNECTIVITY_SERVICE);
            }
            if (mConnectivityManager == null) {
                Log.e(TAG, "network changed, mConnectivityManager is null");
                return;
            }

            long begin = SystemClock.uptimeMillis();

            NetworkInfo activeNetworkInfo = null;
            try {
                activeNetworkInfo = mConnectivityManager.getActiveNetworkInfo();
            } catch (Exception e) {
                e.printStackTrace();
            }
            if (activeNetworkInfo != null) {
                if (activeNetworkInfo.isConnected()) {
                    mNetworkConnected = true;
                } else {
                    mNetworkConnected = false;
                }

                updateNetworkType(activeNetworkInfo);
                updateNetworkSSID();
                updateProviderName();
                updateNetworkInterface();

                Log.i(TAG, "mNetworkConnected: " + mNetworkConnected + ", current network: " + activeNetworkInfo);
            } else {
                mNetworkConnected = false;
                mNetworkType = NETWORK_UNKNOWN;

                Log.e(TAG, "no network connection found");
            }

            updateIPType();

            mInitializeCostTime = SystemClock.uptimeMillis() - begin;

            INetworkChangeListener listener = mListener.get();
            if (listener != null) {
                listener.onNetworkChange(mNetworkConnected, getNetworkType(), getIPType(),
                        getNetworkID(), getWifiNetworkHandle(), getXgNetworkHandle(),
                        getInitializeCostTime());
            }
        }
    }

    private void updateNetworkType(NetworkInfo activeNetInfo) {
        // 获取网络类型，如果为空，返回无网络
        if (activeNetInfo == null || !activeNetInfo.isAvailable()) {
            mNetworkType = NETWORK_UNKNOWN;
            return;
        }

        if (mConnectivityManager == null) {
            mConnectivityManager = (ConnectivityManager) mContext.getSystemService(Context.CONNECTIVITY_SERVICE);
        }

        if (mConnectivityManager == null) {
            mNetworkType = NETWORK_UNKNOWN;
            return;
        }

        // 判断是否为WIFI
        NetworkInfo wifiInfo = mConnectivityManager.getNetworkInfo(ConnectivityManager.TYPE_WIFI);
        if (wifiInfo != null) {
            NetworkInfo.State state = wifiInfo.getState();
            if (null != state) {
                if (state == NetworkInfo.State.CONNECTED || state == NetworkInfo.State.CONNECTING) {
                    mNetworkType = NETWORK_WIFI;
                    return;
                }
            }
        }

        // 若不是WIFI，则去判断是2G、3G、4G网
        TelephonyManager telephonyManager = (TelephonyManager) mContext.getSystemService(Context.TELEPHONY_SERVICE);
        if (telephonyManager == null) {
            mNetworkType = NETWORK_UNKNOWN;
            return;
        }

        int networkType = TelephonyManager.NETWORK_TYPE_UNKNOWN;
        try {
            networkType = telephonyManager.getNetworkType();
        } catch (Exception e) {
            e.printStackTrace();
            Log.w(TAG, "getNetworkType:" + e.getLocalizedMessage());
        }

        switch (networkType) {
            /*
             GPRS : 2G(2.5) General Packet Radia Service 114kbps
             EDGE : 2G(2.75G) Enhanced Data Rate for GSM Evolution 384kbps
             UMTS : 3G WCDMA 联通3G Universal Mobile Telecommunication System 完整的3G移动通信技术标准
             CDMA : 2G 电信 Code Division Multiple Access 码分多址
             EVDO_0 : 3G (EVDO 全程 CDMA2000 1xEV-DO) Evolution - Data Only (Data Optimized) 153.6kps - 2.4mbps 属于3G
             EVDO_A : 3G 1.8mbps - 3.1mbps 属于3G过渡，3.5G
             1xRTT : 2G CDMA2000 1xRTT (RTT - 无线电传输技术) 144kbps 2G的过渡,
             HSDPA : 3.5G 高速下行分组接入 3.5G WCDMA High Speed Downlink Packet Access 14.4mbps
             HSUPA : 3.5G High Speed Uplink Packet Access 高速上行链路分组接入 1.4 - 5.8 mbps
             HSPA : 3G (分HSDPA,HSUPA) High Speed Packet Access
             IDEN : 2G Integrated Dispatch Enhanced Networks 集成数字增强型网络 （属于2G，来自维基百科）
             EVDO_B : 3G EV-DO Rev.B 14.7Mbps 下行 3.5G
             LTE : 4G Long Term Evolution FDD-LTE 和 TDD-LTE , 3G过渡，升级版 LTE Advanced 才是4G
             EHRPD : 3G CDMA2000向LTE 4G的中间产物 Evolved High Rate Packet Data HRPD的升级
             HSPAP : 3G HSPAP 比 HSDPA 快些
             */

            // 2G网络
            case TelephonyManager.NETWORK_TYPE_GPRS:
                mNetworkType = NETWORK_GPRS;
                break;
            case TelephonyManager.NETWORK_TYPE_CDMA:
                mNetworkType = NETWORK_CDMA;
                break;
            case TelephonyManager.NETWORK_TYPE_EDGE:
                mNetworkType = NETWORK_EDGE;
                break;
            case TelephonyManager.NETWORK_TYPE_1xRTT:
                mNetworkType = NETWORK_1xRTT;
                break;
            case TelephonyManager.NETWORK_TYPE_IDEN:
                mNetworkType = NETWORK_IDEN;
                break;
            // 3G网络
            case TelephonyManager.NETWORK_TYPE_EVDO_A:
                mNetworkType = NETWORK_EVDO_A;
                break;
            case TelephonyManager.NETWORK_TYPE_UMTS:
                mNetworkType = NETWORK_UMTS;
                break;
            case TelephonyManager.NETWORK_TYPE_EVDO_0:
                mNetworkType = NETWORK_EVDO_0;
                break;
            case TelephonyManager.NETWORK_TYPE_HSDPA:
                mNetworkType = NETWORK_HSDPA;
                break;
            case TelephonyManager.NETWORK_TYPE_HSUPA:
                mNetworkType = NETWORK_HSUPA;
                break;
            case TelephonyManager.NETWORK_TYPE_HSPA:
                mNetworkType = NETWORK_HSPA;
                break;
            case TelephonyManager.NETWORK_TYPE_EVDO_B:
                mNetworkType = NETWORK_EVDO_B;
                break;
            case TelephonyManager.NETWORK_TYPE_EHRPD:
                mNetworkType = NETWORK_EHRPD;
                break;
            case TelephonyManager.NETWORK_TYPE_HSPAP:
                mNetworkType = NETWORK_HSPAP;
                break;
            // 4G网络
            case TelephonyManager.NETWORK_TYPE_LTE:
                mNetworkType = NETWORK_LTE;
                break;
            default:
                mNetworkType = NETWORK_UNKNOWN;
                break;
        }
    }

    private void updateIPType() {
        if (SystemUtil.getSDKVersion() < Build.VERSION_CODES.M) {
            return;
        }

        mIPType = IP_TYPE_UNKNOWN;

        if (mConnectivityManager == null) {
            mConnectivityManager = (ConnectivityManager) mContext.getSystemService(Context.CONNECTIVITY_SERVICE);
        }
        if (mConnectivityManager == null) {
            Log.e(TAG, "updateIPType, mConnectivityManager is null");
            return;
        }

        try {
            LinkProperties linkProperties = mConnectivityManager.getLinkProperties(mConnectivityManager.getActiveNetwork());
            if (linkProperties == null) {
                Log.i(TAG, "updateIPType, no linkProperties");
                return;
            }

            List<LinkAddress> linkAddresses = linkProperties.getLinkAddresses();

            for (LinkAddress linkAddress : linkAddresses) {
                InetAddress address = linkAddress.getAddress();
                if (address instanceof Inet4Address && !address.isLinkLocalAddress()) {
                    mIPType = mIPType | IP_TYPE_IPV4_ONLY;
                } else if (address instanceof Inet6Address && !address.isLinkLocalAddress()) {
                    mIPType = mIPType | IP_TYPE_IPV6_ONLY;
                } else {
                    continue;
                }
            }
        } catch (Exception e) {
            Log.w(TAG, "updateIPType, e: " + e.getLocalizedMessage());
            e.printStackTrace();
        }

        Log.i(TAG, "updateIPType|type:" + mIPType);
    }

    private void updateNetworkSSID() {
        // 获取WIFI的SSID
        // 使用WifiManager和WifiInfo来获取当前Wifi连接信息
//        if (mWifiManager == null) {
//            mWifiManager = (WifiManager) mContext.getSystemService(WIFI_SERVICE);
//        }
//        if (mWifiManager != null) {
//            WifiInfo wifiInfo = mWifiManager.getConnectionInfo();
//            if (null != wifiInfo) {
//                String ssid = wifiInfo.getSSID();
//                String bssid = wifiInfo.getBSSID();
//                mNetworkSSID = ssid + "-" + bssid;
//            }
//        }
    }

    private void updateProviderName() {
        // 获取运营商名字
        // getSimOperatorName()就可以直接获取到运营商的名字
        // 也可以使用IMSI获取，getSimOperator()，然后根据返回值判断，例如"46000"为移动
        // IMSI相关链接：http://baike.baidu.com/item/imsi

        // 工信部禁止获取运营商行为，应用上架会被拒绝
//        if (mTelephonyManager == null) {
//            mTelephonyManager = (TelephonyManager) mContext.getSystemService(Context.TELEPHONY_SERVICE);
//        }
//        if (mTelephonyManager != null) {
//            // getSimOperatorName就可以直接获取到运营商的名字
//            mServiceProvider = mTelephonyManager.getSimOperatorName();
//        }
    }

    private void updateNetworkInterface() {
        if (mCheckNetworkInterface == false) {
            return;
        }

        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.LOLLIPOP /*5.0*/) {
            Log.w(TAG, "updateNetworkInterface, support from Android LOLLIPOP");
            return;
        }

        mWifiNetworkHandle = 0;
        mXgNetworkHandle = 0;

        if (mConnectivityManager == null) {
            mConnectivityManager = (ConnectivityManager)mContext.getSystemService(Context.CONNECTIVITY_SERVICE);
        }

        if (mConnectivityManager == null) {
            Log.e(TAG, "updateNetworkInterface, mConnectivityManager is null");
            return;
        }

        Network[] allNetworks = mConnectivityManager.getAllNetworks();
        for (Network network : allNetworks) {
            if (network == null) {
                continue;
            }

            NetworkCapabilities networkCapabilities = mConnectivityManager.getNetworkCapabilities(network);
            if (networkCapabilities == null) {
                continue;
            }

            if (networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI) &&
                    networkCapabilities.hasCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET)) {
                mWifiNetworkHandle = networkToNetId(network);
            } else if (networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR) &&
                    networkCapabilities.hasCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET)) {
                mXgNetworkHandle = networkToNetId(network);
            }
        }

        Log.i(TAG, "updateNetworkInterface, wifiNetworkHandle = " + mWifiNetworkHandle + " xgNetworkHandle = " + mXgNetworkHandle);
    }

    private static long networkToNetId(Network network) {
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            // Android M 版本开始支持，通过 getNetworkHandle API 获取网络标识
            return network.getNetworkHandle();
        } else {
            // Andriod L 版本，没有公开的 Network API 获取网络标识，
            // 部分 Android framework 的实现，将 network 对象转换成字符串，得到字符串形式的网络标识，
            // 再将该字符串转换成 long 类型，得到 long 类型的网络标识
            return Integer.parseInt(network.toString());
        }
    }

    public int getNetworkType() {
        return mNetworkType;
    }

    public String getNetworkSSID() {
        return mNetworkSSID;
    }

    private String getProviderName() {
        return mServiceProvider;
    }

    public int getIPType() {
        return mIPType;
    }

    public String getNetworkID() {
        // WIFI: SSID, WWAN: service provider
        int netType = getNetworkType();
        if (NETWORK_WIFI == netType) {
            return getNetworkSSID();
        } else {
            return getProviderName();
        }
    }

    public long getWifiNetworkHandle() { return mWifiNetworkHandle; }

    public long getXgNetworkHandle() { return mXgNetworkHandle; }

    public long getInitializeCostTime() { return mInitializeCostTime; }

    public boolean isNetworkConnected() {
        if (mConnectivityManager == null) {
            if (mContext == null) {
                return false;
            }
            mConnectivityManager = (ConnectivityManager) mContext.getSystemService(Context.CONNECTIVITY_SERVICE);
        }

        if (mConnectivityManager == null) {
            Log.e(TAG, "isNetworkConnected, mConnectivityManager is null");
            return false;
        }

        NetworkInfo activeNetworkInfo = null;
        try {
            activeNetworkInfo = mConnectivityManager.getActiveNetworkInfo();
        } catch (Exception e) {
            e.printStackTrace();
        }
        if (activeNetworkInfo != null) {
            if (activeNetworkInfo.isConnected()) {
                return true;
            } else {
                return false;
            }
        } else {
            Log.e(TAG, "isNetworkConnected, activeNetwork is null");
            return false;
        }
    }
}
